/***************************************************************************/
/*                                                                         */
/* File: VGAHI.C                                                           */
/*                                                                         */
/* Description: This file contains the VGAHI graphics routines used to     */
/*              display and simulate the Fuzzy controller.                 */
/*                                                                         */
/***************************************************************************/
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <graphics.h>
#include <math.h>
#include <mem.h>

#define N_SET 5
extern float T[];
extern float COOL[];
extern float WARM[];
extern float HOT[];

extern float H[];
extern float DRY[];
extern float MOIST[];
extern float WET[];

extern float FS[];
extern float LOW[];
extern float MED[];
extern float HIGH[];

extern float O[];

extern float GetMembership(float Xo, float X[], float A[]);


/* ggetch()
*
*  ggetch is a function that is used to for echoless keyboard entry
*  including CURSOR and FUNCTION keys. Returns int ch.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1990
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1990
*/

/* defines for special chars */
#define HOME            199
#define END             207
#define PGUP            201
#define PGDN            209
#define LEFT            203
#define RIGHT           205
#define UP              200
#define DOWN            208
#define INS             210
#define DEL             211
#define CTRL_HOME       247
#define CTRL_END        245

#define F1              187
#define F2              188
#define F3              189
#define F4              190
#define F5              191
#define F6              192
#define F7              193
#define F8              194
#define F9              195
#define F10             196

int ggetch(void)
{
  int ch;

  ch = getch();

  if ((ch == 0x00) || (ch == 0xE0)) {
    ch = getch() | 0x80;
  }
  ch &= 0xFF;

  return (ch);
}


/* InitGraphics(void)
*
*  InitGraphics is used to initialize the graphics library.
*  
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void InitGraphics(void)
{
  int GraphDriver, GraphMode, GraphError;

  /* initialize graphics system */
  GraphDriver = VGA;
  GraphMode   = VGAHI;
  initgraph(&GraphDriver, &GraphMode, "");

  /* test result of initialization */
  GraphError = graphresult();
  if(GraphError != grOk) {
   printf("FUZZYFAN graphics system error: %s\n", grapherrormsg(GraphError));
   exit(1);
  }
}


/* QuitGraphics(void)
*
*  QuitGraphics is used to reset the screen to it's mode prior to
*  calling InitGraphics.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
static void QuitGraphics(void)
{
  cleardevice();      /* clear graphics screen  */
  closegraph();
}


/* DrawGraph()
*
*  DrawGraph is used to initialize a graph plot area on the vga screen.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
#define GRAPH_WIDTH   319
#define GRAPH_HEIGHT  150
#define GRAPH_LEFT     32
#define GRAPH_BOTTOM  130

void DrawGraph(int x_pos, int y_pos,
               char *name,  char *str1,  char *str2, char *str3)
{
  struct viewporttype viewport;
  int    si;

  /* save current viewport settings */
  getviewsettings(&viewport);

  /* clear viewport. */
  setviewport(x_pos, y_pos, x_pos +GRAPH_WIDTH, y_pos +GRAPH_HEIGHT, 1);
  clearviewport();

  /* draw border around graph */
//  setcolor(LIGHTRED);
//  rectangle(0, 0, GRAPH_WIDTH, GRAPH_HEIGHT);

  /* display title on screen */
  setcolor(WHITE);
  settextjustify(CENTER_TEXT, CENTER_TEXT);
  outtextxy(GRAPH_WIDTH/2,   10, name);
  outtextxy(60,              20, str1);
  outtextxy(GRAPH_WIDTH/2,   20, str2);
  outtextxy(GRAPH_WIDTH -60, 20, str3);

  /* draw graph axis */
  setcolor(YELLOW);
  line(GRAPH_LEFT,     GRAPH_BOTTOM, GRAPH_LEFT,      GRAPH_BOTTOM -105);
  line(GRAPH_LEFT,     GRAPH_BOTTOM, GRAPH_LEFT +256, GRAPH_BOTTOM);
  line(GRAPH_LEFT+256, GRAPH_BOTTOM, GRAPH_LEFT +256, GRAPH_BOTTOM -105);

  /* display Y-axis values */
  outtextxy(GRAPH_LEFT -17, GRAPH_BOTTOM -100, "1.0");
  outtextxy(GRAPH_LEFT -17, GRAPH_BOTTOM - 50, "0.5");
  outtextxy(GRAPH_LEFT -17, GRAPH_BOTTOM,      "0");

  /* restore current viewport settings */
  setviewport(viewport.left, viewport.top,
              viewport.right, viewport.bottom, viewport.clip);
}


/* PlotGraph()
*
*  PlotGraph is used to plot a fuzzy set A[] in the graph plot area on
*  the vga screen.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
#define W_SET  256
void PlotGraph(int x_pos, int y_pos, float X[], float A[], int color)
{
  struct viewporttype viewport;
  int    si;
  float x, xinc, a[W_SET];

  /* Transform the universe X[] and fuzzy set A[]
   * into a universe x = {0,.. 255} having a fuzzy set a[W_SET]. */
  x     = X[0];
  xinc  =(X[N_SET-1]-X[0])/W_SET;

  for(si = 0; si < W_SET; si++) {
    a[si] = GetMembership(x, X, A);
    x    += xinc;
  }

  /* save current viewport settings */
  getviewsettings(&viewport);

  /* Ok.. draw the plot of a[W_SET] */
  setviewport(x_pos, y_pos, x_pos +GRAPH_WIDTH, y_pos +GRAPH_HEIGHT, 1);

  setwritemode(XOR_PUT);
  setcolor(color);
  for(si = 0; si < W_SET; si++) {
     if (a[si] > 0) {
       line(GRAPH_LEFT+si, GRAPH_BOTTOM -((int )(100*a[si])),
            GRAPH_LEFT+si, GRAPH_BOTTOM -((int )(100*a[si])));
     }
  }
  setwritemode(COPY_PUT);

  /* restore current viewport settings */
  setviewport(viewport.left, viewport.top,
              viewport.right, viewport.bottom, viewport.clip);
}

/* PlotSingleTon()
*
*  PlotSingleTon is used to plot a singleton value Ao onto the fuzzy set A[]
*  which was previously plotted in the graph plot area on the vga screen.
*
*  Note: The PlotSingleTon routine uses an setwritemode(XOR_PUT) to draw
*  graphics to the screen.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void PlotSingleTon(int x_pos, int y_pos, float Xo, float X[])
{
  struct viewporttype viewport;
  int    si;
  float x, xinc;

  /* calculate offset for singleton */
  xinc  = (X[N_SET-1]-X[0])/W_SET;
  x     = (Xo - X[0])/xinc;

  si    = (int )x;

  /* Ok.. draw the singleton */
  /* save current viewport settings */
  getviewsettings(&viewport);

  setviewport(x_pos, y_pos, x_pos +GRAPH_WIDTH, y_pos +GRAPH_HEIGHT, 1);

  setwritemode(XOR_PUT);
  setcolor(WHITE);
  line(GRAPH_LEFT+si, GRAPH_BOTTOM,
       GRAPH_LEFT+si, GRAPH_BOTTOM -((int )(100*1.0)));
  setwritemode(COPY_PUT);

  /* restore current viewport settings */
  setviewport(viewport.left, viewport.top,
              viewport.right, viewport.bottom, viewport.clip);
}


/* DrawBar()
*
*  DrawBar is used to initialize a bar plot area on the vga screen.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
#define BAR_WIDTH    80
#define BAR_HEIGHT  145
#define BAR_LEFT     40
#define BAR_BOTTOM  125

void DrawBar(int x_pos, int y_pos, char *name)
{
  struct viewporttype viewport;
  int    si;

  /* save current viewport settings */
  getviewsettings(&viewport);

  /* clear viewport. */
  setviewport(x_pos, y_pos, x_pos +BAR_WIDTH, y_pos +BAR_HEIGHT, 1);
  clearviewport();

  /* draw border around Bar */
  setcolor(LIGHTRED);
  rectangle(0, 0, BAR_WIDTH, BAR_HEIGHT);

  /* display title on screen */
  setcolor(WHITE);
  settextjustify(CENTER_TEXT, CENTER_TEXT);
  outtextxy(BAR_WIDTH/2, BAR_BOTTOM +15, name);

  /* display Y-axis values */
  outtextxy(BAR_LEFT -20, BAR_BOTTOM -100, "100%");
  outtextxy(BAR_LEFT -20, BAR_BOTTOM - 75, " 75%");
  outtextxy(BAR_LEFT -20, BAR_BOTTOM - 50, " 50%");
  outtextxy(BAR_LEFT -20, BAR_BOTTOM - 25, " 25%");
  outtextxy(BAR_LEFT -20, BAR_BOTTOM,      "  0%");

  /* restore current viewport settings */
  setviewport(viewport.left, viewport.top,
              viewport.right, viewport.bottom, viewport.clip);
}


/* PlotBar()
*
*  PlotBar is used to plot a fuzzy set A[] in the Bar plot area on
*  the vga screen.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
#define W_SET  256
void PlotBar(int x_pos, int y_pos, float Xo, float X[], int color)
{
  struct viewporttype viewport;
  int    si;
  char   buffer[20];
  float x, xinc;

  /* calculate offset for singleton */
  xinc  = (X[N_SET-1]-X[0])/100.0;
  x     = (Xo - X[0])/xinc;

  si    = (int )x;

  /* Ok.. draw the singleton */
  /* save current viewport settings */
  getviewsettings(&viewport);

  /* Ok.. draw the plot of a[W_SET] */
  setviewport(x_pos, y_pos, x_pos +BAR_WIDTH, y_pos +BAR_HEIGHT, 1);

  setfillstyle(SOLID_FILL, BLACK);
  bar(BAR_LEFT, BAR_BOTTOM - 120, BAR_LEFT+30, BAR_BOTTOM -si);

  setfillstyle(SOLID_FILL, color);
  bar(BAR_LEFT, BAR_BOTTOM, BAR_LEFT+30, BAR_BOTTOM -si);
  outtextxy(BAR_LEFT +15, BAR_BOTTOM - (si +10), itoa((int )Xo, buffer, 10));

  /* restore current viewport settings */
  setviewport(viewport.left, viewport.top,
              viewport.right, viewport.bottom, viewport.clip);
}


/* InitController()
*
*  InitController() is used to simulate the startup routines that would
*  be performed if this was an actual controller. In this case, it simply
*  initializes the VGA display and sets up the user interface.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
float  temperature =  72.0;
float  humidity    =  45.0;
float  fanspeed    = 250.0;

void InitController(void)
{
   InitGraphics();

   outtextxy(20, 350, "Fuzzy Fan Controller");
   outtextxy(20, 370, "Increase Temperature - CURSOR UP");
   outtextxy(20, 380, "Decrease Temperature - CURSOR DOWN");
   outtextxy(20, 390, "Increase Humidity    - CURSOR RIGHT");
   outtextxy(20, 400, "Decrease Humidity    - CURSOR LEFT");
   outtextxy(20, 410, "Quit program         - Q,q");

   DrawGraph(319,   0, "Temperature (F)", "COOL", "WARM", "HOT");
   PlotGraph(319,   0, T, COOL,  LIGHTGREEN);
   PlotGraph(319,   0, T, WARM,  LIGHTCYAN);
   PlotGraph(319,   0, T, HOT,   LIGHTRED);
   PlotSingleTon(319,   0, temperature,  T);

   DrawGraph(319, 150, "Humidity (%)", "DRY", "MOIST", "WET");
   PlotGraph(319, 150, H, DRY,   LIGHTGREEN);
   PlotGraph(319, 150, H, MOIST, LIGHTCYAN);
   PlotGraph(319, 150, H, WET,   LIGHTRED);
   PlotSingleTon(319,  150, humidity,  H);

   DrawGraph(319, 300, "FanSpeed (rpm)", "LOW", "MED", "HIGH");
   PlotGraph(319, 300, FS, LOW,  LIGHTGREEN);
   PlotGraph(319, 300, FS, MED,  LIGHTCYAN);
   PlotGraph(319, 300, FS, HIGH, LIGHTRED);

   DrawBar(10,  170, "T");
   PlotBar(10,  170, temperature, T,  LIGHTGREEN);

   DrawBar(110, 170, "H");
   PlotBar(110, 170, humidity,    H,  LIGHTCYAN);

   DrawBar(210, 170, "FS");
   PlotBar(210, 170, fanspeed,    FS, YELLOW);

   DrawGraph(0,   0, "Fuzzy FanSpeed (rpm)", "LOW", "MED", "HIGH");
}

/* GetInputs()
*
*  GetInputs() is used to simulate the reading of the input variables for
*  the fuzzy logic controller.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void GetInputs(float *Uo, float *Vo)
{
   *Uo = temperature;
   *Vo = humidity;
}

/* SetOutput()
*
*  SetOutput() is used to simulate the setting of the output variable for
*  the fuzzy logic controller. In this case however, it updates the
*  user display and retrieves new input.
*
*  Designed by: Henry Hurdon      Date: Mar.  15, 1993
*  Reviewed by: ................. Date: ..............
*  Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
#define INC  1.0
void SetOutput(float Yo)
{
   int ch;

   /* plot new fanspeed */
   fanspeed = Yo;
   PlotGraph(0, 0, FS, O,    WHITE);
   PlotSingleTon(  0, 0,    fanspeed, FS);
   PlotSingleTon(319, 300,    fanspeed, FS);
   PlotBar(210,  170, fanspeed, FS,  YELLOW);

   /* Get knew information */
   ch = ggetch();

   switch (ch) {
   case 'Q':
   case 'q':
     QuitGraphics();
     exit(0);
     break;

   case UP:
     /* remove old temperature */
     PlotSingleTon(319, 0, temperature, T);

     if (temperature < T[N_SET-1]) {
        temperature += INC;
     }
     if (temperature > T[N_SET-1]) {
        temperature = T[N_SET-1];
     }
     PlotSingleTon(319, 0, temperature, T);
     PlotBar(10, 170, temperature, T,  LIGHTGREEN);
     break;

   case DOWN:
     /* remove old temperature */
     PlotSingleTon(319, 0, temperature, T);

     if (temperature > T[0]) {
        temperature -= INC;
     }
     if (temperature < T[0]) {
        temperature = T[0];
     }
     PlotSingleTon(319, 0, temperature, T);
     PlotBar(10, 170, temperature, T,  LIGHTGREEN);
     break;

   case RIGHT:
     /* remove old humidity */
     PlotSingleTon(319, 150, humidity,  H);

     if (humidity < H[N_SET -1]) {
        humidity += INC;
     }
     if (humidity > H[N_SET -1]) {
        humidity = H[N_SET -1];
     }
     PlotSingleTon(319, 150, humidity, H);
     PlotBar(110,  170, humidity, H,  LIGHTCYAN);
     break;

   case LEFT:
     /* remove old humidity */
     PlotSingleTon(319, 150, humidity, H);

     if (humidity > H[0]) {
        humidity -= INC;
     }
     if (humidity < H[0]) {
        humidity = H[0];
     }
     PlotSingleTon(319, 150, humidity, H);
     PlotBar(110, 170, humidity, H, LIGHTCYAN);
     break;
   }

   /* remove old fanspeed */
   PlotGraph(0, 0, FS, O,    WHITE);
   PlotSingleTon(  0, 0,    fanspeed, FS);
   PlotSingleTon(319, 300,    fanspeed, FS);
}
