/**********************************************************************/
/*  File - FuzzyFan.c                                                 */
/*                                                                    */
/*  Description: This file contains the code for a fuzzy logic        */
/*  expert fan controller.                                            */
/*                                                                    */
/**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* The following defines the data sets used for 
** the Fuzzy Logic Expert FAN Controller
**
**      T[] - universal discourse for temperature
**   COOL[] - temperatures between 40 to 80
**   WARM[] - temperatures between 60 to 100
**    HOT[] - temperatures between 80 to 120
**
**      H[] - universal discourse for humidity
**    DRY[] - humidity between 20 to 60
**  MOIST[] - humidity between 40 to 80
**    WET[] - humidity between 60 to 100
**
**     FS[] - universal discourse for fan speed
**    LOW[] - fan speed between 250 to 750
** MEDIUM[] - fan speed between 500 to 1000
**   HIGH[] - fan speed between 750 to 1250
**
**   ZERO[] - contains all 0 values..
**
**  Note: The fuzzy sets are linearly defined over the full
**  universe of discourse. Even though only 5 elements are 
**  defined, values between this elements are computed..
**
**                     Temperature
**                 ______            ______
**                |      \    /\    /      |
**                |       \  /  \  /       |
**                |  cool  \/warm\/   hot  |
**                |        /\    /\        |
**                |       /  \  /  \       |
**                |      /    \/    \      |
**                --------------------------
**                40    60    80    100   120
**
*/                             

#define N_ELEM 5         /* N_ELEM defines the number */
                         /* of fuzzy set values       */
float     T[N_ELEM] = {  40,   60,   80,  100,  120};
float  COOL[N_ELEM] = { 1.0,  1.0,    0,    0,    0};
float  WARM[N_ELEM] = {   0,    0,  1.0,    0,    0};
float   HOT[N_ELEM] = {   0,    0,    0,  1.0,  1.0};

float     H[N_ELEM] = {  20,   40,   60,   80,  100};
float   DRY[N_ELEM] = { 1.0,  1.0,    0,    0,    0};
float MOIST[N_ELEM] = {   0,    0,  1.0,    0,    0};
float   WET[N_ELEM] = {   0,    0,    0,  1.0,  1.0};

float    FS[N_ELEM] = { 250,  500,  750, 1000, 1250};
float   LOW[N_ELEM] = { 1.0,  1.0,    0,    0,    0};
float   MED[N_ELEM] = {   0,    0,  1.0,    0,    0};
float  HIGH[N_ELEM] = {   0,    0,    0,  1.0,  1.0};

float  ZERO[N_ELEM] = {   0,    0,    0,    0,    0};

/* standard max() definition, used by a number of routines.. */
#ifndef max(a,b)
#define max(a,b)    (((a) > (b)) ? (a) : (b))
#endif


/* GetMembership()
**
** GetMembership() is used to calculate the membership value Ao
** with respect to a given Xo value based on the universe X[]
** and the fuzzy set A[]. The calculated value is returned.
**
** Designed by: Henry Hurdon      Date: Mar.  15, 1993
** Reviewed by: ................. Date: ..............
** Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
float GetMembership(float xo, float X[], float A[])
{
  float ao;
  int si;

  /* error check bounds on xo..
   * if xo outside of universe X[N_ELEM] then
   * return closest membership value. */
  if (xo < X[0]) {
     return(A[0]);
  } else if (xo > X[N_ELEM -1]) {
    return(A[N_ELEM -1]);
  }

  /* Ok, calculate the membership value */
  for (si = 0; si < N_ELEM-1; si++) {
    /* calculate membership value */
    if ((xo >= X[si])&&(xo <= X[si+1])) {
      if (X[si] == X[si+1]) {
         /* membership value based on maximum value */
         ao = max(A[si], A[si+1]);
      } else {
         /* membership value based on linear interpolation */
         ao = (xo-X[si]) * ((A[si+1]-A[si])/(X[si+1]-X[si])) + A[si];
      }
    }
  }

  return(ao);
}


/* GetMaximum()
**
** GetMaximum() is used to perform the MAX operation on the fuzzy sets
** A[] and B[]. The results are stored in C[].
**
** Designed by: Henry Hurdon      Date: Mar.  15, 1993
** Reviewed by: ................. Date: ..............
** Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void GetMaximum(float A[N_ELEM], float B[N_ELEM], float C[N_ELEM])
{
  int   si;

  for (si = 0; si < N_ELEM; si++) {
    C[si] = max(A[si], B[si]);
  }
}


/* DoInferEngine()
**
** DoInferEngine() is used to perform the inference engine calculation
** based on Ao and Bo and C[]. The results are stored in O[].
**
** Designed by: Henry Hurdon      Date: Mar.  15, 1993
** Reviewed by: ................. Date: ..............
** Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void DoInferEngine(float Wo, float C[], float O[])
{
  int   si;

  for (si = 0; si < N_ELEM; si++) {
    O[si] = Wo*C[si];
  }
}


/* DeFuzzyOutput()
**
** DeFuzzyOutput() is used to de-fuzzify the universe Y[] based on 
** the fuzzy set B[], using the center of gravity theorem.
** The result is stored in *Yo.
**
** Designed by: Henry Hurdon      Date: Mar.  15, 1993
** Reviewed by: ................. Date: ..............
** Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void DeFuzzyOutput(float Y[N_ELEM], float B[N_ELEM], float *Yo)
{
  float yB_sum, B_sum;
  int   si;

  yB_sum = 0.0;
  B_sum  = 0.0;

  /* get summations of Y[]B[] and B[] */
  for (si = 0; si < (N_ELEM -1); si++) {
    yB_sum += ((Y[si] + Y[si+1])/2)*((B[si] + B[si+1])/2);
     B_sum += (B[si] + B[si+1])/2;
  }

  /* check for divide by zero */
  if (B_sum != 0.0) {
    *Yo = yB_sum/B_sum;
  }
}


/* The following functions are found in the VGAHI.C or the DATA.C file,
** and are used to simulate the controller interface.     */
extern void InitController(void );
extern void GetInputs(float *To, float *Ho);
extern void SetOutput(float FSo);

/* The following variables are declared globally, so that they can be used
** by the display functions in VGAHI.C */
float R[N_ELEM];                  /* R[] - interm inference result        */
float O[N_ELEM];                  /* O[] - accumulative inference result  */


/* main()
**
** main() is the main procedure for the "Fuzzy Logic Expert Fan Controller".
**
** The following contains the linguistic matrix used for the FuzzyFan
** controller. Notice that the ideal condition is "warm and dry". If the
** room is cool the fan can be used to draw "warm and dry" air from outside
** the room. Likewise, if the room is hot and/or wet/moist, the fan can be
** used to cool/dry the room by expelling the hot/wet/moist air, drawing in
** the "warm and dry" air from outside the room.
**
**
**                     Temperature
**
**                 DRY    MOIST    WET                
**              +-------+-------+-------+
**         COOL |  MED  | HIGH  | HIGH  |
**              +-------+-------+-------+
**  Temp.  WARM |  LOW  |  MED  | HIGH  |
**              +-------+-------+-------+
**          HOT |  MED  | HIGH  | HIGH  |
**              +-------+-------+-------+
**
** The above linguistic matrix can be reduced to the following fuzzy rules.
**
**     IF H = WET  THEN FS = HIGH               
**     IF T = COOL AND H = DRY   THEN FS = MED   
**     IF T = COOL AND H = MOIST THEN FS = HIGH
**     IF T = WARM AND H = DRY   THEN FS = LOW   
**     IF T = WARM AND H = MOIST THEN FS = MED 
**     IF T = HOT  AND H = DRY   THEN FS = MED    
**     IF T = HOT  AND H = MOIST THEN FS = HIGH 
**
** Designed by: Henry Hurdon      Date: Mar.  15, 1993
** Reviewed by: ................. Date: ..............
** Modified by: Henry Hurdon      Date: Mar.  15, 1993
*/
void main()
{
  float To, Ho, FSo;              /* crisp input (To, Ho) and outut (FSo) */
                                  /* To  - crisp temperature              */
                                  /* Ho  - crisp humidity                 */
                                  /* FSo - crisp fanspeed                 */

  float T_COOL, T_WARM,  T_HOT;   /* membership values for To   */
  float H_DRY,  H_MOIST, H_WET;   /* membership values for Ho   */
  float FS_LOW, FS_MED,  FS_HIGH; /* membership values for FSo  */


  /* Initialize the controller */
  InitController();

  /* loop forever */
  while (1) {

    /* get real world inputs */
    GetInputs(&To, &Ho);
    
    /* convert to membership values.. */
    H_WET   = GetMembership(Ho, H, WET);
    H_MOIST = GetMembership(Ho, H, MOIST);
    H_DRY   = GetMembership(Ho, H, DRY);

    T_COOL  = GetMembership(To, T, COOL);
    T_WARM  = GetMembership(To, T, WARM);
    T_HOT   = GetMembership(To, T, HOT);

    /* initialize output membership values.. */
    FS_LOW  = 0.0;
    FS_MED  = 0.0;
    FS_HIGH = 0.0;


    /* The following executes the rule base used for
    *  the Fuzzy Logic Expert FAN Controller. This algorithm
    * is based on the "max product" implication. */

    /* IF H = WET THEN FS = HIGH                 */
    FS_HIGH = max(H_WET,          FS_HIGH);

    /* IF T = COOL AND H = DRY THEN FS = MED     */
    FS_MED  = max(T_COOL*H_DRY,   FS_MED);

    /* IF T = COOL AND H = MOIST THEN FS = HIGH  */
    FS_HIGH = max(T_COOL*H_MOIST, FS_HIGH);

    /* IF T = WARM AND H = DRY THEN FS = LOW     */
    FS_LOW  = max(T_WARM*H_DRY,   FS_LOW);

    /* IF T = WARM AND H = MOIST THEN FS = MED   */
    FS_MED  = max(T_WARM*H_MOIST, FS_MED);

    /* IF T = HOT AND H = DRY THEN FS = MED      */
    FS_MED  = max(T_HOT*H_DRY,    FS_MED);
       
    /* IF T = HOT AND H = MOIST THEN FS = HIGH   */
    FS_HIGH = max(T_HOT*H_MOIST,  FS_HIGH);


    /* Calculate inferences.. */
    DoInferEngine(FS_LOW,  LOW, R);
    GetMaximum(R, ZERO, O);

    DoInferEngine(FS_MED,  MED, R);
    GetMaximum(R, O, O);

    DoInferEngine(FS_HIGH, HIGH, R);
    GetMaximum(R, O, O);

    /* Ok.. defuzzify and output crisp result */
    DeFuzzyOutput(FS, O, &FSo);
    SetOutput(FSo);
  }
}
