/*	This file is part of the software similarity tester SIM.
	Written by Dick Grune, Vrije Universiteit, Amsterdam.
	$Header: compare.c,v 2.1 91/06/18 22:00:59 dick Exp $
*/

#include	"private.h"
#include	"buff.h"
#include	"sim.h"

/* from the Language Department: */
extern int MayBeStartOfRun();
extern unsigned int CheckRun();

extern char options[];
extern int ntexts;
extern int nnewtexts;
extern struct text *text;
extern int min_run_size;
extern unsigned int *forw_ref;
extern add_run();

PRIVATE compare1text();
PRIVATE unsigned int lcs();

/*	The overall structure of the routine compare() is:

	for all new files
		for all texts it must be compared to
			for all positions in the new file
				for all positions in the text
					for ever increasing sizes
						try to match and keep the best
*/

compare()
{
	/*	Compares each new text to the appropriate texts.
	*/
	register int n;
	
	for (n = 0; n < nnewtexts; n++) {
		register int first =
			(options['S'] ? nnewtexts+1 : options['s'] ? n+1 : n);

		if (options['e']) {
			/* from first to ntexts in steps */
			register int m;

			for (m = first; m < ntexts; m++) {
				compare1text(n, m, m+1);
			}
		}
		else {
			/* from first to ntexts in one action */
			if (first < ntexts) {
				compare1text(n, first, ntexts);
			}
		}
	}
}

PRIVATE
compare1text(n, first, limit)
	int n;				/* text to be compared */
	int first;			/* first text to be compared to */
	int limit;			/* limit text in comparison */
{
	register unsigned int i_first = text[first].tx_start;
	register unsigned int i_limit =
		(limit < ntexts ? text[limit].tx_start : text_length());
	register struct text *txt0 = &text[n];
	register unsigned int i0 = txt0->tx_start;
	
	while (	/* there may still be a useful substring */
		i0 + min_run_size < txt0->tx_limit + 1
	) {
		/* see if there really is one */
		struct text *txt_best;
		unsigned int i_best;
		register unsigned int size_best =
			lcs(txt0, i0, &txt_best, &i_best, i_first, i_limit);
		
		if (size_best) {
			/* good run found; enter it */
			add_run(txt0, i0, txt_best, i_best, size_best);
			/* and skip it */
			i0 += size_best;
		}
		else {
			/* we try our luck at the next token */
			i0++;
		}
	}
}

PRIVATE unsigned int
lcs(txt0, i0, tbp, ibp, i_first, i_limit)
	struct text *txt0;		/* input: starting position */
	unsigned int i0;
	struct text **tbp;		/* output: position of best run */
	unsigned int *ibp;
	unsigned int i_first;		/* no comparison before this pos. */
	unsigned int i_limit;		/* no comparison after this pos. */
{
	/*	Finds the longest common substring (not -sequence) in:
			txt0, starting precisely at i0 and
			the text between i_first and i_limit.
		Writes the position in tbp and ibp and returns the size.
		Returns 0 if no common substring is found.
	*/
	register struct text *txt1 = txt0;
	register unsigned int i1 = i0;
	register unsigned int size_best = 0;
	register unsigned int txt0limit = txt0->tx_limit;
	register unsigned int txt1limit = txt1->tx_limit;

	while (	/* there is a next opportunity */
		(i1 = (forw_ref ? forw_ref[i1] : i1 + 1))
	&&	/* it is still in range */
		i1 < i_limit
	) {
		register unsigned int min_size;
		register unsigned int new_size;
		register unsigned int j0;
		register unsigned int j1;
		
		if (i1 < i_first) {	/* not in range */
			continue;
		}

		/* bump txt1; we may have skipped a text or two */
		while (i1 >= txt1->tx_limit) {
			txt1++;
		}
		txt1limit = txt1->tx_limit;

		min_size = size_best ? size_best+1 : min_run_size;
		/* are we looking at something better than we have got? */
		{
			j0 = i0 + min_size - 1;
			j1 = i1 + min_size - 1;
			if (	/* j0 still inside txt0 */
				j0 < txt0limit
			&&	/* j1 still inside txt1 */
				j1 < txt1limit
			&&	/* j0 and j1 don't overlap */
				j0 < j1 - min_size + 1
			) {
				/* there would be room enough */
				register int cnt = min_size;

				/* does the text match? */
				while (	cnt
				&&	TOKEN_EQ(tk_buff[j0], tk_buff[j1])
				) {
					cnt--, j0--, j1--;
				}
				if (cnt) continue;	/* forget it */
			}
			else continue;			/* forget it */
		}

		/* yes, we are; how long can we make it? */
		{
			register unsigned int size = min_size;

			j0 = i0 + min_size;
			j1 = i1 + min_size;
			while (	/* j0 still inside txt0 */
				j0 < txt0limit
			&&	/* j1 still inside txt1 */
				j1 < txt1limit
			&&	/* j0 and j1 don't overlap */
				j0 + size < j1
			&&	/* tokens are the same */
				TOKEN_EQ(tk_buff[j0], tk_buff[j1])
			) {
				j0++, j1++, size++;
			}
			new_size = size;
		}
		
		/*	offer the run to the Language Department which may
			reject it or may cut its tail
		*/
		new_size = MayBeStartOfRun(tk_buff[i0]) ?
				CheckRun(&tk_buff[i0], new_size) : 0;
		
		if (	/* we still have something acceptable */
			new_size >= min_run_size
		&&	/* it is better still than what we had */
			new_size > size_best
		) {
			/* record it */
			*tbp = txt1;
			*ibp = i1;
			size_best = new_size;
		}
	}

	return size_best;
}

