#include "global.h"


/*
 * Structure to hold information for heuristics with domain ordering.
 */
struct info {
    int index, value;
};


/*
 * Calculates the convexity value for a row of the constraint and returns the
 * value.  A "convexity value" is simply a string of 0's in between strings
 * of 1's.  Thus strings of 0's at both ends of the row are eliminated first.
 * Then as long as the pointers do not overlap, strings of 0's in between
 * strings of 1's are counted and returned.
 */
int domain_convexity(c, row, k)
CONSTRAINT c;
int row, k;
{
    int i, j, count = 0;

    i = 0;
    while (i < k && c[row][i] == 0)
        i++;
    j = k - 1;
    while (j > i && c[row][j] == 0)
        j--;
    while (j > i && c[row][j])
        j--;
    while (i < j) {
        while (i < j && c[row][i])
            i++;
        while (i < j && c[row][i] == 0)
            i++;
        count++;
    }
    return(count);
}


/*
 * Return whether a certain row of a given edge contains a constraint or not.
 * This simply amounts to scanning a constraint for a "0" entry.
 */
int domain_degree(c, row, k)
CONSTRAINT c;
int row, k;
{
    int i;

    for (i = 0; i < k; i++)
        if (c[row][i] == 0)
            return(1);
    return(0);
}


/*
 * Calculates the number of constraints associated with a row in an edge and
 * returns the result.  That is, simply count the number of 0's in the
 * constraint and return this value.
 */
int domain_satisfy(c, row, k)
CONSTRAINT c;
int row, k;
{
    int i, count = 0;

    for (i = 0; i < k; i++)
        if (c[row][i] == 0)
            count++;
    return(count);
}


/*
 * Comparison function for qsort to order the domain values in ascending
 * order.
 */
static int ascending(a, b)
struct info *a, *b;
{
    if	    (a->value < b->value) return(-1);
    else if (a->value > b->value) return( 1);
    else			  return( 0);
}


/*
 * Function to change the network to conform to the domain ordering specified.
 * The ordering is first applied by copying the matrix, in the new order, into
 * the "temp" network array.  Then, this temporary network has its contents
 * copied back to the original network array.
 */
static void apply_order(C, n, k, order)
NETWORK C;
int n, k, order[N][K];
{
    NETWORK temp;
    int i, j, x, y;
    
    for (i = 1; i <= n; i++)
        for (j = 1; j <= n; j++)
            for (x = 0; x < k; x++)
                for (y = 0; y < k; y++)
                    temp[i][j][x][y] = C[i][j][order[i][x]][order[j][y]];
    for (i = 1; i <= n; i++)
        for (j = 1; j <= n; j++)
            for (x = 0; x < k; x++)
                for (y = 0; y < k; y++)
                    C[i][j][x][y] = temp[i][j][x][y];
}


/*
 * Create an order to instantiate the domains by applying a heuristic to the
 * original problem.  First, using the "heristic" as passed in by the user,
 * count the heuristic value for each domain value by considering the entire
 * "row" across all edges in the constraint network.  Then, sort the values
 * using the "qsort" library function.  Then store the results for later use.
 * These steps are followed for each variable.  Finally, call the routine to
 * apply the heuristic to each variable's domains.
 */
void order_domains(heuristic, C, n, k)
int (* heuristic)(), n, k;
NETWORK C;
{
    struct info current_info[K];
    int i, j, x, order[N][K];

    for (i = 1; i <= n; i++) {
        for (x = 0; x < k; x++) {
            current_info[x].index = x;
            current_info[x].value = 0;
            for (j = 1; j <= n; j++)
                current_info[x].value += heuristic(C[i][j], x, k);
        }
        qsort((char *)&current_info[0], k, sizeof(struct info), ascending);
        for (x = 0; x < k; x++)
            order[i][x] = current_info[x].index;
    }
    apply_order(C, n, k, order);
}
