#include "global.h"


/*
 * Stack functions for arc consistency where only variables have to be stored.
 * Push pushes a variable on the stack and marks it as being on the stack,
 * unless the variable was already on the stack, in which case it is not added
 * again.  Pop gets the top value off the stack.  The stack_empty function
 * returns whether the stack is empty or not.
 */
#define push(a)       if (on_stack[a] == 0) { \
                          stack[top] = a; \
                          top++; \
                          on_stack[a] = 1; \
                       }

#define pop(a)        top--; \
                      a = stack[top]; \
                      on_stack[a] = 0;

#define stack_empty() (top == 0)


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.  The
 * "domains" structure works as follows.  If an entry contains a 0, then the
 * value is still allowed.  Otherwise, it contains the variable whose
 * instantiation eliminated it.  Note that this data structure has no
 * information from preprocessing (which is kept directly in the "C" matrix).
 * Note that the stack is for the previously mentioned "functions".
 */ 
static int domains[N][K];
static int checking[N][N];
static int stack[VARS], on_stack[N], top = 0;


/*
 * Clear the setup data structures.
 */
static void clear_setup(n, k)
int n, k;
{
    int i, j;

    for (i = 1; i <= n; i++) {
        for (j = 0; j < k; j++)
            domains[i][j] = 0;
        for (j = 1; j <= n; j++)
            checking[i][j] = 0;
    }
}


/*
 * This function checks the edge between variables "i" and "j" to see which
 * values in the domain of "j" can be eliminated due to the current
 * instantiation of "i" as "value".  Note that given a variable "i" and an
 * instantiation "v", domains[i][v] is 0 if the value is still allowed,
 * otherwise it contains the variable which caused it to be eliminated.  If any
 * value in "j" is deleted, this is noted in the "checking" matrix, and since
 * the edge changed, all affected edges are pushed onto the stack.  The number
 * of values in the domain of "j" after all deletions is returned.
 */
static int check_forward(k, current, j, value)
int k, current, j, value;
{
    int i, delete_count = 0, old_count = 0;

    checks++;
    for (i = 0; i < k; i++)
        if (domains[j][i] == 0)
            old_count++;
    if (domains[j][value] == 0) {
        delete_count = 1;
        domains[j][value] = current;
        checking[current][j] = 1;
        push(j);
    }
    return(old_count - delete_count);
}


/*
 * Function to restore the domain of variables which were eliminated due to the
 * instantiation of variable "i".  A variable is known to have had a value
 * deleted by looking at the "checking" matrix.
 */
static void restore(current, n, k)
int current, n, k;
{
    int i, j;

    for (i = current + 1; i <= n; i++)
        if (checking[current][i]) {
            for (j = 0; j < k; j++)
                if (domains[i][j] == current)
                    domains[i][j] = 0;
            checking[current][i] = 0;
        }
}


/*
 * This function determines whether given a specific edge, there exists some
 * instantiation for "y" given the instantiation for "x".
 */
static int exists(k, i, value)
int k, i, value;
{
    int j;

    for (j = 0; j < k; j++)
        if (domains[i][j] == 0 && (j != value))
            return(1);
    return(0);
}


/*
 * Determine which domain values for variable "x" can be eliminated by
 * considering the edge between "x" and "y".  The number of values left in the
 * domain of "x" is returned. (-1 indicates no change to the edge).  The
 * "checking" matrix is appropriately updated.
 */
static int revise(k, i, j, current)
int k, i, j, current;
{
    int n, delete = 0, lookat = 0;

    checks++;
    for (n = 0; n < k; n++) {
        if (domains[i][n])
            continue;
        lookat++;
        if (exists(k, j, n) == 0) {
            domains[i][n] = current;
            delete++;
        }
    }
    if (delete)
        checking[current][i] = 1;
    return((delete) ? (lookat - delete) : -1);
}


/*
 * Check if the current instantiation of the variables is consistent by
 * checking if the current instantiation of a variable will reduce some future
 * variable's domain to the empty set, in which case the appropriate result is
 * returned.  For each domain change in a future variable, the appropriate
 * edges are pushed onto the stack.
 */
static int consistent(C, n, k, current, value)
NETWORK C;
int n, k, current, value;
{
    int i, j, lookahead;

    for (i = current + 1; i <= n; i++)
        if (C[current][i] == U)
            continue;
        else if (check_forward(k, current, i, value) == 0)
            goto END;
    while (stack_empty() == 0) {
        pop(j);
        for (i = current + 1; i <= n; i++) {
            if (C[i][j] != NE)
                continue;
            lookahead = revise(k, i, j, current);
            if (!lookahead)
                goto END;
            else if (lookahead != -1)
                push(i);
        }
    }
    return(1);
    END: while (stack_empty() == 0)
        pop(j);
    return(0);
}


/*
 * Solve the constraint network using the FORWARD CHECKING with FULL ARC
 * CONSISTENCY (FC-arc) method.  If this is the first variable then initialize
 * the data structures.  If a solution is found then update the "found"
 * variable, call the function to process the solution, and return the value
 * according to whether the first solution is desired or all solutions.  Then,
 * check if the timer has expired, and if it has, return immediately.
 * Otherwise, begin checking each possible instantiation of the variable.  For
 * each domain value, perform the following steps.  First, if preprocessing
 * eliminated the value, disregard the rest of the loop.  Otherwise,
 * instantiate the variable, check if the network is still consistent, and then
 * call the backtracking routine recursively.  After checking each domain
 * value, restore the domains which were eliminated by the instantiation.
 * After checking all possible domain values, return.
 */
int FC_arc(C, n, k, solution, current, found)
NETWORK C;
int n, k, current, *found;
SOLUTION solution;
{
    int i;

    if (current == 1) {
        clear_setup(n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        return(1);
    }
    if (time_expired())
        return(1);
    for (i = 0; i < k; i++) {
        if (domains[current][i])
            continue;
        solution[current] = i;
        if (consistent(C, n, k, current, solution[current]))
            if (FC_arc(C, n, k, solution, current + 1, found))
                return(1);
        restore(current, n, k);
    }
    return(0);
}
