
#include "global.h"


/*
 *  Stack functions for path consistency where only edges have to be stored.
 *  Push stores an edge on the stack and marks it as being on the stack.  If it
 *  is already on the stack, it is not added again.  Pop removes the top edge
 *  from the stack and unmarks it appropriately.  The stack_empty macro returns
 *  whether the stack is empty or not.
 */
#define push(x,y)	{\
				if( !on_stack[x][y] ) {\
					top++;\
					stack[top][0] = x;\
					stack[top][1] = y;\
					on_stack[x][y] = 1;\
					on_stack[y][x] = 1;\
				}\
			}

#define pop(x,y)	{\
				x = stack[top][0];\
				y = stack[top][1];\
				on_stack[x][y] = 0;\
				on_stack[y][x] = 0;\
				top--;\
			}

#define init_stack()	{\
				top = 0;\
				for( i = 1; i <= n; i++ )\
				for( j = 1; j <= n; j++ )\
					on_stack [i][j] = 0;\
			}

#define stack_empty()	(top == 0)


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.  The
 * "domains" structure works as follows.  If an entry contains a 0, then the
 * value is still allowed.  Otherwise, it contains the variable whose
 * instantiation eliminated it.  Note that this data structure has no
 * information from preprocessing (which is kept directly in the "C" matrix).
 * The stack functions are for the forementioned "functions".
 */
static int domains[N][K];
static int checking[N][N];
static HEAD *changes = NULL;
static int stack[(N*(N+1))/2][2], on_stack[N][N], top = 0;


/*
 * Function to add the entry (i, j, x, y) to the list meaning it was changed
 * from a 1 to a 0 (also, entry (j, i, y, x) was changed).
 */
static void add_list(i, j, x, y)
int i, j, x, y;
{
    LIST *temp;

    new_malloc(temp, LIST);
    temp->i = i;
    temp->j = j;
    temp->x = x;
    temp->y = y;
    temp->next = changes->list;
    changes->list = temp;
}


/*
 * Restore an entire list of entries to changes entries (i, j, x, y) and
 * (j, i, y, x) from a 0 to a 1.
 */
static void restore_list(C)
NETWORK C;
{
    LIST *curr, *prev;

    curr = changes->list;
    changes->list = NULL;
    while (curr) {
        C[curr->i][curr->j][curr->x][curr->y] = 1;
        C[curr->j][curr->i][curr->y][curr->x] = 1;
        prev = curr;
        curr = curr->next;
        free((char *)prev);
    }
}


/*
 * Create a new list head for a new level.
 */
static void start_head()
{
    HEAD *temp;

    new_malloc(temp, HEAD);
    temp->list = NULL;
    temp->next = changes;
    changes = temp;
}


/*
 * Destroy the topmost head of the list.
 */
static void remove_head()
{
    HEAD *temp;

    temp = changes;
    changes = changes->next;
    free((char *)temp);
}

 
/*
 * Clear the setup data structures.
 */
static void clear_setup(n, k)
int n, k;
{
    int i, j;

    init_stack();

    for (i = 1; i <= n; i++) {
        for (j = 0; j < k; j++)
            domains[i][j] = 0;
        for (j = 1; j <= n; j++)
            checking[i][j] = 0;
    }
}


/*
 * Function to restore the domain of variables which were eliminated due to the
 * instantiation of variable "i".  A variable is known to have had a value
 * deleted by looking at the "checking" matrix.
 */
static void restore(C, n, k, i)
NETWORK C;
int n, k, i;
{
    int j, l;

    for (j = i + 1; j <= n; j++)
        if (checking[i][j]) {
            checking[i][j] = 0;
            for (l = 0; l < k; l++)
                if (domains[j][l] == i)
                    domains[j][l] = 0;
        }
    restore_list(C);
}


/*
 * This function checks the edge between variables "i" and "j" to see which
 * values in the domain of "j" can be eliminated due to the current
 * instantiation of "i" as "value".  First, all rows not possible in the
 * current edge are set to zero.  Then, for each value not allowed, that entry
 * is set to zero.  If any entry is changed to a zero throughout this process,
 * this is noted in the "checking" matrix.  The number of values in the domain
 * of "j" after all deletions is returned.
 */
static int check_forward(C, k, i, j, value)
NETWORK C;
int k, i, j, value;
{
    int m, n, changed, old_count = 0, delete_count = 0;

    for (m = 0; m < k; m++)
        if (m != value)
            for (n = 0; n < k; n++) {
                checks++;
                if (C[i][j][m][n]) {
                    checks++;
                    changed = 1;
                    add_list(i, j, m, n);
                    C[i][j][m][n] = 0;
                    C[j][i][n][m] = 0;
                }
            }
    for (m = 0; m < k; m++)
        if (C[j][j][m][m] && (domains[j][m] == 0)) {
            old_count++;
            checks++;
            if (C[i][j][value][m] == 0) {
                domains[j][m] = i;
                delete_count++;
            }
        }
    if (delete_count)
        checking[i][j] = 1;
    if (changed || delete_count)
        push(i, j);
    return(old_count - delete_count);
}


/*
 * This function performs composition between edges "ab" and "bc" and compares
 * the results with edge "ac".  The first if statement performs the "union" (if
 * the entry was a 0, then no change will ever result, but if it was a 1, then
 * the entry may change).  The "for" loop determines whether the composition of
 * a row from edge "ab" with a column from "bc" produces a 1.  If a 1 is not
 * produced, the following if statement makes the appropriate changes since it
 * knows that the "for" loop loop counter is now "out-of-range"; otherwise a
 * 1 is produced and no changes from this part of the composition are created.
 * This is done for every entry in the edge.
 */
static int revise(C, k, a, b, c)
NETWORK C;
int k, a, b, c;
{
    int i, j, x, changed = 0;

    for (i = 0; i < k; i++)
        for (j = 0; j < k; j++) {
            checks++;
            if (C[a][c][i][j]) {
                x = 0;
                while (x < k && (C[a][b][i][x] == 0 || C[b][c][x][j] == 0))
                    x++;
                checks += 2 * (x + 1);
                if (x == k) {
                    changed = 1;
                    C[a][c][i][j] = 0;
                    C[c][a][j][i] = 0;
                    add_list(a, c, i, j);
                   }
            }
        }
    return(changed);
}


/*
 * Check if the current instantiation of the variables is consistent by
 * checking if the current instantiation of a variable will reduce some future
 * variable's domain to the empty set, in which case the appropriate result is
 * returned.  For any edge changed, it is pushed back on the stack to be used
 * further in the path consistency.
 */
static int consistent(C, n, k, current, value)
NETWORK C;
int n, k, current, value;
{
    int i, x, y;

    for (i = current + 1; i <= n; i++) {
        if (!check_forward(C, k, current, i, value)) {
            while (stack_empty() == 0)
                pop(x, y);
            restore_list(C);
            return(0);
        }
    }
    while (stack_empty() == 0) {
        pop(x, y);
        for (i = 1; i <= n; i++) {
            if (revise(C, k, x, y, i))
                push(x, i);
            if (revise(C, k, i, x, y))
                push(i, y);
        }
    }
    return(1);
}


/*
 * Solve the constraint network using the FORWARD CHECKING with FULL PATH
 * CONSISTENCY (FCpath) method.  If this is the first variable then initialize
 * the data structures.  If a solution is found then update the "found"
 * variable, call the function to process the solution, and return the value
 * according to whether the first solution is desired or all solutions.  Then,
 * check if the timer has expired, and if it has, return immediately.
 * Otherwise, begin checking each possible instantiation of the variable.  For
 * each domain value, perform the following steps.  First, if preprocessing
 * eliminated the value, disregard the rest of the loop.  Otherwise,
 * instantiate the variable, check if the network is still consistent, and then
 * call the backtracking routine recursively.  After checking each domain
 * value, restore the domains which were eliminated by the instantiation.
 * After checking all possible domain values, reset and return.
 */
int FCpath(C, n, k, solution, current, number, found)
NETWORK C;
int n, k, current, number, *found;
SOLUTION solution;
{
    int i;

    if (current == 1) {
        clear_setup(n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        count++;
        if (number == 1)
            while (changes) {
                restore_list(C);
                remove_head();
            }
        return(number == 1 ? 1 : 0);
    }
    if (time_expired()) {
        while (changes) {
            restore_list(C);
            remove_head();
        }
        return(1);
    }
    start_head();
    for (i = 0; i < k; i++) {
        if (C[current][current][i][i] == 0 || domains[current][i])
            continue;
        solution[current] = i;
        if (consistent(C, n, k, current, solution[current]))
            if (FCpath(C, n,k, solution, current + 1, number, found))
                return(1);
        restore(C, n, k, current);
    }
    remove_head();
    return(0);
}
