
#include "global.h"


/*
 * Stack functions for arc consistency where only variables have to be stored.
 * Push pushes a variable on the stack and marks it as being on the stack,
 * unless the variable was already on the stack, in which case it is not added
 * again.  Pop gets the top value off the stack.  The stack_empty function
 * returns whether the stack is empty or not.
 */
#define push(a)       if (!on_stack[a]) { \
                          top++; \
                          stack[top] = a; \
                          on_stack[a] = 1; \
                      }

#define pop(a)        { \
			  a = stack[top]; \
			  on_stack[a] = 0; \
			  top--; \
		      }

#define init_stack()  {\
			  top = 0;\
			  for( i = 1; i <= n; i++ )\
			      on_stack [i] = 0;\
		      }

#define stack_empty() (top == 0)


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.  The
 * "domains" structure works as follows.  If an entry contains a 0, then the
 * value is still allowed.  Otherwise, it contains the variable whose
 * instantiation eliminated it.  Note that this data structure has no
 * information from preprocessing (which is kept directly in the "C" matrix).
 * Note that the stack is for the previously mentioned "functions".
 */
static int domains[N][K];
static int checking[N][N];
static int stack[N], on_stack[N], top = 0;


/*
 * Clear the setup data structures.
 */
static void clear_setup(n, k)
int n, k;
{
    int i, j;

    init_stack();

    for (i = 1; i <= n; i++) {
        for (j = 0; j < k; j++)
            domains[i][j] = 0;
        for (j = 1; j <= n; j++)
            checking[i][j] = 0;
    }
}


/*
 * Function to restore the domain of variables which were eliminated due to the
 * instantiation of variable "i".  A variable is known to have had a value
 * deleted by looking at the "checking" matrix.
 */
static void restore(i, n, k)
int i, n, k;
{
    int j, l;

    for (j = i + 1; j <= n; j++)
        if (checking[i][j]) {
            checking[i][j] = 0;
            for (l = 0; l < k; l++)
                if (domains[j][l] == i)
                    domains[j][l] = 0;
        }
}


/*
 *  This function checks the edge between variables "current" and "j" to
 *  see which values in the domain of "j" can be eliminated due to the
 *  instantiation of "current" as "value".  Note that given a variable
 *  "current" and an instantiation "a", domains[current][a] is 0 if the value
 *  is still allowed, otherwise it contains the variable which caused it to
 *  be eliminated.  If any value in the domain of "j" is deleted, this is
 *  noted in the "checking" matrix, and since its domain changed, "j" is
 *  pushed onto the stack.  The number of values in the domain of "j" after
 *  all deletions is returned.
 */
static int check_forward(C, k, current, j, value)
NETWORK C;
int k, current, j, value;
{
    int a, domain_count = 0, delete_count = 0;

    for (a = 0; a < k; a++)
        if (C[j][j][a][a] && (domains[j][a] == 0)) {
            domain_count++;
            checks++;
            if (C[current][j][value][a] == 0) {
                domains[j][a] = current;
                delete_count++;
            }
        }
    if (delete_count) {
        if (domain_count > delete_count) /* there's still domain values left */
            push(j);
        checking[current][j] = 1;
    }
    return(domain_count - delete_count);
}


/*
 * This function determines whether given a specific edge, there exists some
 * instantiation for "y" given the instantiation for "x".
 */
static int exists(con, k, y, a)
CONSTRAINT con;
int a, k, y;
{
    int b;

    for (b = 0; b < k; b++)
        if (domains[y][b] == 0) {
            checks++;
            if (con[a][b])
                return(1);
        }
    return(0);
}


/*
 * Determine which domain values for variable "i" can be eliminated by
 * considering the edge between "i" and "j".  The number of values left in the
 * domain of "i" is returned. (-1 indicates no change to the edge).  The
 * "checking" matrix is appropriately updated.
 */
static int revise(C, k, i, j, current)
NETWORK C;
int k, i, j, current;
{
    int a, domain_count = 0, delete_count = 0;

    for (a = 0; a < k; a++)
        if (domains[i][a] == 0) {
            domain_count++;
            if (!exists(C[i][j], k, j, a)) {
                domains[i][a] = current;
                delete_count++;
            }
	}
    if (delete_count)
        checking[current][i] = 1;
    return((delete_count) ? (domain_count - delete_count) : -1);
}


/*
 * Check if the current instantiation of the variables is consistent by
 * checking if the current instantiation of a variable will reduce some future
 * variable's domain to the empty set, in which case the appropriate result is
 * returned.  For each domain change in a future variable, the appropriate
 * edges are pushed onto the stack.
 */
static int consistent(C, n, k, current, value)
NETWORK C;
int n, k, current, value;
{
    int i, j, lookahead;

    for (j = current + 1; j <= n; j++)
        if (!check_forward(C, k, current, j, value))
            goto END;
    while (!stack_empty()) {
        pop(j);
        for (i = current + 1; i <= n; i++)
            if (i != j) {
                lookahead = revise(C, k, i, j, current);
                if (!lookahead)
                    goto END;
                else if (lookahead != -1)
                    push(i);
            }
    }
    return(1);
    END: while (!stack_empty())
        pop(j);
    return(0);
}


/*
 * Solve the constraint network using the FORWARD CHECKING with FULL ARC
 * CONSISTENCY (FCarc) method.  If this is the first variable then initialize
 * the data structures.  If a solution is found then update the "found"
 * variable, call the function to process the solution, and return the value
 * according to whether the first solution is desired or all solutions.  Then,
 * check if the timer has expired, and if it has, return immediately.
 * Otherwise, begin checking each possible instantiation of the variable.  For
 * each domain value, perform the following steps.  First, if preprocessing
 * eliminated the value, disregard the rest of the loop.  Otherwise,
 * instantiate the variable, check if the network is still consistent, and then
 * call the backtracking routine recursively.  After checking each domain
 * value, restore the domains which were eliminated by the instantiation.
 * After checking all possible domain values, return.
 */
int FCarc(C, n, k, solution, current, number, found)
NETWORK C;
int n, k, current, number, *found;
SOLUTION solution;
{
    int i;

    if (current == 1) {
        clear_setup(n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        count++;
        return(number == 1 ? 1 : 0);
    }
    if (time_expired())
        return(1);
    for (i = 0; i < k; i++) {
        if (C[current][current][i][i] == 0 || domains[current][i])
            continue;
        solution[current] = i;
        if (consistent(C, n, k, current, solution[current]))
            if (FCarc(C, n, k, solution, current + 1, number, found))
                return(1);
        restore(current, n, k);
    }
    return(0);
}
