#include "global.h"


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.  The
 * "domains" structure works as follows.  If an entry contains a 0, then the
 * value is still allowed.  Otherwise, it contains the variable whose
 * instantiation eliminated it.  Note that this data structure has no
 * information from preprocessing (which is kept directly in the "C" matrix).
 */
static int domains[N][K];
static int checking[N][N];
static int jump_place[N];


/*
 * Clear the setup data structures.
 */
static void clear_setup(n, k)
int n, k;
{
    int i, j;

    for (i = 1; i <= n; i++) {
        jump_place[i] = 0;
        for (j = 0; j < k; j++)
            domains[i][j] = 0;
        for (j = 1; j <= n; j++)
            checking[i][j] = 0;
    }
}


/*
 * This function checks the edge between variables "i" and "j" to see which
 * values in the domain of "j" can be eliminated due to the current
 * instantiation of "i" as "value".  Note that given a variable "i" and an
 * instantiation "v", domains[i][v] is 0 if the value is still allowed,
 * otherwise it contains the variable which caused it to be eliminated.  If any
 * value in "j" is deleted, this is noted in the "checking" matrix.  The number
 * of values in the domain of "j" after all deletions is returned.
 */
static int check_forward(C, k, i, j, value)
NETWORK C;
int k, i, j, value;
{
    int n, old_count = 0, delete_count = 0;

    for (n = 0; n < k; n++)
        if (C[j][j][n][n] && (domains[j][n] == 0)) {
            old_count++;
            checks++;
            if (C[i][j][value][n] == 0) {
                domains[j][n] = i;
                delete_count++;
            }
        }
    if (delete_count)
        checking[i][j] = 1;
    return(old_count - delete_count);
}


/*
 * Function to restore the domain of variables which were eliminated due to the
 * instantiation of variable "i".  A variable is known to have had a value
 * deleted by looking at the "checking" matrix.
 */
static void restore(i, n, k)
int i, n, k;
{
    int j, l;

    for (j = i + 1; j <= n; j++)
        if (checking[i][j])
            for (l = 0; l < k; l++) {
                checking[i][j] = 0;
                if (domains[j][l] == i)
                    domains[j][l] = 0;
            }
    checking[i][i] = 0;
}


/*
 * Check if the current instantiation of the variables is consistent by
 * checking if the current instantiation of a variable will reduce some future
 * variable's domain to the empty set, in which case the appropriate result is
 * returned.
 */
static int consistent(C, n, k, current, value)
NETWORK C;
int n, k, current, value;
{
    int i;

    for (i = current + 1; i <= n; i++) {
        if (check_forward(C, k, current, i, value) == 0)
            return(i);
    }
    return(0);
}


/*
 * Solve the CSP using the FORWARD CHECKING with BACKJUMPING (FC-BJ) method.
 * Solve the constraint network using the FORWARD CHECKING with BACKJUMPING
 * (FC-BJ) method.  If this is the first variable then initialize the data
 * structures.  If a solution is found then update the "found" variable, call
 * the function to process the solution, and return the value according to
 * whether the first solution is desired or all solutions.  Then, check if the
 * timer has expired, and if it has, return immediately.  Otherwise, begin
 * checking each possible instantiation of the variable.  For each domain
 * value, perform the following steps.  First, if preprocessing eliminated the
 * value, disregard the rest of the loop.  Otherwise, instantiate the variable,
 * check if the network is still consistent, and then call the backtracking
 * routine recursively.  After checking each domain value, restore the domains
 * which were eliminated by the instantiation.  After checking all possible
 * domain values, determine the variable to jump back to, update the data
 * structures, and return.
 */
int FC_BJ(C, n, k, solution, current, number, found)
NETWORK C;
int n, k, current, number, *found;
SOLUTION solution;
{
    int i, j, jump, fail;

    jump_place[current] = 0;
    if (current == 1) {
        clear_setup(n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        count++;
        return(number == 1 ? 0 : n);
    }
    if (time_expired())
        return(0);
    for (i = 0; i < k; i++) {
        if (C[current][current][i][i] == 0 || domains[current][i])
            continue;
        solution[current] = i;
        fail = consistent(C, n, k, current, solution[current]);
        if (!fail) {
            jump_place[current] = current - 1;
            jump = FC_BJ(C, n, k, solution, current + 1, number, found);
            if (jump != current)
                return(jump);
        }
        restore(current, n, k);
        if (fail)
            for (j = 1; j < fail; j++)
                if (checking[j][fail] && jump_place[current] < j)
                    jump_place[current] = j;
    }
    jump = jump_place[current];
    for (j = 1; j < current; j++)
        if (checking[j][current] && jump < j)
            jump = j;
/*
Replaced by three lines above.
    for (i = 1; i <= current; i++)
        if (jump < checking[i][current])
            jump = checking[i][current];
*/
    for (i = current; i >= jump; i--)
        restore(i, n, k);
    return(jump);
}
