#include "global.h"


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.
 */
static int parents[N][N];
static int P[N];
static int mcl[N][K];
static int mbl[N];


/*
 * Return whether there exists some constraint between two variables, which is
 * simple searching an edge for a "0" entry.
 */
static int constraint(con, k)
CONSTRAINT con;
int k;
{
    int i, j;

    for (i = 0; i < k; i++)
        for (j = 0; j < k; j++)
            if (con[i][j] == 0)
                return(1);
    return(0);
}


/*
 * Clear the setup data structures.  Then, compute the parent tree and store
 * this information in the data structure to be used when backjumping is to be
 * done.  See the "Dechter" paper for a full description of the data structure.
 */
static void clear_setup(C, n, k)
NETWORK C;
int n, k;
{
    int i, j, l;

    for (i = 1; i <= n; i++) {
        P[i] = 0;
        mbl[i] = 1;
        for (j = 1; j <= n; j++)
            parents[i][j] = 0;
        for (j = 0; j <= k; j++)
            mcl[i][j] = 1;
    }
    for (i = 1; i <= n; i++)
        for (j = 1; j < i; j++)
            if (constraint(C[i][j], k)) {
                for (l = 1; l < j; l++)
                    parents[i][l] = parents[i][l] || parents[j][l];
                parents[i][j] = 1;
                parents[i][i] = j;
            }
}


/*
 * Create the union of two parent sets.  This is necessary to be able to
 * continue graph-based backjumping correctly to ensure the correct variable
 * will not be missed.
 */
static int union_parents(i)
int i;
{
    int j, temp = 0;

    for (j = 1; j < i; j++) {
        P[j] = P[j] || parents[i][j];
        if (P[j])
            temp = j;
    }
    P[i] = 0;
    return(temp);
}


/*
 * Check if the current instantiation of the variables is consistent by
 * looking at the edge between the current variable and all previous variables.
 * However, this is not done if the algorithm has not backed up enough to
 * prevent a previous conflict from occurring again.  If a check is done
 * between two variables, this information is retained for later use.
 */
static int consistent(C, solution, current)
NETWORK C;
int current;
SOLUTION solution;
{
    int i;

    if (mcl[current][solution[current]] < mbl[current])
        return(0);
    for (i = mbl[current]; i < current; i++) {
        checks++;
        mcl[current][solution[current]] = i;
        if (C[current][i][solution[current]][solution[i]] == 0)
            return(0);
    }
    return(1);
}


/*
 * Solve the constraint network using the BACKMARKING with Dechter's GRAPH-BASED
 * BACKJUMPING (BM-GBJ) method.  If this is the first variable then initialize
 * the data structures.  If a solution is found then update the "found"
 * variable, call the function to process the solution, and return the value
 * according to whether the first solution is desired or all solutions.  Then,
 * check if the timer has expired, and if it has, return immediately.
 * Otherwise, begin checking each possible instantiation of the variable.  For
 * each domain value, perform the following steps.  First, if preprocessing
 * eliminated the value, disregard the rest of the loop.  Otherwise,
 * instantiate the variable, check if the network is still consistent, and then
 * call the backtracking routine recursively.  After checking all possible
 * domain values, determine the variable to jump back to (return this value),
 * and update the data structures to reflect that this is the variable being
 * returned.
 */
int BM_GBJ(C, n, k, solution, current, number, found)
NETWORK C;
int n, k, current, number, *found;
SOLUTION solution;
{
    int i, jump, curr = count;

    if (current == 1) {
        clear_setup(C, n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        count++;
        return(number == 1 ? 0 : n);
    }
    if (time_expired())
        return(0);
    for (i = 0; i < k; i++) {
        if (C[current][current][i][i] == 0)
            continue;
        solution[current] = i;
        if (consistent(C, solution, current)) {
            jump = BM_GBJ(C, n, k, solution, current + 1, number, found);
            if (jump != current)
                return(jump);
        }
    }
    jump = (curr == count) ? union_parents(current) : current - 1;
    for (i = jump + 1; i <= n; i++)
        if (mbl[i] > jump)
            mbl[i] = jump;
    mbl[current] = jump;
    return(jump);
}
