;;; -*- LISP -*-

;;.@chapter Reading A Story: An Example

(in-package :restaurant)

;;.We would like to have a system which can parse the following
;;.story; unfortunately, we need to define some more works, actions
;;.and case frames in order to do so.  As an example, we will use the
;;.following example and extend HUH to succesfully parse it.
;;.The story we use is about an uneventful visit to a restaurant:
;;...@example
(defparameter *restaurant-text*
  "John went to the restaurant.  He asked the waiter for a rare steak.
   The waiter brought the steak to John.  John ate the steak.
   John paid the money to the waiter.  John left the restaurant.")
;;...@end example
(export '(*restaurant-text*))


;;;;.Defining new nouns.

;;.When we call the function @code{read-text} on a string like this,
;;.it normally prints out each sentence as it is parsed and then
;;.evaluates the result of the parse.  When it fails, it reports the
;;.sentence it failed to parse.  Examining this sentence may tell us
;;.things about words which HUH did not recognize; for instance, in
;;.the following example HUH successfully parses the first sentence,
;;.but fails with the remainder of the text.
;;.@example
;;.> (read-text *restaurant-text*)
;;.; Successfully parsed: JOHN WENT TO THE RESTAURANT . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'BEING-MOVED)
;;.  ((INITIATOR THING-MOVED)
;;.    (HUH::CREATE-NAMED-OBJECT (GENSYMBOL 'WORDS::JOHN)
;;.                              (MEMBER-OF 'HUMANS)
;;.                              (MEMBER-OF 'MEN)
;;.                              (FIRST-NAME 'WORDS::JOHN)))
;;.  ((DESTINATION) (CREATE-OBJECT
;;.                   (MEMBER-OF 'RESTAURANT))))
;;.; Couldn't parse (WORDS::HE WORDS::ASKED WORDS::THE WAITER WORDS::FOR WORDS::A RARE STEAK WORDS::|.| .....
;;.(STORY BEING-MOVED.1 RESTAURANT.4 JOHN.1)
;;.@end example
;;.In looking the failure description (`Couldn't parse ...'), we note
;;.that the word `waiter' appears without a prefixing `WORDS::'.  This
;;.means that HUH was unable to recognize the word @code{waiter};
;;.because of this, it was unable to succesfully parse `the waiter'.
;;.We can verify this by using the function @code{parse} which
;;.takes a string and a `structure type' (e.g. `noun-phrase,'
;;.`verb-phrase,' etc) and returns the parses of the string as that
;;.sort of structure, or NIL if there are none.  Applied to the
;;.fragment, `the waiter':
;;.@example
;;.> (parse "the waiter" 'noun-phrase)
;;.NIL
;;.@end example
;;.it returns NIL, indicating that our diagnosis is probably correct.
;;.To rectify this, we define a collection of `waitrons' with subsets
;;.`waiters' and `waitresses'
;;...@example
(define-collection waitrons (humans)
  (english-class-name (register-word "WAITPERSON")))
(define-collection waiters (waitrons men)
  (supersets 'waitrons)
  (english-class-name (register-word "WAITER")))
(define-collection waitresses (waitrons women)
  (supersets 'waitrons)
  (english-class-name (register-word "WAITRESS")))
;;...@end example
;;.The crucial part of these definitions from HUH's point of view is the
;;.@code{english-class-name} slot which indicates to HUH that the noun
;;.`waiter' refers to membership in the class @code{waiters}.  To test
;;.that our definitions have worked, we call @code{parse} again:
;;.@example
;;.> (parse "the waiter" 'noun-phrase)
;;.(#S(HUH::PARSE-RESULT OBJECT (CREATE-OBJECT (MEMBER-OF (QUOTE WAITERS))) YET-TO-PARSE NIL))
;;.@end example
;;.Given this, we can try and parse our story again:
;;.@example
;;.> (read-text *restaurant-text*)
;;.; Successfully parsed: JOHN WENT TO THE RESTAURANT . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'BEING-MOVED)
;;.  ((INITIATOR THING-MOVED)
;;.    (HUH::CREATE-NAMED-OBJECT (GENSYMBOL 'WORDS::JOHN)
;;.                              (MEMBER-OF 'HUMANS)
;;.                              (MEMBER-OF 'MEN)
;;.                              (FIRST-NAME 'WORDS::JOHN)))
;;.  ((DESTINATION) (CREATE-OBJECT
;;.                   (MEMBER-OF 'RESTAURANT))))
;;.; Couldn't parse (WORDS::HE WORDS::ASKED WORDS::THE WORDS::WAITER WORDS::FOR WORDS::A RARE STEAK WORDS::|.| .....
;;.(STORY BEING-MOVED.1 RESTAURANT.4 JOHN.1)
;;.@end example
;;.and the system again fails to parse that second sentence.  Looking
;;.again for words not prefixed by @code{WORDS::}, we see that it
;;.doesn't recognize the phrase `rare steak'.  To remedy this, we go a
;;.little bit overboard and give it an entire vocabulary of
;;.foodstuffs.
;;.Here we use the @code{define-collection} form to construct a
;;.collection; this simply expands into a @code{define-unit} which
;;.includes @code{(member-of 'collections)} and a @code{(supersets
;;.'things)} clauses.  We also assert that this collection is --- with
;;.some cultural bias --- exclusive of agents and money; we try not to
;;.eat intelligent creatures nor use food as a medium of exchange.
;;...@example
(define-collection foodstuff (things)
  (english-class-name 'food)
  ; Culturally biased, I know.
  (mutually-excludes 'agents)
  (mutually-excludes 'money))

;;.Foodstuffs are specialized in @code{meat}, @code{seafood},
;;.@code{vegetable}, and @code{starch} with corresponding english
;;.class names.
(define-collection meat (foodstuff)
  (english-class-name 'meat))
(define-collection seafood (foodstuff)
  (english-class-name 'seafood)
  (mutually-excludes 'meat))
(define-collection vegetable (foodstuff)
  (english-class-name 'vegetable)
  (mutually-excludes 'seafood)
  (mutually-excludes 'meat))
(define-collection starch (foodstuff)
  (english-class-name 'starch)
  (mutually-excludes 'seafood)
  (mutually-excludes 'meat)
  (mutually-excludes 'vegetable))

;;.Particular sorts of foods are steak, cauliflower, eggplant, fish,
;;.and lobster; we are going through all of this rigamarole to define
;;.`steak' for the sake of our example...
(define-collection steak (meat)
  (english-class-name 'steak))
(define-collection cauliflower (vegetable)
  (english-class-name 'cauliflower))
(define-collection eggplant (vegetable)
  (english-class-name 'eggplant)
  (mutually-excludes 'cauliflower))
(define-collection fish (seafood)
  (english-class-name 'fish))
(define-collection lobster (seafood)
  (english-class-name 'lobster)
  (mutually-excludes 'fish))
;;...@end example

;;.We can know parse ``the steak'':
;;.@example
;;.> (parse "the steak" 'noun-phrase)
;;.(#S(HUH::PARSE-RESULT OBJECT (CREATE-OBJECT (MEMBER-OF (QUOTE RESTAURANT:STEAK))) YET-TO-PARSE NIL))
;;.@end example
;;.yet ``the rare steak'' is still a stumper:
;;.@example
;;.> (parse "the rare steak" 'noun-phrase)
;;.NIL
;;.@end example

;;;;.Defining new adjectives

;;.Given a definition of @code{steak}, we still need to define the
;;.adjective @code{rare}; we could do this by defining a collection
;;.@code{degrees-of-doneness} and a slot @code{doneness}:
;;...@example
(define-collection degrees-of-doneness ()
  (members 'rare) (members 'well-done)
  ;; Members of questionable pedigree...
  (members 'bloody) (members 'mooing) (members 'charred))

(define-slot doneness single-valued-slots
     meat degrees-of-doneness
  (english-description "This is the degree of doneness of a meat."))
;;...@end example
;;.And defining a corresponding adjective with the form @code{define-adjective}:
;;...@example
(define-adjective "rare" doneness rare)
;;...@end example

;;.Now, when we parse the story, we get further:
;;.@example
;;.> (read-text *restaurant-text*)
;;.; Successfully parsed: JOHN WENT TO THE RESTAURANT . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'BEING-MOVED)
;;.  ((INITIATOR THING-MOVED)
;;.    (HUH::CREATE-NAMED-OBJECT (GENSYMBOL 'WORDS::JOHN)
;;.                              (MEMBER-OF 'HUMANS)
;;.                              (MEMBER-OF 'MEN)
;;.                              (FIRST-NAME 'WORDS::JOHN)))
;;.  ((DESTINATION) (CREATE-OBJECT
;;.                   (MEMBER-OF 'RESTAURANT))))
;;.; Successfully parsed: HE ASKED THE WAITER FOR A RARE STEAK . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'DEMANDING)
;;.  ((SPEAKER) 'JOHN.3)
;;.  ((LISTENER) (CREATE-OBJECT
;;.                (MEMBER-OF 'WAITERS)))
;;.  ((DEMANDED-OBJECT) (CREATE-OBJECT
;;.                       (MEMBER-OF 'STEAK)
;;.                       (DONENESS 'RARE))))
;;.; Successfully parsed: THE WAITER BROUGHT THE STEAK TO JOHN . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'GIVING)
;;.  ((GIVER) 'WAITERS.1)
;;.  ((THING-GIVEN) 'STEAK.1)
;;.  ((RECIPIENT) 'JOHN.3))
;;.; Couldn't parse (WORDS::JOHN ATE WORDS::THE WORDS::STEAK WORDS::|.| WORDS::JOHN WORDS::PAID  WORDS::THE ....
;;.(STORY GIVING.1 DEMANDING.1 STEAK.1 WAITERS.1 BEING-MOVED.3 RESTAURANT.6 JOHN.3)
;;.@end example
;;.Here we parse the second sentence, and our work in defining slabs
;;.of beef got us the additional mileage of parsing the third sentence
;;.without problems.  We note here that the problem is that HUH does
;;.not recognize the word @code{ate}; to tell HUH about this word, we
;;.must first introduce the categories of actions corresponding to eating.


;;;;.Defining new verbs and actions

;;.We define @code{ate} by defining the infinitive `eat' and its past tense:
;;...@example
(huh::define-word "eat"
    (member-of 'infinitives)
  (past-tense (huh::register-word "ate")))
;;...@end example

;;.The action of @code{eating} we take as merely the physical motion
;;.of food into ones stomach; note that this ignores the nutritive
;;.value of the food being eaten.  From HUH's point of view, the
;;.crucial thing about the definition of @code{eating} is its
;;.@code{english-cases} slot which indicates that the verb @code{eat}
;;.denotes an eating action and the subject of the verb denotes the
;;.initiator and the object denotes the thing eaten.
;;...@example
(define-unit eating
  (member-of 'collections)
  (supersets 'being-moved)
  (english-cases '(words::eat (eater) (thing-eaten))))
;;...@end example

;; An important accessory for eating.
(define-unit stomach
  (member-of 'single-valued-slots)
  (makes-sense-for 'animals))

;;.The slots @code{eater} and @code{eaten} are defined for eating
;;.actions; the eater must be an animal (Elijah Bailey to the
;;.contrary) and since eating is a moving action, we indicate that the
;;.eater's stomach is the destination of the movement.
;;...@example
(define-unit eater
  (member-of 'single-valued-slots)
  (makes-sense-for 'eating)
  (must-be 'animals)
  (genl-slots 'initiator)
  (structure '(stomach destination)))
;;...@end example
;;.We also note that the thing moved in eating is the
;;.@code{thing-eaten} (the genl-slot of @code{thing-eaten} is
;;.@code{thing-moved}) and that people only eat foods (a
;;.quasi-falsehood asserted by the @code{must-be} slot of
;;.@code{thing-eaten} being @code{foodstuffs}).
;;...@example
(define-unit thing-eaten
  (member-of 'single-valued-slots)
  (makes-sense-for 'eating)
  (must-be 'foodstuff)
  (genl-slots 'thing-moved))
;;...@end example

;;.Given these definitions, we resume our attempt to read the text.
;;.Note that we begin in the middle of parsing the story, rather than
;;.repeating all of the text and descriptions above.
;;.@example
;;.; Successfully parsed: THE WAITER BROUGHT THE STEAK TO JOHN . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'GIVING)
;;.  ((GIVER) 'WAITERS.1)
;;.  ((THING-GIVEN) 'STEAK.1)
;;.  ((RECIPIENT) 'JOHN.3))
;;.; Successfully parsed: JOHN ATE THE STEAK . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'EATING) ((EATER) 'JOHN.4) ((THING-EATEN) 'STEAK.2))
;;.; Couldn't parse (WORDS::JOHN PAID WORDS::THE MONEY WORDS::TO WORDS::THE WORDS::WAITER WORDS::|.| ....
;;.(STORY EATING.1 GIVING.2 DEMANDING.2 STEAK.2 WAITERS.2 BEING-MOVED.4 RESTAURANT.7 JOHN.4)
;;.@end example

;;.It now handles the eating, but balks at the sentence describing the
;;.payment; we note that the symbol @code{PAID} lacks the prefixing
;;.@code{WORDS::} which suggests (as above) that HUH doesn't recognize
;;.the word.  To support this, we must construct classes actions denoting
;;.payment and the corresponding case frames and slots.

;;.Defining the verb `paid' is simple:
;;...@example
(huh::define-word "pay"
    (member-of 'infinitives)
  (past-tense (huh::register-word "paid")))
;;...@end example
;;.We have it map into a new sort of action: @code{paying}.
;;...@example
(define-unit paying
  (member-of 'collections)
  (supersets 'giving)
  (english-cases
   `(words::pay (giver) (thing-given) (recipient)
     (to recipient) (for payment-for))))
;;...@end example
;;.The collection of `paying' actions is a subset of the collection of
;;.`giving' actions; paying is a kind of giving.  Giving is defined in
;;.ARLOtje's initial ontology.  The object of @code{paying} is the
;;.@code{thing-given}, the subject is the @code{giver}, and the
;;.indirect object is the recipient.  The prepositions `to' and `for'
;;.refer to the recipient and the goods the payment is recieved for.
;;.@code{Recipient} is defined as part of @code{giving} in ARLOtje's
;;.base ontology; @code{payment-for} is defined especially for payments:
;;...@example
(define-unit payment-for
  (works-like 'prototypical-slot)
  (makes-sense-for 'paying)
  (must-be 'actions))
(define-unit paid-for-in
  (works-like 'prototypical-slot)
  (makes-sense-for 'actions)
  (must-be 'paying)
  (inverse-slot 'payment-for))
;;...@end example
;;.with its inverse @code{paid-for-in} going in the other direction;
;;.parsing the sentence now yields the response 
;;.@example
;;.; Couldn't parse (WORDS::JOHN WORDS::PAID WORDS::THE MONEY WORDS::TO WORDS::THE WORDS::WAITER WORDS::|.| ....
;;.(STORY EATING.2 GIVING.3 DEMANDING.3 STEAK.3 WAITERS.3 BEING-MOVED.5 RESTAURANT.8 JOHN.5)
;;.@end example
;;.revealing that HUH has a problem with money (who doesn't?) which in
;;.HUH's case can be easily resolved by:
;;...@example
(define-unit money
  (member-of 'collections)
  (supersets 'artifacts)
  (mutually-excludes 'processes)
  (english-class-name 'money)
  (english-class-name 'cash))
;;...@end example
;;.which allows us to finally parse:
;;.@example
;;.> (read-text *restaurant-text*)
;;.; Successfully parsed: JOHN WENT TO THE RESTAURANT . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'BEING-MOVED)
;;.  ((INITIATOR THING-MOVED)
;;.    (HUH::CREATE-NAMED-OBJECT (GENSYMBOL 'WORDS::JOHN)
;;.                              (MEMBER-OF 'HUMANS)
;;.                              (MEMBER-OF 'MEN)
;;.                              (FIRST-NAME 'WORDS::JOHN)))
;;.  ((DESTINATION) (CREATE-OBJECT
;;.                   (MEMBER-OF 'RESTAURANT))))
;;.; Successfully parsed: HE ASKED THE WAITER FOR A RARE STEAK . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'DEMANDING)
;;.  ((SPEAKER) 'JOHN.3)
;;.  ((LISTENER) (CREATE-OBJECT
;;.                (MEMBER-OF 'WAITERS)))
;;.  ((DEMANDED-OBJECT) (CREATE-OBJECT
;;.                       (MEMBER-OF 'STEAK)
;;.                       (DONENESS 'RARE))))
;;.; Successfully parsed: THE WAITER BROUGHT THE STEAK TO JOHN . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'GIVING)
;;.  ((GIVER) 'WAITERS.1)
;;.  ((THING-GIVEN) 'STEAK.1)
;;.  ((RECIPIENT) 'JOHN.3))
;;.; Successfully parsed: JOHN ATE THE STEAK . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'EATING) ((EATER) 'JOHN.4) ((THING-EATEN) 'STEAK.2))
;;.; Successfully parsed: JOHN PAID THE MONEY TO THE WAITER . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'PAYING)
;;.  ((GIVER) 'JOHN.7)
;;.  ((THING-GIVEN) (CREATE-OBJECT
;;.                   (MEMBER-OF 'MONEY)))
;;.  ((RECIPIENT) 'WAITERS.5))
;;.; Successfully parsed: JOHN LEFT THE RESTAURANT . 
;;.; Translation to LISP:
;;.(CREATE-OBJECT
;;.  (MEMBER-OF 'PAYING)
;;.  ((GIVER) 'JOHN.7)
;;.  ((THING-GIVEN) (CREATE-OBJECT
;;.                   (MEMBER-OF 'MONEY)))
;;.  ((RECIPIENT) 'WAITERS.5))
;;.(STORY BEING-MOVED.8 PAYING.1 MONEY.1 EATING.4 GIVING.5 DEMANDING.5 STEAK.5
;;.       WAITERS.5 BEING-MOVED.7 RESTAURANT.10 JOHN.7)
;;.@end example
;;.successfully parsing the whole story.  Of course, understanding what
;;.has been said --- though difficult --- is only a portion of the
;;.problem.  The harder part is resolving what wasn't said!
