;;; -*- LISP -*-

(in-package :huh)

;;.@chapter Generating Text

;;.ARLOtje has a very simple generator which inverts the case frame
;;.knowledge about events and their rendering with verbs.  The central
;;.user-level functions of the generator are @code{say} and
;;.@code{answer}.

;;.The procedure @code{say} takes an ARLOtje object and outputs an
;;.english description of it to a stream (specified by the
;;.@code{:stream} keyword); if the supplied stream is @code{nil},
;;.@code{say} returns the english text as a string.

;;.The procedure @code{answer} describes an event (an ARLOtje
;;.description of a process) with a descriptive focus on a particular
;;.property of the event; for instance, suppose @code{BRINGING.1}
;;.describes the event of John giving a book to Bob:
;;.@example
;;.(answer 'bringing.1 'initiator)
;;.John gave the book to Bob
;;.(answer 'bringing.1 'destination)
;;.Bob got the book from John.
;;.@end example
;;.If @code{answer} is given a third argument, its output is preceded
;;.by a `yes' or `no' depending on whether the corresponding slot
;;.value actually is the second value.  For instance,
;;.@example
;;.(answer 'bringing.1 'initiator 'John)
;;.Yes, John gave the book to Bob
;;.(answer 'bringing.1 'initiator 'Bob)
;;.No, John gave the book to Bob
;;.@end example

;;.This generator is very simpled minded and occasionally will produce
;;.funny phrasings; improvements, as ever, will be more than welcome.
;;.It does nothing else special in terms of focus than the emphasis
;;.introduced by @code{answer}.  It's version of providing word
;;.variety is that when there are multiple choices, one is chosen at
;;.random.


;;;;.The top level: SAY

;;.The top level function for producing english text is @code{SAY}
;;.which takes an ARLOtje object and a variety of keyword arguments.
;;.The @code{:stream} argument indicates a stream to which @code{SAY}
;;.directs its output; if @code{nil}, @code{say} returns a string.
;;.The @code{:annotated?} keyword indicates that the object itself
;;.should be provided (in parentheses) following the argument.  The
;;.@code{:preamble} and @code{:punctuation} keywords identify strings
;;.which precede and follow the output; if there is some punctuation
;;.but no preamble, @code{SAY} takes the initiative of capitalizing
;;.the sentence.

;;.For example, a unit @code{being-moved.1} (described in examples
;;.below) might produce the following output from @code{SAY}.
;;.@example
;;.> (say 'being-moved.1)
;;.John went to the restaurant by the taxi. (BEING-MOVED.1)
;;.NIL
;;.> (say 'being-moved.1 :annotated? NIL)
;;.John left to the restaurant by the taxi.
;;.NIL
;;.> (say 'being-moved.1 :preamble "Yikes,")
;;.Yikes, John went to the restaurant by the taxi. (BEING-MOVED.1)
;;.NIL
;;.> (say 'being-moved.1 :punctuation "!!")
;;.John took to the restaurant by the taxi!! (BEING-MOVED.1)
;;.NIL
;;.> (say 'being-moved.1 :stream NIL)
;;."John left to the restaurant by the taxi. (BEING-MOVED.1)"
;;.@end example
;;.The variation in phrasings reflect HUH's random selection of case
;;.frames from those superfically approriate for a description.

;;.The variable @code{*annotated-generated-text*} determines whether
;;.text is annotated by default (e.g. it provides the default value
;;.for the @code{:annotated?} keyword.)
(defparameter *annotate-generated-text* T
  "If set to T, things which are `said' have their output annotated with their representations.")

(defun say (thing &rest keywords
		  &key  (stream *standard-output*) preamble (punctuation ".")
		  (annotated? huh::*annotate-generated-text*)
		  &allow-other-keys)
  (let ((sentence
	 (cond ((satisfies? thing 'things)
		(word-list->string (object->noun-phrase thing)))
	       ((satisfies? thing 'processes)
		(apply #'event->sentence thing keywords))
	       (T (word-list->string (object->noun-phrase thing))))))
    (unless (or preamble (null punctuation))
      (setf (char sentence 0) (char-upcase (char sentence 0))))
    (format stream "~&~@[~A ~]~A~@[~A~]~@[ (~S)~]"
	    preamble sentence punctuation (and annotated? thing))))

(defun random-ref (list)
  "Returns a random element of LIST."
  (nth (random (length list)) list))
;;.@findex{random-ref}


;;;;.Descriptions to text

;;.ARLOtje units which do not describe processes are handled by the
;;.function @code{object->noun-phrase}; it handles proper names and
;;.first names of people and uses the @code{english-class-name} slot
;;.of collections to produce constructions like `the dog' or `the
;;.waiter'.  If all else fails, it renders the paltry `the unit FOO.3'
;;.as a description.  

(defun object->noun-phrase (thing)
  "Takes an object and renders a noun phrase describing it."
  (cond ((or (null thing) (failurep  thing)) '())
	((listp thing) (object->noun-phrase (car thing)))
	((satisfies? thing 'words)
	 `(WORDS::THE WORDS::WORD ,(string-downcase (symbol-name thing))))
	((get-value thing 'proper-name) (list (random-ref (get-value thing 'proper-name))))
	((and (satisfies? thing 'humans) (not (failurep (get-value thing 'first-name))))
	 (list (make-symbol (string-capitalize (symbol-name (get-value thing 'first-name))))))
	(T (docollections (class thing `(WORDS::THE WORDS::UNIT ,thing))
	     (unless (null (get-value class 'english-class-name))
	       (return `(WORDS::THE ,(random-ref (get-value class 'english-class-name)))))))))
;;.@findex{object->noun-phrase}

;;.For instance, it would yield the following results:
;;.@example
;;.(define-unit george.3
;;.  (member-of 'men)
;;.  (first-name (register-word "George"))
;;.  (last-name (register-word "Washington")))
;;.(object->noun-phrase 'george.3)
;;.===> (#:|George|) ; It returns an uninterned symbol to preserve case information.
;;.(define-unit washington-dc
;;.  (member-of 'places)
;;.  (proper-name "Washington D.C.")
;;.  (proper-name "the Nation's Capitol"))
;;.(object->noun-phrase 'washington-dc)
;;.===> (WORDS::|WASHINGTON D.C.|)
;;.(object->noun-phrase 'washington-dc)
;;.===> (WORDS::|WASHINGTON D.C.|)
;;.(object->noun-phrase 'washington-dc)
;;.===> (WORDS::|THE NATION'S CAPITOL|)
;;.@end example

;;.Events (processes) are rendered into sentences by first finding an
;;.appropriate case frame and then applying this case frame to the
;;.sentence to return a list identifying subject, object, indirect
;;.object, and prepositions.  The procedure
;;.@code{event->sentence-props} yields this description.
(defun event->sentence-props
    (event &key focus (case-frame (choose-case-frame event focus)) &allow-other-keys)
  "Takes an event and turns it into a a-list of sentence properties."
  (let ((subject-slots (cadr case-frame)) (object-slots (caddr case-frame))
	(indirect-object-slots (cadddr case-frame))
	(preposition-slots (cddddr case-frame))
	(objects-mentioned '()))
    (flet ((has-value (x slot)
	     (and slot (not (failurep (get-value x slot)))
		  (let ((value (get-value x slot)))
		    (unless (member (get-value x slot) objects-mentioned)
		      (push value objects-mentioned)
		      value)))))
      `(,(car case-frame)
	,(has-value event (car subject-slots))
	,(has-value event (car object-slots))
	,(has-value event (car indirect-object-slots))
	,@(remove NIL
	   (mapcar #'(lambda (x)
		       (some #'(lambda (slot)
				 (let ((value (has-value event slot)))
				   (if value `(,(car x) ,value))))
			     (cdr x)))
	    preposition-slots))))))
;;.@findex{event->sentence-props}

;;;.For instance, the following responses would be yielded (here we
;;;.use the parser to build the descriptions).
;;.@example
;;.(read-text "John went to the restaurant by taxi.")
;;.(HUH::STORY BEING-MOVED.1 TAXI.1 RESTAURANT.4 JOHN.1)
;;.
;;.> (du 'BEING-MOVED.1)
;;.
;;.-------------------------------------------------------------------------------
;;.  Slot              Value
;;.  ----              -----
;;.  INSTRUMENT        TAXI.1
;;.  VEHICLE           TAXI.1
;;.  CONTAINS          (RESTAURANT.4)
;;.  DESTINATION       RESTAURANT.4
;;.  PARTICIPANTS      (JOHN.1)
;;.  THING-MOVED       JOHN.1
;;.  OVERLAPS          (JOHN.1 TAXI.1)
;;.  INITIATOR         JOHN.1
;;.  MEMBER-OF         (BEING-MOVED MOVING-PROCESSES              STATE-CHANGE...
;;.  TYPE-CODE         517006786560
;;.  CREATION-ID       ARLOTJE::LISZT-13.36.57-31NOV1990
;;.-------------------------------------------------------------------------------
;;.BEING-MOVED.1
;;.> (huh::event->sentence-props 'being-moved.1)
;;.(WORDS::GO JOHN.1 NIL NIL (HUH::TO RESTAURANT.4) (HUH::BY TAXI.1))
;;.@end example
;;.with the final returned value meaning `make a sentence with the
;;.verb `go' with subject @code{JOHN.1}, and TO preposition of
;;.@code{RESTAURANT.4} and a BY preposition of @code{TAXI.1}.

;;.Case frames are selected by the function @code{choose-case-frame}
;;.whose first argument is an event; the optional second argument is a
;;.@var{focus} and the function will select case frames for which the
;;.focus would be the subject.

(defun choose-case-frame (event &optional focus)
  "Selects a case frame to describe an event.
 A case frame is chosen by looking at all applicable case frames
 (indexed by collection) and dividing them into good and okay cases.
 A good case frame has a focus which agrees with the FOCUS given
 as the second (optional) argument to CHOOSE-CASE-FRAME."
  (let ((okay-cases '()) (good-cases '()))
    (docollections (class event ())
      (when (and (satisfies? class 'collections) (query class 'supersets 'processes))
	(aj::do-members (case class 'english-cases)
	  (when (suitable-case? case event)
	    (push case okay-cases)
	    (when (and focus (focus? event case focus))
	      (push case good-cases))))))
    (if good-cases (random-ref good-cases)
      (and okay-cases (random-ref okay-cases)))))

(defun focus? (event case-frame focus)
  "Returns true if FOCUS is the subject of EVENT described with CASE-FRAME."
  (let ((subject-slot (car (cadr case-frame))))
    (eq (get-value event subject-slot) focus)))

(defun suitable-case? (case event)
  "Returns true if CASE is a good match for EVENT."
  (every #'(lambda (entry)
	     (or (null entry)
		 (let* ((slot (car entry)) (value (get-value event slot)))
		   (or (failurep value)
		       (every #'(lambda (sl) (query event sl value)) (cdr entry))))))
	 (list* (cadr case) (caddr case) (cadddr case)
		(mapcar #'cdr (cddddr case)))))

;;.The sentence props returned by @code{event->sentence-props} can be
;;.transformed into a list of words by
;;.@code{sentence-props->word-list}; this primarily expands the
;;.component objects into noun phrases (or, if nominalizations are
;;.implemented, action descriptions).
(defun sentence-props->word-list (sentence-props)
  "Flattens a list of sentence properties into a word list."
  (flet ((sget (x) (cadr (assoc x sentence-props))))
    `(,@(object->noun-phrase (cadr sentence-props))
      ,(if (failurep (get-value (car sentence-props) 'past-tense-of))
	   (get-value (car sentence-props) 'past-tense)
	 (car sentence-props))
      ,@(object->noun-phrase (cadddr sentence-props))
      ,@(object->noun-phrase (caddr sentence-props))
      ,@(apply #'append
	 (mapcar #'(lambda (cl) `(,(car cl) ,@(object->noun-phrase (cadr cl))))
	  (cddddr sentence-props))))))

;;.For instance, given the structure returned above, we can do.
;;.@example
;;.> (huh::event->sentence-props 'being-moved.1)
;;.(WORDS::GO JOHN.1 NIL NIL (HUH::TO RESTAURANT.4) (HUH::BY TAXI.1))
;;.> (huh::sentence-props->word-list
;;.   '(WORDS::GO JOHN.1 NIL NIL (HUH::TO RESTAURANT.4) (HUH::BY TAXI.1)))
;;.(#:|John| WORDS::WENT HUH::TO WORDS::THE WORDS::RESTAURANT HUH::BY WORDS::THE WORDS::TAXI)
;;.@end example
;;.This returned word list is transformed into a string by the
;;.function @code{word-list->string}; the word list contains
;;.uninterned symbols (like @code{#:|John|}) when case is important.
;;.It also takes explicit strings in the input stream as direct
;;.strings to be output.

(defun word-list->string (word-list)
  "Yields a pretty string from a word list."
  (let ((string-list
	 (mapcar #'(lambda (word)
		     (if (stringp word) (format NIL "~S" word)
		       (if (and (satisfies? word 'words) (get-value word 'proper-name-of))
			   (string-capitalize (symbol-name word))
			 (if (null (symbol-package word)) (symbol-name word)
			   (string-downcase (symbol-name word))))))
		 word-list)))
    (format NIL "~A~{ ~A~}" (car string-list) (cdr string-list))))

;;.The word list generated above is transformed thus:
;;.@example
;;.(#:|John| WORDS::WENT HUH::TO WORDS::THE WORDS::RESTAURANT HUH::BY WORDS::THE WORDS::TAXI)
;;.(huh::word-list->string
;;. '(#:|John| WORDS::WENT HUH::TO WORDS::THE WORDS::RESTAURANT HUH::BY WORDS::THE WORDS::TAXI))
;;.==> "John went to the restaurant by the taxi"
;;.@end example
(defun event->sentence (event &rest keywords)
  (let* ((props (apply #'event->sentence-props event keywords))
	 (word-list (if (null (car props))
			(object->noun-phrase event)
		      (sentence-props->word-list props))))
    (word-list->string word-list)))


;;;;.Answering Questions

;;.HUH answers questions by an interaction with the discourse
;;.reference mechanisms; the function @code{answer} takes a list of
;;.properties as a `topic' and tries to find an existing or expected
;;.description having a particular property.  It then yields a
;;.description of the topic focussing on that property.  If given a
;;.third argument, it replies with an answer to a yes/no question
;;.regarding the identity of the requested property with the third
;;.argument.  For instance, suppose that the discourse context
;;.contained a reference to @code{SHOOTING.1} an instance of
;;.@code{SHOOTING} (not a default part of ARLOtje's ontology) whose
;;.@code{target} was @code{JR-EWING}; the question `Who shot J.R.?'
;;.would be translated into the following @code{answer} expression:
;;.@example
;;.(answer '((member-of 'shooting) (target 'jr-ewing)) 'initiator)
;;.@end example
;;.while a question like `Did the cameraman shoot J.R.?' would be
;;.transformed into an expression like
;;.@example
;;.(answer '((member-of 'shooting) (target 'jr-ewing)) 'initiator
;;.        'cameraman.1)
;;.@end example

;;.The @code{answer} procedure first calls the function
;;.@code{find-reference} (described below) on its first argument; if
;;.this fails, it asserts its identity by the useful reply `Huh?'.  If
;;.it succeeds, it isolates a @emph{focus} by getting the
;;.@var{property} of the returned reference (@var{property} is the
;;.second argument to @code{answer}).  If there is no such property,
;;.it sends out another call to @code{find-reference} looking for an
;;.appropriate filler for the slot.  This is designed to interact with
;;.some expectation mechanism to yield expected (but unmentioned)
;;.fillers for particular slots.  If a focus is found, it is passed to
;;.@code{say} in describing the match found for @code{SAY}'s first
;;.argument.  Finally, if the @var{value} argument to @code{answer}
;;.has been specified, the preamble is preceded by a `yes' or `no'
;;.depending on whether the focus matches the expected value.

;;.One additional feature of @code{answer} is that units which have
;;.the slot @code{putative} are described in cautious language; for
;;.instance, `Yes, I @emph{assume} that the cameraman shot J.R.?'
;;.Various expectation mechanisms may choose to create objects which
;;.have a @code{putative} property assigned to them to indicate that
;;.they were not @emph{actually} mentioned.

(defun answer (topic property &optional value)
  (let ((about (find-reference topic)))
    (if (null about) (format t "~&Huh?")
      (let ((focus (get-value about property)))
	(when (failurep focus)
	  (setq focus
		(find-reference `((member-of ',(get-value property 'must-be))
				  (,(get-value property 'inverse-slot) ',about)))))
	(if (or (null focus) (not (match? about topic)))
	    (say about :preamble "Huh? All I know is that")
	  (if (null value)
	      (say about :focus focus
		   :preamble (and (get about 'putative) "I assume that"))
	    (if (equal focus value)
		(say about :focus focus
		     :preamble (if (get about 'putative)
				   "Yes, I assume that" "Yes,"))
	      (say about :focus focus
		   :preamble (if (get about 'putative)
				 "No, I assume that" "No,")))))
	(list about focus)))))

;;.As described above, much of the `figuring' out work of
;;.@code{answer} is done by @code{find-reference}.  This function
;;.takes the list of properties and first evaluates all of the right
;;.hand sides; it then looks for any discourse references which fit
;;.the given properties.  If any such references exist, the latest one
;;.is returned (this is a naive mechanism).  If no existing references
;;.exist, a `guess' (named @code{GUESS.}something) is created with the
;;.appropriate properties and the object is @emph{introduced} with the
;;.function @code{introduce!} which is supposed to find some fitting
;;.expectation and install the corresponding inferences.  If
;;.@code{INTRODUCE!} returns non-nil, @code{find-reference} checks
;;.that the `understood reference' matches the original specification;
;;.if so, it is returned.  If @code{INTRODUCE!} returns NIL or the
;;.object returned doesn't match the originally specified properties,
;;.the properties of the guess are retracted (to keep things clean)
;;.and NIL is returned by @code{find-reference}.

(defun find-reference (props)
  "Finds or creates (based on expectations) a reference fitting PROPS."
  (let* ((props (mapcar #'(lambda (x) `(,(car x) ',(eval (cadr x)))) props))
	 (known (find-discourse-reference props)))
    (if known (car known)
      (let ((object (eval `(extend-object (make-unit (gensymbol 'guess) (putative T))
			    ,@props))))
	(multiple-value-bind (object expected?) (introduce! object)
	  (if (and expected?
		   (every #'(lambda (prop)
			      (if (listp (car prop))
				  (every #'(lambda (sl) (query object sl (cadr (cadr prop))))
					 (car prop))
				(query object (car prop) (cadr (cadr prop)))))
			  props))
	      (extend-object object (putative T))
	    (progn (dovalues object (slot value)
		     (retraction object slot value))
		   NIL)))))))



