;Welcome to the ARLOtje/LISP code section of our show.
;The following code sets up the knowledge base for the Mary Chung scenario.
;First a domain of "comestibles" is established.
;This domain consists of simple edible substances
;which are then "compounded" into more complex edible substances.
;The goal here is to be able to reduce any food down to its base ingredients
;so that dietary restrictions inferences can be exact.
;Then the domain of dietary restrictions is represented by dividing up the
;various reasons people have for restricting their diets into religious-practices,
;moral-stances, and health-conditions.  Relevant examples for each of these reasons
;are defined with their corresponding dietary restrictions 
;(e.g. religious-practice "buddhist" has dietary-restrictions "animal-products,
;animal-by-products and alcohol").
;Then (about a third of) the Mary-Chung-menu itself is represented with its various dishes.
;Finally, a unit called "my-dinner-party" is defined which allows one to bring together
;a group of guests and infer, based on their dietary restrictions and in relation to a 
;given menu (Mary Chung's being the current example), which dishes they can all share.


;To avoid name conflicts with the arlotje package around such base things as "meat"
;I define a package called "comestibles" which makes use of the arlotje and lisp packages. 
;A package is roughly a "symbol space" in which the values of the various lisp symbols 
;are internal to it.  It is like a boundaried name space which can communicate with 
;other boundaried name spaces "packages" by importing and exporting symbols.
(defpackage :comestibles (:nicknames cmst) (:use lisp arlotje))

;The package "comestibles" is loaded.
(in-package :comestibles)

;The test function "collectionp" is imported from the arlotje package.
(import '(aj::collectionp) 'comestibles)

;The collection "comestibles" is defined.
(define-unit comestibles
  (member-of 'collections))

;The test function "kind-of-comestiblep" is defined which tests whether a given unit which
;is a collection itself has the collection "comestibles" as its superset.
(deffcn kind-of-comestiblep (type)
  (annotations (works-like prototypical-test))
  (and (unitp type) (aj::collectionp type)
       (query type 'supersets 'comestibles)))

;The slot "comestible-components" is defined of which "ingredients," 
;"processed-elements" and "compound-elements" are spec-slots.  The slot
;"comestible-components" is also a transitive-slot.  This will be used later to
;create a list of all the comestible-components of a given dish.
(define-unit comestible-components
  (works-like 'transitive-slot)
  (makes-sense-for 'collections)
  (must-be 'collections))

;The slot "all-comestible-components" is defined which is the kleene-star of comestible-components, 
;and expands out the list of a given dish's comestible-components to include the most basic 
;comestible-components of which its comestible-components are composed.
(define-unit all-comestible-components
  (works-like 'prototypical-cached-slot)
  (kleene-star-of 'comestible-components))

;The following code sets up the collection of edible animal products.
(define-unit animal-products
  (member-of 'collections)
  (supersets 'comestibles))

;The collection of meat is defined.
(define-unit meat
  (member-of 'collections)
  (supersets 'animal-products))

;The various kinds of meat are set up as collections.
(define-unit pig-meat
  (member-of 'collections)
  (supersets 'meat))

(define-unit pork
  (member-of 'collections)
  (supersets 'pig-meat))

(define-unit piglet
  (member-of 'collections)
  (supersets 'pig-meat))

(define-unit cow-meat
  (member-of 'collections)
  (supersets 'meat))

(define-unit beef
  (member-of 'collections)
  (supersets 'cow-meat))

(define-unit veal
  (member-of 'collections)
  (supersets 'cow-meat))

(define-unit sheep-meat
  (member-of 'collections)
  (supersets 'meat))

(define-unit mutton
  (member-of 'collections)
  (supersets 'sheep-meat))

(define-unit lamb
  (member-of 'collections)
  (supersets 'sheep-meat))

;The collection of seafood is defined.
(define-unit seafood
 (member-of 'collections)
 (supersets 'animal-products))

;The various kinds of seafood are set up as collections.
(define-unit fish
  (member-of 'collections)
  (supersets 'seafood))

(define-unit fish-with-fins-and-scales
  (member-of 'collections)
  (supersets 'fish))

(define-unit fish-without-fins-and-scales
  (member-of 'collections)
  (supersets 'fish))

(define-unit shellfish
  (member-of 'collections)
  (supersets 'seafood))

(define-unit lobster
  (member-of 'collections)
  (supersets 'shellfish))

(define-unit crab
  (member-of 'collections)
  (supersets 'shellfish))

(define-unit shrimp
  (member-of 'collections)
  (supersets 'shellfish))

(define-unit clams
  (member-of 'collections)
  (supersets 'shellfish))

(define-unit oysters
  (member-of 'collections)
  (supersets 'shellfish))

(define-unit scallops
  (member-of 'collections)
  (supersets 'shellfish))

(define-unit squid
  (member-of 'collections)
  (supersets 'shellfish))

;The collection of poultry is defined.
(define-unit poultry
  (member-of 'collections)
  (supersets 'animal-products))

;The various kinds of poultry are set up as collections.
(define-unit chicken
  (member-of 'collections)
  (supersets 'poultry))

(define-unit duck
  (member-of 'collections)
  (supersets 'poultry))

(define-unit turkey
  (member-of 'collections)
  (supersets 'poultry))

(define-unit pheasant
  (member-of 'collections)
  (supersets 'poultry))

(define-unit quail
  (member-of 'collections)
  (supersets 'poultry))

(define-unit cornish-game-hen
  (member-of 'collections)
  (supersets 'poultry))

;The following code sets up the domain of edible animal by-products, 
;i.e. edible products made by animals such as milk and eggs.
(define-unit animal-by-products
  (member-of 'collections)
  (supersets 'comestibles))

;The collection of poultry-by-products is defined.
(define-unit poultry-by-products
  (member-of 'collections)
  (supersets 'animal-by-products))

(define-unit eggs
  (member-of 'collections)
  (supersets 'poultry-by-products))

;The collection of dairy-by-products is defined.
(define-unit dairy-products
  (member-of 'collections)
  (supersets 'animal-by-products))

(define-unit milk
  (member-of 'collections)
  (supersets 'dairy-products))

;The following code sets up the domain of edible plant products.
(define-unit plant-products
  (member-of 'collections)
  (supersets 'comestibles))

;The collection of vegetables is defined.
(define-unit vegetables
  (member-of 'collections)
  (supersets 'plant-products))

;The collection of vegetables is subdivided into those vegetables which grow above ground.
(define-unit above-ground-vegetables
  (member-of 'collections)
  (supersets 'vegetables))

(define-unit bamboo-shoots
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit broccoli
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit cabbage
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit green-beans
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit hot-peppers
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit pea-pods
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit cucumbers
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit baby-corn
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit sugar-cane
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

;The collection of above-ground-vegetables is subdivided into those vegetables which are 
;referred to as "night shade" vegetables.
(define-unit night-shade-vegetables
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit peppers
  (member-of 'collections)
  (supersets 'night-shade-vegetables))

(define-unit green-peppers
  (member-of 'collections)
  (supersets 'peppers))

(define-unit red-peppers
  (member-of 'collections)
  (supersets 'peppers))

(define-unit yellow-peppers
  (member-of 'collections)
  (supersets 'peppers))

(define-unit eggplant
  (member-of 'collections)
  (supersets 'night-shade-vegetables))

(define-unit tomatoes
  (member-of 'collections)
  (supersets 'night-shade-vegetables))

;The collection of above-ground-vegetables is subdivided into those vegetables which are 
;referred to as "fungi" and specifically mushrooms (of which there are many types) and 
;morels.
(define-unit fungi
  (member-of 'collections)
  (supersets 'above-ground-vegetables))

(define-unit mushrooms
  (member-of 'collections)
  (supersets 'fungi))

(define-unit black-mushrooms
  (member-of 'collections)
  (supersets 'mushrooms))

(define-unit white-mushrooms
  (member-of 'collections)
  (supersets 'mushrooms))

(define-unit wood-ears
  (member-of 'collections)
  (supersets 'mushrooms))

(define-unit straw-mushrooms
  (member-of 'collections)
  (supersets 'mushrooms))

(define-unit morels
  (member-of 'collections)
  (supersets 'fungi))

;The collection of vegetables is subdivided into those vegetables which grow below ground.
(define-unit below-ground-vegetables
  (member-of 'collections)
  (supersets 'vegetables))

;The collection of vegetables which grow below ground is further subdivided into those
;which are classified as bulbs and tubers (caveat: I am not a botanist).
(define-unit bulbs
  (member-of 'collections)
  (supersets 'below-ground-vegetables))

(define-unit onions
  (member-of 'collections)
  (supersets 'bulbs))

(define-unit scallions
  (member-of 'collections)
  (supersets 'bulbs))

(define-unit garlic
  (member-of 'collections)
  (supersets 'bulbs))

(define-unit tubers
  (member-of 'collections)
  (supersets 'below-ground-vegetables))

(define-unit ginger
  (member-of 'collections)
  (supersets 'tubers))

(define-unit water-chestnuts
  (member-of 'collections)
  (supersets 'tubers))

(define-unit lotus-roots
  (member-of 'collections)
  (supersets 'tubers))

;The collection of plant-products is also divided into the collection of fruits.
(define-unit fruits
  (member-of 'collections)
  (supersets 'plant-products))

(define-unit apples
  (member-of 'collections)
  (supersets 'fruits))

;The collection of fruits is further divided into the collections of tropical-fruits and
;citrus fruits.
(define-unit tropical-fruits
  (member-of 'collections)
  (supersets 'fruits))

(define-unit pineapple
  (member-of 'collections)
  (supersets 'tropical-fruits))

(define-unit lychee
  (member-of 'collections)
  (supersets 'tropical-fruits))

(define-unit citrus-fruits
  (member-of 'collections)
  (supersets 'fruits))

(define-unit lemons
  (member-of 'collections)
  (supersets 'citrus-fruits))

(define-unit oranges
  (member-of 'collections)
  (supersets 'citrus-fruits))

;The collection of plant-products is also divided into the collection of legumes.
(define-unit legumes
  (member-of 'collections)
  (supersets 'plant-products))

;The collection of legumes is further divided into the collections of beans, peanuts, and peas.
(define-unit beans
  (member-of 'collections)
  (supersets 'legumes))

(define-unit soybeans
  (member-of 'collections)
  (supersets 'beans))

(define-unit azuki-beans
  (member-of 'collections)
  (supersets 'beans))

(define-unit mung-beans
  (member-of 'collections)
  (supersets 'beans))

(define-unit peanuts
  (member-of 'collections)
  (supersets 'legumes))

(define-unit peas
  (member-of 'collections)
  (supersets 'legumes))

(define-unit green-peas
  (member-of 'collections)
  (supersets 'peas))

;The collection of plant-products is also divided into the collection of grains
(define-unit grain
  (member-of 'collections)
  (supersets 'plant-products))

(define-unit rice
  (member-of 'collections)
  (supersets 'grain))

(define-unit wheat
  (member-of 'collections)
  (supersets 'grain))

;The collection of plant-products is also divided into the collection of (edible) seeds.
(define-unit seeds
  (member-of 'collections)
  (supersets 'plant-products))

(define-unit sesame-seeds
  (member-of 'collections)
  (supersets 'seeds))


;The following code sets up the domain of edible mineral products.
(define-unit mineral-products
  (member-of 'collections)
  (supersets 'comestibles))

(define-unit liquid-mineral-products
    (member-of 'collections)
    (supersets 'mineral-products))

(define-unit water
    (member-of 'collections)
    (supersets 'liquid-mineral-products))

(define-unit solid-mineral-products
    (member-of 'collections)
    (supersets 'mineral-products))

(define-unit salt
    (member-of 'collections)
    (supersets 'solid-mineral-products))

(define-unit nigari
    (member-of 'collections)
    (supersets 'solid-mineral-products))

(define-unit rennet
    (member-of 'collections)
    (supersets 'solid-mineral-products))

(define-unit gas-mineral-products
    (member-of 'collections)
    (supersets 'mineral-products))

(define-unit carbon-dioxide
    (member-of 'collections)
    (supersets 'gas-mineral-products))


;The following code sets up the domain of processed comestibles.
;Processed comestibles specifically refer to single base comestibles which have been altered
;by any number of processes (usually one).  The idea here is to allow us to represent, 
;for example, "peanut butter" as ground peanuts.
(define-unit processed-comestibles
  (member-of 'collections)
  (supersets 'comestibles))

;The test function "kind-of-processed-comestiblep" is defined which tests whether 
;a given unit which is a collection itself has the collection "processed-comestibles" 
;as its superset.
(deffcn kind-of-processed-comestiblep (type)
  (annotations (works-like prototypical-test))
  (and (unitp type) (aj::collectionp type)
       (query type 'supersets 'processed-comestibles)))

;In the following code we are creating slots to be used in defining 
;processed-comestible units. The processed-element slot stores what base comestible 
;is being processed into a processed-comestible.
(define-unit processed-element
  (member-of 'single-valued-slots)
  (makes-sense-for 'kind-of-processed-comestiblep)
  (must-be 'collections)
  (genl-slots 'comestible-components))

;The food-processes-used slot stores what food-processes are being used in processing a 
;base comestible into a processed-comestible.
(define-unit food-processes-used
  (member-of 'many-valued-slots)
  (makes-sense-for 'kind-of-processed-comestiblep)
  (must-be 'collections))

;The collection "food-processes" is defined.
(define-unit food-processes
  (member-of 'collections))

;The following code defines various kinds of food-processes.
(define-unit pressing
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit grinding
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit crystalizing
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit dehydrating
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit distilling
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit fermenting
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit sprouting
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit curing
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit molding
  (member-of 'collections)
  (supersets 'food-processes))

(define-unit rarifying
  (member-of 'collections)
  (supersets 'food-processes))

;With the slots and food-processes defined above we can now define various 
;processed-comestibles in terms of their base comestibles (processed-elements)
;and the processes being done to them.
(define-unit sesame-oil
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'sesame-seeds)
  (food-processes-used 'pressing))

(define-unit sesame-butter
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'sesame-seeds)
  (food-processes-used 'grinding))

(define-unit peanut-oil
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'peanuts)
  (food-processes-used 'pressing))

(define-unit peanut-butter
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'peanuts)
  (food-processes-used 'grinding))

(define-unit ground-meat
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'meat)
  (food-processes-used 'grinding))

(define-unit veal-sausage
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'veal)
  (food-processes-used 'grinding)
  (food-processes-used 'molding))

(define-unit ham
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'pork)
  (food-processes-used 'curing))

(define-unit sugar
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'sugar-cane)
  (food-processes-used 'crystalizing))

(define-unit soybean-milk
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'soybeans)
  (food-processes-used 'pressing))

(define-unit mung-bean-sprouts
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'mung-beans)
  (food-processes-used 'sprouting))

(define-unit tomato-paste
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'tomatoes)
  (food-processes-used 'dehydrating)
  (food-processes-used 'grinding))

(define-unit alcohol
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'grain)
  (food-processes-used 'distilling))

(define-unit apple-juice
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'apples)
  (food-processes-used 'pressing))

;Note that in the case of vinegar we are able to define it as a processed-comestible
;which is itself the result of processing (fermenting) another processed-comestible 
;(apple-juice).
(define-unit vinegar
  (member-of 'collections)
  (supersets 'processed-comestibles)
  (processed-element 'apple-juice)
  (food-processes-used 'fermenting))


;The following code sets up the domain of compound comestibles.
;Compound-comestibles express merely the combination of various base or processed
;comestibles.  A weakness of the present representation is that one would like to 
;be able to represent the processes which go into the combination of the various
;comestibles to form complex comestibles like "tofu" or "white wine sauce".  One
;would want to know which processes were applied to which comestibles (or groups of 
;comestibles) and in which order. Due to my being a novice at ARLOtje and time constraints
;I have sacrificed these levels of complexity and precision in order to create a 
;representation which reasonably well allows me to solve the task and hand and could be 
;extended to allow more precise and complex representations of the domain.
(define-unit compound-comestibles
  (member-of 'collections)
  (supersets 'comestibles))

;The test function "kind-of-compound-comestiblep" is defined which tests whether 
;a given unit which is a collection itself has the collection "compound-comestibles" 
;as its superset.
(deffcn kind-of-compound-comestiblep (type)
  (annotations (works-like prototypical-test))
  (and (unitp type) (aj::collectionp type)
       (query type 'supersets 'compound-comestibles)))

;The compound-elements slot stores is defined which stores what comestibles 
;are being combined into a compound-comestible.
(define-unit compound-elements
  (member-of 'many-valued-slots)
  (makes-sense-for 'kind-of-compound-comestiblep)
  (must-be 'collections)
  (genl-slots 'comestible-components))

;The following code defines various compound comestibles.
(define-unit noodles
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'grain)
  (compound-elements 'water))

(define-unit rice-noodles
  (member-of 'collections)
  (supersets 'noodles)
  (compound-elements 'rice)
  (compound-elements 'water))

(define-unit wheat-noodles
  (member-of 'collections)
  (supersets 'noodles)
  (compound-elements 'wheat)
  (compound-elements 'water))

(define-unit soy-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'soybeans) 
  (compound-elements 'water) 
  (compound-elements 'salt))

(define-unit fried-rice
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'rice) 
  (compound-elements 'peanut-oil)
  (compound-elements 'soy-sauce))

(define-unit batter
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'wheat)
  (compound-elements 'water)
  (compound-elements 'eggs) 
  (compound-elements 'salt))

(define-unit hot-oil
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'hot-peppers) 
  (compound-elements 'peanut-oil)
  (compound-elements 'sesame-oil))

(define-unit tofu
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'soybean-milk) 
  (compound-elements 'nigari) 
  (compound-elements 'water))

(define-unit moo-shi-pancakes
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'rice)
  (compound-elements 'water))

(define-unit moo-shi
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'cabbage)
  (compound-elements 'wood-ears) 
  (compound-elements 'bamboo-shoots) 
  (compound-elements 'eggs)
  (compound-elements 'moo-shi-pancakes) 
  (compound-elements 'soy-sauce))

(define-unit peking-sweet-bean-paste
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'azuki-beans) 
  (compound-elements 'soy-sauce)
  (compound-elements 'vinegar)
  (compound-elements 'sugar)
  (compound-elements 'water))

(define-unit yu-hsiang-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'hot-peppers) 
  (compound-elements 'garlic) 
  (compound-elements 'soy-sauce) 
  (compound-elements 'vinegar)
  (compound-elements 'sugar) 
  (compound-elements 'water))

(define-unit kung-pao-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'peanuts) 
  (compound-elements 'hot-peppers) 
  (compound-elements 'garlic)
  (compound-elements 'soy-sauce) 
  (compound-elements 'sugar) 
  (compound-elements 'water))

(define-unit sweet-and-sour-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'ginger) 
  (compound-elements 'hot-peppers) 
  (compound-elements 'garlic) 
  (compound-elements 'tomato-paste) 
  (compound-elements 'soy-sauce) 
  (compound-elements 'vinegar) 
  (compound-elements 'sugar) 
  (compound-elements 'water))

(define-unit oyster-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'oysters)
  (compound-elements 'soy-sauce) 
  (compound-elements 'water))

(define-unit dun-dun-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'sesame-oil) 
  (compound-elements 'peanut-butter) 
  (compound-elements 'garlic) 
  (compound-elements 'ginger) 
  (compound-elements 'hot-oil) 
  (compound-elements 'soy-sauce) 
  (compound-elements 'water))

(define-unit tomato-sauce
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'tomatoes) 
  (compound-elements 'water) 
  (compound-elements 'garlic)
  (compound-elements 'salt))

(define-unit pizza-dough
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'wheat) 
  (compound-elements 'water)
  (compound-elements 'salt))

(define-unit cheese
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'milk) 
  (compound-elements 'rennet))

(define-unit mozzarella-cheese
  (member-of 'collections)
  (supersets 'cheese)
  (compound-elements 'milk) 
  (compound-elements 'rennet))


(define-unit base-pizza
  (member-of 'collections)
  (supersets 'compound-comestibles)
  (compound-elements 'tomato-sauce) 
  (compound-elements 'pizza-dough) 
  (compound-elements 'mozzarella-cheese))

;We now model dietary restrictions as prohibitions against eating 
;certain foods according to various religious practices, moral stances, and 
;health conditions.
;We create the slot "dietary-restrictions" as a many valued slot which makes sense
;for animals and, as a subset thereof, humans.
(define-unit dietary-restrictions
  (member-of 'many-valued-slots)
  (makes-sense-for 'animals)
  (must-be 'collections))

;The slot "comprehensive-dietary-restrictions" is created which generates a list
;comprised of all specializations of a given set of dietary-restrictions.
(define-unit comprehensive-dietary-restrictions
  (works-like 'prototypical-slot)
  (operation-per-element 'all-specializations-of)
  (combine-results-by 'lunion)
  (operate-on-slot 'dietary-restrictions))

;We then create collections of various religious-practices, moral-stances, 
;and health-conditions and their related dietary restrictions.
(define-unit religious-practices
  (member-of 'collections))

(define-unit kind-of-religious-practice
  (member-of 'collections)
  (supersets 'collections)
  (members-have '(supersets religious-practices)))

(define-unit orthodox-jewish
  (member-of 'kind-of-religious-practice)
  (members-have '(dietary-restrictions meat))
  (members-have '(dietary-restrictions poultry)) 
  (members-have '(dietary-restrictions fish-without-fins-and-scales))
  (members-have '(dietary-restrictions shellfish)))

(define-unit muslim
  (member-of 'kind-of-religious-practice)
  (members-have '(dietary-restrictions pork)) 
  (members-have '(dietary-restrictions fish-without-fins-and-scales)) 
  (members-have '(dietary-restrictions alcohol)))

(define-unit buddhist
  (member-of 'kind-of-religious-practice)
  (members-have '(dietary-restrictions animal-products)) 
  (members-have '(dietary-restrictions animal-by-products)) 
  (members-have '(dietary-restrictions alcohol)))

(define-unit brahmin
  (member-of 'kind-of-religious-practice)
  (members-have '(dietary-restrictions beef)))


(define-unit moral-stances
  (member-of 'collections))

(define-unit kind-of-moral-stance
  (member-of 'collections)
  (supersets 'collections)
  (members-have '(supersets moral-stances)))

(define-unit lacto-ovo-vegetarian
  (member-of 'kind-of-moral-stance)
  (members-have '(dietary-restrictions animal-products)))

(define-unit pesce-vegetarian
  (member-of 'kind-of-moral-stance)
  (members-have '(dietary-restrictions meat))
  (members-have '(dietary-restrictions poultry)))

(define-unit vegan
  (member-of 'kind-of-moral-stance)
  (members-have '(dietary-restrictions animal-products))
  (members-have '(dietary-restrictions animal-by-products)))

(define-unit macrobiotic
  (member-of 'kind-of-moral-stance)
  (members-have '(dietary-restrictions animal-by-products))
  (members-have '(dietary-restrictions meat))
  (members-have '(dietary-restrictions poultry)))

(define-unit free-range-livestock-only
  (member-of 'kind-of-moral-stance)
  (members-have '(dietary-restrictions veal)))


(define-unit health-conditions
  (member-of 'collections))

(define-unit kind-of-health-condition
  (member-of 'collections)
  (supersets 'collections)
  (members-have '(supersets health-conditions)))

(define-unit diabetic
  (member-of 'kind-of-health-condition)
  (members-have '(dietary-restrictions sugar)))

(define-unit hypertensive
  (member-of 'kind-of-health-condition)
  (members-have '(dietary-restrictions salt)))

;We then define slots which humans and animals can have which state their
;respective religious practices, moral stances, and health conditions.
(define-unit religious-practices
  (member-of 'many-valued-slots)
  (makes-sense-for 'humans)
  (must-be 'collections))

(define-unit moral-stances
  (member-of 'many-valued-slots)
  (makes-sense-for 'humans)
  (must-be 'collections))
  
(define-unit health-conditions            
  (member-of 'many-valued-slots)
  (makes-sense-for 'animals)
  (must-be 'collections))


;We now come to finally represent the Mary Chung Menu.
;First a collection of menus is defined.
(define-unit menus
  (member-of 'collections))

;The Mary-Chung-menu is defined as a collection which is a member of the collection menus.
(define-unit Mary-Chung-menu
  (member-of 'collections)
  (member-of 'menus))

;The collection menu-dishes is defined which is a subset of the collection menus.
(define-unit menu-dishes
  (member-of 'collections)
  (supersets 'comestibles)
  (supersets 'menus))

;The test function "kind-of-menu-dishp" is defined which tests whether 
;a given unit which is a collection itself is a member of menu-dishes.
(deffcn kind-of-menu-dishp (type)
  (annotations (works-like prototypical-test))
  (and (unitp type) (aj::collectionp type)
       (query type 'member-of 'menu-dishes)))


;The collection Mary-Chung-dishes is defined which is a member-of the collection
;Mary-Chung-menu and whose members are members of the Mary-Chung-menu as well as of
;the collection menu-dishes.
(define-unit Mary-Chung-dishes
  (member-of 'collections)
  (supersets 'comestibles)
  (supersets 'Mary-Chung-menu)
  (members-have '(member-of menu-dishes))
  (members-have '(supersets Mary-Chung-menu)))

;The test function "kind-of-Mary-Chung-dishp" is defined which tests whether 
;a given unit which is a collection itself is a member of Mary-Chung-dishes.
(deffcn kind-of-Mary-Chung-dishp (type)
  (annotations (works-like prototypical-test))
  (and (unitp type) (aj::collectionp type)
       (query type 'member-of 'Mary-Chung-dishes)))

;The slot "hot-and-spicy" is defined which indicates whether a dish is hot-and-spicy
;based on the presence of comestible-components whose hot-and-spicy slot is T being
;in the dish's ingredients list.
(define-unit hot-and-spicy
  (member-of 'single-valued-slots)
  (makes-sense-for 'collections)
  (must-be 'booleanp)
  (transfers-through 'comestible-components))

;The collection of hot-peppers is redefined to have a hot-and-spicy slot with the value T.
(define-unit hot-peppers
  (member-of 'collections)
  (supersets 'above-ground-vegetables)
  (hot-and-spicy T))

;The following code creates the slots for the units ("dishes") which are members 
;of the Mary-Chung-dishes.

(define-unit Mary-Chung-menu-number
  (member-of 'single-valued-slots)
  (makes-sense-for 'kind-of-Mary-Chung-dishp)
  (must-be 'numberp))

(define-unit Mary-Chung-menu-price
  (member-of 'single-valued-slots)
  (makes-sense-for 'kind-of-Mary-Chung-dishp)
  (must-be 'numberp))

;The slot ingredients is defined which makes sense for a kind-of-menu-dishp, which tests whether
;a given unit is a member of the collection menu-dishes.
;ARLOtje figures out that the dishes listed below are members of the collection menu-dishes by
;inferring that members of Mary-Chung-dishes are members of the collection menu-dishes.
(define-unit ingredients
  (member-of 'many-valued-slots)
  (makes-sense-for 'kind-of-menu-dishp)
  (must-be 'collections)
  (genl-slots 'comestible-components))

;Each dish is defined as a collection with slots for its English name, 
;menu number, menu price, and ingredients. (Caveat: I had to guess about
;the ingredients of most of these dishes).
(define-unit fried-jumbo-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Fried Jumbo Shrimp")
  (Mary-Chung-menu-number 7)
  (Mary-Chung-menu-price 5.25)
  (ingredients 'shrimp) 
  (ingredients 'batter) 
  (ingredients 'peanut-oil))

(define-unit hot-pickled-cabbage
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Hot Pickled Cabbage")
  (Mary-Chung-menu-number 13)
  (Mary-Chung-menu-price 2.00)
  (ingredients 'cabbage) 
  (ingredients 'vinegar) 
  (ingredients 'sugar) 
  (ingredients 'soy-sauce) 
  (ingredients 'hot-peppers) 
  (ingredients 'salt))

(define-unit refreshing-bean-sprouts
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Refreshing Bean Sprouts")
  (Mary-Chung-menu-number 14)
  (Mary-Chung-menu-price 1.20)
  (ingredients 'mung-bean-sprouts) 
  (ingredients 'scallions) 
  (ingredients 'ginger) 
  (ingredients 'vinegar) 
  (ingredients 'soy-sauce) 
  (ingredients 'hot-peppers) 
  (ingredients 'hot-oil))

(define-unit moo-shi-vegetables
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Vegetables")
  (Mary-Chung-menu-number 31)
  (Mary-Chung-menu-price 5.95)
  (ingredients 'moo-shi) 
  (ingredients 'lotus-roots) 
  (ingredients 'scallions)
  (ingredients 'baby-corn))

(define-unit moo-shi-vegetables-with-peking-sweet-bean-paste
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Vegetables with Peking Sweet Bean Paste")
  (Mary-Chung-menu-number 32)
  (Mary-Chung-menu-price 6.25)
  (ingredients 'moo-shi-vegetables)
  (ingredients 'peking-sweet-bean-paste))

(define-unit moo-shi-pork
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Pork")
  (Mary-Chung-menu-number 33)
  (Mary-Chung-menu-price 5.95)
  (ingredients 'moo-shi-vegetables)
  (ingredients 'pork))

(define-unit moo-shi-pork-with-peking-sweet-bean-paste
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Pork with Peking Sweet Bean Paste")
  (Mary-Chung-menu-number 34)
  (Mary-Chung-menu-price 6.25)
  (ingredients 'moo-shi-pork)
  (ingredients 'peking-sweet-bean-paste))

(define-unit moo-shi-chicken
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Chicken")
  (Mary-Chung-menu-number 35)
  (Mary-Chung-menu-price 5.95)
  (ingredients 'moo-shi-vegetables)
  (ingredients 'chicken))

(define-unit moo-shi-chicken-with-peking-sweet-bean-paste
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Chicken with Peking Sweet Bean Paste")
  (Mary-Chung-menu-number 36)
  (Mary-Chung-menu-price 6.25)
  (ingredients 'moo-shi-chicken)
  (ingredients 'peking-sweet-bean-paste))

(define-unit moo-shi-beef
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Beef")
  (Mary-Chung-menu-number 37)
  (Mary-Chung-menu-price 5.95)
  (ingredients 'moo-shi-vegetables)
  (ingredients 'beef))

(define-unit moo-shi-beef-with-peking-sweet-bean-paste
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Beef with Peking Sweet Bean Paste")
  (Mary-Chung-menu-number 38)
  (Mary-Chung-menu-price 6.25)
  (ingredients 'moo-shi-beef) 
  (ingredients 'peking-sweet-bean-paste))

(define-unit moo-shi-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Shrimp")
  (Mary-Chung-menu-number 39)
  (Mary-Chung-menu-price 6.25)
  (ingredients 'moo-shi-vegetables)
  (ingredients 'shrimp))

(define-unit moo-shi-shrimp-with-peking-sweet-bean-paste
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Moo Shi Shrimp with Peking Sweet Bean Paste")
  (Mary-Chung-menu-number 40)
  (Mary-Chung-menu-price 6.50)
  (ingredients 'moo-shi-shrimp) 
  (ingredients 'peking-sweet-bean-paste))

(define-unit dry-cooked-spicy-jumbo-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Dry Cooked, Spicy Jumbo Shrimp")
  (Mary-Chung-menu-number 110)
  (Mary-Chung-menu-price 8.75)
  (ingredients 'shrimp) 
  (ingredients 'hot-peppers)
  (ingredients 'soy-sauce) 
  (ingredients 'garlic) 
  (ingredients 'onions) 
  (ingredients 'peanut-oil))

(define-unit chungs-spicy-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Chung's Spicy Shrimp")
  (Mary-Chung-menu-number 111)
  (Mary-Chung-menu-price 8.75)
  (ingredients 'shrimp) 
  (ingredients 'hot-peppers)
  (ingredients 'soy-sauce) 
  (ingredients 'garlic) 
  (ingredients 'pea-pods) 
  (ingredients 'black-mushrooms)
  (ingredients 'bamboo-shoots) 
  (ingredients 'peanut-oil))

(define-unit fried-jumbo-jumbo-shrimp-with-yu-hsiang-sauce
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Fried Jumbo Shrimp with Yu Hsiang Sauce")
  (Mary-Chung-menu-number 112)
  (Mary-Chung-menu-price 7.75)
  (ingredients 'shrimp) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit kung-pao-sliced-fish
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Kung Pao Sliced Fish")
  (Mary-Chung-menu-number 113)
  (Mary-Chung-menu-price 6.75)
  (ingredients 'fish-with-fins-and-scales) 
  (ingredients 'kung-pao-sauce) 
  (ingredients 'batter)
  (ingredients 'peanut-oil))

(define-unit sliced-fish-with-yu-hsiang-sauce
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Sliced Fish with Yu Hsiang Sauce")
  (Mary-Chung-menu-number 114)
  (Mary-Chung-menu-price 6.75)
  (ingredients 'fish-with-fins-and-scales)
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit chungs-spicy-scallops
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Chung's Spicy Scallops")
  (Mary-Chung-menu-number 115)
  (Mary-Chung-menu-price 8.75)
  (ingredients 'scallops) 
  (ingredients 'hot-peppers)
  (ingredients 'soy-sauce) 
  (ingredients 'garlic) 
  (ingredients 'pea-pods) 
  (ingredients 'black-mushrooms)
  (ingredients 'bamboo-shoots) 
  (ingredients 'peanut-oil))

(define-unit scallops-with-yu-hsiang-sauce
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Scallops with Yu Hsiang Sauce")
  (Mary-Chung-menu-number 116)
  (Mary-Chung-menu-price 8.25)
  (ingredients 'scallops) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit sweet-and-sour-jumbo-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Sweet & Sour Jumbo Shrimp")
  (Mary-Chung-menu-number 117)
  (Mary-Chung-menu-price 7.75)
  (ingredients 'shrimp) 
  (ingredients 'sweet-and-sour-sauce) 
  (ingredients 'green-peppers)
  (ingredients 'onions) 
  (ingredients 'peanut-oil))

(define-unit jumbo-shrimp-with-pea-pods
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Jumbo Shrimp with Pea Pods")
  (Mary-Chung-menu-number 118)
  (Mary-Chung-menu-price 8.75)
  (ingredients 'shrimp)
  (ingredients 'pea-pods) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit scallops-with-pea-pods
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Scallops with Pea Pods")
  (Mary-Chung-menu-number 119)
  (Mary-Chung-menu-price 8.25)
  (ingredients 'scallops)
  (ingredients 'pea-pods) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit sweet-and-sour-sliced-fish
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Sweet & Sour Sliced Fish")
  (Mary-Chung-menu-number 120)
  (Mary-Chung-menu-price 6.75)
  (ingredients 'fish-with-fins-and-scales) 
  (ingredients 'sweet-and-sour-sauce) 
  (ingredients 'green-peppers)
  (ingredients 'onions) 
  (ingredients 'peanut-oil))

(define-unit jumbo-shrimp-with-broccoli
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Jumbo Shrimp with Pea Pods")
  (Mary-Chung-menu-number 121)
  (Mary-Chung-menu-price 8.50)
  (ingredients 'shrimp) 
  (ingredients 'broccoli)
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit soy-sauce-eggplant
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Soy Sauce Eggplant")
  (Mary-Chung-menu-number 131)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'eggplant)
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit broccoli-with-oyster-sauce
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Broccoli with Oyster Sauce")
  (Mary-Chung-menu-number 132)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'broccoli) 
  (ingredients 'oyster-sauce) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit pea-pods-with-black-mushrooms-and-bamboo-shoots
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Pea Pods with Black Mushrooms and Bamboo Shoots")
  (Mary-Chung-menu-number 133)
  (Mary-Chung-menu-price 5.50)
  (ingredients 'pea-pods)
  (ingredients 'black-mushrooms) 
  (ingredients 'bamboo-shoots) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit vegetarians-delight
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Vegetarian's Delight")
  (Mary-Chung-menu-number 134)
  (Mary-Chung-menu-price 5.50)
  (ingredients 'green-peppers) 
  (ingredients 'onions) 
  (ingredients 'baby-corn) 
  (ingredients 'water-chestnuts)
  (ingredients 'pea-pods)
  (ingredients 'black-mushrooms) 
  (ingredients 'bamboo-shoots) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit dry-cooked-green-beans-with-pork
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Dry Cooked Green Bean with Pork")
  (Mary-Chung-menu-number 135)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'green-beans) 
  (ingredients 'pork) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit yu-hsiang-broccoli
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Yu Hsiang Broccoli")
  (Mary-Chung-menu-number 136)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'broccoli) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit yu-hsiang-eggplant
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Yu Hsiang Eggplant")
  (Mary-Chung-menu-number 137)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'eggplant) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit spicy-eggplant
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Spicy Eggplant")
  (Mary-Chung-menu-number 138)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'eggplant) 
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'ginger) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit yu-hsiang-green-beans-with-pork
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Yu Hsiang Green Beans with Pork")
  (Mary-Chung-menu-number 139)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'green-beans) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'pork) 
  (ingredients 'peanut-oil))

(define-unit hot-sweet-and-sour-cabbage
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Hot, Sweet and Sour Cabbage")
  (Mary-Chung-menu-number 140)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'cabbage) 
  (ingredients 'sweet-and-sour-sauce) 
  (ingredients 'hot-peppers)
  (ingredients 'garlic) 
  (ingredients 'ginger) 
  (ingredients 'green-peppers) 
  (ingredients 'onions) 
  (ingredients 'peanut-oil))

(define-unit yu-hsiang-cabbage
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Yu Hsiang Cabbage")
  (Mary-Chung-menu-number 141)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'cabbage) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit tofu-with-mushrooms
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Bean Curd with Mushrooms")
  (Mary-Chung-menu-number 141)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu)
  (ingredients 'black-mushrooms) 
  (ingredients 'straw-mushrooms) 
  (ingredients 'wood-ears)
  (ingredients 'morels) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit tofu-with-oyster-sauce
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Bean Curd with Oyster Sauce")
  (Mary-Chung-menu-number 142)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu)
  (ingredients 'black-mushrooms) 
  (ingredients 'oyster-sauce) 
  (ingredients 'peanut-oil))

(define-unit red-cooked-tofu
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Red Cooked Bean Curd")
  (Mary-Chung-menu-number 143)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu) 
  (ingredients 'ground-meat) 
  (ingredients 'wood-ears) 
  (ingredients 'bamboo-shoots) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit tofu-with-black-mushrooms-and-bamboo-shoots
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Bean Curd with Black Mushrooms and Bamboo Shoots")
  (Mary-Chung-menu-number 144)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu) 
  (ingredients 'black-mushrooms) 
  (ingredients 'bamboo-shoots) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit tofu-with-pea-pods-and-straw-mushrooms
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Bean Curd with Pea Pods and Straw Mushrooms")
  (Mary-Chung-menu-number 145)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu) 
  (ingredients 'pea-pods) 
  (ingredients 'straw-mushrooms) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit tofu-with-chicken-velvet-and-ham
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Bean Curd with Chicken Velvet and Ham")
  (Mary-Chung-menu-number 146)
  (Mary-Chung-menu-price 5.95)
  (ingredients 'tofu) 
  (ingredients 'chicken) 
  (ingredients 'ham) 
  (ingredients 'soy-sauce) 
  (ingredients 'peanut-oil))

(define-unit spicy-tofu-with-chicken-velvet
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Spicy Bean Curd with Chicken Velvet")
  (Mary-Chung-menu-number 147)
  (Mary-Chung-menu-price 5.95)
  (ingredients 'tofu) 
  (ingredients 'chicken) 
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'ginger) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit ma-paw-tofu
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Ma Paw Tofu")
  (Mary-Chung-menu-number 148)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu)
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit spicy-tofu-with-beef
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Spicy Bean Curd with Beef")
  (Mary-Chung-menu-number 149)
  (Mary-Chung-menu-price 5.50)
  (ingredients 'tofu) 
  (ingredients 'beef) 
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'ginger) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit spicy-tofu-with-chicken
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Spicy Bean Curd with Chicken")
  (Mary-Chung-menu-number 150)
  (Mary-Chung-menu-price 5.50)
  (ingredients 'tofu) 
  (ingredients 'chicken) 
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'ginger) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit spicy-tofu-with-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Spicy Bean Curd with Shrimp")
  (Mary-Chung-menu-number 151)
  (Mary-Chung-menu-price 5.75)
  (ingredients 'tofu) 
  (ingredients 'shrimp) 
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'ginger) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit spicy-tofu-with-beef-chicken-and-shrimp
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Spicy Bean Curd with Beef, Chicken and Shrimp")
  (Mary-Chung-menu-number 152)
  (Mary-Chung-menu-price 5.75)
  (ingredients 'tofu) 
  (ingredients 'beef) 
  (ingredients 'chicken) 
  (ingredients 'shrimp) 
  (ingredients 'hot-peppers) 
  (ingredients 'garlic)
  (ingredients 'ginger) 
  (ingredients 'soy-sauce)
  (ingredients 'peanut-oil))

(define-unit chungs-spicy-tofu
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Chung's Spicy Bean Curd")
  (Mary-Chung-menu-number 153)
  (Mary-Chung-menu-price 5.75)
  (ingredients 'tofu) 
  (ingredients 'pork) 
  (ingredients 'hot-peppers)
  (ingredients 'soy-sauce) 
  (ingredients 'garlic) 
  (ingredients 'pea-pods) 
  (ingredients 'black-mushrooms)
  (ingredients 'bamboo-shoots) 
  (ingredients 'peanut-oil))

(define-unit yu-hsiang-tofu
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Yu Hsiang Bean Curd")
  (Mary-Chung-menu-number 154)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'tofu) 
  (ingredients 'yu-hsiang-sauce) 
  (ingredients 'peanut-oil))

(define-unit plain-fried-rice
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Plain Fried Rice")
  (Mary-Chung-menu-number 161)
  (Mary-Chung-menu-price 3.95)
  (ingredients 'fried-rice) 
  (ingredients 'scallions) 
  (ingredients 'green-peas) 
  (ingredients 'eggs))

(define-unit pork-fried-rice
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Pork Fried Rice")
  (Mary-Chung-menu-number 162)
  (Mary-Chung-menu-price 4.25)
  (ingredients 'plain-fried-rice) 
  (ingredients 'pork))

(define-unit chicken-fried-rice
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Chicken Fried Rice")
  (Mary-Chung-menu-number 163)
  (Mary-Chung-menu-price 4.25)
  (ingredients 'plain-fried-rice) 
  (ingredients 'chicken))

(define-unit beef-fried-rice
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Beef Fried Rice")
  (Mary-Chung-menu-number 164)
  (Mary-Chung-menu-price 4.50)
  (ingredients 'plain-fried-rice) 
  (ingredients 'beef))

(define-unit shrimp-fried-rice
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Shrimp Fried Rice")
  (Mary-Chung-menu-number 165)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'plain-fried-rice) 
  (ingredients 'shrimp))

(define-unit yang-chow-fried-rice
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Yang Chow Fried Rice")
  (Mary-Chung-menu-number 166)
  (Mary-Chung-menu-price 4.95)
  (ingredients 'plain-fried-rice) 
  (ingredients 'pork)
  (ingredients 'chicken)
  (ingredients 'beef)
  (ingredients 'shrimp))

(define-unit steamed-rice-small
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Steamed Rice (Small)")
  (Mary-Chung-menu-number 167)
  (Mary-Chung-menu-price .80)
  (ingredients 'rice) 
  (ingredients 'water))

(define-unit steamed-rice-large
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Steamed Rice (Large)")
  (Mary-Chung-menu-number 167)
  (Mary-Chung-menu-price 2.20)
  (ingredients 'rice) 
  (ingredients 'water))

(define-unit tzar-cheung-mein
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Tzar Cheung Mein (Peking Special Sauce Noodles)")
  (Mary-Chung-menu-number 168)
  (Mary-Chung-menu-price 3.95)
  (ingredients 'wheat-noodles)
  (ingredients 'pork)
  (ingredients 'scallions) 
  (ingredients 'sesame-butter) 
  (ingredients 'sesame-oil)
  (ingredients 'garlic) 
  (ingredients 'ginger) 
  (ingredients 'soy-sauce) 
  (ingredients 'water))

(define-unit dun-dun-noodles
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Dun Dun Noodle")
  (Mary-Chung-menu-number 169)
  (Mary-Chung-menu-price 3.95)
  (ingredients 'wheat-noodles) 
  (ingredients 'dun-dun-sauce) 
  (ingredients 'cucumbers) 
  (ingredients 'mung-bean-sprouts))

(define-unit dun-dun-noodles-with-shredded-chicken
  (member-of 'collections)
  (member-of 'Mary-Chung-dishes)
  (english-description "Dun Dun Noodle with Shredded Chicken")
  (Mary-Chung-menu-number 170)
  (Mary-Chung-menu-price 4.25)
  (ingredients 'dun-dun-noodles) 
  (ingredients 'chicken))

;Similarly to the Mary-Chung-menu a miniscule subset of the Bertucci's menu is
;defined to allow Amy (see below) to compute what dishes she can share at a lunch date.
(define-unit Bertuccis-menu
  (member-of 'collections)
  (member-of 'menus))

(define-unit Bertuccis-dishes
  (member-of 'collections)
  (supersets 'comestibles)
  (supersets 'Bertuccis-menu)
  (members-have '(member-of menu-dishes))
  (members-have '(supersets Bertuccis-menu)))

;The test "kind-of-Bertuccis-dishp" is defined.
(deffcn kind-of-Bertuccis-dishp (type)
  (annotations (works-like prototypical-test))
  (and (unitp type) (aj::collectionp type)
       (query type 'member-of 'Bertuccis-dishes)))

(define-unit Bertuccis-menu-price
  (member-of 'single-valued-slots)
  (makes-sense-for 'kind-of-Bertuccis-dishp)
  (must-be 'numberp))

;Each dish is defined as a collection with slots for its English name, 
;menu price, and ingredients.
(define-unit panino-small
  (member-of 'collections)
  (member-of 'Bertuccis-dishes)
  (english-description "Panino (small)")
  (Bertuccis-menu-price 6.75)
  (ingredients 'veal-sausage) 
  (ingredients 'white-mushrooms) 
  (ingredients 'base-pizza))

(define-unit romano-small
  (member-of 'collections)
  (member-of 'Bertuccis-dishes)
  (english-description "Romano (small)")
  (Bertuccis-menu-price 7.50)
  (ingredients 'peppers) 
  (ingredients 'white-mushrooms) 
  (ingredients 'onions)
  (ingredients 'base-pizza))

(define-unit fra-diavolo-small
  (member-of 'collections)
  (member-of 'Bertuccis-dishes)
  (english-description "Fra Diavolo (small)")
  (Bertuccis-menu-price 6.75)
  (ingredients 'hot-peppers) 
  (ingredients 'veal-sausage) 
  (ingredients 'base-pizza))


;Now we can define a unit "my-dinner-party" which allows us to compute what the guests
;can all eat together based on the union of all of their dietary restrictions.

;First we define a collection social events in order to create a larger sensible context 
;in terms of which "my-dinner-party" can be defined.
(define-unit social-events
  (english-description "social events")
  (member-of 'collections))

;The following code defines the slots for "guests" and for the "menu" being served 
;at "my-dinner-party". The "dinner party" is here thought of as taking place at
;Mary Chung's.
(define-unit guests
  (english-description "the guests at a dinner party")
  (member-of 'many-valued-slots)
  (makes-sense-for 'social-events)
  (must-be 'humans))

(define-unit menu
  (english-description "a given social event's possible menu")
  (member-of 'single-valued-slots)
  (makes-sense-for 'social-events)
  (must-be 'menus))

;Our guests have various religious practices, moral stances, and health conditions.
;The code below defines slots used in creating our "guest list" which allows us to
;ascribe religious practices, moral stances, and health conditions to particular individuals. 
(define-unit religious-practice
  (english-description "a given person's religious practice")
  (member-of 'many-valued-slots)
  (makes-sense-for 'humans)
  (must-be 'kind-of-religious-practice)
  (genl-slots 'member-of))

(define-unit moral-stance
  (english-description "a given person's moral stance")
  (member-of 'many-valued-slots)
  (makes-sense-for 'humans)
  (must-be 'kind-of-moral-stance)
  (genl-slots 'member-of))

(define-unit health-condition
  (english-description "a given person's health condition")
  (member-of 'many-valued-slots)
  (makes-sense-for 'animals)
  (must-be 'kind-of-health-condition)
  (genl-slots 'member-of))

;The various possible guests are defined.
(define-unit Aaron-Isiah-Goldberg
  (english-description "Aaron Isiah Goldberg")
  (member-of 'men)
  (religious-practice 'orthodox-jewish))

(define-unit Siddharta-Nirvana
  (english-description "Siddharta Nirvana")
  (member-of 'women)
  (religious-practice 'buddhist))

(define-unit Abdul-Al-Hameni
  (english-description "Abdul Al-Hameni")
  (member-of 'men)
  (religious-practice 'muslim))

(define-unit Radu-Shiva-Krishna
  (english-description "Radu Shiva Krishna")
  (member-of 'women)
  (religious-practice 'brahmin))

(define-unit Shigeru-Takana
  (english-description "Shigeru Takana")
  (member-of 'men)
  (moral-stance 'macrobiotic))

(define-unit Amy-Susan-Bruckman
  (english-description "Amy Susan Bruckman")
  (member-of 'women)
  (moral-stance 'free-range-livestock-only))

(define-unit Asana-Veda
  (english-description "Asana Veda")
  (member-of 'men)
  (moral-stance 'vegan))

(define-unit Marc-Eliot-Davis
  (english-description "Marc Eliot Davis")
  (member-of 'men)
  (moral-stance 'lacto-ovo-vegetarian))

(define-unit Alice-Judith-Zucker
  (english-description "Alice Judith Zucker")
  (member-of 'women)
  (health-condition 'diabetic))

(define-unit Buster-Hart
  (english-description "Buster Hart")
  (member-of 'men)
  (health-condition 'hypertensive))

(define-unit Simson-Garfinkel
  (english-description "Simson Garfinkel")
  (member-of 'men)
  (moral-stance 'pesce-vegetarian))

(define-unit Alan-Ruttenberg
  (english-description "Alan Ruttenberg")
  (member-of 'men)
  (moral-stance 'pesce-vegetarian))

;Now we can define the unit my-dinner party.
(define-unit my-dinner-party
  (english-description "my dinner party")
  (member-of 'collections)
  (member-of 'social-events)
  (guests 'Aaron-Isiah-Goldberg)
  (guests 'Shigeru-Takana)
  (guests 'Alice-Judith-Zucker)
  (menu 'Mary-Chung-menu))

;The following slots set up the inferences which compute (1) a list of the dietary
;restrictions of all the guests, (2) a list of the available dishes in the menu (this
;is simply a list of all the dishes on the menu), (3) a list of the dishes all of the 
;guests can share computed by the function juggle-dietary-restrictions.
;The slot all-guests-dietary-restrictions is defined.
(define-unit all-guests-dietary-restrictions
  (works-like 'prototypical-slot)
  (operation-per-element 'comprehensive-dietary-restrictions)
  (combine-results-by 'lunion)
  (operate-on-slot 'guests))

;The slot available-dishes is defined.
(define-unit available-dishes
  (works-like 'prototypical-slot)
  (composition-of '(members menu)))

;The slot possible-dishes is defined.
(define-unit possible-dishes
  (works-like 'prototypical-slot)
  (to-compute-value 'juggle-dietary-restrictions))

;The function juggle-dietary-restrictions is defined which takes the list of 
;all the guests' dietary restrictions and creates a list of possible dishes by
;removing dishes from the list of available dishes which are restricted by
;the cumulative dietary restrictions of all the guests.
(defun juggle-dietary-restrictions (unit slot)
  (declare (ignore slot))
  (let ((restrictions (get-value unit 'all-guests-dietary-restrictions))
        (dishes (get-value unit 'available-dishes)))
    (remove-if #'(lambda (dish) (some #'(lambda (restriction) 
                                          (query dish 'comestible-components restriction))
                                      restrictions))
               dishes)))

;To get a list of all the possible dishes our guests can share at the dinner party, simply
;evaluate the buffer (this will take a very, very long time) and then type 
;(get-value my-dinner-party possible-dishes) at the lisp prompt.  Then wait for a very
;long time and a list of the dishes our guests can share will be returned.

;Given the code above Amy can also define the unit for her lunch date.
;In similar fashion, you can create social events with various guests
;and menus and use the knowledge represented here to figure out what 
;dishes all your guests can eat together.
(define-unit Amys-lunch-date
  (english-description "Amy's lunch date")
  (member-of 'collections)
  (member-of 'social-events)
  (guests 'Amy-Susan-Bruckman)
  (guests 'Marc-Eliot-Davis)
  (menu 'Bertuccis-menu))
     
