;;; -*- LISP -*-

(in-package :CONSTRAINTS)

(import '(aj::annotated-value aj::run-method aj::*recursion-checking* aj::computing
	  aj::return-dependencies aj::annotated-value-put aj::demonic-put-value
	  aj::dependency!))
(export '(annotate-value suggest-value post-constraint acceptable-value))

(defun dependent-demonic-put-value (unit slot value)
  (let ((av (demonic-put-value unit slot value)))
    (aj::put-value av 'depends-on aj::*assertion-context*)
    av))

;;.Constraints are declarations of relations which support the further
;;.description of those values.  A constraint is an assertion about
;;.how one value is related to another.  If enough constraints exist,
;;.a precise characterization of that value is possible; even if there
;;.is not enough constraint, such knowledge allows the specification
;;.of a space in to which search for, hypothesize, or invent values.

;;.Straightforward sorts of constraints were presented when we spoke
;;.about ARLOtje's @code{constraint-methods}; these were algorithmic
;;.specifications of how to compute one value's slots from the values
;;.of other slots.  For instance, we might figure out the height or
;;.width of a box from either its edges or from its areas.  Constraint
;;.methods in ARLOtje allowed the statement of these algorithmic
;;.interrelations in a way that let the system find the combination of
;;.inferences which actually worked.

;;.A more complicated sort of inference is implemented here with the
;;.assertion of relations which both infer other relations and infer
;;.some algorithmic underpinning; the central problem of building
;;.constraint systems is understanding how to translate relational
;;.declarations into algorithmic specifications.  If one knows how two
;;.values are related, how do we sort out how one value is computed
;;.from another?

(defvar *CONSTRAINT-TRACE-STREAM* nil
  "The stream to which object constraint tracing reports go.")
(defmacro constraint-trace (format-string &rest format-args)
  "Prints a trace from the constraint mechanism if tracing is activated."
  `(when *CONSTRAINT-TRACE-STREAM*
    (format *CONSTRAINT-TRACE-STREAM* "~&\;\;\; ")
    (format *CONSTRAINT-TRACE-STREAM* ,format-string ,@format-args)
    (force-output  *CONSTRAINT-TRACE-STREAM*)))

(defun trace-constraint (&optional (stream (and (not *CONSTRAINT-TRACE-STREAM*) *standard-output*)))
  "Turns on constraint tracing."
  (if *constraint-trace-stream*
      (cond ((eq stream *constraint-trace-stream*))
	    (stream (format *constraint-trace-stream*
			    "~&\; Switching constraint trace from ~S to ~S~&"
			    stream *constraint-trace-stream*)
		    (format stream
			    "~&\; Switching constraint trace from ~S to ~S~&"
			    stream *constraint-trace-stream*))
	    (T (format *constraint-trace-stream*
		       "~&\; Switching off constraint trace (from ~S)~&"
		       *constraint-trace-stream*)))
    (format stream "~&\; Tracing constraint on stream ~S~&" stream))
  (setq *CONSTRAINT-TRACE-STREAM* stream))

;;;;.Implementing constraints

(define-unit local-constraints
  (member-of 'many-valued-slots)
  (to-put-value 'store-local-constraint)
  (makes-sense-for 'annotated-valuep)
  (must-be 'listp))
(define-unit slot-constraints
  (member-of 'many-valued-slots)
  (makes-sense-for 'slotp)
  (must-be 'listp))

(defun constrained-get (unit slot)
  (let ((av (annotated-value unit slot)))
    (let ((local-value (get-value av 'current-value)))
      (if (failurep local-value)
	  (multiple-value-bind (value dependencies)
	      (return-dependencies
	       (multiple-value-bind (value failures successes)
		   (apply-all-constraints unit slot)
		 (if (failurep value) value
		   (dolist (failure failures value)
		     (constraint-violation unit slot value successes failure)))))
	    (if (failurep value) value
	      (let ((av (demonic-put-value unit slot value)))
		(assert-value av 'depends-on dependencies)
		(dependency! av)
		value)))
	(progn (dependency! av) local-value)))))

(defun constrained-put (unit slot value)
  (unless (or (invalidated-p (annotated-value unit slot))
	      (aj::%hasnt  (annotated-value unit slot) 'current-value))
    (aj::invalidate-assertion (annotated-value unit slot)))
  (multiple-value-bind (computed failures successes)
      (apply-all-constraints unit slot value)
    (declare (ignore computed))
    (dolist (failure failures)
      (constraint-violation unit slot value successes failure))
    (dependent-demonic-put-value unit slot value)))

(defun constraint-violation (unit slot value passed-constraints failure)
  (error "Constraint violation: ~S (~S of ~S) fails for ~S [~S]~{~&~S [~S]~}"
	 value slot unit (cadr failure) (car failure)
	 passed-constraints))

(define-unit prototypical-constrained-slot
  (works-like 'prototypical-slot)
  (to-get-value 'constrained-get)
  (to-put-value 'constrained-put))

(define-unit constrained-slots
  (member-of 'collections)
  (members-have '(works-like prototypical-constrained-slot)))



(defun apply-all-constraints (unit slot &optional (attempt (fail unit slot)))
  "Applies a variety of methods to compute a slot value."
  (if *recursion-checking*
      (apply-constraints unit slot attempt
			 (append (get-value (annotated-value unit slot) 'local-constraints)
				 (get-value slot 'slot-constraints)))
    (let ((*recursion-checking* T))
      (computing (get-value unit slot)
	 (apply-constraints unit slot attempt
			    (append (get-value (annotated-value unit slot) 'local-constraints)
				    (get-value slot 'slot-constraints)))))))

(defun apply-constraints (unit slot value constraints)
  (let ((relational-constraints '()) (passed-constraints '()) (failed-constraints '()))
    (dolist (constraint constraints)
      (if (not (eq (car constraint) '=))
	  (push constraint relational-constraints)
	(let ((result (run-method (cadr constraint) unit slot)))
	  (unless (failurep result)
	    (if (and (not (failurep value)) (not (equal value result)))
		(push (list result constraint) failed-constraints)
	      (setq passed-constraints
		    (cons (list result constraint) passed-constraints)
		    value result))))))
    (unless (failurep value)
      (dolist (constraint relational-constraints)
	(let ((result (run-method (cadr constraint) unit slot)))
	  (unless (failurep result)
	    (if (funcall (car constraint) value result)
		(setq passed-constraints
		      (cons (list value constraint) passed-constraints))
	      (push (list result constraint) failed-constraints))))))
    (values value failed-constraints passed-constraints)))



(defun store-local-constraint (av slot constraint)
  (declare (ignore slot))
  (as-a-side-effect
   (let ((unit (annotated-value-unit av)) (slot (annotated-value-slot av)))
     (multiple-value-bind (value failures successes)
	 (apply-constraints unit slot (get-value av 'current-value) 
			    (append (get-value av 'local-constraints)
				    (get-value slot 'slot-constraints)
				    (list constraint)))
       (and failures
	    (cerror "Accept it anyway"
		    "Can't add constraint ~S: it doesn't accept ~S"
		    constraint value)))))
  (dependent-demonic-put-value av slot constraint))

(defun post-constraint (av slot constraint)
  (declare (ignore slot))
  (as-a-side-effect
   (let ((unit (annotated-value-unit av)) (slot (annotated-value-slot av)))
     (multiple-value-bind (value failures successes)
	 (apply-constraints unit slot (get-value av 'current-value)
			    (append (get-value (annotated-value unit slot) 'local-constraints)
				    (get-value slot 'slot-constraints)
				    (list constraint)))
       (if failures
	   (progn (constraint-trace "Can't add constraint ~S: it doesn't accept ~S"
				    constraint value)
		  NIL)
	   (dependent-demonic-put-value av 'local-constraints constraint))))))

(defun suggest-value (unit slot value)
  (and (not (some #'(lambda (sl) (query unit sl value)) (get-value slot 'denies)))
       (multiple-value-bind (computed failures successes)	    
	   (apply-all-constraints unit slot value)
	 (declare (ignore computed))
	 (cond (failures (constraint-trace "Failed to accept value ~S ~S ~S"
					   unit slot value)
			 NIL)
	       (T (dependent-demonic-put-value unit slot value)
		  (constraint-trace "Succesfully accepted value ~S ~S ~S"
				    unit slot value)
		  T)))))

(defun acceptable-value (unit slot value)
  (and (not (some #'(lambda (sl) (query unit sl value)) (get-value slot 'denies)))
       (multiple-value-bind (computed failures successes)	    
	   (apply-all-constraints unit slot value)
	 (null failures))))

;;;;.Simple algebraic manipulation

;;.We introduce a set of four slots: @code{sum-of},
;;.@code{difference-of}, @code{product-of}, and @code{ratio-of} which
;;.each refer to pairs of slots from which a particular slot's values
;;.may be computed.  Asserting any of these relations immediately
;;.installs a corresponding @code{method} on the slot and also
;;.installs two new relations between the triumvate.

;;.For instance, when we say a slot @code{Z} is the @code{SUM-OF}
;;.@code{X} and @code{Z}:
;;.@example
;;.(define-unit Z
;;.  (member-of 'single-valued-slots)
;;.  (sum-of '(X Y)))
;;.@end example
;;.ARLOtje first asserts a method that allows us to compute @code{Z} from
;;.@code{Y} and @code{X}:
;;.@example
;;.(assert-value 'Z 'slot-constraints
;;.              (+ (get-value %unit% X) (get-value %unit% Y)))
;;.@end example
;;.subsequent attempts to get @code{Z} from a unit with @code{X} and
;;.@code{Y} will succeed:
;;.@example
;;.(define-unit xyz (x 5) (y 3))
;;.(get-value 'xyz 'z) ==> 8
;;.@end example
;;.In addition to this relation, we know several other relations
;;.between @code{X}, @code{Y}, and @code{Z}.  For instance, we now
;;.that @code{Y} is the difference of @code{Z} and @code{X} and
;;.similarly @code{X} is the difference of @code{Z} and @code{Y}.
;;.ARLOtje thus asserts:
;;.@example
;;.(assert-value 'X 'difference-of '(Z Y))
;;.(assert-value 'Z 'difference-of '(Z X))
;;.@end example
;;.Each of these @code{difference-of} assertions is just like
;;.@code{sum-of} in that it supplies methods (@code{slot-constraints}) and also
;;.infers correpsonding @code{sum-of} and @code{difference-of} relations.@refill

;;.This example provides four slots --- for sum, difference, product,
;;.and ratio --- with interconnected functions.  Any slot can
;;.participate in multiple such relations and each constitutes a
;;.method or constraint which might be applied.  Initially, only one
;;.has to succeed and if multiple methods succeed, their values do not
;;.have to agree; later in the example, we will change this behavior.

;;..@code{SUM-OF} sets up a method which calls @code{+} to combine the
;;..specified slots.  It makes corresponding @code{difference-of}
;;..relations between its components and the slot it is stored on.@refill
(define-unit sum-of
  (member-of 'many-valued-slots)
  (must-be 'listp)
  (put-demons '(assert-value %unit% 'slot-constraints
		`(= (+ (get-value %unit% ,(car %value%)) (get-value %unit% ,(cadr %value%))))))
  (put-demons '(assert-value (car %value%) 'difference-of `(,%unit% ,(cadr %value%))))
  (put-demons '(assert-value (cadr %value%) 'difference-of `(,%unit% ,(car %value%)))))

;;..@code{DIFFERENCE-OF} sets up a method which calls @code{-} to combine the
;;..specified slots.  It makes corresponding @code{difference-of} and @code{sum-of}
;;..relations between its components and the slot it is stored
;;..on.@refill
(define-unit difference-of
  (member-of 'many-valued-slots)
  (must-be 'listp)
  (put-demons '(assert-value %unit% 'slot-constraints
		`(= (- (get-value %unit% ,(car %value%)) (get-value %unit% ,(cadr %value%))))))
  (put-demons '(assert-value (car %value%) 'sum-of `(,%unit% ,(cadr %value%))))
  (put-demons '(assert-value (cadr %value%) 'difference-of `(,(car %value%) ,%unit% ))))

;;..@code{PRODUCT-OF} sets up a method which calls @code{*} to combine the
;;..specified slots.  It makes corresponding @code{ratio-of} relations
;;..between its components and the slot it is stored on.@refill
(define-unit product-of
  (member-of 'many-valued-slots)
  (must-be 'listp)
  (put-demons '(assert-value %unit% 'slot-constraints
		`(= (* (get-value %unit% ,(car %value%)) (get-value %unit% ,(cadr %value%))))))
  (put-demons '(assert-value (car %value%) 'ratio-of `(,%unit% ,(cadr %value%))))
  (put-demons '(assert-value (cadr %value%) 'ratio-of `(,%unit% ,(car %value%)))))

;;..@code{RATIO-OF} sets up a method which calls @code{/} to combine the
;;..specified slots.  It makes corresponding @code{ratio-of} and @code{product-of}
;;..relations between its components and the slot it is stored
;;..on.@refill
(define-unit ratio-of
  (member-of 'many-valued-slots)
  (must-be 'listp)
  (put-demons '(assert-value %unit% 'slot-constraints
		`(= (/ (get-value %unit% ,(car %value%)) (get-value %unit% ,(cadr %value%))))))
  (put-demons '(assert-value (car %value%) 'product-of `(,%unit% ,(cadr %value%))))
  (put-demons '(assert-value (cadr %value%) 'ratio-of `(,(car %value%) ,%unit% ))))


;;;;.Boxes: An Example

;;.As a simple example of these constraints in action, we look at the
;;.dimensions of boxes; we first implement boxes as a collection and
;;.then define slots @code{height}, @code{top}, @code{bottom},
;;.@code{width}, @code{left}, @code{right}, and @code{area} with the
;;.following model
;;.@example
;;.                ^  ^
;;.            Top |  |
;;.                |  |Bottom
;;.                V  |
;;.              _____|__
;;.     Left    |     |  |   ^           Area=Height X Width
;;.<----------->|     |  |   |
;;.             |     |  |   | Height
;;.      Right  |     |  |   |
;;.<------------------|->|   |
;;.             |_____V__|   V
;;.
;;.             <-------->
;;.                Width
;;.;;.@end example

;;.Each of theses slots is a single valued slot which @code{must-be}
;;.@code{numberp} and @code{makes-sense-for} @code{boxes}.  The
;;.constraints are asserted as a @code{SUM-OF} constraint on
;;.@code{BOTTOM}, a @code{DIFFERENCE-OF} constraint on
;;.@code{WIDTH), and a @code{product-of} constraint on
;;.@code{area}.@refill

(define-unit boxes
  (member-of 'collections))

(define-unit height
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp))

(define-unit top
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp))

(define-unit bottom
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp)
  ;; The bottom is the TOP plus the HEIGHT.
  (sum-of '(top height)))

(define-unit left
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp))

(define-unit right
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp))

(define-unit width
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp)
  ;; The width is the difference of the right and the left.
  (difference-of '(right left)))

(define-unit area
  (member-of 'constrained-slots)
  (makes-sense-for 'boxes)
  (must-be 'numberp)  
  ;; The area is the product of the height and width.
  (product-of '(height width)))


;;;;.Representing Boxes

;;.We can construct a box whose left, top, width, and area we have
;;.specified:
;;...@example
(define-internal-unit box-1
  (member-of 'boxes)
  (left 20)  (top 10)
  (width 20) (area 1000))
;;...@end example
;;.and ask for its @code{bottom}, which ARLOtje easily resolves:
;;...@example
(get-value 'box-1 'bottom) ; ===> 60
;;...@end example
;;.and a description reveals that it also computed the @code{width}
;;.slot in this process:
;;.@example
;;.(du 'box-1)
;;.-------------------------------------------------------------------------------
;;.  Slot            Value
;;.  ----            -----
;;.  HEIGHT          50
;;.  BOTTOM          60
;;.  AREA            1000
;;.  TOP             10
;;.  WIDTH           20  ; This was computed to compute @code{BOTTOM}.
;;.  LEFT            20
;;.  MEMBER-OF       (BOXES)
;;.  TYPE-CODE       34359738368
;;.  CREATION-ID     ARLOTJE::LISZT-8.49.42-6DEC1990
;;.-------------------------------------------------------------------------------
;;.@end example

;;.One issue with the representation of boxes above is that for any
;;.box one must have precise knowledge of at least some of the boxes
;;.parameters and it is impossible to represent the fact that a box is
;;.within another box or that one of the properties of a box is
;;.related to another box.  This can be done in a few special cases;
;;.for instance, we might define the relation @code{same-area-as-box}:
;;...@example
(define-unit same-area-as-box
  (member-of 'single-valued-slots)
  (makes-sense-for 'boxes)
  (must-be 'boxes))
(assertion 'area 'slot-constraints
	   '(get-value (get-value %unit% same-area-as-box) area))
;;...@end example

;;.A more general solution allows us to say for any particular value
;;.where it might get its value; this is done by @code{local-constraints}.


;;;;.Local constraints

;;.The constraints above allow the specification of methods which compute
;;.the values of any value for a particular slot; here we use
;;.@code{local-constraints} to specify that a particular value may be
;;.computed in a particular way.  The @code{local-constraints} slot is
;;.stored on an annotated value and its methods are look like regular
;;.methods; they have priority over the methods assigned for all slots
;;.of a particular type.

;;.We can define a function @code{annotate-value} to add an annotation
;;.to a value:
;;...@example
(defun annotate-value (unit slot annotation value)
  (assert-value (annotated-value unit slot) annotation value))
;;...@end example

;;.Using local constraints, we can assert that @code{BOX-2} has the
;;.same area as @code{BOX-1} by:
;;.@example
(define-internal-unit box-2
  (member-of 'boxes)
  (left 20)  (top 10)
  (width 10))
(annotate-value 'box-2 'area 'local-constraints '(= (get-value box-1 area)))
;;.@end example
;;.and when we get the bottom of this box, it goes to get the area
;;.from @code{box-1}
;;...@example
(get-value 'box-2 'bottom) ;===> 110
;;...@end example
;;.We don't have to make the relation a strict identity, however.  We
;;.could define @code{box-3} to have twice the area of @code{box-1}:
;;...@example
(define-internal-unit box-3
  (member-of 'boxes)
  (left 20)  (top 10)
  (width 10))
(annotate-value 'box-3 'area
		'local-constraints '(= (* 2 (get-value box-1 area))))
(get-value 'cstr::box-3 'cstr::bottom) ;===> 210
;;...@end example

;;.We can regularize the `programming cliches' above by defining a
;;.slot @code{identical-to};
;;...@example
(define-unit identical-to
  (member-of 'many-valued-slots)
  (makes-sense-for 'annotated-valuep)
  (inverse-slot 'identical-to)
  (kleene-star-of 'identical-to)
  (put-demons '(assert-value %unit% 'local-constraints
		`(= (get-value ,(annotated-value-unit %value%) ,(annotated-value-slot %value%))))))
;;...@end example
;;.Allowing us to define @code{box-4} by something like:
;;...@example
(define-internal-unit box-4
  (member-of 'boxes)
  (left 20)  (top 10)
  (width 10))
(assert-value (annotated-value 'box-4 'area) 'identical-to
	      (annotated-value 'box-1 'area))
;===> #<ELT #<AV BOX-4 AREA> IDENTICAL-TO #<AV BOX-1 AREA>)>
(get-value 'box-4 'bottom) ;===> 110
;;...@end example


;;;;.Symbolic constraints into numeric constraints.

;;.We define here a collection @code{columns} with slots
;;.@code{right-column} and @code{left-column}.

(define-unit columns
  (member-of 'collections)
  (supersets 'boxes))

(define-unit left-column
  (member-of 'single-valued-slots)
  (makes-sense-for 'columns)
  (must-be 'boxes)
  (to-compute-value 'aj::fill-in-slot)
  (put-demons '(annotate-value %value% 'left
		'identical-to (annotated-value %unit% left)))
  (put-demons '(annotate-value %value% 'top
		'identical-to (annotated-value %unit% top)))
  (put-demons '(annotate-value %value% 'height
		'identical-to (annotated-value %unit% height)))
  (put-demons '(annotate-value %value% 'width
		'local-constraints `(= (/ (get-value ,%unit% width) 2)))))

(define-unit right-column
  (member-of 'single-valued-slots)
  (makes-sense-for 'columns)
  (must-be 'boxes)
  (to-compute-value 'aj::fill-in-slot)
  (put-demons '(annotate-value %value% 'right
		'identical-to (annotated-value %unit% right)))
  (put-demons '(annotate-value %value% 'top
		'identical-to (annotated-value %unit% top)))
  (put-demons '(annotate-value %value% 'height
		'identical-to (annotated-value %unit% height)))
  (put-demons '(annotate-value %value% 'width
		'local-constraints `(= (/ (get-value ,%unit% width) 2)))))

(define-unit cl1
    (member-of 'columns)
    (left 20) (right 120)
    (height 100) (top 10))
(setq lc (get-value 'cl1 'left-column)) ; ===> LEFT-COLUMN.1
(get-value lc 'top) ; ===> 10
(get-value lc 'right) ; ===> 70


(defmacro altogether-now (expression)
  "Evaluates expression whose side effects will only endure if it returns T."
  (let ((assumption-var (make-symbol "ASSUMPTION")))
    `(let ((,assumption-var (aj::make-assertion)))
      (let ((result (let ((aj::*assertion-context* (cons ,assumption-var aj::*assertion-context*)))
		      ,expression)))
	(unless result (aj::invalidate-assertion ,assumption-var))
	result))))