;;; -*- LISP -*-

;;.@chapter ARLOtje''s User Level
(in-package :arlotje)


;;;.User level functions

;;.@code{Assertion} is a user level interface function which does an
;;.@code{assert-value} and also records the assertion as an event in
;;.the current session; the call to @code{assertion} will be reexecuted
;;.when the session is restored.
(defun assertion (unit slot value &rest annotations)
  "Asserts that SLOT of UNIT has value and records it on the session history."
  (prog1 (apply #'assert-value unit slot value annotations)
    (session-event! `(assertion ,unit ,slot ,value ,@annotations)
		    (list unit value))))
;;.@code{Query} is a user level interface to @code{check-value}.
(defun query (unit slot value)
  "Checks if UNIT of SLOT has VALUE, and returns the assertion's description if true."
  (let ((value (check-value unit slot value)))
    (if (annotated-valuep value) (dependency! value))
    value))

;;.Both @code{retraction} and @code{retract-value} take a @var{unit},
;;.@var{slot}, and @var{value} and can be thought of as ensuring that
;;.@code{(check-value @var{unit} @var{slot} @var{value})} returns NIL.
;;.When the stored value is an annotated value, these functions take
;;.things a step further and proceed to call @code{retract-dependents}
;;.to descend the dependency network.
(defun retraction (unit slot &optional (value nil value-supplied?))
  (if value-supplied?
      (prog1 (retract-value unit slot value)
	(session-event! `(retraction ,unit ,slot ,value) (list unit value)))
    (prog1 (retract-value unit slot)
      (session-event! `(retraction ,unit ,slot) (list unit)))))

(defun kill-changes (to-unit &optional (in-session *session*))
  "Retracts all the changes made to TO-UNIT in the session IN-SESSION (defaulting)."
  (format T "~&\; Removing previous definition information regarding ~S" to-unit)
  (flet ((retractor (event)
	   (and (eq (cadr event) to-unit)
		(case (car event)
		  (ASSERTION (format T "~&\;  Retracting ~S" event)
			     (apply #'RETRACTION (cdr event)) T)
		  (RETRACTION (format T "~&\;  Retracting ~S" event)
			      (apply #'ASSERTION (cdr event))
			      T)
		  (DECLARE-UNIT NIL)
		  (T (format T "~&\;Don't know how to retract ~S." event) NIL)))))
    (setf (get in-session 'events)
	  (remove-if #'retractor (get in-session 'events)))))

(defmacro define-unit (name &body slot-specs)
  "Defines an external unit NAME and stores SLOT-SPECS on it with ASSERTION.
Each SLOT-SPEC has the form (slot-name value-form . annotations)
where value-form is evaluated and asserted and the resulting assertion
is annotated with ANNOTATIONS."
  (flet ((transform-slot (s)
	   (if (symbolp s) `',s `(get-value ',(first s) ',(second s)))))
    (let ((unit-var (make-symbol "UNIT")))
      `(bundling
	(if (and (eq (get ',name 'creation-id) *session*) (fboundp 'kill-changes))
	    (kill-changes ',name)
	  (if (get ',name 'creation-id)
	      (format T "~&\; Warning, the unit ~S is already defined from session ~S."
		      ',name (get ',name 'creation-id))))
	(let ((,unit-var (declare-unit ',name)))
	  (record-source-file ',name :unit)
	  ,@(mapcar #'(lambda (sl) `(assertion ,unit-var ,(transform-slot (first sl)) ,@(rest sl)))
		     slot-specs)
	  ,unit-var)))))


;;;;.Listing units

(defun declared-unitp (x) 
  (and (symbolp x) (get x 'creation-id) (not (annotated-valuep x))))
(defun list-all-units (&key (package *arlotje-package*) (test 'declared-unitp))
  (do-symbols (symbol package)
    (when (funcall test symbol)
      (format T "~&\; ~S ~@[~A~]" symbol 
	      (if (failurep (get-value symbol 'english-description)) '()
		(get-value symbol 'english-description))))))


;;;;.User level macros

(defmacro define-slot (slot-name slot-type makes-sense-for must-be
				 &body other-slots)
  "Defines a slot of a type with particular domain and range restrictions."
  `(define-unit ,slot-name
     (member-of ',slot-type)
     (makes-sense-for ',makes-sense-for)
     (must-be ',must-be)
     ,@other-slots))

(defmacro define-collection
  (collection-name supersets &body other-slots)
  "Defines a collection with particular generalizations and other properties."
  `(define-unit ,collection-name
     (member-of 'collections)
     ,@(mapcar #'(lambda (c) `(supersets ',c)) supersets)
     ,@other-slots))

(defmacro define-individual
  (individual-name member-of &body other-slots)
  "Defines a unit which is a member of particular collections."
  `(define-unit ,individual-name
     ,@(mapcar #'(lambda (c) `(member-of ',c)) member-of)
     ,@other-slots))

(defmacro define-structure
  (structure-name generalizations &body part-specs)
  "Defines a set of interrelated slots with a structure constraint holding between them."
  `(progn (define-unit ,structure-name
	    (member-of 'collections)
	    ,@(mapcar #'(lambda (c) `(supersets ',c)) generalizations))
    ,@(mapcar #'(lambda (part-spec)
		  `(define-unit ,(car part-spec)
		    (works-like 'prototypical-slot)
		    (makes-sense-for ',structure-name)))
       part-specs)
    ,@(mapcar #'(lambda (part-spec)
		  (flet ((assertionize (slot-spec)
			   `(assertion ',(car part-spec)
			     ',(car slot-spec) ,@(cdr slot-spec))))
		    `(progn ,@(mapcar #'assertionize (cdr part-spec)))))
		  part-specs)))

(defmacro define-units (unit-names &body slots)
  `(progn ,@(mapcar #'(lambda (name) `(define-unit ,name ,@slots))
	     unit-names)))


;;;;.Popular collections

(define-unit single-valued-slots
  (member-of 'collections)
  (generalizations 'slotp)
  (members-have '(works-like prototypical-slot)))
(define-unit many-valued-slots
  (member-of 'collections)
  (generalizations 'slotp)
  (members-have '(works-like prototypical-set-slot)))
(define-unit transitive-slots
  (member-of 'collections)
  (supersets 'many-valued-slots)
  (local-members-demons
   '(assert-value %value% kleene-plus-of %value%)))



;;;;.Useful for debugging.

(defun why? ()
  (show-arlotje-stack))

(defun yes-no-or-why-p (format-string &rest format-args)
  (loop (let ((stream *query-io*))
	  (format stream "~&\;?? ")
	  (apply #'format stream format-string format-args)
	  (format stream "~&\;??    (Yes, No, or Why) ")
	  (let ((input (read)))
	    (cond ((listp input) (print (eval input)))
		  ((not (symbolp input)) (format stream "~&\;!! Don't understand ~S" input))
		  ((get input 'creation-id) (fresh-line stream) (du input))
		  ((string-equal (symbol-name input) "YES") (return T))
		  ((string-equal (symbol-name input) "NO") (return NIL))
		  ((string-equal (symbol-name input) "WHY") (show-arlotje-stack))
		  (T (format stream "~&\;!! Don't understand ~S" input)))))))
