;;; -*- LISP -*-

;;.@chapter Constraining slot assignments
(in-package :arlotje)

;;.Humans make mistakes and more often, programs written by humans make
;;.mistakes.  In order to minimize the impact of such mistakes, it is
;;.useful to have systems which check themselves and understand what
;;.sorts of actions make sense and which sorts of actions are marginal
;;.or dangerous to perform.  In ARLOtje, we try and catch some of these
;;.mistakes by assigning type restrictions to some slots; we say that
;;.values of a particular slot `must be' of some particular type or that
;;.particular slots only `make sense for' particular types of units.

;;.ARLOtje's basic functions are extended to support the use of these
;;.slots.  In `packaged' ARLOtje, these mechanisms signal errors which
;;.must be handled by the user; however, there might be schemes which
;;.do more intelligent error recovery.  For instance, following CYC, we
;;.might have type failures search for alternative slots or units whose
;;.subsitution would both make some sense and make the corresponding
;;.action sensible.

;; Functionp kludge
; The problem: functionp means different things in various dialects of
; Common Lisp.  In early CL, it was always true of symbols.  In modern
; (X3J13) CL, it is never true of symbols.  In Symbolics CL, it is
; true of symbols only if it's true of the symbol's function value.
; To deal with this, we shadow the functionp symbol and define the
; local version to be always true of symbols.
#+GENERA
(shadow 'functionp)
#+GENERA
(defun aj::functionp (x) (symbolp x))

;;;;. Slots describing Slot Types

;;.There are two slots for typing slots: @code{MUST-BE} and
;;.@code{MAKES-SENSE-FOR}.  As ARLOtje boots, these are all LISP
;;.predicate functions; when ARLOtje's @emph{collections} are defined,
;;.the range of these slots is extended to include
;;.collections.@xref{Collections}@refill

;;.The @code{MAKES-SENSE-FOR} slot makes sense for slots and can take
;;.functions or (after they have been defined) ARLOtje
;;.@emph{collections} as values.  It determines what sorts of units can
;;.have this slot; since only units can have slots, the predicates or
;;.collections stored hereon must specialize the type of all units, @code{unitp}.
;;.The @code{MAKES-SENSE-FOR} slot is used to verify
;;.@code{STANDARD-PUT-VALUE} operations and to check that a default
;;.computation makes sense.  The inverse of this slot is
;;.@code{SENSIBLE-SLOTS} @xref{Describing Sensible Slots}.@refill
(define-primitively makes-sense-for
  (english-descrition "The type of objects for which a slot makes sense.")
  (works-like 'prototypical-slot-slot)
  ; These have to be here before typing is installed so as to avoid an
  ; infinite recursion.
  (to-get-value 'get-slot-method)
  (to-put-value 'typed-put-value)
  (makes-sense-for 'slotp)
  (must-be 'function-namep))
;;.@vindex{MAKES-SENSE-FOR (slot)}.

;;.The @code{MUST-BE} slot makes sense for slots and can take
;;.functions or (after they have been defined) ARLOtje
;;.@emph{collections} as values.  This slot determines what the
;;.approriate values for this slot might be.  The inverse of this slot
;;.is the slot @code{SENSIBLE-FOR-SLOTS} @xref{Describing Sensible Slots}.
(define-primitively must-be
  (engilsh-descrition "The type of objects which a slot can have as values.")
  (works-like 'prototypical-slot-slot)
  ; These have to be here before typing is installed so as to avoid an
  ; infinite recursion.
  (to-get-value 'get-slot-method)
  (to-put-value '%put)
  (makes-sense-for 'slotp)
  (must-be 'function-namep))
;;.@vindex{MUST-BE (slot)} 


;;;;. Typing Mechanisms

;;.Typing is first introduced with the slot prototype
;;.@code{prototypical-typed-slot} which provides default get and put
;;.functions that check the type of their arguments against the
;;.@code{MAKES-SENSE-FOR} and @code{MUST-BE} slots of the slots which
;;.they get or put.  As ARLOtje is progressively constructed, new get
;;.and put functions all have typing --- with reference to the
;;.@code{MUST-BE} and @code{MAKES-SENSE-FOR} slots --- built into
;;.them.@refill

(define-primitively prototypical-typed-slot
  (works-like 'prototypical-primitive-slot)
  (to-get-value 'typed-get)
  (to-put-value 'typed-put)
  (makes-sense-for 'unitp)
  (must-be 'anythingp))
;;.@vindex{prototypical-typed-slot (slot prototype)}

;;.The extension to @code{standard-put-value} is called
;;.@code{typed-put} and simply calls the function
;;.@code{check-put-value} on the @var{unit}, @var{slot}, and
;;.@var{value} being stored. The @var{annotations} passed in are
;;.ignored for purposes of type checking.
(defun typed-put-value (unit slot value &rest annotations)
  "A put function which checks the domain and range of the slot being stored."
  (declare (ignore annotations))
  (check-put-value unit slot value)
  (%put unit slot value))


;;;;.Bundling

;;.Bundling is the execution of a series of operations with type
;;.checking defferred until the series is completed; it makes certain
;;.nasty interdependenc problems seem a little bit simpler.  It is
;;.also potentially dangerous in some cases; though those cases grow
;;.fewer as the commutativity of ARLOtje's mechanisms improves.

(defvar *bundling* nil
  "Whether type checking is being deferred.")

(defvar *bundling-assumptions* '()
  "A list of assumptions made so far.")

(defmacro bundling (&body body)
  `(if *bundling* (progn ,@body)
    (let ((*bundling-assumptions* '()))
      (unwind-protect (let ((*bundling* t)) ,@body)
	(dolist (assumption *bundling-assumptions*)
	  (let ((object (car (car assumption))) (test (cadr (car assumption)))
		(misteak (cadr assumption)))
	    (unless (satisfies? object test)
	      (if (yes-or-no-p "Morning After Errror: ~S doesn't satisfy ~S.  Should I retract ~S"
			       object test misteak)
		  (apply #'retract-value (cdr misteak))
		(funcall (get-value test 'to-enforce-test) object test)))))))))

;;.The function @code{check-put-value} uses the @code{MAKES-SENSE-FOR}
;;.and @code{MUST-BE} slots and calls the function @code{satisfies?}
;;.with the corresponding values.  If either of these calls to
;;.@code{satisfies?} returns NIL, ARLOtje first checks if it is
;;.@emph{bundling}.  If it is bundling, it stores the object and the
;;.failed test on @code{*bundling-assumptions*} along with the
;;.operation being executed; when ARLOtje stops bundling, these are
;;.processed one by one.  If ARLOtje is not bundling, it checks to see
;;.if it is already `assuming' that the predicate is satisfied; if it
;;.is not so assuming it signals a proceedable error whose procession
;;.(?) asserts the corresponding assumption.@refill
(defun check-put-value (unit slot value)
  "Checks the slot assignment of VALUE to SLOT of UNIT.
Uses the MAKES-SENSE-FOR and MUST-BE slots as arguments to SATISFIES?."
  (let ((domain (get-value slot 'makes-sense-for))
	(range  (get-value slot 'must-be)))
    (unless (satisfies? unit domain)
      (unless (or *bundling* (assuming? (satisfies? unit domain)))
	(cerror "Assume that ~*~S satisfies ~S"
		"The ~S slot doesn't make sense for ~S:~%it must pass ~S"
		slot unit domain))
      (if *bundling* 
	  (push (list (list unit domain) (list 'put-value unit slot value))
		*bundling-assumptions*)
	  (assuming (satisfies? unit domain)
		    "~S satisfies ~S (and can validly have a ~S slot of ~S)"
		    unit domain slot value)))
    (unless (satisfies? value range)
      (unless (or *bundling* (assuming? (satisfies? value range)))
	(cerror "Assume that ~*~S satisfies ~S"
		"The ~A slot can't be ~A:~%it must pass ~A"
		slot value range))
      (if *bundling* 
	  (push (list (list value range) (list 'put-value unit slot value))
		*bundling-assumptions*)
	(assuming (satisfies? value range)
		  "~S satisfies ~S (and can be the ~S slot of ~S)"
		  value range slot unit)))))
;;.@findex{check-put-value}

;;.@code{CHECK-GET-VALUE} operates in much the same fashion, but only
;;.checks the domain of the slot, since obviously the value does not
;;.yet exist.
(defun check-get-value (unit slot)
  "Checks that UNIT makes sense for SLOT."
  (let ((domain (get-value slot 'makes-sense-for)))
    (unless (satisfies? unit domain)
      (unless (or *bundling* (assuming? (satisfies? unit domain)))
	(cerror "Assume that ~*~S satisfies ~S"
		"The ~S slot doesn't make sense for ~S:~%it must pass ~S"
		slot unit domain))
      (if *bundling* 
	  (push (list (list unit domain) (list 'get-value unit slot))
		*bundling-assumptions*)
	(assuming (satisfies? unit domain)
		  "~S satisfies ~S (and can validly have a ~S slot)"
		  unit domain slot)))))
;;.@findex{check-get-value}

;;.Until ARLOtje's `collections' have been defined, @code{satisfies?}
;;.simply calls its second argument @var{type} on its first argument
;;.@var{object}.
(defun satisfies? (object type)
  "Returns true if OBJECT is of TYPE.  This simple version just calls TYPE on OBJECT."
  ;; If TYPE isn't a function, it is assumed satisfied till a later
  ;; version of SATISFIES? is implemented.
  (or (not (fboundp type)) (funcall type object)))
;;.@findex{satisfies? (initial version)}

;;.To check value accesses, we extend the implementation of
;;.@code{%get} to call @code{check-get-value} which tests @var{unit}
;;.against the @code{MAKES-SENSE-FOR} slot of @var{slot}.
(defun typed-get (unit slot)
  (check-get-value unit slot)
  (%get unit slot))


;;;;. Typing Examples

;;.A most trivial example:
;;...@example
(defvar *physical-objects* '(clyde elephant))
(defun physical-objectp (x) (member x *physical-objects*))

(define-primitively xweight
  (english-description "An example of a typed slot.")
  (works-like 'prototypical-typed-slot)
  (makes-sense-for 'physical-objectp)
  (makes-sense-for 'numberp))
;;...@end example

;;.And we would see, for instance, that while this is perfectly
;;.respectable:
;;.@example
;;.(put-value 'clyde 'weight 20000)
;;.(get-value 'clyde 'weight) ==> 20000
;;.@end example
;;.We get complaints if the value is syntactically inappropriate:
;;.@example
;;. (put-value 'clyde 'weight 'heavy)
;;. Error: >> The WEIGHT slot of CLYDE can't be HEAVY; 
;;.           it must pass NUMBERP
;;.@end example
;;.As well as if we use inappropriate objects:
;;.@example
;;. (get-value 'love 'weight)
;;. Error: >> The WEIGHT slot doesn't make sense for LOVE: 
;;.           it must pass PHYSICAL-OBJECTP
;;.@end example

