;;; -*- Mode:  LISP; Package: ARLOTJE -*-

;;.@chapter The keyboard based unit editor
(in-package :arlotje)

;;.ARLOtje provides a basic, keyboard-centered, interface for
;;.examining units.  Particular implementations or packages may
;;.provide more sophisticated functionality using windows, speech
;;.interfaces, etc.

;;.The keyboard based interface is invoked by the @code{EU} function
;;.with a single argument of the ARLOtje unit to edit.  The editor
;;.begins by calling the function @code{DU} to produce a description
;;.of the unit being edited.  @code{DU}'s display of properties
;;.consists of a table of properties with three columns: slot name,
;;.slot value, and annotated description.
;;.For instance, this is a description of Ken Haase:
;;.@smallexample
;;.----------------------------------------------------------------------------------------------------
;;.  Slot                        Value                                             Value Description
;;.  ----                        -----                                             -----------------
;;.  CREATION-ID:                ARLOTJE::|NOISE.MEDIA.MIT.EDU-8.20.59-14June1990| 
;;.  MEMBER-OF:                  (MEN QUAKERS)                                     ; (ARLOTJE::AV.5256)
;;.  TYPE-CODE:                  130944                                            ; (AV.5428)
;;.  GENDER:                     ARLOTJE::MALE                                     ; (ARLOTJE::AV.5312)
;;.  DEFAULT-SLOT-FILTER:        SLOTP                                             ; (AV.5425)
;;.----------------------------------------------------------------------------------------------------
;;.@end smallexample
;;.where Ken is a member of the collections @code{MEN} and
;;.@code{QUAKERS} and has the @code{GENDER} slot @code{MALE}.  The
;;.values in the rightmost column are the annotated values describing
;;.the row's slot assignment.  For instance, @code{ARLOTJE::AV.5312}
;;.is an annotated value describing the @code{GENDER} slot of Ken's
;;.description.

;;.After @code{EU} calls @code{DU} to produce the table of properties,
;;.it enters a command loop, reading s-expressions as commands (with
;;.@code{READ}).  If the command is a list, it is evaluated and the
;;.result printed out.  If it is a symbol, there are two
;;.possibilities: it is either a command (from a list of possible
;;.commands) or it is the name of a unit.  If it is a name of a
;;.unit, the editor shifts to editing that unit and prints out a
;;.corresponding description (again with @code{DU}) of the unit being
;;.edited.  If it is a command, it is invoked: commands do things like
;;.extracting values of slots, modifying slots, or browsing various
;;.structures (i.e. class structures of dependency structures) in
;;.which the edited unit is embedded.


;;;;. Sorting out important slots

(define-unit important-slots
  (english-description "These are the slots of this unit which are particular important.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'unitp)
  (must-be 'slotp))

(defparameter *de-facto-important-slots*
  '(english-description important-slots member-of works-like makes-sense-for must-be lisp-coder
    generalizations specializations)
  "These are slots which are always pretty important.")

(defparameter *de-facto-unimportant-slots* '(creation-id type-code)
  "These are slots which are seldom very important.")

(defvar *currently-important-slots* '()
  "Slots important currently.")

(defun important-slots (unit)
  (append *currently-important-slots*
	  (if (%hasnt unit 'important-slots) '() 
	      (get-value unit 'important-slots))
	  *de-facto-important-slots*))


;;;;. DU: How Units Are Described

(defvar *du-width* 79
  "This is the default width of the terminal to use in describing ARLOtje units.")
(defvar *du-single-line-format* T
  "Whether DU uses multiple lines for large entries.")
(defvar *default-slot-filter* 'slotp
  "Tests determining which slots are shown or not shown.")

;;.The @code{DU} function prints a table of properties describing a
;;.unit (specified as its first argument).  It accepts three keyword
;;.arguments:
;;.@table @code
;;.@item :stream
;;.The stream to which the description is sent, defaulting to the
;;.standard output.
;;.@item :note-annotated-values
;;.Whether or not a third column --- noting the annotated value
;;.describing the row's value --- is printed.
;;.The default is @code{T}.
;;.@item :slot-filter
;;.A LISP predicate or ARLOtje @emph{test} denoting the slots which
;;.should be printed or not printed.  The default for this is the
;;.@code{default-slot-filter} slot of the result of
;;.@code{(arlotje-user)}.  This slot, by default, is simply the
;;.predicate @code{slotp} which is true of any declared slot (i.e. any
;;.unit from which ARLOtje can extract a @code{to-get-value} function).
;;.@end table
;;.@findex{du}
#-GENERA
(defun du (unit &optional slot &key
		(stream *standard-output*)
		(single-line-format *du-single-line-format*)
		(slot-filter *default-slot-filter*))
  "Describes a unit, with options for listing annotated value descriptions, or
recursively describing them."
  (let* ((unit (if slot (%get unit slot) unit))
	 (important-slots (important-slots unit))
	 (max-slot-length
	  (do ((props (properties unit) (cddr props))
	       (size 0 (max size (length (symbol-name (car props))))))
	      ((null props) size)))
	 (split (min (+ max-slot-length 7) (floor (/ *du-width* 2))))
	 (columns (format NIL "~~%~~2T~~A~~~DT~~A" split)))
    (labels ((stringify (value)
	       (with-output-to-string (string-stream)
		 (let ((*print-pretty* T)
		       (*print-right-margin* (- *du-width* split)))
		   (declare (special *print-pretty* *print-right-margin*))
		   (write value :stream string-stream))))
	     (print-row (slot value)				  
	       (let* ((slot-string (format NIL "~A" slot))
		      (value-string (stringify value)))
		 (cond ((and (or (> (length value-string) (- *du-width* split))
				 (find #\Newline value-string))
			     single-line-format)
			(format stream columns slot-string
				(nsubstitute #\space #\newline 
					     (concatenate 'string (subseq value-string 0 (- *du-width* split 4)) "...")
					     :test 'char=)))
		       ((or (> (length value-string) (- *du-width* split))
				 (find #\Newline value-string))
			(format stream columns slot-string "vvvvvvvv")
			(let ((*print-pretty* T) (*print-right-margin* *du-width*))
			  (declare (special *print-pretty* *print-right-margin*))
			  (if (fboundp 'pprint) (pprint value) (print value))))
		       (T (format stream columns slot-string value-string))))))
      (terpri)
      (dotimes (i *du-width*) (write-char #\- stream))
      (format stream columns "Slot" "Value")
      (format stream columns "----" "-----")
      (dolist (sl important-slots)
	(unless (failurep (%get unit sl))
	  (print-row (symbol-name sl) (get-value unit sl))))
      (dolist (sl (get (type-of unit) 'invisible-slots))
	(unless (member sl important-slots)
	  (print-row (symbol-name sl) (get-value unit sl) )))
      (doslots unit (slot value)
	(unless (or (member slot important-slots) (member slot *de-facto-unimportant-slots*)
		    (failurep value) (not (funcall slot-filter slot)))
	  (print-row slot value)))
      (dolist (sl *de-facto-unimportant-slots*)
	(unless (failurep (%get unit sl))
	  (print-row (symbol-name sl) (get-value unit sl))))
      (terpri)
      (dotimes (i *du-width*) (write-char #\- stream))
      unit)))

#-GENERA
(defparameter *hairy-formatter-for-du*
  (if (fboundp 'formatter)
    (formatter "~%~2T~S:  ~30T~@[~A~] ~@[~80T\; (~S)~]~@[~%~@<     ~@\;~W~:>~]")
    ;; standard old-cl version
    #'(lambda (stream &rest stuff)
        (apply #'format 
               stream 
               "~%~2T~S:  ~30T~@[~A~] ~@[~80T\; (~S)~]~@[~%~@<     ~\;~S~>~]" 
               stuff))))

#+GENERA
(defun du (unit &key
		(stream *standard-output*)
		ignore
		(single-line-format *du-single-line-format*)
		(slot-filter *default-slot-filter*))
  "Describes a unit, with options for listing annotated value descriptions, or
recursively describing them."
  (declare (ignore single-line-format))
  (let ((*standard-output* stream)
	(*print-case* :downcase))
    (fresh-line)
    (dw:formatting-table ()
      (dw:formatting-column-headings ()
	(scl:with-character-face (:italic)
	  (dw:with-underlining ()
	    (dw:formatting-cell ()
	      (write-string "Slot"))
	    (dw:formatting-cell ()
	      (write-string "Value"))
	    (dw:formatting-cell ()
	      (write-string "Value-Description")))))
      (doslots unit (slot value description)
	(unless (or (failurep value) (not (funcall slot-filter slot)))
	  (dw:formatting-row ()
	    (dw:formatting-cell () (dw:present slot))
	    (dw:formatting-cell ()
	      (if (listp value)
		  (progn
		    (princ "(")
		    (dw:format-item-list value :fresh-line nil :return-at-end nil
					 :additional-indentation 0 :optimal-number-of-rows 1
					 :max-width 600)
		    (princ ")"))
		  (dw:present value)))
	    (dw:formatting-cell () (dw:present description))))))))

; #+CLIM
; (defun du (unit &key
; 		(stream *standard-output*)
; 		ignore
; 		(single-line-format *du-single-line-format*)
; 		(slot-filter *default-slot-filter*))
;   "Describes a unit, with options for listing annotated value descriptions, or
; recursively describing them."
;   (declare (ignore single-line-format))
;   (clim-lisp:fresh-line)
;   (let ((*standard-output* stream)
; 	(*print-case* :downcase))
;     (clim:formatting-table ()
;       (clim:formatting-row ()
; 	(clim:with-text-face (:italic)
; 	  (clim:formatting-cell ()
; 	    (clim-lisp:write-string "Slot"))
; 	  (clim:formatting-cell ()
; 	    (clim-lisp:write-string "Value"))
; 	  (clim:formatting-cell ()
; 	    (clim-lisp:write-string "Value-Description"))))
;       (doslots unit (slot value description)
; 	(unless (or (failurep value) (not (funcall slot-filter slot)))
; 	  (clim:formatting-row ()
; 	    (clim:formatting-cell () (clim:present slot))
; 	    (clim:formatting-cell ()
; 	      (if (listp value)
; 		  (progn
; 		    (clim-lisp:write-char #\()
; 		    (clim:formatting-item-list (*standard-output* :max-width 600)
; 		      (dolist (v value)
; 			(clim:formatting-cell ()
; 			  (clim:present v))))
; 		    (clim-lisp:write-char #\)))
; 		  (clim:present value)))
; 	    (clim:formatting-cell () (clim:present description))))))))

;;.The @code{default-slot-filter} slot stores the test or predicate
;;.which filters the slots described by @code{DU}.
;;.@vindex{default-slot-filter (slot)}
(define-unit default-slot-filter
  (works-like 'prototypical-slot)
  (must-be 'function-namep)
  (value-defaults-to 'slotp))


(setf (get 'annotated-value 'invisible-slots)
      '(annotated-value-unit annotated-value-slot
	invalidated-p depends-on depended-on-by))
(setf (get 'element 'invisible-slots)
      '(element-in-set element-value
	invalidated-p depends-on depended-on-by))


;;;;.Defining EU Commands

;;.@code{EU} commands are defined by the macro
;;.@code{define-unit-editor-command}.  Such definitions have the form:
;;.@example
;;.(define-unit-editor-command @var{name} (@var{alias1} @var{alias2} ....)
;;.  @var{body referring to }@code{unit})
;;.@end example
;;.which defines a command invokable by @var{name} or any of
;;.@var{alias}es.
;;.@findex{define-unit-editor-command (macro)}
(defmacro define-unit-editor-command (name nicknames &body body)
  "Defines a unit editor command name with nicknames NICKNAMES that
executes BODY with `unit' bound to the unit being edited.  The value
returned by body becomes the new unit being edited."
  (let* ((nicknames (mapcar 'symbol-name nicknames))
	 (menu-command (member "MENU" nicknames :test 'equal)))
    `(progn
      (defun ,(fsymbol *package* "EU-~A" name) (unit)
	,@body)
      (pushnew '((,(symbol-name name) ,@(remove "menu" nicknames :test 'equal))
		 ,(fsymbol *package* "EU-~A" name))
       *eu-commands* :test 'equal)
      ,@(if menu-command `((pushnew ',name *eu-menu* :test 'equal)) '())
      ',(fsymbol (symbol-package name) "EU-~A" name))))

;;.One special alias, @code{menu}, places the defined command on the
;;.default menu of commands listed after each unit description; e.g.:
;;.@smallexample
;;.Description Of KEN.HAASE (Q to QUIT)
;;.----------------------------------------------------------------------------------------------------
;;.  Slot                        Value                                             Value Description
;;.  ----                        -----                                             -----------------
;;.  CREATION-ID:                ARLOTJE::|NOISE.MEDIA.MIT.EDU-8.20.59-14June1990| 
;;.  MEMBER-OF:                  (MEN QUAKERS)                                     ; (ARLOTJE::AV.5256)
;;.  TYPE-CODE:                  130944                                            ; (AV.5428)
;;.  GENDER:                     ARLOTJE::MALE                                     ; (ARLOTJE::AV.5312)
;;.  DEFAULT-SLOT-FILTER:        SLOTP                                             ; (AV.5425)
;;.----------------------------------------------------------------------------------------------------
;;.  Commands:   COPY   ASSERT   RETRACT   GET   WALK   DEPENDENTS   PUSH   TYPES   HELP   QUIT
;;.----------------------------------------------------------------------------------------------------
;;.Editing KEN.HAASE >
;;.@end smallexample

;;.In this framework, one might define the command @code{GET} as follows:
;;.@example
;;.(define-unit-editor-command GET (GET-VALUE G MENU)
;;.  "EU Command for getting a slot's value."
;;.  (format T "~&Slot to get?")
;;.  (let ((value (get-value unit (read))))
;;.    (if (failurep value)
;;.	(progn (format T "~&Failed: ~A" value) unit)
;;.	(if (unitp value) value
;;.	    (progn (format T "~&Value is: ~S" value) unit)))))
;;.@end example
;;.specifying a command which prompts for and reads a slot name and
;;.then gets that slot's value from the current unit.  The current
;;.unit is referred to by the variable @code{unit} exported by the
;;.ARLOtje package.

;;.Global unit editor commands are maintained in the list
;;.@code{*EU-COMMANDS*}; they are maintained in a-list of the form:
;;.@example
;;.(@var{aliases} @var{implementation})
;;.@end example
;;.where @var{aliases} consists of the command name and its aliases,
;;.and the @var{implementation} slot contains the name of the function
;;.implementing the command, usually of the form @code{EU-@var{name}}
;;.for a command defined with a primary name @var{name}.
(defvar *EU-COMMANDS* '()
  "This is an association list of EU commands.")

;;.The variable @code{*EU-MENU*} is a list of those commands displayed
;;.in the menu list.  A future version of @code{EU} will also include
;;.specialized commands for the particular unit being edited.
(defvar *eu-menu* '()
  "This is list of EU commands in the menu.")



;;;;.The EU editor loop

;;.When @code{EU} is first invoked, it examines its first argument,
;;.the unit being edited.  It calls @code{try-to-load-unit} to load up
;;.the unit's description (from the @file{units} and @file{sessions}
;;.directories), providing the unit has not already been loaded or
;;.otherwise declared.  It then calls the function @code{DU} on the
;;.unit, prints out a list of `menu commands', and prompts with
;;.@example
;;.Editing KEN.HAASE >
;;.@end example
;;.From this prompt, it reads a command with the function @code{READ}
;;.in whatever package is current; if the command is a list, it is
;;.evaluated and the result printed with @code{pprint}.  If the
;;.command is a symbol, the names @samp{Q}, @samp{QUIT}, or
;;.@samp{EXIT} are all especially identified as exit commands and
;;.handled by leaving @code{EU}.  Otherwise, if the command is a
;;.symbol it is first looked up (by name without package) on
;;.@code{*EU-COMMANDS*}; if a corresponding command is found, its
;;.implementation is called on the unit being edited.  The resulting
;;.value, if it is a unit, is then made the current edit object;
;;.otherwise, the result of the command is ignored and the command
;;.loop continues by describing the current unit again and prompting
;;.for another command.  If the command entered is not a command on
;;.@code{*EU-COMMANDS*}, it tries to coerce it into a unit by calling
;;.@code{guess-at-unit} and then changes the edited object to this
;;.unit.
;;.@findex{EU}
(defun eu (unit)
  "Edits the unit UNIT."
  (try-to-load-unit unit)
  (loop (format t "~&Description Of ~S (Q to QUIT)" unit)
    (if (and (unitp unit) (or (annotated-valuep unit) (get unit 'creation-id)))
	(du unit)
      (describe unit))
    (format t "~%  Commands:~{   ~a~}" (reverse (append '(quit help) *eu-menu*)))
    (terpri) (dotimes (i *du-width*) (write-char #\-))
    (format t "~&Editing ~s > " unit)
    (let* ((command (read)))
      (cond ((and (symbolp command)
		  (member (symbol-name command) '("Q" "QUIT" "EXIT") :test 'equal))
	     (return unit))
	    ((symbolp command)
	     (do ((command-name (symbol-name command))
		  (commands *eu-commands* (cdr commands)))
		 ((or (null commands)
		      (member command-name (car (first commands)) :test 'equal))
		  (if (null commands)
		      (setq unit (or (or (and (slotp command) (%get-default unit command NIL))
					 (and (slotp (guess-at-unit command))
					      (%get-default unit (guess-at-unit command) NIL)))
				     (guess-at-unit command)
				     unit))
		    (let ((result (funcall (cadr (first commands)) unit)))
		      (if (unitp result) (setq unit result)))))))
	    (T (pprint (eval command) ))))))

;;.The function @code{guess-at-unit} tries to coerce a symbol into a
;;.valid unit name.  It first checks if its argument (which should be
;;.a symbol) is already a unit, annotated value, or session
;;.description.  If so, it is just returned.  Otherwise, it looks in
;;.all the packages which use the @code{ARLOtje} package for genuine
;;.units with the name given.  If any is found, it is returned.
;;.Finally, it tries to load the unit by calling
;;.@code{try-to-load-unit} with the keyword argument
;;.@code{:declare-otherwise} which ensures that the unit will be
;;.declared as a unit even if no description of it exists to load.
;;.@findex{guess-at-unit}
(defun guess-at-unit (unit-name)
  "Tries to find a declared unit with a name matching UNIT-NAME."
  (block guessing-at-unit
    (when (or (get unit-name 'creation-id) (get unit-name 'session-start)
	  (get unit-name 'annotated-value-token))
      (return-from guessing-at-unit unit-name))
    (dolist (pkg (package-used-by-list *arlotje-package*))
      (let ((unit-name (find-symbol (symbol-name unit-name) pkg)))
	(when (and unit-name
		   (or (get unit-name 'creation-id) (get unit-name 'session-start)
			 (get unit-name 'annotated-value-token)))
	  (return-from guessing-at-unit unit-name))))
    (try-to-load-unit unit-name :declare-otherwise? T)))


;;;;.Useful functions in commands.

;;.There are several useful functions for defining unit editor commands
;;.which interact with the user.  They all prompt the user with a
;;.format-string and arguments and return a particular sort of value.

;;.@code{Eu-read} reads an s-expression from the standard input,
;;.preceding the read with a prompt based on the format string and
;;.arguments it is passed.
(defun eu-read (&rest format-args)
  "Prompts (with FORMAT-ARGS) and reads (with READ) an object from the standard output."
  (unless (null format-args)
    (apply #'format T format-args))
  (read *standard-input*))

;;.@code{Eu-read-unit} reads a unit name from the standard output,
;;.prompting with the specified format-string and arguments.  It tries
;;.to load the unit if possible and otherwise asks whether the symbol
;;.read should be declared as a unit.  If the user replies No, it
;;.loops to prompt and read again.
(defun eu-read-unit (&rest format-args)
  "Prompts (with FORMAT-ARGS) and reads (with READ) a unit name from the standard output.
It tries to load the unit if possible and otherwise asks whether the
symbol read should be declared as a unit.  If the user replies No, it
loops to prompt and read again."
  (unless (null format-args)
    (apply #'format t format-args))
  (let ((unit (read *standard-input*)))
    (cond ((not (symbolp unit))
	   (format T "~&Type the name of a unit...")
	   (apply #'eu-read-unit format-args))
	  (T (try-to-load-unit unit)
	     (if (get unit 'creation-id) unit
	       (progn (format T "~&~S is not a known unit. Declare it?" unit)
		      (if (yes-or-no-p) unit
			(apply #'eu-read-unit format-args))))))))

;;.@code{Eu-read-slot} reads a slot name from the standard output,
;;.prompting with the specified format-string and arguments.  It tries
;;.to load the unit if possible and otherwise asks whether the symbol
;;.read should be declared as both a unit and a slot.  If the user
;;.replies No, it loops to prompt and read again.
(defun eu-read-slot (&rest format-args)
  "Prompts (with FORMAT-ARGS) and reads (with READ) a slot name from the standard output.
It tries to load the unit if possible and otherwise asks whether the
symbol read should be declared as a unit and slot.  If the user replies No, it
loops to prompt and read again."
  (unless (null format-args)
    (apply #'format t format-args))
  (let ((unit (read *standard-input*)))
    (cond ((not (symbolp unit))
	   (format T "~&Type the name of a slot...")
	   (apply #'eu-read-unit format-args))
	  ((slotp unit) unit)
	  (T (try-to-load-unit unit)
	     (if (get unit 'creation-id) unit
	       (if (yes-or-no-p "~&~S is not a known unit. Declare it?" unit)
		   (make-unit unit
		     (works-like 'prototypical-slot))
		 (apply #'eu-read-unit format-args)))))))


;;;;.EU Commands

;;.The command @samp{COPY} (with alias @samp{X} and appearing on the
;;.command menu) creates and edits a new unit which copies the
;;.properties of the current unit.
;;.@findex{COPY (unit editor command)}
(define-unit-editor-command COPY (X MENU)
  "Copy and edit for units."
  (format T "~&Name for new unit: ")
  (copy-unit (declare-unit (read)) unit))
;;.It calls the function @code{copy-unit} which copies all of the
;;.properties of its second argument onto its first argument.
(defun copy-unit (new-unit unit)
  (dovalues unit (slot value description)
    (unless (or (member slot '(creation-id member-of))
		(check-value new-unit slot value))
      (format T "~&\; Copying ~S: ~S" slot value)
      (assertion new-unit slot value)))
  new-unit)

;;.The command @samp{ASSERT} (with aliases @samp{A}, @samp{SET},
;;.@samp{S} and appearing in the menu) stores a value on the current
;;.unit.  It prompts for a slot name, a value (read with READ) and
;;.stores it (by @code{ASSERTION}) on the current unit.
(define-unit-editor-command ASSERT (A SET S MENU)
  "EU Command for putting a slot's value."
  (let ((slot (eu-read-slot "~&Slot of ~S to modify? " unit)))
    (let ((value (eu-read "Value to store in ~S of ~S? " slot unit)))
      (when (symbolp value) (try-to-load-unit value))
      (assertion unit slot value))
    unit))

;;.The command @samp{RETRACT} (with aliases @samp{RETRACT-VALUE},
;;.@samp{R} and appearing in the command menu), retracts the value for
;;.a slot.  It prompts for the slot and (if the slot is many-valued)
;;.and the value of the slot to retract.
(define-unit-editor-command RETRACT (RETRACT-VALUE R MENU)
  "EU Command for retracting a slot's value."
  (let ((slot (eu-read-slot "~&Slot of ~S to retract? " unit)))
    (if (many-valued-slotp slot)
	(retraction unit slot (eu-read "~&Value to retract from the ~S slot of ~S?" slot unit)))
    (retraction unit slot (get-value unit slot)))
  unit)

;;.The command @samp{GET} (with aliases @samp{GET-VALUE}, @samp{G}
;;.and displayed in the menu) gets a slot from the current unit.  If
;;.the returned value is a unit, the unit is edited; if it is not a
;;.unit, it is printed and the current unit remains the focus of
;;.editing activity.
(define-unit-editor-command GET (GET-VALUE G MENU)
  "EU Command for getting a slot's value."
  (let ((value (get-value unit (eu-read-slot "~&Slot of ~S to get? " unit))))
    (if (failurep value)
	(progn (format T "~&Failed: ~A" value) unit)
	(if (unitp value) value
	    (progn (format T "~&Value is: ~S" value) unit)))))

;;.The command @samp{WALK} (with aliases @samp{W}, @samp{BROWSE},
;;.@samp{B} and appearing in the menu) prompts for a slot and enters
;;.the `lattice walker' over the lattice determined by the slot.
(define-unit-editor-command WALK (W BROWSE B MENU)
  "EU Command for walking an arbitrary lattice."
  (let ((slot (eu-read-slot "~&Walk Lattice of: ")))
    (walk-lattice unit slot)))

;;.The command @samp{DEPENDENTS} (with alias @samp{D} and appearing in
;;.the menu) walks the dependency lattice (@xref{Walking Lattices})
;;.from the current unit.  If the current unit is an annotated value
;;.(and thus directly has dependents or dependencies of its own), the
;;.lattice walk begins there.  Otherwise, a slot is prompted for and
;;.the lattice walk begins with the annotated value for that slot.
(define-unit-editor-command DEPENDENTS (D MENU)
  "EU Command for walking the dependency lattice."
  (walk-lattice (if (or (get unit 'depends-on) (get unit 'depended-on-by)) unit
		  (let ((slot (eu-read-slot "~&Dependencies of which slot? ")))
		    (get-value unit slot)
		    (%get unit slot)))
		'depends-on
		'depended-on-by))

;;.The @samp{HELP} command (with the alias @samp{??} and appearing in
;;.the menu) prints out a list of the current commands.
(define-unit-editor-command HELP (??)
  "EU Command for describing EU commands."
  (terpri) (dotimes (i *du-width*) (write-char #\-))
  (let ((documentation '()))
    (dolist (command *eu-commands*)
      (push (list (format NIL "~A~{, ~A~}" (first (car command)) (rest (car command)))
		  (documentation (cadr command) 'function))
	    documentation))
    (let ((format-string
	   (format NIL "~~&~~5T~~A~~~DT~~A"
		   (+ 10 (apply #'max (mapcar #'length (mapcar #'car documentation)))))))
      (dolist (line documentation)
	(apply #'format T format-string line))))
  (terpri) (dotimes (i *du-width*) (write-char #\-))
  unit)

;;.The command @samp{PUSH} (appearing on the menu) starts a new unit
;;.editor editing a particular unit.  When the user quits from this
;;.embedded unit editor, she will find herself editing the current
;;.unit again.
(define-unit-editor-command PUSH (MENU)
  (eu (eu-read-unit "~&Push with what unit? "))
  unit)


;;;;.Commands for modifying slot filters.

(define-unit-editor-command INCLUDE (INC)
  "EU Command for describing EU commands."
  (let ((test (eu-read "Include which class of slots? ")))
    (setq *default-slot-filter*
	  (if (check-value *default-slot-filter* 'generalizations test) test
	    (if (check-value test 'generalizations *default-slot-filter*) *default-slot-filter*
	      (make-internal-unit (gensymbol 'filter)
		(lisp-coder 'test-union)
		(union-of *default-slot-filter*)
		(union-of test)))))
    unit))

(define-unit-editor-command EXCLUDE (INC)
  "EU Command for describing EU commands."
  (let ((test (eu-read "Exclude which class of slots? ")))
    (setq *default-slot-filter*
	  (make-internal-unit (gensymbol 'filter)
	    (lisp-coder 'test-intersection)
	    (intersection-of *default-slot-filter*)
	    (intersection-of (make-unit (gensymbol 'fliter)
			       (lisp-coder 'test-complement)
			       (complement-of test))))))
  unit)

;;;;.Commands for sessions

;;.A few unit editor commands for dealing with sessions are also
;;.provided.

;;.The command @samp{PUSH-SESSION} creates a new session beneath the
;;.current session and `pushes' to the session within a new unit
;;.editor.  When the user exits that unit editor, she will return to
;;.the session at which the push occured and be editing the unit under
;;.which she pushed the session.
(define-unit-editor-command PUSH-SESSION ()
  (format T "~& What is this sub-session for? ")
  (within-session (read-line)
    (eu unit)))

;;.The command @samp{NEW-SESSION} prompts for a purpose (one line of
;;.text) and creates a new session with that purpose, making the new
;;.session current.  This results in the saving of the previous session.
(define-unit-editor-command NEW-SESSION ()
  (format T "~& What is this new session for? ")
  (new-session (read-line))
  unit)

;;.The command @samp{RESTORE-SESSIONS} (with the alias @samp{RESTORE})
;;.restores all of the sessions in which the current unit was modified.
(define-unit-editor-command RESTORE-SESSIONS (RESTORE)
  (dolist (session (get unit 'references))
    (restore-session session))
  unit)


;;;;.Walking Lattices

;;.ARLOtje provides a simple keyboard based lattice walker invoked
;;.from @code{EU} or by the function @code{walk-lattice}.  The lattice
;;.walker is a command loop which outputs a display like this:
;;.@example
;;.Walking the GENERALIZATIONS lattice around ANIMALS: (Q to QUIT)
;;.       SPECIALIZATIONS       GENERALIZATIONS
;;.       <<<<<<<<<<<<<<<       >>>>>>>>>>>>>>>
;;.       HUMANS                THINGS
;;.       MONKEY                AGENTS
;;.What next?
;;.@end example
;;.indicating that the current point in the lattice is @code{ANIMALS}
;;.and displaying @code{SPECIALIZATIONS} and @code{GENERALIZATIONS} of
;;.@code{ANIMALS}.  The lattice walker is exited by typing either
;;.@samp{Q}, @samp{QUIT} or the name of the current point in the
;;.lattice (e.g. @samp{ANIMALS} in the example above); it then returns
;;.the current point in the lattice.  Any other input to the walker
;;.becomes a new point in the lattice to start from and the lattice
;;.walk continues.

;;.The two lattice walking commands provided by default are the
;;.@samp{TYPES} walker and the @samp{DEPENDENTS} walker.  The
;;.@samp{TYPES} walker was shown above (it walks the
;;.`generalizations/specializations' lattice).  The @samp{DEPENDENTS}
;;.walker walks the dependency network and looks something like this:
;;.@example
;;.I need to make this up.
;;.@end example

;;.The lattice walker can be invoked by a program through the function
;;.@code{walk-lattice} which requires two arguments: a @var{root} and
;;.a @var{up} in the lattice.  An optional third argument --- the
;;.@var{down} in the lattice --- is by default the inverse of the
;;.@var{up}.
;;.@findex{walk-lattice}
(defun walk-lattice (root up &optional (down (get-value up 'inverse-slot)))
  (loop (format T "~&Walking the ~A lattice around ~A: (Q to QUIT)" up root)
    (let ((above (get-value root up)) (below (get-value root down))
	  (*package* *arlotje-package*))
      (let ((header-format-string
	     (format NIL "~~%~~7T~~A~~~DT~~A"
		     (+ (max-symbol-length (cons down below)) 14)))
	    (element-format-string
	     (format NIL "~~%~~7T~~@[~~S~~]~~~DT~~@[~~S~~]"
		     (+ (max-symbol-length (cons down below)) 14))))
	(format T header-format-string down up)
	(format T header-format-string
		(make-string (length (symbol-name down)) :initial-element #\<)
		(make-string (length (symbol-name up)) :initial-element #\>))
	(do ((above above (and above (rest above)))
	     (below below (and below (rest below))))
	    ((and (null above) (null below)))
	  (format T element-format-string
		  (if (null below) '() (car below))
		  (if (null above) '() (car above)))))
      (let ((next (eu-read "~&What next?")))
	(if (or (member next '(QUIT Q)) (eq next root))
	    (return root)
	  (if (satisfies? next (get-value up 'makes-sense-for))
	      (setq root next)))))))

(defun max-symbol-length (symbols)
  (do ((symbols symbols (rest symbols))
       (max-length 0 (max (length (symbol-name (car symbols))) max-length)))
      ((null symbols) max-length)))