;;; -*- LISP -*-

;;.@chapter Collections
(in-package :arlotje)

(proclaim '(inline collectionp))

;;.ARLOtje implements a special sort of test called the
;;.@dfn{collection}. Collections are finite sets of enumerated
;;.members; memership in a collection can be asserted explicitly and
;;.tested.  ARLOtje builds its basic ontology @xref{The Base Ontology}
;;.on top of these collections, since they provide a good way of
;;.distinctly labelling units with no other particular properties.
;;.Collections are generally used when there is no single decisive
;;.test that indicates a property; we thus merely assert the property
;;.and let other things follow from that assertion.


;;;;.Distinguishing collections

;;.Collections are distinguished by the @code{collection} property;
;;.the predicate @code{collectionp} checks for this property; unlike
;;.@code{test} which inherits through the @code{works-like} slot,
;;.@code{collection} is generally assigned by the assertion of a
;;.collections membership in the collection @code{collections}.
;;.The significant distinction from this is the collection
;;.@code{collections} itself.

;;.Collections are also tests and they inherit the @code{test}
;;.property through the @code{works-like} hierarchy from
;;.@code{prototypical-collection} and then @code{prototypical-test}.

(define-unit collection
  (english-description "This is true for units which are collections.")
  (works-like 'prototypical-primitive-slot)
  (to-put-value '%put)
  (to-get-value 'get))
(defun collectionp (x) (get x 'collection))
(define-unit collectionp
  (works-like 'prototypical-test))

(define-unit collections
  (works-like (make-unit 'prototypical-collection
		(works-like 'prototypical-test)
		(to-enforce-test 'enforce-membership)))
  (collection t))

(defun enforce-membership (x collection)
  "Asserts that X is a MEMBER-OF the collection COLLECTION."
  (assertion x 'member-of collection))


;;;;.Slots of collections

;;.Collections are organized into a lattice by the relations
;;.@code{supersets} and @code{subsets} which are @code{spec-slots}
;;.@pxref{Slot Generalizations} of @code{generalizations} and
;;.@code{specializations} respectively.

(define-unit supersets
  (english-description "These are the sets in which this set is contained.")
  (works-like 'prototypical-set-slot)
  (kleene-plus-of 'supersets)
  (makes-sense-for 'collections)
  (must-be 'collections)
  (genl-slots 'generalizations))

(define-unit subsets
  (english-description "These are the sets which are contained in this one.")
  (works-like 'prototypical-set-slot)
  (kleene-plus-of 'subsets)
  (makes-sense-for 'collections)
  (must-be 'collections)
  (inverse-slot 'supersets))

;;.The members of collections are determined by the @code{member-of}
;;.and @code{members} slots respectively.  The @code{members} slot
;;.pushes through the @code{supersets} slot to maintain the transitive
;;.closure of @code{members} through the @code{supersets} lattice.

(define-unit member-of
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'unitp)
  (must-be 'collections))

(define-unit members
  (works-like 'prototypical-set-slot)
  (inverse-slot 'member-of)
  (pushes-through 'supersets))


;;;;.Subset tests

;;.Subset tests are tests that return true for subsets of another
;;.test.  They are automatically generated by the @code{has-value}
;;.coder and lazily computed as values of a collection's
;;.@code{subset-test} slot.  For instance, all subsets of
;;.@code{things} could be tested by:
;;.@example
;;.(get-value 'things 'subset-test)
;;.==> HAS-VALUE.1
;;.(HAS-VALUE.1 'ANIMALS) ==> T 
;;.(HAS-VALUE.1 'ARTIFACTS) ==> T
;;.(HAS-VALUE.1 'SPATIAL-EXTENTS) ==> NIL
;;.@end example

(define-unit subset-test
  (works-like 'prototypical-slot)
  (makes-sense-for 'collections)
  (must-be 'function-namep)
  (to-compute-value 'compute-subset-test))

(defun compute-subset-test (collection slot)
  "Computes a test which checks whether or not a collection is
underneath another collection in the subset lattice."
  (declare (ignore slot))
  (fcn has-value
       (tests-slot 'supersets)
       (tests-for-value collection)))


;;;;.Commuting put-demons

;;.Put demons can be made commutative in most cases by having a put
;;.demon on put demons which looks at all possible units which might
;;.have the slot in question and attempting to apply the put demon to
;;.its value.  This method will only work if it is possible to
;;.enumerate the possible units which might possess the slot; this is
;;.possible in general only if the domain (the @code{makes-sense-for}
;;.slot of the slot) is a collection.  But in these cases, the put
;;.demon will be applied retroactively to all the members of this
;;.collection.

(defun retroactive-put-demon (put-demon test slot)
  "Retroactively runs a put-demon on asserted values."
  (when (get slot 'to-put-value)
    (let ((collection (find-superior-collection test)))
      (if collection
	  (do-members (member collection 'members)
	    ;; Go over each of the domains current members,
	    (do-members (value member slot)
	      ;; and each of the member's values for the slot,
	      ;;  calling the put demon on each one.
	      (run-put-demon put-demon member slot value)))
	(dolist (unit *all-units*)
	  (when (and (funcall test unit) (not (%hasnt unit slot)))
	    (do-members (value unit slot)
	      ;; and each of the member's values for the slot,
	      ;;  calling the put demon on each one.
	      (run-put-demon put-demon unit slot value))))))))

(defun find-superior-collection (test)
  "Returns a collection containing TEST."
  (if (collectionp test) test
      (some #'find-superior-collection
	    (get-value (get test 'generalizations) 'elements-as-set))))

;; This works this way because otherwise it retroactively runs all put
;; demons (doing a good, logical, horribly expensive job).
(annotated-value-put 'put-demons 'put-demons
		     '(retroactive-put-demon
		       %value% (get-value %unit% 'makes-sense-for) %unit%))


;;;;.Redefining satisfaction

;;.Once collections are defined, @code{satisfies?} is extended to
;;.handle them.  In addition, it is finally extended to signal an
;;.error if the @code{test} argument is neither a test function nor a
;;.collection. 

;;Do this first.
(put-value 'collections 'member-of 'collections)

(defun satisfies? (x test)
  "Returns true if X satisfies the test or is in the collection TEST."
  (cond (*reckless?* t)
	((collectionp test)
	 (and (symbolp x) (not (%hasnt x 'creation-id))
	      (find-value x 'member-of test)))
	((testp test) (funcall test x))
	(T (error "~S isn't a valid test" test)))) 
;;.@findex{satisfies? (version for collections)}


;;;;.Inheritance from collections

;;.Generic inheritance @xref{Generic Inheritance} is used to assign
;;.properties to members of collections.  The @code{members-have} slot
;;.contains the associated inferences for the @code{all-members} slot.
(define-unit members-have
  (english-description "Default properties for members of this slot.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'collections)
  (associated-inferences-for 'members))
;;.@vindex{members-have (slot)}

;;.Thus we would say, for instance,
;;...@example
(put-value 'collections 'members-have '(works-like prototypical-collection))
(put-value 'collections 'members-have '(collection t))
;;...@end example
;;.to assert that all members of @code{collections} have a
;;.@code{works-like} of @code{prototypical-collection}.  Note that
;;.it is important that the @code{members-have} slot is defined
;;.@emph{after} the @code{propogate-all-members} put-demon is assigned
;;.to all members.  This ensures that one climbs the lattice before
;;.processing @code{members-have} so that more general `defaults' are
;;.run first.


;;;;.The MINIMAL-MEMBER-OF slot
;
;;.Often it is useful to get a generating set for the lattice of
;;.collections above an instance.  The slot @code{Minimal-Member-of}
;;.returns a minimal subset of the @code{member-of} slot of a unit;
;;.this means that no member of this set is a specialization of any
;;.other member.
(define-unit Minimal-Member-Of
  (english-description
   "The collections of which the object which a unit describes is a member.")
  (works-like 'prototypical-cached-slot)
  (makes-sense-for 'anythingp)
  (must-be         'listp)
  (to-compute-value 'compute-minimal-member-of))
;;.@findex{Minimal-Member-Of (slot)}

;;.The minimal members of a set are computed by the defaulting
;;.function @code{compute-minimal-member-of}.
(defun compute-minimal-member-of (unit slot)
  "Computes the minimal set of collections which a unit satisfies."
  (declare (ignore slot))
  (let ((member-of (get-value unit 'member-of)))
    (remove-if #'(lambda (x)
		   (some #'(lambda (y)
			     (and (not (eq x y)) (find-value y 'generalizations x)))
			 member-of))
	       member-of)))
;;.@findex{compute-minimal-member-of (put-demon)}


;;;;.Useful Collection Macros

;;.There are several macros for using collections.  The macro
;;.@code{new-unit} takes a class name and any number of slot/value
;;.specs; it constructs a unit name by tacking an integer suffix to
;;.the class name and then asserts that the unit is a member of the
;;.corresponding class.  This established, it assigns the specified
;;.slots and returns the modified unit.

(defmacro new-unit (class &body slot-specs)
  "Generates a new unit whose name is a gensym of CLASS, which is
asserted to be MEMBER-OF the collection CLASS and has slots SLOT-SPECS."
  `(let ((unit (declare-unit (gensymbol ',class))))
    (assertion unit 'member-of ',class)
    ,@(mapcar #'(lambda (sl) `(assertion unit ',(first sl) ,@(rest sl)))
       slot-specs)
    unit))
;;.@findex{new-unit (macro)}

;;.For instance,
;;.@example
;;.(let ((top (new-unit 'brick))
;;.      (right (new-unit pillar (below top) (above table)))
;;.      (left  (new-unit pillar (below top) (above table) (left-of right))))
;;.  (list top right left))
;;.===> (BRICK.1 PILLAR.1 PILLAR.2)
;;.@end example

;;.A functional version of @code{new-unit} is @code{instantiate} which
;;.takes a collection and makes a unit which is a @code{member-of} the
;;.corresponding collection.
(defun instantiate (collection)
  "Returns a unit which is a MEMBER-OF the collection COLLECTION."
  (make-internal-unit (gensymbol collection)
    (member-of collection)))

;;.The macro @code{docollections} has the form of a @code{dolist} or
;;.@code{dotimes} and executes @code{body} with an iterative variable
;;.bound to each collection which an object satisfies.
(defmacro docollections ((class-var object return-form) &body body)
  `(let ((queue (get-value ,object 'minimal-member-of))
	 (marks '()))
    (loop (if (null queue) (return ,return-form))
      (if (member (car queue) marks) (setq queue (cdr queue))
	  (let ((,class-var (car queue)))
	    (setq marks (cons ,class-var marks)
		  queue (append (cdr queue) (get-value ,class-var 'generalizations)))
	    ,@body)))))
;;.@findex{docollections (macro)}


;;;;.Editor Commands

;;.Several unit editor commands use collections.  The command
;;.@code{MAKE} (not on the menu) makes an instance of the current
;;.class with a name which is prompted for.  The command
;;.@code{COLLECTIONS} walks the lattice of generalizations and
;;.specializations.  This was described in @pxref{Walking Lattices}.

(defun create-unit (unit class)
  "Creates a unit named UNIT as a member of CLASS."
  (try-to-load-unit class)
  (if (and (get class 'creation-id) (satisfies? class 'collections))
      (let ((unit (declare-unit unit)))
	(assertion unit 'member-of class)
	(eu unit))
      (error "~S isn't a class I can instantiate" class)))
;;.@findex{create-unit}
(defun cu (unit class) ; Short alias
  (create-unit unit class))
;;.@findex{cu}

(define-unit-editor-command INSTANTIATE (MAKE)
  "Instantiates an instance of a collection."
  (if (satisfies? unit 'collections)
      (let ((name (eu-read "~&Name for instantiation of ~A: " unit)))
	(declare-unit name)
	(assertion name 'member-of unit)
	name)
      (progn (format T "Can't figure out what to instantiate!") unit)))
;;.@findex{INSTANTIATE (unit editor command)}

(define-unit-editor-command COLLECTIONS (LATTICE L CLASSES C MENU)
  "EU Command for walking the generalizations lattice."
  (walk-lattice (cond ((satisfies? unit 'collections) unit)
		      (T (format T "~&Collections of ~S" unit)
			 (dolist (collection (get-value unit 'member-of))
			   (format T "~%~5T~A" collection))
			 (eu-read-unit "~&Where do I start? ")))
		'generalizations))
;;.@findex{COLLECTIONS (unit editor command)}
