;;; -*- LISP -*-

;;.@chapter Annotated values
(in-package :arlotje)

;;.ARLOtje uses @dfn{annotated values} to allow the description of
;;.meta-level information regarding slots and values computed or
;;.asserted by the system.  This meta-level information might include
;;.the person who asserted a slot, the other slots on which a value
;;.depends, or the degree to which an assertion is believed.  For
;;.every annotated slot of a unit, the system defines a unit (usually
;;.of the form @code{@var{unit}*@var{slot}}) to describe that value;
;;.trying to get the value of the slot accesses this
;;.@code{description} to yield a value.

;;.Annotated values are like the active values of @cite{Loops}, the
;;.`SeeUnits' of @cite{RLL} and @cite{CYC}, the dependency network
;;.maintained by @cite{ARLO} (the predecessor of ARLOtje), and the
;;.tweedleboxes of @cite{KRS}.  In addition to being used to make
;;.claims about the relations between particular slot values, they are
;;.used in @pxref{Many valued slots} to implement multiple valued
;;.slots.

;;.Annotated values introduce a different level of access to ARLOtje,
;;.the @dfn{assertion level}.  At the assertion level, we refer not to
;;.units, slots or values, but to the composite which is asserting (or
;;.verifying) that a unit possesses a slot with a certain value.  While
;;.@code{get-value} is still used to extract values, we can often check
;;.that an assertion holds or does not hold with less effort than
;;.getting the value of a slot.

(proclaim '(inline dependency! annotated-value))

;; A kludge for MACL to force inline coding...
#+(AND CL2 MACL)
(define-compiler-macro dependency! (value)
  (let ((v (gensym "value")))
    `(let ((,v ,value))
       (declare (optimize (speed 3) (safety 0)))
       (when *computing-default*
         (when (assertionp ,v)
           (when (%hasnt ,v 'constant-value)
             (jamnew ,v *current-computation*))))
       ,v)))



;;;;.Dependencies

(defstruct (assertion (:include whonit) (:predicate assertionp)
		      (:constructor make-assertion ()))
  (invalidated NIL)
  (depends-on '())
  (depended-on-by '()))

(defun store-dependencies (assertion slot depends-on)
  "Records the annotated value VALUE as depending on ON-VALUES."
  (declare (ignore slot)) ; This is always DEPENDS-ON.
  (push depends-on (assertion-depends-on assertion))
  (dolist (support depends-on)
    (jamnew assertion (assertion-depended-on-by support))))

(defun invalidate-dependents (assertion)
  "Invalidates ASSERTION and all its dependents, recursively.
The annotated values which are depended on by a value are stored as a
list in its DEPENDED-ON-BY slot, recursively."
  (dolist (dependent (assertion-depended-on-by assertion))
    (when (not (null (assertion-depends-on dependent)))
      (setf (assertion-depends-on dependent)
	    (remove-if #'(lambda (l) (member assertion l :test #'eq))
		       (assertion-depends-on dependent)))
      (when (null (assertion-depends-on dependent))
	; Equivalent to (invalidate-assertion dependent)
	(setf (assertion-invalidated dependent) T)
	(invalidate-dependents dependent)))))

(defun invalidate-assertion (assertion)
  "Invalidates ASSERTION and all its dependents, recursively.
The annotated values which are depended on by a value are stored as a
list in its DEPENDED-ON-BY slot, recursively."
  (setf (assertion-invalidated assertion) T)
  (invalidate-dependents assertion))


;;;;. Annotated Values

;;.Annotated values are special sorts of ARLOtje units.  One use of
;;.this is that it allows one to know when a given slot value was
;;.asserted or inferred; another is that --- when automatically
;;.tracing dependencies --- it is possible to see which sessions
;;.contributed to a particular conclusion.

(defstruct (annotated-value (:include assertion) (:predicate annotated-valuep)
			    (:print-function print-annotated-value)
			    (:constructor make-annotated-value (unit slot)))
  unit slot)
(defun print-annotated-value (av stream &rest options)
  (declare (ignore options))
  (format stream "#<AV ~S ~S>"
	  (annotated-value-unit av) (annotated-value-slot av)))

(defun install-annotated-value (unit slot)
  (let ((*computing-default* nil))
    (declare (special *computing-default*))
    (check-get-value unit slot))
  (let ((new-av (make-annotated-value unit slot)))
    (%put unit slot new-av)
    new-av))

(defun annotated-value (unit slot)
  (let ((av (%get unit slot)))
    (if (failurep av) (install-annotated-value unit slot) av)))

(define-primitively annotated-value-unit
  (english-description "The assertions which this annotated value depends on.")
  (makes-sense-for 'assertionp)
  (must-be 'list-of-assertionsp)
  (to-get-value 'whonit-get)
  (to-put-value 'whonit-put)
  (whonit-type 'annotated-value)
  (whonit-accessor 'annotated-value-unit))

(define-primitively annotated-value-slot
  (english-description "The assertions which this annotated value depends on.")
  (makes-sense-for 'assertionp)
  (must-be 'list-of-assertionsp)
  (to-get-value 'whonit-get)
  (to-put-value 'whonit-put)
  (whonit-type 'annotated-value)
  (whonit-accessor 'annotated-value-slot))

;;.The predicate @code{list-of-annotated-valuesp} returns true of
;;.lists whose elements are all annotated values.@refill
(defun list-of-annotated-valuesp (x)
  "Returns true for lists of annotated values."
  (and (listp x) (every 'annotated-valuep x)))
;;.@findex{list-of-annotated-valuesp}


;;.The Assertion Level

;;.The function @code{find-value} is a reflexive operator which calls
;;.a slot's @code{to-find-value} function.  This is a function which
;;.`proves' that a particular value holds; it is more exacting than
;;.@code{get-value} because it focusses on a particular value.  Thus,
;;.if a slot has many values, @code{find-value} will find the
;;.annotated value corresponding to exactly one of them.
(defun find-value (unit slot value)
  (if (computing? find-value unit slot value) NIL
    (computing (find-value unit slot value)
       (let ((method (get-slot-method slot 'to-find-value)))
	 (if (failurep method) method
	   (funcall method unit slot value))))))

;;.The @dfn{assertion level} is maintained by a pair of ARLOtje
;;.internal functions @code{find-value} and @code{assert-value} (the
;;.outside versions of these will be called @code{query} and
;;.@code{assertion}).  These external versions also do event posting
;;.to the current session.

;;.@code{Assert-value} gets the current annotated value describing the
;;.assertion that @var{slot} of @var{unit} has @var{value}.  If there
;;.is such a description it asserts @code{annotations} upon it; if
;;.there is not, it calls @code{put-value} to create one with the
;;.appropriate annotations.
(defun assert-value (unit slot value &rest annotations)
  "Does a PUT-VALUE (with ANNOTATIONS) if SLOT of UNIT is not VALUE."
  (let ((assertion (or (find-value unit slot value) (put-value unit slot value))))
    (cond ((failurep assertion) (put-value unit slot value))
	  (T (do ((props  annotations (cddr props))
		  (values (cdr annotations) (cddr values))
		  (dependencies? nil (or dependencies? (eq (car props) 'depends-on))))
		 ((null props) 
		  (unless dependencies?
		    (store-dependencies assertion 'depends-on (support-set)))
		  assertion)
	       (put-value assertion (first props) (first values)))))))
;;.@findex{assert-value}

;;.@code{Retract-value} gets the description of a value (with
;;.@code{find-value}) and invalidates it, thus forcing the
;;.invalidation of its dependents an so forth.
(defun retract-value (unit slot &optional value)
  (let ((assertion (find-value unit slot value)))
    (cond ((failurep assertion)
	   (if (equal value (%get unit slot))
	       (%remove unit slot)))
	  ((null assertion) NIL)
	  (T (invalidate-assertion assertion)
	     assertion))))
;;.@findex{retract-value}

;;.The @code{to-find-value} slot is used with the assertion level
;;.interface to ARLOtje.  It contains a function which will `prove'
;;.that a given value holds for a particular slot.
(define-primitively to-find-value
  (english-description "This is the slot in which values are stored on an annotated value.")
  (to-get-value 'get-slot-method)
  (to-put-value 'typed-put-value)
  (makes-sense-for 'slotp)
  (must-be 'function-namep))


;;;;.Slots whose values are annotated

;;.Annotated slots are slots whose `fillers' --- the data structures
;;.stored for them --- are actually @emph{descriptions} of values; the
;;.actual `value' of the slot is derived and modified by accessing
;;.this description.  In particular, every `annotated slot' has the
;;.slots @code{value-in-slot} and @code{accumulates-in-slot} which determine
;;.where on the value description to get or store values.

;;.In order to support the assertion level, an additional reflexive
;;.operation is provided @code{find-value} which returns a description
;;.of a particular value of a slot, if it exists.  This is implemented
;;.through the @code{to-find-value} slot; in some sense,
;;.@code{to-find-value} is a @code{to-compute-value} which tries to
;;.find a @emph{particular} value.

;;.For instance, the filler of a slot might store a set of values;
;;.adding a value to the set would add the value to the elements of
;;.the set.  However, extracting a value from the set might compute
;;.some special function on the accumulated elements: it might filter
;;.out duplicates (making a set representation), remove values with
;;.some property, or even combine the values by some function like an
;;.arithmetic mean.  Note that the annotated descriptions are full
;;.fledged units; the properties evaluated to yield or respond to
;;.values can call on all of the features of ARLOtje's units and
;;.slots.  Commonly, an annotated description will have some
;;.properties which are themselves annotated descriptions; at some
;;.point this terminates, but in both principle and practice value
;;.descriptions have the full status of ARLOtje units.@refill

;;.Annotated slots usually are descendants of
;;.@code{prototypical-slot} or other prototypes which
;;.@code{work-like} it.  This slot provides @code{to-get-value} and
;;.@code{to-put-value} functions which interpret annotated values and
;;.also provide default values for @code{value-in-slot}, and
;;.@code{accumulates-in-slot}, and @code{probe-in-slot}.  All of these
;;.descriptors inherit through @code{works-like}.@refill
(define-primitively prototypical-slot
  (english-description "This is the prototype for most annotated slots.")
  (works-like 'prototypical-typed-slot)
  (to-get-value 'annotated-value-get)
  (to-put-value 'annotated-value-put)
  (to-find-value 'simple-find-value)
  (value-in-slot 'current-value)
  (accumulates-in-slot 'current-value))

;;.@code{Annotated-value-get} gets the current description stored in
;;.a slot (using @code{annotated-value}, which creates a description
;;.if neccessary) and the gets the appropriate @code{value-in-slot}
;;.slot and calls @code{get-value} on the value description and this
;;.slot.  Or, as it is actually implemented:@refill
(defun annotated-value-get (unit slot)
  "Extracts the value of an annotated value."
  ;; We can afford to remove this by shifting it to ANNOTATED-VALUE and
  ;; only checking for appropriateness when creating the annotated value.
  ;; (check-get-value unit slot)
  (let ((av (annotated-value unit slot)))
    (dependency! av)
    (get-value av (get-value slot 'value-in-slot))))
;;.The function @code{dependency!} adds its argument to the current
;;.computation context, indicating that any current computation
;;.depends upon it.  It is defined below.@refill

;;.@code{Annotated-value-put} stores a value in an annotated slot by
;;.first getting a description in the slot (like
;;.@code{annotated-value-get}, by calling @code{annotated-value}) and
;;.then storing a value in its @code{accumulates-in-slot} slot.  In
;;.some cases, this slot is identical to the @code{value-in-slot}
;;.slot, but sometimes it a different slot from which
;;.@code{value-in-slot} computes the actual stored value.@refill

;;.@code{Annotated-value-put} is implemented as:
;;...@example
(defun annotated-value-put (unit slot value)
  "Stores an annotated value assigning the given annotations."
  (let ((*computing-default* nil))
    (declare (special *computing-default*))
    (check-put-value unit slot value))
  (let ((av (annotated-value unit slot)))
    (put-value av (get-value slot 'accumulates-in-slot) value)))
;;...@end example
;;.There is an implicit assumption that the call to @code{put-value}
;;.on the accumulating slot yields a value description which denotes
;;.the assertion made by the @code{put-value}.  This is to support the
;;.assertion level of ARLOtje and gives the accumulating slot the
;;.freedom to construct and return special descriptions.

;;.The @code{value-in-slot} slot stores the slot from which an
;;.annotated slot extracts its slot.  This inherits through
;;.@code{works-like} and the default for
;;.@code{prototypical-slot}s is the slot
;;.@code{current-value} @pxref{Vanilla values}.@refill
(define-primitively value-in-slot
  (english-description "This is the slot to be extracted from an annotated value.")
  (to-get-value 'get-slot-method)
  (to-put-value 'typed-put-value)
  (makes-sense-for 'slotp)
  (must-be 'slotp))

;;.The @code{accumulates-in-slot} slot stores the slot in which an
;;.annotated slot accumulates values.  The @code{value-in-slot} slot
;;.stores the slot from which an annotated slot extracts its slot.
;;.This inherits through @code{works-like} and the default for
;;.@code{prototypical-slot}s is the slot
;;.@code{current-value} @pxref{Vanilla values}.@refill
(define-primitively accumulates-in-slot
  (english-description "This is the slot in which values are stored on an annotated value.")
  (to-get-value 'get-slot-method)
  (to-put-value 'typed-put-value)
  (makes-sense-for 'slotp)
  (must-be 'slotp))
;;.As mentioned above, the put function of a given
;;.@code{accumulates-in-slot} slot must return a description of the
;;.corresponding value.@refill


;;;;.Vanilla values

;;.By default, annotated values store their actual values in the slot
;;.@code{current-value} of a description.  The special character of
;;.this slot has to do with the @emph{invalidation} of values
;;.indicated by the @code{invalidatedp} slot; getting the
;;.@code{current-value} of a description returns a failure token if
;;.the description also has an @code{invalidated-p} slot.  
(define-primitively current-value
  (english-description "The real value of this value description.")
  (works-like 'annotated-value-token)
  (makes-sense-for 'annotated-valuep)
  (maKes-sense-for 'booleanp)
  (to-get-value 'get-if-valid)
  (to-put-value 'invalidate-and-store))
;;.@vindex{current-value (slot)}

;;.The invalidated-p slot is propogated through a dependency network
;;.established with the @code{depends-on} and @code{depended-on-by}
;;.slots; this is described in @pxref{Dependencies and invalidation}.
;;.The get function for @code{current-value} is @code{get-if-valid};
;;.the put function is @code{invalidate-and-store} which invalidates
;;.any current value (thus setting off an invalidation chain) and then
;;.stores the current value.

(defun get-if-valid (unit slot)
  "Gets the CURRENT-VALUE property of a unit if not invalidated."
  (if (invalidated-p unit) (fail unit slot)
    (%get unit slot)))
;;.@findex{get-if-valid}
(defun invalidate-and-store (unit slot value)
  "Invalidates the current value of a slot (and hence its dependents)
and then stores a new value."
  (declare (ignore slot))
  (invalidate-assertion unit)
  (%put unit slot value)
  (setf (assertion-invalidated unit) NIL)
  unit)
;;.@findex{invalidate-and-store}

;;.@code{Simple-find-value} returns an annotated value for a slot if
;;.the corresponding slot's value is equal to the requested value.
(defun simple-find-value (unit slot value)
  "This is the simplest FIND-VALUE function, returning an annotated
value if its represented value is EQUAL to the requested value."
  (let ((av (%get unit slot))
	(value-slot (get-value slot 'value-in-slot)))
    (and (not (failurep av)) (not (failurep value-slot))
	 (not (failurep (%get av value-slot)))
	 (if (and *recursion-checking* (computing? put-value unit slot value))
	     av
	   (and (equal (computing (get-value unit slot) (get-value av value-slot))
		       value)
		av)))))


;;;;.Assertion and Computation contexts

;;.ARLOtje maintains an @dfn{assertion context} describing the
;;.`outstanding assumptions' which are current when other inferences
;;.are made.  This assertion context is used in the procedure
;;.@code{support-set} which combines a number of value descriptions
;;.into a support set (a list) which also contains all of the members
;;.of the current assertion context.

;;.The variable @code{*assertion-context*} stores the current
;;.assertion context.  It is usually dynamically bound by various
;;.inference mechanisms.
(defvar *assertion-context* '()
  "This is the list of annotated values upon which any current values depend.")
;;.@findex{*assertion-context* (variable)}

;;.The function @code{support-set} creates a
;;.support set of annotated descriptions which includes the current
;;.assertion context.
;;.@findex{support-set}
(defun support-set (&optional depends-on)
  (append depends-on *assertion-context*))

;;.While the assertion context records what lead to a particular
;;.assertion being made (supporting forward inferences), the `current
;;.computation context' records those assertions accessed in
;;.determining a need-based backward inference.

;;.The variable @code{*current-computation*} is used to keep track of
;;.the current computation.  It is dynamically bound by
;;.various default computation mechanisms.
(defvar *current-computation* nil
  "This keeps track of a default computation for installation in the
dependency network.")
;;.@findex{*current-computation* (variable)}

;;.The variable @code{*computing-default*}
;;.is a flag identifying whether a default value is currently being
;;.computed.  When it is nil, value accesses are not recorded on
;;.@code{*current-computation*}; it is bound to @code{T} by
;;.various defaulting mechanisms and bound to @code{NIL} by the
;;.@code{as-a-side-effect} macro.
(defvar *computing-default* nil
  "This is true when a default is being computed.")
;;.@findex{*computing-default* (variable)}

;;.A value is recored in the current computation context by calling
;;.the function @code{dependency!} on the value description.  This is
;;.called by @code{annotated-value-get} as well as other procedures.
(defun dependency! (value)
  (declare (optimize (speed 3) (safety 0)))
  (when (and *computing-default* (assertionp value)
	     (%hasnt value 'constant-value))
    (jamnew value *current-computation*))
  value)
;;.@findex{dependency!}

;;.Dependency tracking may be temporarily turned off by wrapping some
;;.expressions in the macro @code{as-a-side-effect}.  It is often
;;.useful to wrap break loops, debugging checks, type checks, or other
;;.non-functional (in a strict sense) computations within this form.
;;.It works by dynamically binding @code{*computing-default*} to NIL
;;.for the execution of its body.
(defmacro as-a-side-effect (&body body)
  "Executes BODY while not tracking dependencies."
  `(let ((*computing-default* nil))
    (declare (special *computing-default*))
    ,@body))
;;.@findex{as-a-side-effect (macro)}

;;.A user can use the dependency tracking mechanism directly through
;;.the @code{return-dependencies} form which executes a form and
;;.returns two values: the value actually returned by the form and a
;;.list of the slots accessed during the forms evaluation.
(defmacro return-dependencies (form)
  (let ((temp (make-symbol "DEPENDENT-RESULT")))
    `(let ((*computing-default* t) (*current-computation* '()))
      (declare (special *computing-default* *current-computation*))
      (let ((,temp ,form))
	(values ,temp *current-computation*)))))

(defmacro forget-if-fail (form)
  "Evaluates FORM, tracking dependencies but forgetting them if FORM fails."
  `(if *computing-default* 
    (multiple-value-bind (result dependencies)
	(return-dependencies ,form)
      (unless (failurep result)
	(setq *current-computation*
	      (append dependencies *current-computation*)))
      result)
    ,form))


;;;;.Slots describing assertions

(define-primitively depends-on
  (english-description "The assertions which this annotated value depends on.")
  (makes-sense-for 'assertionp)
  (must-be 'list-of-assertionsp)
  (to-get-value 'whonit-get)
  (to-put-value 'store-dependencies)
  (whonit-type 'assertion)
  (whonit-accessor 'assertion-depends-on))

(define-primitively depended-on-by
  (english-description "The assertions which depend on this annotated value.")
  (makes-sense-for 'assertionp)
  (must-be         'assertionp)
  (to-get-value 'whonit-get)
  (to-put-value '%error-to-put)
  (whonit-type 'assertion)
  (whonit-accessor 'assertion-depended-on-by))

(define-primitively invalidated-p
  (english-description "Whether this assertion has been invalidated.")
  (makes-sense-for 'assertionp)
  (maKes-sense-for 'booleanp)
  (to-get-value 'get-invalidated-p)
  (to-put-value 'store-invalidated-p))

(defun get-invalidated-p (unit slot)
  "Gets the invalidation slot of an assertion."
  (declare (ignore slot))
  (check-type unit assertion "an assertion")
  (assertion-invalidated unit))
(defun put-invalidated-p (unit slot value)
  "Stores the invalidation slot of an assertion, propogating invalidation."
  (declare (ignore slot))
  (check-type unit assertion "an assertion")
  (if (and value (not (assertion-invalidated unit)))
      (invalidate-assertion unit)
    (setf (assertion-invalidated unit) value)))

;;.The function @code{invalidated-p} returns true of value descriptions
;;.which no longer represent valid values.
(defun invalidated-p (x)
  "Returns true if X is an invalidated annotated value."
  (assertion-invalidated x))
;;.@findex{invalidated-p}


;;;;. Unit defining macros.

;;.Units are usually defined not by @code{declare-unit} expressions
;;.followed by series of @code{assertion}s, but by one from a family
;;.of unit defining macros.  The general form of these macros takes a
;;.name specification (perhaps an unquoted symbol, or a symbol to be
;;.the root of a gensym (e.g. `@code{FOO}' yielding @code{FOO.1}))
;;.followed by a series of `slot specs', each of which is a list of a
;;.slot identifier, a value, and an alternating list of annotations.
;;.The slot identifier is either a slot name (unevaluated) or a list
;;.(@var{unit} @var{meta-slot}) which denotes the slot stored as the
;;.@var{meta-slot} slot of @var{unit}. The value is evaluated in the
;;.current context and the alternating list of annotations intersperse
;;.slot names and values to be stored on the description of the
;;.unit/slot specified.  So for instance, using the
;;.@code{Define-internal-unit} macro and some hypothetical annotation
;;.slots, one might say:
;;.@example
;;.(define-unit Madonna-Example
;;.  (age 32 'certainy-factor .7)
;;.  (hair-color 'blonde 'true-after  'like-a-virgin
;;.                      'true-before 'like-a-prayer)
;;.  (worth 5E7 'source 'Variety))
;;.@end example
;;.indicating some skepticism about age assesments, some time
;;.constraints on hair color and the published source of a worth
;;.estimate.  (This example is wholly fabricated; ARLOtje does not
;;.natively have either a certainty factor or a good interval
;;.representation.)

;;.@code{Define-unit} doesn't evaluate its first argument, but uses it
;;.directly as a unit name which is declared as an external unit (and
;;.thus exported from the current package).@findex{define-unit (macro)}
(defmacro define-unit (name &body slot-specs)
  "Defines an external unit NAME and stores SLOT-SPECS on it with ASSERTION.
Each SLOT-SPEC has the form (slot-name value-form . annotations)
where value-form is evaluated and asserted and the resulting assertion
is annotated with ANNOTATIONS."
  (flet ((transform-slot (s)
	   (if (symbolp s) `',s `(get-value ',(first s) ',(second s)))))
    (let ((unit-var (make-symbol "UNIT")))
      `(let ((,unit-var (declare-unit ',name)))
	(record-source-file ',name :unit)
	(bundling
	 ,@(mapcar #'(lambda (sl) `(assertion ,unit-var ,(transform-slot (first sl)) ,@(rest sl)))
	    slot-specs))
	,unit-var))))
;;.A hypothetical example of @code{define-unit} was illustrated above.

;;.@code{Define-internal-unit} is just like @code{define-unit} except
;;.that the unit created is internal to the current package (and thus
;;.not exported).@findex{define-internal-unit (macro)}
(defmacro define-internal-unit (name &body slot-specs)
  "Defines an internal (not exported) unit NAME and stores SLOT-SPECS on it with ASSERTION.
Each SLOT-SPEC has the form (slot-name value-form . annotations)
where value-form is evaluated and asserted and the resulting assertion
is annotated with ANNOTATIONS."
  (flet ((transform-slot (s)
	   (if (symbolp s) `',s `(get-value ',(first s) ',(second s)))))
    (let* ((unit-var (make-symbol "UNIT"))
	   (local? (or (null (symbol-package name))
		       (eq (symbol-package name) *package*)
		       (not (find-symbol (symbol-name name) *package*))))
	   (name (cond (local? name)
		       (T (shadow name) (intern (symbol-name name) *package*)))))
      `(progn
	,@(if (not local?) `((shadow ',name)) '())
	(eval-when (eval load)
	  (let ((,unit-var (declare-unit ',name :external? nil)))
	    (record-source-file ',name :unit)
	    (bundling
	     ,@(mapcar #'(lambda (sl) `(assertion ,unit-var ,(transform-slot (first sl)) ,@(rest sl)))
		       slot-specs))
	    ,unit-var))))))
;;.Uses of @code{define-internal-unit} look just like uses of
;;.@code{define-unit} above.

;;.@code{Make-unit} is designed for use by LISP functions; its first
;;.parameter, @var{name-form} is evaluated to yield a name and this
;;.symbol is declared as an external unit with properties assigned from
;;.the following slot specifications.@findex{make-unit (macro)}
(defmacro make-unit (name-form &body slot-specs)
  "Constructs and intitializes an external unit named by the result of NAME-FORM,
storing SLOT-SPECS on it with ASSERT-VALUE.
Each SLOT-SPEC has the form (slot-name value-form . annotations)
where value-form is evaluated and asserted and the resulting assertion
is annotated with ANNOTATIONS."
  (flet ((transform-slot (s)
	   (if (symbolp s) `',s `(get-value ',(first s) ',(second s)))))
    (let ((unit-var (make-symbol "UNIT")))
      `(let ((,unit-var (declare-unit ,name-form)))
	,@(mapcar #'(lambda (sl) `(assert-value ,unit-var ,(transform-slot (first sl)) ,@(rest sl)))
	   slot-specs)
	,unit-var))))
;;.For instance, we might define a function @code{difference-slot} which constructs
;;.a slot that computes the numeric difference of two other slots:
;;.@example
;;.(defun difference-slot (slot-1 slot-2)
;;.  (make-unit (fsymbol "~A-MINUS-~A" slot-1 slot-2)
;;.     (works-like 'prototypical-slot)
;;.     (constraint-methods `(- ,slot-1 ,slot-2))))
;;.(setq federal-deficit (difference-slot 'federal-revenues 'federal-expenditures))
;;.==> FEDERAL-REVENUES-MINUS-FEDERAL-EXPENDITURES
;;.(get-value 'United.States federal-deficit)
;;.==> 4E9
;;.@end example

;;.@code{Make-internal-unit} is like @code{make-unit} except that the
;;.unit name is not exported from the current package.
;;.@findex{make-internal-unit (macro)}
(defmacro make-internal-unit (name-form &body slot-specs)
  "Constructs and intitializes an internal unit named by the result of NAME-FORM,
storing SLOT-SPECS on it with ASSERT-VALUE.
Each SLOT-SPEC has the form (slot-name value-form . annotations)
where value-form is evaluated and asserted and the resulting assertion
is annotated with ANNOTATIONS."
  (flet ((transform-slot (s)
	   (if (symbolp s) `',s `(get-value ',(first s) ',(second s)))))
    (let ((unit-var (make-symbol "UNIT")))
      `(let ((,unit-var (declare-unit ,name-form :external? nil)))
	,@(mapcar #'(lambda (sl) `(assert-value ,unit-var ,(transform-slot (first sl)) ,@(rest sl)))
	   slot-specs)
	,unit-var))))
;;.Uses of @code{make-internal-unit} look just like uses of
;;.@code{make-unit} above.

;;.@code{Assertion} is a user level interface function which does an
;;.@code{assert-value} and also records the assertion as an event in
;;.the current session; the call to @code{assertion} will be reexecuted
;;.when the session is restored.
(defun assertion (unit slot value &rest annotations)
  "Asserts that SLOT of UNIT has value and records it on the session history."
  (session-event! `(assertion ,unit ,slot ,value ,@annotations)
		  (list unit value))
  (apply #'assert-value unit slot value annotations))


;;;;.Enumerating Slots

;;.The macro @code{doslots} iterates over all of the slots currently
;;.defined for a unit.  Its first parameter is evaluated to yield a
;;.unit, while its second parameter is a list of variable names
;;.@var{slot-var}, @var{value-var}, @var{description-var}.  The
;;.remaining parameters are forms constituting a @var{body} which is
;;.repeatedly evaluated with bindings for the @var{slot-var}, etc for
;;.each slot/value pair.  If the value is annotated, this annotated
;;.description is bound to @var{description-var}; otherwise
;;.,@code{description-var} is @code{NIL}.
(defmacro doslots (unit-var (slot-name-var value-var &optional
					   (description-var (make-symbol "DESC"))
					   (return-form unit-var))
			    &body body)
  "Executes BODY for each slot of UNIT-VAR, binding SLOT-NAME-VAR,
VALUE-VAR, and DESCRIPTION-VAR to the slot name, the actual value
stored and (if it exists) the annotated value for the slot."
  `(doproplist (,slot-name-var ,description-var ,unit-var ,return-form)
    (cond ((not (slotp ,slot-name-var)))
	  ((failurep (get-value ,slot-name-var 'value-in-slot))
	   (let ((,value-var ,description-var) (,description-var nil))
	     ,@body))
	  (T (let ((,value-var
		    (computing (get-value ,unit-var ,slot-name-var)
		       (get-value ,description-var (get-value ,slot-name-var 'value-in-slot)))))
	       ,@body)))))
;;.@findex{doslots (macro)}

;;.For instance, a better unit describer describe might be defined thus:
;;...@example
(defun du (unit &rest other-args)
  "A prototype versio of DU."
  (declare (ignore other-args))
  (format T "~&Description of ~S" unit)
  (doslots unit (slot value description)
    (format T "~&~S:~40T~S" slot value)
    (when description
      (doslots description (slot value)
	(unless (get slot 'dull-slot)
	  (format t "~&~3T~S:~43T~S" slot value))))))
;;...@end example
;;.yielding, for instance,
;;.@example
;;.(du 'madonna)
;;.Description of MADONNA
;;.AGE:  32
;;.  CERTAINTY-FACTOR:    .7
;;.HAIR-COLOR:  BLONDE
;;.  TRUE-AFTER:    LIKE-A-VIRGIN
;;.  TRUE-BEFORE:    LIKE-A-PRAYER
;;.WORTH:  5E7
;;.  SOURCE         VARIETY
;;.@end example
;;.Assuming that slots like @code{ANNOTATED-VALUE-FOR-SLOT} had been
;;.given appropriate @code{dull-slot} properties.
;;.Of more interest than @code{doslots} is the macro @code{dovalues}
;;.which is defined once many valued slots are implemented during
;;.ARLOtje's bootstrap process. @xref{Iterating over slot values}

;;.The procedure @code{mapslots} maps a function over all the
;;.slot/value pairs of a unit; it is to @code{doslots} as
;;.@code{mapcar} is to @code{dolist}, down to returning a list of the
;;.results of the given function.
(defun mapslots (fcn unit)
  "Executes BODY for each slot of UNIT-VAR, binding SLOT-NAME-VAR,
VALUE-VAR, and DESCRIPTION-VAR to the slot name, the actual value
stored and (if it exists) the annotated value for the slot."
  (let* ((results (list '())) (results-tail results))
    (doslots unit (slot value)
      (push (funcall fcn slot value) results-tail)
      (setf results-tail (cdr results-tail)))
    (cdr results)))