;;; $Id: port-s2c.scm,v 1.10 1994/02/03 17:36:12 queinnec Exp $
;;; Copyright (c) 1990-93 by Christian Queinnec. All rights reserved.
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;;                        LiSP2TeX
;;;   Christian Queinnec             or to:  Christian Queinnec
;;;   <queinnec@polytechnique.fr>            <Christian.Queinnec@inria.fr>
;;;   Laboratoire d'Informatique de l'X      INRIA -- Rocquencourt
;;;   Ecole Polytechnique                    Domaine de Voluceau, BP 105
;;;   91128 Palaiseau                        78153 Le Chesnay Cedex
;;;   France                                 France
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;; This program is distributed in the hope that it will be useful.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted, so long as the following
;;; conditions are met:
;;;      o credit to the authors is acknowledged following current
;;;        academic behaviour
;;;      o no fees or compensation are charged for use, copies, or
;;;        access to this software
;;;      o this copyright notice is included intact.
;;; This software is made available AS IS, and no warranty is made about
;;; the software or its performance.

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; This file contains the functions that are highly dependent of the
;;; Scheme implementation and, here, Scheme->C from Joel Bartlett.
;;; It also contains some utilities that are not present.

;;; The name of the underlying Scheme system.

(define scm-name "Scheme->C")

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Utilities are reverse!, mapcan, atom?, string-uppercase and
;;; string-lowercase. Some ports do not need to define them as they
;;; may already be present.

(define (reverse! l)
  (define (nreverse l r)
    (if (pair? l)
        (let ((cdrl (cdr l)))
          (set-cdr! l r)
          (nreverse cdrl l) )
        r ) )
  (nreverse l '()) )

(define (mapcan fn l)
  (if (pair? l)
      (append (fn (car l))
              (mapcan fn (cdr l)) )
      '() ) )

(define (atom? e)
  (not (pair? e)) )

(define (string-uppercase string)
  (let* ((n (string-length string))
         (s (make-string n)) )
    (do ((i 0 (+ i 1)))
        ((= i n) s)
      (string-set! s i (char-upcase (string-ref string i))) ) ) )

(define (string-lowercase string)
  (let* ((n (string-length string))
         (s (make-string n)) )
    (do ((i 0 (+ i 1)))
        ((= i n) s)
      (string-set! s i (char-downcase (string-ref string i))) ) ) )

;;; As the Unix utility (already defined in Bigloo)

(define (basename string)
  (let* ((n (string-length string))
         (lastdot (do ((i (- n 1) (- i 1)))
                      ((or (= i -1) (char=? (string-ref string i) #\.))
                       i ) )) )
    (if (= lastdot -1) string (substring string 0 lastdot)) ) )

;;; This is useful to concatenate various things and turn the result into
;;; a symbol. This is already present in Bigloo.

(define (symbol-append . args)
  (string->symbol 
   (apply string-append
          (map (lambda (s)
                 (cond ((string? s) s)
                       ((symbol? s) (symbol->string s))
                       ((number? s) (number->string s))
                       (else (error 'symbol-append args)) ) )
               args ) ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Scheme->C Dependencies.

;;; Save typing.

(define call/cc call-with-current-continuation)

;;; Scheme->C has a working peek-char so define this variable to true.

(define system-can-peek-char? #t)

;;; These streams correspond to the usual streams of Posix.

;(set! stderr-port stderr-port)
;(set! stdout-port stdout-port)

;;; Defines case sensitivity.

(define *respect-input* #t)
(define *uppercase-input* #f)
(define *lowercase-input* #f)

;;; Set how expressions to be evaluated (by Scheme->C) should be read.
;;; Scheme->C reads symbols in uppercase.

(define *evaluator-preferred-respect-input*   #f)
(define *evaluator-preferred-uppercase-input* #t)
(define *evaluator-preferred-lowercase-input* #f)

;;; By default respect case sensitivity.

(define *respect-output* #t)
(define *uppercase-output* #f)
(define *lowercase-output* #f)

;;; After each directive processing, flush buffers if not automatically
;;; provided by the implementation.

(define (flush-all-buffers) 
  (flush-buffer stdout-port)
  (flush-buffer stderr-port) )

;;; probe-file tests if a file exists.
;;; This is very specific to Scheme->C

(define (probe-file file)
  (let* ((handler *error-handler*)
         (r (call/cc
             (lambda (return)
               (set! *error-handler* (lambda args (return #f)))
               (call-with-input-file file (lambda (stream) #t)) ) )) )
    (set! *error-handler* handler)
    r ) )

;;; Dynamically evaluates an expression. If an error occurs, wrap the
;;; error in a list with a distinguished symbol.
;;; This is specific to Scheme->C.

(define (LiSP2TeX-eval exp)
  (let ((handler *error-handler*)
	(result 'wait) )
    (set! result 
	  (call/cc 
	   (lambda (return)
	     (set! *error-handler*
		   (lambda args (return (cons '***EVALUATION-ERROR*** args))) )
	     (eval exp) ) ) )
    (set! *error-handler* handler)
    result ) )

;;; The entry point of LiSP2TeX. The args variable is the list of 
;;; arguments of the shell invokation (a list of strings). This function
;;; has to be called as the initial function.

(define (unix-main . args)
  (set! args (cdr args)) ; strip executable name
  (LiSP2TeX args) )

;;; load the content of file. Load it non verbosely to not clutter the
;;; output stream.  The content of the file may use the various macros
;;; that are defined in emit.scm. It is better to load the file not in
;;; verbose mode to avoid cluttering the output: loadq does this.

(define (load-parameter-file file)
  (loadq file) )
  
;;; The pp.scm file uses some macros that must be present when 
;;; customization files are loaded.
;;; Code snarfed from ~/s2c/s2cmac.scm
;;; Syntax: (define-lisp2tex-macro (name . parameters) . body)

(define-macro define-lisp2tex-macro
  (lambda (e m)
    (define (map-arguments variables parameters)
      (cond ((pair? variables)
             (cons `( ,(car variables) (car ,parameters) )
                   (map-arguments (cdr variables) `(cdr ,parameters)) ) )
            ((symbol? variables)
             (list `( ,variables ,parameters )) )
            ((null? variables) '()) ) )
    (let* ((call (cadr e))
           (body (cddr e))
           (s2c-expander
            `(lambda (e m)
               (when (getprop 'define-macro 'ultra-verbose)
                 (format stderr-port "[call macro ~A]~%" e) )
               (m (let ,(map-arguments (cdr call) '(cdr e))
                    . ,body ) m) ))
           (s2c-installer
            `(let ((expander ,s2c-expander))
               (putprop ',(car call) '*expander* expander)
               (putprop ',(car call) 'scc
                        (cons (cons 'macro expander)
                              (or (getprop ',(car call) 'scc)
                                  '() ) ) )
               (putprop ',(car call) 'initial-scc
                        (cons (cons 'macro expander)
                              (or (getprop ',(car call) 'initial-scc)
                                  '() ) ) )
               ,(if (getprop 'define-macro 'verbose)
                    `(format 
                      stderr-port "[macro ~A installed]~%"
                      ',(car call) )
                    `#t )
               ',(car call) ) ) )
      (when (getprop 'define-macro 'verbose)
        (format stderr-port "[install macro ~A]~%" (car call)) )
      (eval s2c-installer)
      (m (if #t ;; (getprop 'define-macro 'pervasive) ; They are persistent !
             s2c-installer
             `',(car call) ) m) ) ) )

;;; Give define-lisp2tex-macro a chance to be pervasive itself so
;;; they remain in the final a.out.

(define-lisp2tex-macro (define-lisp2tex-macro call . body)
  (define (map-arguments variables parameters)
    (cond ((pair? variables)
           (cons `( ,(car variables) (car ,parameters) )
                 (map-arguments (cdr variables) `(cdr ,parameters)) ) )
          ((symbol? variables)
           (list `( ,variables ,parameters )) )
          ((null? variables) '()) ) )
  (let* ((arity-check
            `(lambda (n)
               (,(if (null? (last-pair call)) `= `<=)
                ,(let count ((variables (cdr call)))
                   (if (pair? variables)
                       (+ 1 (count (cdr variables)))
                       0 ) )
                n ) ) )
         (s2c-expander 
          `(lambda (e m)
             (when (getprop 'define-macro 'ultra-verbose)
                   (format stderr-port "[call macro ~A]~%" e) )
             (unless (,arity-check (length e))
                     (error ',(car call)
                            "Incorrect arity in macro call ~A" e ) )
             (m (let ,(map-arguments (cdr call) '(cdr e))
                  . ,body ) m) ))
         (s2c-installer 
          `(let ((expander ,s2c-expander))
             (putprop ',(car call) '*expander* expander)
             (putprop ',(car call) 'scc 
                      (cons (cons 'macro expander)
                            (or (getprop ',(car call) 'scc)
                                '() ) ) )
             (putprop ',(car call) 'initial-scc 
                      (cons (cons 'macro expander)
                            (or (getprop ',(car call) 'initial-scc)
                                '() ) ) )
             ,(if (getprop 'define-macro 'verbose)
                  `(format stderr-port "[macro ~A installed]~%" 
                           ',(car call) )
                  `#t )
             ',(car call) ) ) )
    (when (getprop 'define-macro 'verbose)
          (format stderr-port "[install macro ~A]~%" (car call)) )
    (eval s2c-installer)
    (if #t ; always persistent
        s2c-installer
        `',(car call) ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Define some macros

;;; PC Scheme has still a binary apply so any port has to define this macro.

(define-lisp2tex-macro (LiSP2TeX-apply . args)
  `(apply . ,args) )

;;; end of port-s2c.scm
