;;; $Id: scan-file.scm,v 1.9 1992/04/03 21:45:10 queinnec beta $
;;; This file contains the functions ro read a file from disk, scan it
;;; to find the bounds of Sexpressions and cache it in memory.

;;; convert a file into a scanned-file object
(define (scan-file filename)
  (call-with-input-file filename
    (lambda (in)
      (let ((extensible-string (make-extensible-string 4096))
	    (eof-object 'wait) )
	;; read the whole file into an extensible-string.
	;; This code is completely inefficient. It should be better
	;; to determine the size of the file and to read it in one go.
	(letrec ((loop 
		  (lambda (char)
		    (cond 
		     ((eof-object? char)
		      (set! eof-object char) )
		     (else 
		      (increment-string char extensible-string)
		      (loop (read-char in)) ) ) ) ))
	  (loop (read-char in)) )
	;; determine the bounds of each Sexpression
	(let* ((string (finalize-string extensible-string))
	       (bounds
		(scan-string 
		 (make-counting-stream 
		  filename
		  string
		  (extensible-string-index extensible-string)
		  0 
		  eof-object 
		  0 )
		 (list) ) ) )
	  ;; build the final object
	  (make-scanned-file
	   filename
	   (apply vector (reverse! bounds))
	   string ) ) ) ) ) )

;;; determine the bounds of each Sexpression in the stream.
;;; The substring between start and real-start may contain useful comments.
(define (scan-string counting-stream result)
  (let* ((start (counting-stream-position counting-stream))
         (real-start 
          (begin (counting-skip counting-stream)
                 (counting-stream-position counting-stream) ) )
         (o (counting-read counting-stream)) )
    (if (eof-object? o)
        result
        (if (atom? o)
            (scan-string counting-stream result)
            (scan-string 
             counting-stream
             (cons (make-scanned-expression
                    o
                    (find-scanned-expression-type o)
                    (find-scanned-expression-key o)
                    start
                    real-start
                    (counting-stream-position counting-stream) )
                   result ) ) ) ) ) )

;;; To ease the simple case where an expression is looked for from 
;;; its name precompute the name and the type (ie the definer) 
;;; of the expression

;;; The type is usually define, defmacro ...
(define (find-scanned-expression-type o)
  (if (symbol? (car o))
      (cond ((symbol? (car o)) (symbol->string (car o)))
            ((string? (car o)) (car o))
            (else #f) )
      #f ) )

;;; the defined name generally appears as the first symbol (or atom)
;;; after the type of the expression.
(define (find-scanned-expression-key o)
  (if (pair? (cdr o))
      (do ((o (cadr o) (car o)))
          ((not (pair? o)) 
           (cond ((symbol? o) (symbol->string o))
                 ((string? o) o)
                 (else #f) ) ) )
      #f ) )


;;; Search a file in a list of directories:
;;; Posix conventions are assumed.
(define (search-file filename path)
  (if (pair? path)
      (let ((fullname (string-append (car path) "/" filename)))
        (if (probe-file fullname)
            fullname
            (search-file filename  (cdr path)) ) )
      #f ) )
     

;;; Define the handler for read errors
;;; Report the line number and the context
(define read-error-format 
  "~%READ error (line ~A-~A): ~A ~A~%READ context: ~A " )

(define (counting-read-error stream start-line message culprits)
  (format stderr-port
          read-error-format
          start-line
          (counting-stream-line stream)
          message culprits 
          (counting-stream-current-read-line stream) )
  ;; resume by returning a bizarre object
  '***READ-ERROR-OBJECT*** )

(define (numbering-read-error stream start-line message culprits)
  (format stderr-port
          read-error-format
          start-line
          (numbering-stream-line stream)
          message culprits 
          (numbering-stream-current-read-line stream) )
  ;; resume by returning a bizarre object
  '***READ-ERROR-OBJECT*** )

;;; The main function that reads expressions from a counting stream
(define counting-read 'wait)
(define counting-skip 'wait)

(let ((fun (make-reader counting-stream-peek-char
                        counting-stream-read-char )))
  (set! counting-read 
        (lambda (stream)
          (call/cc
           (lambda (exit)
             (fun 'skip stream 'useless)
             (let* ((start-line (counting-stream-line stream))
                    (exp (fun 'read stream
                              (lambda (stream message . culprits)
                                (exit
                                 (counting-read-error 
                                  stream start-line message culprits ) ) ) )) )
               exp ) ) ) ) )
  (set! counting-skip 
        (lambda (stream)
          (fun 'skip stream 'useless) ) ) )

;;; The main function that reads expressions from a numbering stream
(define numbering-read 'wait)
(define numbering-skip 'wait)

(let ((fun (make-reader 
            numbering-stream-peek-char
            numbering-stream-read-char )) )
  (set! numbering-read 
        (lambda (stream)
          (call/cc 
           (lambda (exit)
             (fun 'skip stream 'useless)
             (let* ((start-line (numbering-stream-line stream))
                    (exp (fun 'read stream
                              (lambda (stream message . culprits)
                                (exit
                                 (numbering-read-error 
                                  stream start-line message culprits ) ) ) )) )
               exp ) ) ) ) )
  (set! numbering-skip 
        (lambda (stream)
          (fun 'skip stream 'useless) ) ) )
