;;; $Id: pp.scm,v 1.12 1992/04/18 14:34:28 queinnec Exp $

;;; A naive pretty-printer

;;; REcord the current form (this eases to report errors).
(define *the-current-form* 'wait)

;;; Error handling while pretty-printing
(define (pp-error msg . culprits)
  (command-error *pp-error-format* msg *the-current-form* culprits) )

(define *pp-error-format* "
GREEKIFY ERROR: ~A~%
The error occurs while pretty-printing ~A~%
Culprits are: ~A~%" )

;;; The entry point of the pretty-printer viewed from formated-print.
(define (greek-print e . out)
  (let ((out (if (pair? out) (car out) (current-output-port))))
    (set! *the-current-form* e)
    (for-each (lambda (fmt) (formated-print out fmt))
              *pretty-print-prologue* )
    (pp-do e *environment* out)
    (for-each (lambda (fmt) (formated-print out fmt))
              *pretty-print-epilogue* ) ) )

;;; The real printing engine. It is driven by the environment so it
;;; explores the environment to find an appropriate method.
(define (pp-do e r out)
  (define (lookup rr)
    (if (pair? rr)
        (if (procedure? (car rr))
            (let ((bool ((car rr) e)))
              (if bool (bool e r out) (lookup (cdr rr))) )
            (lookup (cdr rr)) )
        (pp-error "No appropriate method") ) )
  (lookup r) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; The default environment contains the default printer which
;;; just writes the expression. The default-environment is a list
;;; of things among which can be
;;;  1) methods with signature
;;;      (lambda (e) ...) returns #f if it is not an appropriate method
;;;                       returns a printer if apropriate
;;;             where a printer is (lambda (e r out) (display...))
;;;  2) symbols like INDENT indicating the indentation level
;;;  3) (VARIABLE image name) indicating a bound lexical variable and
;;;             its associated image (a string).
;;;  4) (IMAGE string names) indicating the images of some variables.
;;; Other terms can belong to the environment provided they do not 
;;; interfere with these terms.

(define *environment*
  (list
   ;; bool
   (lambda (e)
     (if (memq e '(#f #t))
         (lambda (e r out)
           (formated-print out (if e "\\mbox{\\bf true}"
                                   "\\mbox{\\bf false}" )) )
         #f ) )
   ;; symbols
   (lambda (e)
     (if (symbol? e) pp-symbol #f) )
   ;; forms
   (lambda (e)
     (if (pair? e)
         (lambda (e r out)
           (pp-do (car e) r out)
           (display-terms (cdr e) r out "(" "," ")") )
         #f ) )
   ;; character
   (lambda (e)
     (if (char? e)
         (lambda (e r out)
           (formated-print out "`~U'" (make-string 1 e)) )
         #f ) )
   ;; else
   (lambda (e)
     (lambda (e r out)
       (write e out) ) )) )

;;; Display each term of a list with a prefix, a separator and a suffix.
;;; The list can be empty in this case it is printed as prefix-suffix.
(define (display-terms e* r out beg mid end)
  (define (iterate e*)
    (if (pair? (cdr e*))
        (begin (pp-do (car e*) r out)
               (formated-print out mid)
               (iterate (cdr e*)) )
        (pp-do (car e*) r out) ) )
  (formated-print out beg)
  (when (pair? e*) (iterate e*))
  (formated-print out end) )

;;; Extend an environment with a term, return an environment.
(define (pp-extend environment term)
  (cons term environment) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; All macros enrich the current environment.

;;; Define that some variables must be displayed as images.
(define-pervasive-macro (def-image image . variables)
  `(begin 
     (set! *environment*
           (pp-extend *environment*
                      (cons 'IMAGE 
                            (cons ',image  ',variables) ) ) )
     ,image ) )

;;; Associate method to form that begin with keyword. Useful for
;;; special forms.
(define-pervasive-macro (def-form-printer keyword method)
  `(begin
     (set! *environment*
           (pp-extend *environment*
                      (lambda (e)
                        (if (and (pair? e) (equal? (car e) ',keyword))
                            ,method
                            #f ) ) ) )
     ',keyword ) )

;;; Associate a method to a typed object
(define-pervasive-macro (def-type-printer predicate fmt)
  `(begin 
     (set! *environment*
           (pp-extend *environment*
                      (lambda (e) 
                        (if (,predicate e)
                            (lambda (e r out)
                              (formated-print out ,fmt e) )
                            #f ) ) ) )
     ',predicate ) )

;;; Put all these definitions in the environment ?

;;; define a filter that must be satisfied to greekify a form
(define-pervasive-macro (def-excluded . parameters)
  (let ((names '())
        (definers '()) )
    (for-each (lambda (name)
                (if (pair? name)
                    (set! definers (append name definers))
                    (set! names (cons name names)) ) )
              parameters )
    `(begin (set! *excluded-definers*
                  (append ',definers *excluded-definers*) )
            (set! *excluded-names* 
                  (append ',names *excluded-names*) ) ) ) )

;;; At the beginning prints everything
(define *excluded-names* '())
(define *excluded-definers* '())

;;; The fmt will be output at the beginning of each translated form.
(define-pervasive-macro (pretty-print-prologue fmt)
  `(begin 
     (set! *pretty-print-prologue* (cons ,fmt *pretty-print-prologue*))
     ''pretty-print-prologue ) )
(define *pretty-print-prologue* '())

;;; The fmt will be output at the end of each translated form.
(define-pervasive-macro (pretty-print-epilogue fmt)
  `(begin 
     (set! *pretty-print-epilogue* 
           (append *pretty-print-epilogue* (list ,fmt)) )
     ''pretty-print-epilogue ) )
(define *pretty-print-epilogue* '())

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Define some methods for the usual special forms.

(def-form-printer quote
  (lambda (e r out)
    (formated-print out "`\\mbox{\\tt ~A}" (cadr e)) ) )

(def-form-printer progn
  (lambda (e r out)
    (if (pair? (cdr e))
        (if (pair? (cddr e))
            (pp-error "PROGN with more than one form: " e)
            (pp-do (cadr e) r out) )
        (pp-error "PROGN with less than one form: " e) ) ) )

;;; A new version that eliminates when and unless forms (generally used
;;; for tracing things).
(def-form-printer progn pp-progn)
(def-form-printer begin pp-progn)
(define (pp-progn e r out)
  (if (pair? (cdr e))
      (if (pair? (cddr e))
          (if (and (pair? (cadr e))
                   (memq (caadr e) '(unless when)) )
              (pp-do `(progn . ,(cddr e)) r out)
              (pp-error "PROGN with more than one form: " e) )
          (pp-do (cadr e) r out) )
      (pp-error "PROGN with less than one form: " e) ) )

(def-form-printer if
  (lambda (e r out)
    (let ((r (greek-incr-indentation r out)))
      (formated-print out "\\mbox{\\bf\\ if\\ } ")
      (pp-do (cadr e) r out)
      (greek-newline r out)
      (formated-print out "\\mbox{\\bf\\ then\\ } ")
      (pp-do (caddr e) r out)
      (greek-newline r out)
      (formated-print out "\\mbox{\\bf\\ else\\ } ")
      (pp-do (cadddr e) r out)
      (greek-newline r out)
      (formated-print out "\\mbox{\\bf\\ endif\\ } ")
      (greek-decr-indentation r out) ) ) )

(def-form-printer let
  (lambda (e r out)
    (let* ((new1r (greek-incr-indentation r out))
           (new2r (extend-with-variables new1r (map car (cadr e)))) )
      (formated-print out "\\mbox{\\bf\\ let\\ }")
      (display-bindings (cadr e) r new2r out)
      (greek-newline new2r out)
      (formated-print out "\\mbox{\\bf\\ in\\ }")
      (pp-do `(progn . ,(cddr e)) new2r out)
      (greek-decr-indentation r out) ) ) )

(def-form-printer letrec
  (lambda (e r out)
    (let* ((new1r (greek-incr-indentation r out))
           (new2r (extend-with-variables new1r (map car (cadr e)))) )
      (pp-do `(progn . ,(cddr e)) new2r out)
      (greek-newline new2r out)
      (formated-print out "\\mbox{\\bf\\ whererec\\ }")
      (display-bindings (cadr e) new2r new2r out)
      (greek-decr-indentation r out) ) ) )

(def-form-printer let*
  (lambda (e r out)
    (if (pair? (cadr e))
        (pp-do `(let (,(car (cadr e))) 
                  (let* ,(cdr (cadr e)) . ,(cddr e)) )
               r out )
        (pp-do `(progn . ,(cddr e)) r out) ) ) )

(define (display-bindings bindings oldr newr out)
  (when (pair? bindings)
     (pp-do (caar bindings) newr out)
     (formated-print out " = ")
     (pp-do (cadr (car bindings)) oldr out)
     (when (pair? (cdr bindings)) 
           (greek-newline newr out)
           (formated-print out "\\mbox{\\bf\\ and\\ }") )
     (display-bindings (cdr bindings) oldr newr out) ) )

(def-form-printer lambda
  (lambda (e r out)
    (let ((r (extend-with-variables r (cadr e))))
      (formated-print out "\\lambda ")
      (display-terms (cadr e) r out "" "" " . ")
      (pp-do `(progn . ,(cddr e)) r out) ) ) )

(define (extend-with-variables r variables)
  (define (search-image v r)
    (if (pair? r)
        (if (and (pair? (car r))
                 (eq? (caar r) 'IMAGE)
                 (memq v (cddr (car r))) )
            (cadr (car r))
            (search-image v (cdr r)) )
        (format #f "\\mbox{\\it ~A\\/}" 
                (string-lowercase (symbol->string v)) ) ) )
  (if (pair? variables)
      (let* ((v (car variables))
             (image (search-image v r)) )
        (extend-with-variables (pp-extend r `(VARIABLE ,image ,v))
                               (cdr variables) ) )
      r ) )

(define (pp-symbol e r out)
  (define (search-image-collision image r)
    (if (pair? r)
        (if (and (pair? (car r)) 
                 (eq? (caar r) 'VARIABLE)
                 (equal? image (cadr (car r))) )
            (+ 1 (search-image-collision image (cdr r)))
            (search-image-collision image (cdr r)) )
        0 ) )
  (define (search-image e r)
    (if (pair? r)
        (if (pair? (car r)) 
            (cond ((and (eq? (caar r) 'VARIABLE)
                        (eq? e (caddr (car r))) )
                   (let ((image (cadr (car r))))
                     (cons image (search-image-collision image (cdr r))) ) )
                  ((and (eq? (caar r) 'IMAGE)
                        (member e (cddr (car r))) )
                   (let ((image (cadr (car r))))
                     (cons image 0) ) )
                  (else (search-image e (cdr r))) )
            (search-image e (cdr r)) )
        (let ((image (format #f "\\mbox{\\it ~A\\/}" 
                             (string-lowercase (symbol->string e)) )))
          (cons image 0) ) ) )
  (let* ((image+index (search-image e r))
         (image (car image+index))
         (index (cdr image+index)) )
    (pp-indexed-symbol image index r out) ) )

(define *preferred-index* 'index)

(define (pp-indexed-symbol image index r out)
  (if (= 0 index)
      (formated-print out "{~U}" image)
      (case *preferred-index*
        ((index) (formated-print out "{~U}_{~A}" image index))
        ((prime) (formated-print out "{~U}" image)
                 (do ((i 0 (+ 1 i)))
                     ((>= i index))
                   (formated-print out "'") ) )
        (else (pp-error "Unknown indexing mode" *preferred-index*)) ) ) )

(def-form-printer define
  (lambda (e r out)
    (formated-print out "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Definition of ~U~%" 
            (find-scanned-expression-key e) )
    (let ((r (if (pair? (cadr e))
                 (extend-with-variables r (cdadr e))
                 r )))
      (pp-do (cadr e) r out)
      (formated-print out " = \\newline~%")
      (collect-definitions
       (cddr e)
       '()
       (lambda (definitions rest)
         (pp-do `(progn . ,rest) r out)
         (when (pair? definitions)
               (formated-print out "\\newline~% \\mbox{\\bf whererec} ")
               (let ((r (greek-incr-indentation r out)))
                 (for-each (lambda (e)
                             (pp-do e r out)
                             (formated-print out " \\newline~%") )
                           definitions ) ) ) ) ) ) ) )

(define (collect-definitions e* defs k)
  (if (pair? e*)
      (if (and (pair? (car e*))
               (eq? (caar e*) 'define) )
          (collect-definitions (cdr e*)
                               (cons (car e*) defs)
                               k )
          (k (reverse! defs) e*) )
      (pp-error "No body for this definition" defs) ) )

;;;ooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Utility functions to print various things. useful in customization files.

(define (pp-unary beg end)
  (lambda (e r out)
    (formated-print out beg)
    (pp-do (cadr e) r out)
    (formated-print out end) ) )

(define (pp-binary beg mid end)
  (lambda (e r out)
    (formated-print out beg)
    (pp-do (cadr e) r out)
    (formated-print out mid)
    (pp-do (caddr e) r out)
    (formated-print out end) ) )

(define (pp-ternary beg mid1 mid2 end)
  (lambda (e r out)
    (formated-print out beg)
    (pp-do (cadr e) r out)
    (formated-print out mid1)
    (pp-do (caddr e) r out)
    (formated-print out mid2)
    (pp-do (cadddr e) r out)
    (formated-print out end) ) )

(define (pp-nary beg mid end)
  (lambda (e r out)
    (display-terms (cdr e) r out beg mid end) ) )

(define (pp-meaning letter beg mid end)
  (lambda (e r out)
    (formated-print out letter)
    (formated-print out "\\lbrack\\!\\lbrack ")
    (formated-print out beg)
    (display-terms (cdr e) r out "" mid "")
    (formated-print out end) 
    (formated-print out "\\rbrack\\!\\rbrack ") ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Indentation is done via the tabbing environment of LaTeX, so nothing 
;;; needs to be recorded in r. Nevertheless a INDENT is pushed onto r.

;;; set a new margin and return a new environment.
(define (greek-incr-indentation r out)
  (formated-print out " \\setandincrindent~%")
  (pp-extend r 'indent) )

;;; Go to the newline with the current indentation.
(define (greek-newline r out)
  (formated-print out " \\newline~%") )

;;; Reset the margin to the previous indentation. Does not return a new
;;; environment since this is usually restaured through lexical scoping
;;; in the methods which use it (see the method for if).
(define (greek-decr-indentation r out)
  (formated-print out " \\decrindent~%") )

(pretty-print-prologue "~%\\begingroup
\\def\\setandincrindent{ $ \\=\\+ $ }
\\def\\decrindent{ $ \\- $ }
\\def\\newline{ $ \\\\ $ }
\\begin{tabbing} $ ~%" )

(pretty-print-epilogue " $ \\end{tabbing}\\endgroup~%")
