;;; $Id: commands.scm,v 1.17 1992/05/20 20:27:40 queinnec Exp $
;;; This file contains the entry point and 
;;; defines the commands offered by LiSP2TeX
;;; These commands are:
;;;    FromFile                 ( file keys... )
;;;    Print                    Sexpression
;;;    PrettyPrint              Sexpression
;;;    Eval                     Sexpression
;;;    ShowEvaluationOf		Sexpression

;;; Any command is associated to a function named command-name that 
;;; performs the real work. When a directive is found in the input 
;;; stream, a function run-command is invoked on the stream which role
;;; is to read the argument(s) then to invoke command-name.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Print
;;; Read and print a Sexpression

(define (run-Print stream)
  (command-Print (numbering-read stream)) )

(define (command-Print exp)
  (TeX-emit *lisp-excerpt* exp) )

(define *lisp-excerpt*              "{\\WithLispFont{~A}}")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; PrettyPrint
;;; Read and pretty-print a Sexpression

(define (run-PrettyPrint stream)
  (command-PrettyPrint (numbering-read stream)) )

(define (command-PrettyPrint exp)
  (greek-print exp) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; PrettyPrintFromFile
;;; Read from a file and pretty-print a Sexpression

(define (run-PrettyFromFile stream)
  (let ((e (numbering-read stream)))
    (if (pair? e)
        (if (string? (car e))
            (command-PrettyFromFile (car e) (cdr e))
            (command-error
   "~%LiSP2TeX (PrettyFromFile) error: Does not begin with a string: ~A "
             e ) )
        (command-error
   "~%LiSP2TeX (PrettyFromFile) error: Invalid number of arguments ~A "
         e ) ) ) )

(define (command-PrettyFromFile filename keys)
  (let ((sf (if (string=? filename "")
		(or *the-current-scanned-file*
		    (begin (command-error 
                            "~%LiSP2TeX error: No current scanned file ")
                           #f ) )
		(find-scanned-file filename) )))
    (if sf
	(let ((ses (mapcan (lambda (key) (find-scanned-expression key sf))
                           keys )) )
	  (set! *the-current-scanned-file* sf)
	  (for-each (lambda (se)
                      (if (eq? se '***MISSING-EXPRESSION***)
                          (greek-print (symbol->string se))
                          (greek-print (scanned-expression-original se)) ) )
		    ses ) )
	(command-error "~%LiSP2TeX error: Cannot find file ~A~%" 
                       filename ) ) ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; ShowEvaluationOf
;;; Read, print, evaluate and print the result of an Sexpression

(define (run-ShowEvaluationOf stream)
  (command-ShowEvaluationOf (numbering-read stream)) )

(define (command-ShowEvaluationOf exp)
  (TeX-emit *showevaluationof-input* exp)
  (TeX-emit *showevaluationof-output* (LiSP2TeX-eval exp)) )

(define *showevaluationof-input*    "~A ")
(define *showevaluationof-output*   " gives ~A")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Eval
;;; Read and evaluates a Sexpression. If an error is detected then
;;; LiSP2TeX-eval recturns a specific list. LiSP2TeX-eval is a non
;;; portable function defined in port-XXX.scm.

(define eval-error-format
  "~%EVAL error (line ~A-~A): ~A~%EVAL message: ~A " )

(define (run-Eval stream)
  (let* ((start-line (numbering-stream-line stream))
         (exp (numbering-read stream))
         (end-line (numbering-stream-line stream))
	 (result (LiSP2TeX-eval exp)) )
    (when (and (pair? result)
	       (eq? (car result) '***EVALUATION-ERROR***) )
	  (command-error eval-error-format 
                         start-line
			 end-line
			 exp
			 (cdr result) ) ) ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; FromFile
;;; There is a concept of current file. Whenever you use the FromFile
;;; command a null string in lieu of the filename refers to the current file.

(define *the-current-scanned-file* #f)

;;; Find a scanned file and scan it if not yet done. Set it as the
;;; current file. It is left to the implementation to detect if the file
;;; does not exist since it is not possible in standard Scheme to know it.

(define find-scanned-file 
  (let ((already-scanned-files '()))
    (lambda (filename)
      (define (lookup sfs)
        (cond ((null? sfs)
	       (let ((file (search-file filename *path*)))
		 (cond 
		  (file 
		   (command-echo "[Scan ~U" filename)
		   (set! already-scanned-files
			 (cons (scan-file filename)
			       already-scanned-files ) )
		   (command-echo "]")
		   (set! *the-current-scanned-file*
			 (car already-scanned-files) ) )
		  (else (set! *the-current-scanned-file* #f) ) ) )
	       *the-current-scanned-file* )
	      ((string=? filename (scanned-file-filename (car sfs)))
               (car sfs) )
              (else (lookup (cdr sfs))) ) )
      (lookup already-scanned-files) ) ) )

;;; Include a definition, it is reproduced as it is.

(define (run-FromFile stream)
  (let ((e (numbering-read stream)))
    (if (pair? e)
        (if (string? (car e))
	    (command-FromFile (car e) (cdr e))
	    (command-error 
	     "~%LiSP2TeX (FromFile) error: Does not begin with a string: ~A " 
             e ) )
        (command-error 
         "~%LiSP2TeX (FromFile) error: Invalid number of arguments ~A " 
         e ) ) ) )

(define (command-FromFile filename keys)
  (let ((sf (if (string=? filename "")
		(or *the-current-scanned-file*
		    (begin (command-error 
                            "~%LiSP2TeX error: No current scanned file ")
                           #f ) )
		(find-scanned-file filename) )))
    (if sf
	(let ((ses (mapcan (lambda (key) (find-scanned-expression key sf))
                           keys )) )
	  (set! *the-current-scanned-file* sf)
	  (for-each (lambda (se)
                      (unless (eq? se '***MISSING-EXPRESSION***)
                              (TeX-emit *lisp-citation*
                                        (scanned-expression-type se)
                                        (scanned-expression-key se) ) ) )
		    ses )
	  (when (pair? ses) (TeX-emit *beginlisp-block*))
	  (for-each (lambda (se)
                      (if (eq? se '***MISSING-EXPRESSION***)
                          (TeX-emit *lisp-block* (symbol->string se))
                          (TeX-emit 
                           *lisp-block*
                           (substring (scanned-file-string sf)
                                      (scanned-expression-real-start se)
                                      (scanned-expression-end se) ) ) ) )
		    ses )
	  (when (pair? ses) (TeX-emit *endlisp-block*)) )
	(command-error "~%LiSP2TeX error: Cannot find file ~A~%" 
                       filename ) ) ) )

(define *lisp-citation*             "\\LispCite{~U}{~U}~%")
(define *beginlisp-block*           "\\Lisp")
(define *lisp-block*                "~%~U")
(define *endlisp-block*             " \\EndLisp ")

;;; Find the expressions matched by key in scanned-file.
;;; A key can be  
;;;    a symbol : find the Sexpression that defines this name
;;;    a pair (considered as a pattern): find all Sexpressions matched

(define (find-scanned-expression key scanned-file)
  (let* ((ses (scanned-file-scanned-expressions scanned-file))
         (found #f)
         (max (vector-length ses)) )
    (define (lookup i)
      (if (< i max)
          (let ((se (vector-ref ses i)))
            (cond ((and (symbol? key)
                        (scanned-expression-key se)
                        (string-ci=? (symbol->string key)
                                     (scanned-expression-key se) ) )
                   (set! found #t)
                   (list se) )
                  ((and (pair? key)
                        (naive-match (scanned-expression-original se) key) )
                   (set! found #t)
                   (cons se (lookup (+ 1 i))) )
                  (else (lookup (+ 1 i))) ) )
          (if found
              '()
              (begin
                (command-error 
                 "~%LiSP2TeX error: Cannot find definition of ~A " 
                 key )
                '(***MISSING-EXPRESSION***) ) ) ) )
    (lookup 0) ) )

;;; This naive pattern matcher recognizes ?- and ... at any level 

(define (naive-match expression pattern)
  (define (naive-match-list expressions patterns)
    (if (pair? patterns)
        (if (equal? (car patterns) '...) ; accepts any sequence of things
            (or (naive-match-list expressions (cdr patterns))
                (and (pair? expressions)
                     (naive-match-list (cdr expressions) patterns) ) )
            (and (pair? expressions)
                 (naive-match (car expressions) (car patterns))
                 (naive-match-list (cdr expressions) (cdr patterns)) ) )
        (equal? expressions patterns) ) )
  (or (equal? pattern '?-)              ; accepts anything
      (if (equal? pattern '...)
          (command-error "Bad use of ... in pattern" pattern)
          (if (pair? pattern) 
              (naive-match-list expression pattern)
              (equal? expression pattern) ) ) ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Utility functions
;;; Emit some Sexpressions with an appropriate format (see emit.scm)

(define (TeX-emit fmt . arguments)
  (apply formated-print stdout-port fmt arguments) )

;;; These options allow to display the progress of the LiSP2TeX filter.
(define *verbose* #f)
(define *ultra-verbose* #f)

;;; Reports some progress
(define (command-echo fmt . args)
  (when *verbose*
        (apply formated-print stderr-port fmt args)
        (flush-buffer stderr-port) ) )

;;; reports some error
(define (command-error fmt . arguments)
  (apply format stderr-port fmt arguments)
  (flush-buffer stderr-port) )

;;; The list of directories where to search files. 
;;; Only the current one by default.
(define *path* '("."))

;;; Emit a header (a TeX comment) identifying the version of LiSP2TeX
(define (emit-header)
  (format stdout-port
          "%%% This file was automatically produced by LiSP2TeX ~A
%%%                   --- PLEASE DO NOT EDIT --- ~%" 
          version ) )

;;; Identify the current version of LiSP2TeX.
(define (emit-version-header)
  (format stderr-port
          "LiSP2TeX version ~A [Christian.Queinnec@INRIA.fr]~%" 
          version ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; MAIN
;;; The main entry point, handles options then filters stdin.
;;; Scheme->C calls it from the shell with the shell arguments in args.

(define (LiSP2TeX args)
  ;; Make CAR and car have the same value
  (replicate-symbols)
  ;; Analyse options
  (let ((does--appear? #f)
        (settings '())
        (files '()) )
    (do ((args args (cdr args)))
        ((null? args) #f)
      (cond 
       ;; refuses empty options
       ((= 0 (string-length (car args)))
        (command-error "~%LiSP2TeX error: Empty argument ") )
       ;; outputs the version identifier
       ((equal? (car args) "-version")(emit-version-header))
       ;; separates the parameter files from the files to filter
       ((equal? (car args) "--")
        (cond (does--appear? 
               (command-error "~%LiSP2TeX error: too much -- options ") )
              (else
               (set! does--appear? #t)
               (set! settings files)
               (set! files '()) ) ) )
       ;; A real option (or combination of options)
       ((char=? (string-ref (car args) 0) #\-)
        (if (> (string-length (car args)) 1)
            (analyze-option (car args) 1)
            (command-error "~%LiSP2TeX error: No option after dash") ) )
       (else
        (set! files (cons (car args) files)) ) ) )
    ;; Reverse the order of directories pushed onto *path*
    (set! *path* (reverse! *path*))
    ;; Identifies the current version of LiSP2TeX on stdout
    (emit-header)
    ;; Load the parameter files. They use *path* 
    (when does--appear?
          (for-each (lambda (filename) 
                      (let ((file (search-file filename *path*)))
                        (if file 
                            (begin
                              (command-echo "[Read customization ~U..."
                                            file )
                              (load-parameter-file file)
                              (command-echo "]~%") )
                            (command-error
                             "~%LiSP2TeX error: Cannot find file ~A "
                             filename ) ) ) )
                    (reverse! settings) ) )
    ;; Now handle files to filter or the standard input
    (if (pair? files)
        ;; Handles all mentionned files
        (for-each (lambda (filename)
                    (let ((file (search-file filename *path*)))
                      (if file
                          (call-with-input-file
                              file
                            (lambda (in)
			      (command-echo "[Process ~U..." filename)
                              (look (create-numbering-stream in)
                                    numbering-stream-read-char )
			      (command-echo "]~%") ) )
                          (command-error
                           "~%LiSP2TeX error: Cannot find file ~A "
                           filename ) ) ) )
                  (reverse! files) )
        ;; Handles stdin
        (look (create-numbering-stream stdin-port)
              numbering-stream-read-char ) )
    ;; Terminates.
    (flush-all-buffers) ) )

;;; As the Unix utility
(define (basename string)
  (let* ((n (string-length string))
         (lastdot (do ((i (- n 1) (- i 1)))
                      ((or (= i -1) (char=? (string-ref string i) #\.))
                       i ) )) )
    (if (= lastdot -1) string (substring string 0 lastdot)) ) )

;;; analyze options (options start with "-")
(define (analyze-option arg i)
  (when (< i (string-length arg))
        (cond 
         ;; Add a directory to search
	 ((char=? (string-ref arg i) #\I)
	  (set! *path* (cons (substring arg 
					(+ 1 i)
					(string-length arg) )
			     *path* )) )
         ;; be verbose
	 ((char=? (string-ref arg i) #\v)
	  (set! *verbose* #t)
	  (analyze-option arg (+ 1 i)) )
         ;; be more verbose if possible
	 ((char=? (string-ref arg i) #\V)
	  (set! *ultra-verbose* #t)
	  (set! *verbose* #t)
	  (analyze-option arg (+ 1 i)) )
         ;; converts what is read to uppercase
	 ((char=? (string-ref arg i) #\U)
          (set! *respect-input*    #f)
	  (set! *lowercase-input*  #f)
	  (set! *uppercase-input*  #t) )
         ;; converts what is printed to uppercase
	 ((char=? (string-ref arg i) #\u)
          (set! *respect-output*   #f)
	  (set! *lowercase-output* #f)
	  (set! *uppercase-output* #t) )
	 ((char=? (string-ref arg i) #\L)
          (set! *respect-input*    #f)
	  (set! *lowercase-input*  #t)
	  (set! *uppercase-input*  #f) )
	 ((char=? (string-ref arg i) #\l)
          (set! *respect-output*   #f)
	  (set! *lowercase-output* #t)
	  (set! *uppercase-output* #f) )
	 ((char=? (string-ref arg i) #\=)
          (set! *respect-input*    #t)
	  (set! *lowercase-input*  #f)
	  (set! *uppercase-input*  #f)
          (set! *respect-output*   #t)
	  (set! *lowercase-output* #f)
	  (set! *uppercase-output* #f) )
         ;; Do not emit header
	 ((char=? (string-ref arg i) #\h)
	  (set! emit-header (lambda () #f)) )
	 (else 
	  (command-error "~%LiSP2TeX error: Unrecognized option: ~A " 
			 (string-ref arg i) ) ) ) ) )
