/*
 * storage.c -- Implementation of a storage allocation scheme with
 *              stop-and-copy-based garbage collection
 *
 * (C) m.b (Matthias Blume); Jun 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * (C) m.b (Matthias Blume); Nov 1993, PU; (complete redesign)
 *         Princeton University, Dept. of Computer Science
 *
 * ident "@(#) storage.c (C) M.Blume, Princeton University, 2.9"
 */

# ident "@(#)storage.c	(C) M.Blume, Princeton University, 2.9"

# include <stdio.h>
# include <stdlib.h>
# include <assert.h>
# include <string.h>

# define MEM_IMPLEMENTATION

# include "align.h"
# include "storage.h"
# include "storage-t.h"
# include "except.h"

# include "realloc.h"

/* size increment for root array */
# define RA_SIZE_INCR 32

enum {
  NULL_IDENTIFIER = 253,
  MARK_IDENTIFIER = 254,
  REFERTO_IDENTIFIER = 255
};

struct broken_heart {
  MEM_descriptor _;
  void *new_location;
};

/* fake object description for broken hearts */
static MEM_description broken_heart_description = {
  MEM_NORMAL,
  0,
  MEM_UNITS (sizeof (struct broken_heart)),
  MEM_NULL_measure,
  MEM_NULL_iterator,
  MEM_NULL_dumper,
  MEM_NULL_excavator,
  MEM_NULL_revisor,
  MEM_NULL_task,
  MEM_NULL_task,
  MEM_NULL_task,
  0
};

static int gc_prohibited = 0;

/* rubber band array of root pointer locations */
static MEM_cnt ra_length = 0;
static MEM_cnt ra_free = 0;
static struct ra {
  void *location;
  MEM_iterator iterator;
} *ra = NULL;

void MEM_root (void *location, MEM_iterator iterator)
{
  struct ra *tmp;

  if (ra_free >= ra_length) {
    tmp = REALLOC (ra, sizeof (struct ra) * (RA_SIZE_INCR + ra_length));
    if (tmp == NULL)
      fatal ("Out of memory in register_global_object");
    ra = tmp;
    ra_length += RA_SIZE_INCR;
  }

  ra [ra_free].location = location;
  ra [ra_free].iterator = iterator;
  ra_free++;
}

struct heap_block {
  MEM_cnt scan, free, length;
  MEM_align_t array[1];
};

struct heap {
  MEM_cnt scan, free, length, allocated;
  struct heap_block **array;
  MEM_cnt total_size;
};

static struct heap heap = {
  0, 0, 0, 0,
  NULL,
  0
};

static struct heap passive_heap = {
  0, 0, 0, 0,
  NULL,
  0
};

MEM_cnt MEM_min_heap_size = MIN_HEAP_SIZE;

static MEM_cnt satisfying_block_index (MEM_cnt elem)
{
  MEM_cnt i;

  for (i = heap.free; i < heap.length; i++)
    if (elem <= heap.array [i]->length - heap.array [i]->free)
      return i;

  return ~0;
}

static MEM_cnt new_block_index (MEM_cnt elem)
{
  MEM_cnt length;
  struct heap_block *block;
  struct heap_block **tmp;

  length = (elem > HEAP_BLOCK_SIZE) ? elem : HEAP_BLOCK_SIZE;

  if (heap.length == heap.allocated) {
    tmp =
      REALLOC (heap.array,
	       (heap.allocated + HEAP_SIZE_INCR) *
	       sizeof (struct heap_block *));
    if (tmp == NULL)
      return ~0;
    heap.array = tmp;
    heap.allocated += HEAP_SIZE_INCR;
  }

  block = malloc (sizeof (struct heap_block) +
		  sizeof (MEM_align_t) * (length - 1));
  if (block == NULL)
    return ~0;

  heap.total_size += length;

  block->length = length;
  block->free = block->scan = 0;

  heap.array [heap.length] = block;
  return heap.length++;
}

static void *gc_getmem (MEM_cnt elem)
{
  MEM_cnt i;
  struct heap_block *hb;
  void *res;

  if ((i = satisfying_block_index (elem)) == ~0)
    if ((i = new_block_index (elem)) == ~0)
      fatal ("Out of memory (while in GC)");

  hb = heap.array [i];
  res = hb->array + hb->free;
  hb->free += elem;

  return res;
}

/*ARGSUSED*/
static void update_address (void **location, void *ignore)
{
  struct broken_heart *obj;
  void *new_location;
  MEM_cnt size;
  MEM_measure measure;

  if ((obj = *location) == NULL) {
    return;
  }

  if (obj->_ == &broken_heart_description) {
    *location = obj->new_location;
  } else {
    if ((size = obj->_->size) == 0) {
      if ((measure = obj->_->measure) == MEM_NULL_measure)
	/* constant non-heap object -- don't move it! */
	return;
      else
	size = (* measure) (obj);
    }
    new_location = gc_getmem (size);
    memcpy (new_location, (void *) obj, size * sizeof (MEM_align_t));
    obj->new_location = *location = new_location;
    obj->_ = &broken_heart_description;
  }
}

static MEM_cnt live_size, live_count;

static void *scan_next (void)
{
  struct heap_block *hb = NULL;
  struct broken_heart *res;
  MEM_cnt size, i;

  /* advance major scan pointer */
  for (i = heap.scan;
       i < heap.free && heap.array [i]->scan >= heap.array [i]->free;
       i++)
    ;
  heap.scan = i;

  /* search for some copy */
  for (/* i = heap.scan */; i < heap.length; i++) {
    hb = heap.array [i];
    if (hb->scan < hb->free)
      break;
  }
  if (i == heap.length)
    /* nothing */
    return NULL;

  /* advance minor scan pointer */
  res = (void *) (hb->array + hb->scan);
  if ((size = res->_->size) == 0)
    size = (* res->_->measure) (res);
  hb->scan += size;
  live_size += size;
  live_count++;

  return res;
}

enum {
  BEFORE_GC,
  AFTER_GC,
  MODULE_INIT
};

static void all_modules (int what)
{
  unsigned i;
  MEM_task task;

  for (i = 0; i < MEM_identifier_map_length; i++) {
    switch (what) {
    case BEFORE_GC:
      task = MEM_identifier_map [i]->before_gc;
      break;
    case AFTER_GC:
      task = MEM_identifier_map [i]->after_gc;
      break;
    default: /* MODULE_INIT */
      task = MEM_identifier_map [i]->init;
      break;
    }
    if (task != MEM_NULL_task)
      (* task) ();
  }
}

static MEM_cnt current_block;
MEM_align_t *MEM_free, *MEM_ceiling;

static void re_cache (MEM_cnt i)
{
  current_block = i;
  MEM_free = heap.array [i]->array + heap.array [i]->free;
  MEM_ceiling = heap.array [i]->array + heap.array [i]->length;
}

static void init_cache (void)
{
  MEM_cnt i;

  i = satisfying_block_index (SIZE_MINIMUM);
  if (i == ~0) {
    i = new_block_index (SIZE_MINIMUM);
    if (i == ~0)
      fatal ("Out of memory (init_cache in storage.c)");
  }
  re_cache (i);
}

static void exchange_heaps (void)
{
  struct heap tmp;
  MEM_cnt i;

  /* swap heaps */
  tmp = passive_heap;
  passive_heap = heap;
  heap = tmp;

  /* reset all free- and scan-members */
  for (i = 0; i < heap.length; i++)
    heap.array [i]->free = heap.array [i]->scan = 0;
  heap.free = heap.scan = 0;
}

static clock_t gc_clock_accu = 0;

clock_t MEM_total_gc_clock (void)
{
  return gc_clock_accu;
}

static void uncache (void)
{
  struct heap_block *hb;

  hb = heap.array [current_block];
  hb->free = MEM_free - hb->array;
  if (current_block != heap.free &&
      hb->length - hb->free < SIZE_MINIMUM) {
    heap.array [current_block] = heap.array [heap.free];
    heap.array [heap.free++] = hb;
  } else if (hb->length - hb->free < SIZE_MINIMUM)
    heap.free++;
}

static clock_t garbage_collection (void)
{
  MEM_cnt i;
  struct broken_heart *next;
  MEM_iterator iterator;
  clock_t tmp_clock;


  tmp_clock = clock ();
  MEM_announce_gc_start ();
  all_modules (BEFORE_GC);

  exchange_heaps ();

  /* run update_address on root pointers */
  for (i = 0; i < ra_free; i++)
    if ((iterator = ra [i].iterator) != MEM_NULL_iterator)
      (* iterator) (ra [i].location, update_address, NULL);
    else
      update_address (ra [i].location, NULL);

  /* just do it! */
  live_size = 0;
  live_count = 0;
  while ((next = scan_next ()) != NULL) {
    live_count++;
    if ((iterator = next->_->iterator) != MEM_NULL_iterator)
      (* iterator) (next, update_address, NULL);
  }

  all_modules (AFTER_GC);
  MEM_announce_gc_end ();
  tmp_clock = clock () - tmp_clock;
  gc_clock_accu += tmp_clock;

  return tmp_clock;
}

void *MEM_getmem (MEM_cnt elem)
{
  MEM_cnt i;
  void *ret;
  clock_t gc_time = 0;

  MEM_free -= elem;
  uncache ();

  if ((i = satisfying_block_index (elem)) == ~0)
    if (gc_prohibited || heap.total_size < MEM_min_heap_size) {
      i = new_block_index (elem);
      if (i == ~0)
	if (gc_prohibited)
	  fatal ("Out of memory (while GC was prohibited)");
	else {
	  /* decrease min_heap_size to avoid further warnings */
	  MEM_min_heap_size /= 2;
	  warning ("memory allocation problem causes premature GC");
	  goto the_normal_way;
	}
    } else {
the_normal_way:
      gc_time = garbage_collection ();
      if ((i = satisfying_block_index (elem)) == ~0)
	i = new_block_index (elem);
      if (i == ~0)
	reset ("Out of heap (even after GC)");
    }

  re_cache (i);

  if (gc_time != 0)
    MEM_gc_statistics (live_count, passive_heap.total_size,
		       live_size, gc_time);

  MEM_ALLOC (ret, elem);

  return ret;
}

/*
 * Implementation of dump and restore
 */

static void recur_dump_ul (unsigned long l, FILE *file)
{
  if (l != 0) {
    recur_dump_ul (l / 0200, file);
    putc ((l % 0200) | 0200, file);
  }
}

void MEM_dump_ul (unsigned long l, FILE *file)
{
  recur_dump_ul (l / 0200, file);
  putc (l % 0200, file);
}

unsigned long MEM_restore_ul (FILE *file)
{
  unsigned long l = 0;
  int c;

  while ((c = getc (file)) != EOF && c >= 0200)
    l = l * 0200 + c - 0200;
  if (c == EOF)
    fatal ("bad dump file format (EOF in unsigned long)");
  return l * 0200 + c;
}

# define refer_tag(x) ((unsigned long) (x))
# define hash_tag(x) (refer_tag (x) % DUMP_HASHTAB_SIZE)

static unsigned long again_count [DUMP_HASHTAB_SIZE];

static struct loctab_entry {
  unsigned long tag;
  void *location;
} *location_table [DUMP_HASHTAB_SIZE];

static unsigned long dump_tag;

static void register_tag (void *obj)
{
  int i = hash_tag (obj);
  int pos = again_count [i] ++;
  location_table [i][pos].tag = dump_tag++;
  location_table [i][pos].location = obj;
}

static unsigned long lookup_tag (void *obj)
{
  int i = hash_tag (obj);
  int pos;

  for (pos = 0; location_table [i][pos].location != obj; pos++);
  return location_table [i][pos].tag;
}

/*ARGSUSED*/
static void do_struct_analysis (void **object_ptr, void *ignore)
{
  struct broken_heart *obj = *object_ptr;
  MEM_descriptor current;

  if (obj == NULL)
    return;
  current = obj->_;
  if (current->size == 0 && current->measure == MEM_NULL_measure)
    return;
  switch (current->kind) {
  case MEM_NORMAL:
    obj->_ = current->vector [MEM_MARKED];
    if (current->iterator != MEM_NULL_iterator)
      (* current->iterator) (obj, do_struct_analysis, NULL);
    break;
  case MEM_MARKED:
    obj->_ = current->vector [MEM_AGAIN];
    ++again_count [hash_tag (obj)];
    break;
  default:
    break;
  }
}

static void structure_analysis (void)
{
  MEM_cnt i;
  MEM_iterator iterator;

  for (i = 0; i < ra_free; i++)
    if ((iterator = ra [i].iterator) != MEM_NULL_iterator)
      (* iterator) (ra [i].location, do_struct_analysis, NULL);
    else
      do_struct_analysis (ra [i].location, NULL);
}

/*ARGSUSED*/
static void do_cleanup (void **object_ptr, void *ignore)
{
  struct broken_heart *obj = *object_ptr;
  MEM_descriptor current;

  if (obj == NULL)
    return;
  current = obj->_;
  if (current->kind != MEM_NORMAL) {
    obj->_ = current->vector [MEM_NORMAL];
    if (current->iterator != MEM_NULL_iterator)
      (* current->iterator) (obj, do_cleanup, NULL);
  }
}

static void cleanup (void)
{
  MEM_cnt i;
  MEM_iterator iterator;

  for (i = 0; i < ra_free; i++)
    if ((iterator = ra [i].iterator) != MEM_NULL_iterator)
      (* iterator) (ra [i].location, do_cleanup, NULL);
    else
      do_cleanup (ra [i].location, NULL);
}

/*ARGSUSED*/
static void object_dump (void **object_ptr, void *vfile)
{
  struct broken_heart *obj = *object_ptr;
  FILE *file = (FILE *) vfile;
  MEM_descriptor d;
  MEM_iterator iterator;

  if (obj == NULL) {
    putc (NULL_IDENTIFIER, file);
    return;
  }
  d = obj->_;
  switch (d->kind) {
  case MEM_AGAIN:
    obj->_ = d->vector [MEM_WRITTEN];
    putc (MARK_IDENTIFIER, file);
    register_tag (obj);
    /* here is flow between cases, this is not an error! */
  case MEM_MARKED:
    putc (d->identifier, file);
    if (d->dumper != MEM_NULL_dumper)
      (* d->dumper) (obj, file);
    if ((iterator = d->iterator) != MEM_NULL_iterator)
      (* iterator) (obj, object_dump, vfile);
    break;
  case MEM_NORMAL:
    assert (d->size == 0 && d->measure == 0);
    putc (d->identifier, file);
    if (d->dumper != MEM_NULL_dumper)
      (* d->dumper) (obj, file);
    break;
  default: /* MEM_WRITTEN */
    putc (REFERTO_IDENTIFIER, file);
    MEM_dump_ul (lookup_tag (obj), file);
    break;
  }
}

static void dump (FILE *file)
{
  MEM_cnt i;
  MEM_iterator iterator;

  for (i = 0; i < ra_free; i++)
    if ((iterator = ra [i].iterator) != MEM_NULL_iterator)
      (* iterator) (ra [i].location, object_dump, (void *)file);
    else
      object_dump (ra [i].location, (void *)file);
}

static void allocate_location_table (void)
{
  int i;
  unsigned long s;
  struct loctab_entry *ptr;

  for (i = 0; i < DUMP_HASHTAB_SIZE; i++) {
    s = again_count [i];
    if (s > 0) {
      ptr = malloc (s * sizeof (struct loctab_entry));
      if (ptr == NULL)
	fatal ("Out of memory");
      location_table [i] = ptr;
      again_count [i] = 0;
    }
  }
}

static void free_location_table (void)
{
  int i;

  for (i = 0; i < DUMP_HASHTAB_SIZE; i++)
    if (location_table [i] != NULL) {
      free (location_table [i]);
      location_table [i] = NULL;
    }
}

void MEM_dump_storage (FILE *file, const char *prefix_string)
{
  int i;
  unsigned long sum;

  assert (sizeof (void *) <= sizeof (unsigned long));

  fputs (prefix_string, file);	/* prefix_string cannot contain '\n' */
  putc ('\n', file);
  putc ('\0', file);		/* the end marker... */

  for (i = 0; i < DUMP_HASHTAB_SIZE; i++)
    again_count [i] = 0;
  structure_analysis ();
  for (sum = 0, i = 0; i < DUMP_HASHTAB_SIZE; sum += again_count [i++])
    ;
  allocate_location_table ();
  dump_tag = 0;
  MEM_dump_ul (sum, file);
  dump (file);
  free_location_table ();
  cleanup ();
}

static void **refer_table;

static void init_refer_table (FILE *file)
{
  unsigned long l = MEM_restore_ul (file);

  refer_table = malloc (l * sizeof (void *));
  if (refer_table == NULL)
    fatal ("Out of space");
}

static void free_refer_table (void)
{
  free (refer_table);
}

static void restore_object (void **object_ptr, void *vfile)
{
  FILE *file = (FILE *) vfile;
  int identifier;
  unsigned long tag = 0;	/* to shut up the compiler */
  int marked;
  void *location;
  MEM_descriptor d;

  marked = 0;
  switch (identifier = getc (file)) {
  case EOF:
eof:
    fatal ("bad dump file format");
    break;
  case NULL_IDENTIFIER:
    *object_ptr = NULL;
    break;
  case REFERTO_IDENTIFIER:
    tag = MEM_restore_ul (file);
    *object_ptr = refer_table [tag];
    break;
  case MARK_IDENTIFIER:
    marked = 1;
    tag = dump_tag++;
    if ((identifier = getc (file)) == EOF)
      goto eof;
  default:
    d = MEM_identifier_map [identifier];
    if (d->excavator == MEM_NULL_excavator) {
      assert (d->size > 0);
      MEM_NEW (location, d, d->size);
    } else {
      location = (*d->excavator) (file);
      assert (((struct broken_heart *)location)->_ == d);
    }
    *object_ptr = location;
    if (marked)
      refer_table [tag] = location;
    if (d->iterator != MEM_NULL_iterator)
      (*d->iterator) (location, restore_object, vfile);
    if (d->revisor != MEM_NULL_revisor)
      (*d->revisor) (location);
    break;
  }
}

void MEM_restore_storage (FILE *file)
{
  int i, c;
  MEM_iterator iterator;

  while ((c = getc (file)) != EOF && c != '\0')
    ;
  if (c == EOF)
    fatal ("bad first line in dump file");

  /* prevent gc from running (otherwise location table gets corrupted) */
  gc_prohibited = 1;

  /* use passive heap */
  exchange_heaps ();
  init_cache ();

  dump_tag = 0;
  init_refer_table (file);
  for (i = 0; i < ra_free; i++)
    if ((iterator = ra [i].iterator) != MEM_NULL_iterator)
      (* iterator) (ra [i].location, restore_object, (void *)file);
    else
      restore_object (ra [i].location, (void *)file);
  free_refer_table ();
  /* reset gc_prohibited flag */
  gc_prohibited = 0;
}

void *MEM_new_location_of (void *obj)
{
  if (((struct broken_heart *) obj)->_ == &broken_heart_description)
    return ((struct broken_heart *) obj)->new_location;
  return obj;
}

/*
 * Module initialization
 */

void MEM_init_all_modules (void)
{
  init_cache ();
  all_modules (MODULE_INIT);
}
