/*
 * s l i b . c				-- Misc functions
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: ??-Oct-1993 ??:?? 
 * Last file update: 21-Dec-1993 22:38
 *
 */

#include "stk.h"
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/times.h>


/* Define constants used everywhere */
static struct obj VNIL     = {0, tc_nil};
static struct obj VUNDEF   = {0, tc_undefined}; 
static struct obj VUNBOUND = {0, tc_unbound};
static struct obj Vtruth   = {0, tc_boolean};
static struct obj Vntruth  = {0, tc_boolean};

static jmp_buf global_jmp_buf; /* Jump buffer denoting toplevel context */

long nointerrupt = 1;
long interrupt_differed = 0;

void *must_malloc(unsigned long size)
{
  void *tmp;

  tmp = malloc(size);
  if (tmp == NULL)
    err("failed to allocate storage from system", NIL);
 return(tmp);
}

void *must_realloc(void *ptr, unsigned long size)
{
  void *tmp;

  tmp = (char *) realloc(ptr, size);
  if (tmp == NULL)
    err("failed to re-allocate storage from system",NIL);
 return(tmp);
}


int strcmpi(register char *p1, register char *p2)
{
  for( ; tolower(*p1) == tolower(*p2); p1++, p2++) 
    if (!*p1) return 0;

  return tolower(*p1) - tolower(*p2);
}

void whence(char *exec, char *path)
{
  char *p, *q, dir[MAX_PATH_LENGTH];
  struct stat buf;
 
  if (*exec == '/') strncpy(path, exec, MAX_PATH_LENGTH);

  p = getenv("PATH");
  while (*p) {
    /* Copy the stuck of path in dir */
    for (q = dir; *p && *p != ':'; p++, q++) *q = *p;
    *q = '\000';

    sprintf(path, "%s/%s", dir, exec);
    if (access(path, X_OK) == 0) {
      stat(path, &buf);
      if (!S_ISDIR(buf.st_mode)) return;
    }
    
    /* Passer au chemin suivant */
    if (*p) p++;
  }
  path[0] = '\0';
}

SCM makecell(int type)
{
  register SCM z;
  NEWCELL(z, type);
  return z;
}

static void err_ctrl_c(void)
{
  err("Control-C interrupt",NIL);
}

long no_interrupt(long n)
{
  long x;
  x = nointerrupt;
  nointerrupt = n;
  if ((nointerrupt == 0) && (interrupt_differed == 1)){
    interrupt_differed = 0;
    err_ctrl_c();
  }
  return(x);
}

static void handle_sigfpe(int sig)
{
  signal(SIGFPE, handle_sigfpe);
  err("Floating point exception",NIL);
}

static void handle_sigint(int sig)
{
  signal(SIGINT,handle_sigint);
  if (nointerrupt)
    interrupt_differed = 1;
  else
    err_ctrl_c();
}

static void load_init_file(void)
{
  /* Try to load init.stk in ".", $STK_LIBRARY or a default directory */
  static char init_file[] = "init.stk";
  char file[256];
  char *tklib= getenv("STK_LIBRARY");

  sprintf(file, "./%s", init_file);
  if (loadfile(file, 0) == truth) return;

  if (tklib) {
    sprintf(file, "%s/%s",tklib, init_file);
    if (loadfile(file, 0) == truth) return;
  }

  sprintf(file, "%s/%s", STK_LIBRARY, init_file);
  loadfile(file, 1);
}


static void repl_driver()
{
  int k;

  reset_eval_stack();
  top_jmp_buf = &global_jmp_buf; 

  k = setjmp(*top_jmp_buf);
  
  signal(SIGINT, handle_sigint);
  signal(SIGFPE, handle_sigfpe);
  
  interrupt_differed = 0;
  nointerrupt 	     = 0;
  error_context      = ERR_OK;  
  interactivep 	     = isatty(fileno(stdin));
  if (k == 0) { 
    /* It's not an error. So Initialize interpreter */
    if (!dumped_core) load_init_file();
#ifdef USE_TK
    Tk_main(Argc, Argv);
#endif
    /* Evaluate contents of *init-hook* */
    leval(VCELL(intern(INIT_HOOK)), NIL);
    
    /* Put a message on stdout if interactive */
    if (interactivep) {
      fprintf(stderr, "Welcome to the STk interpreter version %s\n", STK_VERSION);
      fprintf(stderr, "Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA "
	      	      "<eg@unice.fr>\n\n");
    }
  }

  /* Start the print/eval/read loop */
  for( ; ; ) {
    SCM x;

    if (interactivep) fprintf(stderr, "STk> ");
    if (EQ(x=lreadf(stdin, FALSE), eof_object)) break;
    lprint(leval(x, NIL), stdout, WRT_MODE);
    Putc('\n', stdout);
  }
  if (interactivep) fprintf(stderr, "Bye.\n");
  quit_interpreter(UNBOUND);
}

SCM internal_eval_string(char *s, int context, SCM env)
{
  jmp_buf jb, *prev_jb = top_jmp_buf;
  int prev_context     = error_context;
  SCM result, port;
  int k;
  
  /* Create a string port to read the command and evaluate it in a new context */
  port = internal_open_input_string(s);

  /* save normal error jmpbuf  so that eval error don't lead to toplevel */
  if ((k = setjmp(jb)) == 0) {
    top_jmp_buf   = &jb;
    error_context = context;

    result = leval(lreadf(port->storage_as.port.f, FALSE), env);
  }
  
  top_jmp_buf   = prev_jb;;
  error_context = prev_context;

  return (k == 0) ? result : EVAL_ERROR;
}

SCM internal_read_from_string(SCM port, int *eof, int case_significant)
{
  jmp_buf jb, *prev_jb = top_jmp_buf;
  int prev_context     = error_context;
  SCM result;
  int k;

  /* save normal error jmpbuf  so that read error don't lead to toplevel */
  if ((k = setjmp(jb)) == 0) {
    top_jmp_buf   = &jb;
    error_context = ERR_READ_FROM_STRING;
    result 	  = lreadf(port->storage_as.port.f, case_significant);
    *eof   	  = Eof(port->storage_as.port.f);
  }
  top_jmp_buf   = prev_jb;;
  error_context = prev_context;

  return (k == 0)? result : EVAL_ERROR;
}

PRIMITIVE lcatch(SCM expr, SCM env)
{
  jmp_buf jb, *prev_jb = top_jmp_buf;
  int prev_context     = error_context;
  SCM l;
  int k;
  
  if (llength(expr) == -1) err("catch: bad list of expressions", expr);
  /* save normal error jmpbuf  so that eval error don't lead to toplevel */
  if ((k = setjmp(jb)) == 0) {
    top_jmp_buf   = &jb;
    error_context = ERR_IGNORED;

    /* Evaluate the list of expressions */
    for (l = expr; NNULLP(l); l = CDR(l)) leval(CAR(l), env);
  }
  top_jmp_buf   = prev_jb;
  error_context = prev_context;

  return (k == 0)? ntruth: truth;
}

PRIMITIVE quit_interpreter(SCM retcode)
{
  int ret = 0;

  if (retcode != UNBOUND) {
    if ((ret=integer_value(retcode)) == LONG_MIN)
      err("quit: bad return code", retcode);
  }
  lunwind_all();
  exit(ret);
}

PRIMITIVE lsystem(SCM com)
{
  if (NSTRINGP(com)) err("system: not a string", com);
  return makeinteger(system(CHARS(com)));
}

    
PRIMITIVE lgetenv(SCM str)
{
  char *tmp;
  if (NSTRINGP(str)) err("getenv: not a string", str);
  tmp = getenv(CHARS(str));
  return tmp ? makestrg(strlen(tmp), tmp) : ntruth;
}
  
PRIMITIVE lversion(void)
{
  return makestrg(strlen(STK_VERSION), STK_VERSION);
}

PRIMITIVE lrandom(SCM n)
{
  if (NINTEGERP(n)) err("random: bad number", n);
  return lmodulo(makeinteger(rand()), n);
}

PRIMITIVE set_random_seed(SCM n)
{
  if (NINTEGERP(n)) err("set-random-seed!: bad number", n);
  srand(INTEGER(n));
  return UNDEFINED;
}


static double _time()
{
  struct tms time_buffer;
  times(&time_buffer);
  return 1000 * (time_buffer.tms_utime + time_buffer.tms_stime) / 60.0;
}


PRIMITIVE ltime(SCM expr, SCM env)
{
  double rt;
  SCM res;

  if (llength(expr) != 1) err("stats: bad expression to stat" , expr);
  
  alloccells = 0;
  rt = _time();
  res = EVALCAR(expr);
  fprintf(stderr, ";;  Time: %.2fms\n;; Cells: %g\n", _time() - rt, alloccells);
  return res;
}


/* When STk evaluates an expression, it recode it in a manner which permits it
   to be more efficient for further evaluations. The uncode functions permits to 
   do the reverse job: it takes an exppression and returns a form similar to the 
   original one. 
   Warning: when a macro has been expanded, there is no mean to "revert" it to 
   its original form 
*/


static SCM associate(SCM l1, SCM l2)
{
  SCM z;

  if (NULLP(l1)) return NIL;
  
  for(z= NIL; NNULLP(l1); l1=CDR(l1), l2=CDR(l2))
    z = cons(LIST2(CAR(l1), luncode(CAR(l2))), z);
  
  return reverse(z);
}

static SCM uncode_let(char *type, SCM expr)
{
  return cons(intern(type),
	      LIST2(associate(CAR(expr), CAR(CDR(expr))),
		    luncode(CAR(CDR(CDR(expr))))));
}
  
PRIMITIVE luncode(SCM expr)
{
  switch (TYPE(expr)) {
    case tc_cons: switch (TYPE(CAR(expr))) {
    		    case tc_let:     return uncode_let("let",    CDR(expr)); 
      		    case tc_letstar: return uncode_let("let*",   CDR(expr)); 
		    case tc_letrec:  return uncode_let("letrec", CDR(expr)); 
		    case tc_if: 
		         expr = CDR(expr);
		         if (EQ(CAR(CDR(CDR(expr))), UNDEFINED)) 
			   return cons(intern("if"),
				       LIST2(luncode(CAR(expr)),
					     luncode(CAR(CDR(expr)))));
			 else
			   return cons(intern("if"),
				       LIST3(luncode(CAR(expr)),
					     luncode(CAR(CDR(expr))),
					     luncode(CAR(CDR(CDR(expr))))));
		    default: return cons(luncode(CAR(expr)), luncode(CDR(expr)));
		  }
    case tc_quote:	return intern("quote");
    case tc_lambda:	return intern("lambda");
    case tc_if:		return intern("if");
    case tc_setq:	return intern("set!");
    case tc_cond:	return intern("cond");
    case tc_and:	return intern("and");
    case tc_or:		return intern("or");
    case tc_let:	return intern("let");
    case tc_letstar:	return intern("letstar");
    case tc_letrec: 	return intern("letrec");
    case tc_begin:	return intern("begin");
    case tc_globalvar:  return VCELL(expr);
    case tc_localvar:   return expr->storage_as.localvar.symbol;
    default:		return expr;
  }
}

#ifdef USE_TK
/******************************************************************************
 *
 * Trace var
 *
 ******************************************************************************/

static char *TraceVarFct(ClientData clientData, Tcl_Interp *interp, 
			 char *name1, char *name2,
			 int flags)
{
  /* 
   * ClientData is the only field which insterest us here. It contains a 
   * string to evaluate 
   */
  Tcl_GlobalEval(interp, (char*) clientData);
  return NULL; /* to make the compiler happy */
}

PRIMITIVE ltrace_var(SCM var, SCM code)
{
  char *s1, *s2;
  
  if (NSYMBOLP(var)) err("trace-var: bad variable name", var);

  s1 = convert_for_tk(code);
  /* 
   * Build a copy of s1 and set it as the ClientData for Tcl_TraceVar
   * Duplication of s1 permits to be sure that the hash table has its 
   * own copy of the code to evaluate since original code can be garbaged.
   * TCL_TRACE_READS flags is used here to indicate to untrace to free
   * the code string.
   */
  s2 = (char *) must_malloc(strlen(s1) + 1); 
  strcpy(s2, s1);

  Tcl_TraceVar(main_interp, PNAME(var), TCL_TRACE_READS|TCL_TRACE_WRITES, 
	       TraceVarFct, (ClientData) s2);
  return UNDEFINED;
}

PRIMITIVE luntrace_var(SCM var)
{
  if (NSYMBOLP(var)) err("untrace-var: bad variable name", var);
  Tcl_CompleteUntraceVar(main_interp, PNAME(var));
  return UNDEFINED;
}
#endif


/******************************************************************************
 *
 * Inits + toplevel
 *
 ******************************************************************************/
void init_interpreter(int argc, char **argv)
{
  long j;

  /* Global variables which must be always initialized (even if it is a core) */
  Argc		= argc;
  Argv		= argv;
  whence(*Argv, Argv0);
  if (dumped_core) return;

  /* Global variables to initialize if not already done */
  NIL           = &VNIL;
  UNDEFINED     = &VUNDEF;
  UNBOUND	= &VUNBOUND;
  tkbuffer 	= (char *) must_malloc(TKBUFFERN+1);

  /* Initialize GC */
  init_gc();

  /* Initialize symbol table */
  for(j=0; j<OBARRAY_SIZE; j++) obarray[j] = NIL;

  gc_protect(truth  = &Vtruth);
  gc_protect(ntruth = &Vntruth);

  gc_protect(sym_progn  	  = intern("begin"));
  gc_protect(sym_lambda 	  = intern("lambda"));
  gc_protect(sym_quote  	  = intern("quote"));
  gc_protect(sym_imply  	  = intern("=>"));
  gc_protect(sym_dot    	  = intern(".")); 
  gc_protect(sym_debug  	  = intern(DEBUG_MODE));
  gc_protect(sym_else   	  = intern("else"));  
  gc_protect(sym_define 	  = intern("define"));
  gc_protect(sym_letrec 	  = intern("letrec"));
  gc_protect(sym_quasiquote 	  = intern("quasiquote"));
  gc_protect(sym_unquote	  = intern("unquote"));
  gc_protect(sym_unquote_splicing = intern("unquote-splicing")); 

  gc_protect(globenv    = makeenv(NIL));
  gc_protect(wind_stack = NIL);

  /* Initialize standard ports */
  init_standard_ports();

  /* Initialize Scheme primitives */
  init_primitives();
}

void toplevel(void)
{
  char stack_start; /* Unused variable. Its the first stack allocated varaiable */

  stack_start_ptr = &stack_start;
  repl_driver();
}
