/*
 *
 * n u m b e r . c				-- Numbers management
 *
 *
 * Numbers recognized by the interpreter are:
 *		- integer (which fit in a C long)
 *		- bignum  (arbitrary precision integer)
 *		- flonum  (represented as a C double)
 *
 * Bignum use the GNU gmp API. However to avoid to fall under the GPL terms
 * you can use the FGMP package (FGMP is a public domain implementation of
 * a subset of the GNU gmp library with the same API, written by Mark 
 * Henderson <markh@wimsey.bc.ca>). If your concern is speed, and if the
 * GPL is not a problem for you, use the gmp package. 
 *
 *
 *****
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: 12-May-1993 10:34
 * Last file update: 24-Nov-1993 22:19
 */

#include "stk.h"

#define SMALLNUMP(x)	(fabs(x) <= LONG_MAX)
#define ISINT(x)	(floor(x) == (x))

/******************************************************************************
 * 
 * Conversion Functions 
 *
 ******************************************************************************/

static int digitp(char c, int base)
{
  c = ('0' <= c && c <= '9') ? c - '0':
      ('a' <= c && c <= 'f') ? c - 'a' + 10:
      ('A' <= c && c <= 'F') ? c - 'A' + 10:
      (c == '#')             ? 0      :
      100;
  return (c <= base);
}

static SCM int2bignum(long n)
{
  SCM z;
  NEWCELL(z, tc_bignum); 
  BIGNUM(z) = must_malloc(sizeof(MP_INT));
  mpz_init_set_si(BIGNUM(z), n);
  return z;
}

static double bignum2double(MP_INT *bn)
{
  char   *s= mpz_get_str(NULL, 10, bn);
  double d = (double) atof(s);
  
  free(s);
  return d;
}

static SCM double2bignum(double n)	/* Very inefficient */
{
  int i, j, size = 20;
  char *tmp = must_malloc(size);
  SCM z;

  if (!ISINT(n)) err("cannot convert this number to an integer", makenumber(n));
  
  i = 0;
  if (n < 0.0) { tmp[i++] = '-'; n = -n; }
  do {
    if (i >= size) tmp = must_realloc(tmp, size *= 2);
    tmp[i++] = (int) fmod(n, (double) 10) + '0';
    n = floor(n / 10.0);
  }
  while (n > 0.0);
  tmp[i] = 0;

  /* Reverse the content of string tmp */
  for (i=i-1, j=(tmp[0]=='-'); i > j; i--, j++) {
    int  c = tmp[i];
    tmp[i] = tmp[j];
    tmp[j] = c;
  }

  /* tmp contains a textual representation of n. Convert it to a bignum */
  z = makebignum(tmp);
  free(tmp);
  return z;
}



/* Convert a number to a C-string. Result must be freed if != from buffer */
char *number2Cstr(SCM n, long base, char buffer[])
{
  char *s = buffer;
  
  switch (TYPE(n)) {
    case tc_flonum:
         if (base != 10) err("base must be 10 for this number", n);
         sprintf(buffer, "%.15g", FLONM(n));
         if (strchr(buffer, '.') == NULL && strchr(buffer, 'e') == NULL) 
	   strcat(buffer, ".0");
	 return buffer;
    case tc_integer:
	 {
	   long tmp, val = INTEGER(n);
	   int u;

	   if (base==2 || base==8 || base==10 || base==16) {
	     if (val < 0) {
	       val  = -val;
	       *s++ = '-';
	     }
	     for (s++, tmp=val; tmp >= base; tmp /= base) s++;
	     *s = '\000'; tmp = val;
	     do {
	       u = tmp % base;
	       *(--s) = u + ((u < 10) ? '0' : 'a'-10);
	       tmp   /= base;
	     }
	     while (tmp);
	     return buffer;
	   }
	 }
    case tc_bignum:
	   if (base==2 || base==8 || base==10 || base==16) {
	     s = must_malloc(mpz_sizeinbase(BIGNUM(n), base) + 2);
	     s = mpz_get_str(s, base, BIGNUM(n));
	     return s;
	   }
  }
BadBase:
  err("base must be 2, 8, 10 or 16", (base == LONG_MIN) ? NIL: makeinteger(base));
}

/* Return #f if the C string doesn't contain a valid number. */
/* UGLY! must be rewritten */
SCM Cstr2number(char *str, long base)
{
  int i, adigit=0, isint=1, exact=' ', radix = 0;
  char *start      = str;
  register char *p = str;

  for (i = 0; i < 2; i++) {
    if (*p == '#') {
      p += 1;
      switch (*p++) {
        case 'e': if (exact == ' ') { exact = 'e'; break; }  else return ntruth;
	case 'i': if (exact == ' ') { exact = 'i'; break; }  else return ntruth;
	case 'b': if (!radix) {base = 2;  radix = 1; break;} else return ntruth;
	case 'o': if (!radix) {base = 8;  radix = 1; break;} else return ntruth;
	case 'd': if (!radix) {base = 10; radix = 1; break;} else return ntruth;
	case 'x': if (!radix) {base = 16; radix = 1; break;} else return ntruth;
	default:  err("bad # syntax", makestrg(strlen(str), str));
      }
      str += 2;
    }
    if (*p != '#') break;
  }

  if (base != 2 && base != 8 && base != 10 && base != 16)
    err("base must be 2, 8, 10 or 16", (base == LONG_MIN) ? NIL: makeinteger(base));
  
  if (*p == '-' || *p == '+') p+=1;
  if (*p == '#') goto End;
  while(digitp(*p, base)) { p+=1; adigit=1; if (*p == '#') isint = 0; }

  if (*p=='.'){
    isint = 0; p += 1;
    while(digitp(*p, base)) { p+=1; adigit=1; }
  }

  if (!adigit) goto End;

  if (*p && strchr("eEsSfFdDlL", *p)) {
    isint = 0;
    p += 1;
    if (*p == '-' || *p == '+') p+=1;
    if (!digitp(*p, base)) goto End; 
    p+=1;
    while (digitp(*p, base)) p+=1;
  }
  if (*p) goto End;

  /* Now we are sure it is a number. Find the more adequate type */
  if (isint) {
    MP_INT n;

    if (mpz_init_set_str(&n, str, base) < 0) return ntruth;
    if (mpz_cmp_si(&n, -LONG_MAX) >=0 && mpz_cmp_si(&n, LONG_MAX) <= 0) {
      long num = mpz_get_si(&n);
      return (exact == 'i') ? makenumber(num) : makeinteger(num);
    }
    /* It's a bignum */
    if (exact == 'i') return makenumber(bignum2double(&n));
    else {
      SCM z;

      NEWCELL(z, tc_bignum);
      BIGNUM(z) = must_malloc(sizeof(MP_INT));
      mpz_init_set(BIGNUM(z), &n);
      mpz_clear(&n);
      return z;
    }
  }
  
  /* It's a float */
  if (exact == 'e')
    err("#e cannot be specified on this number", makestrg(strlen(str), str));
  if (base == 10) {
    /* Replace sharp signs by 0 */
    for(p=str; *p; p++) 
      switch (*p) {
	case '#': *p = '0'; break;
	case 's': case 'S': case 'f': case 'F':
	case 'd': case 'D': case 'l': case 'L': *p = 'e';
      }
    return makenumber((double) atof(str));
  }
End:
  if (*start ==  '#') err("Bad # syntax", makestrg(strlen(start), start));
  return ntruth;
}

long integer_value(SCM x) /* Returns LONG_MIN if not representable as int */
{
  if (INTEGERP(x)) return INTEGER(x);
  if (BIGNUMP(x)) {
    if (mpz_cmp_si(BIGNUM(x), -LONG_MAX)>=0 && mpz_cmp_si(BIGNUM(x), LONG_MAX)<=0)
      return mpz_get_si(BIGNUM(x));
  }
  return LONG_MIN;
}


/******************************************************************************
 *
 * Simple operations 
 *
 ******************************************************************************/

static SCM clone(SCM number)
{
  /* clone a number */
  switch (TYPE(number)) {
    case tc_integer: return makeinteger(INTEGER(number));
    case tc_bignum:  {
      		       SCM c = makebignum("0");
		       mpz_set(BIGNUM(c), BIGNUM(number));
		       return c;
		     }
    case tc_flonum:  return makenumber(FLONM(number));
  }
}

static void do_multiply(SCM *x, SCM y)
{
  switch (TYPE(*x)) {
    case tc_integer:
         switch (TYPE(y)) {
	   case tc_integer:
	        {
		  double prod;
		  if (SMALLNUMP(prod=(double) INTEGER(*x) * INTEGER(y)))
		    INTEGER(*x) = (long) prod;
		  else {
		    *x = int2bignum(INTEGER(*x));
		    mpz_mul_ui(BIGNUM(*x), BIGNUM(*x), INTEGER(y));
		  }
		}
		break;
	   case tc_bignum:
		*x = int2bignum(INTEGER(*x));
		mpz_mul(BIGNUM(*x), BIGNUM(*x), BIGNUM(y));
		break;
	   case tc_flonum:
		*x = makenumber(INTEGER(*x) * FLONM(y));
		break;
	 }
	 break;
    case tc_bignum:
	 switch (TYPE(y)) {
	   case tc_integer: 
	        {
		  SCM tmp = int2bignum(INTEGER(y));
		  mpz_mul(BIGNUM(*x), BIGNUM(*x), BIGNUM(tmp));
		}
		break;
	   case tc_bignum:
		mpz_mul(BIGNUM(*x), BIGNUM(*x),  BIGNUM(y));
		break;
	   case tc_flonum:
		*x = makenumber(bignum2double(BIGNUM(*x)) * FLONM(y));
		break;
	 }
	 break;
    case tc_flonum:
	 switch (TYPE(y)) {
	   case tc_integer:
	        FLONM(*x) *= INTEGER(y);
		break;
	   case tc_bignum:
		FLONM(*x) *= bignum2double(BIGNUM(y));
		break;
	   case tc_flonum:
		FLONM(*x) *= FLONM(y);
		break;
	 }
	 break;
  }
}

static void do_addition(SCM *x, SCM y)
{
  switch (TYPE(*x)) {
    case tc_integer:
         switch (TYPE(y)) {
	   case tc_integer:
	        {
		  double add;
		  if (SMALLNUMP(add=(double) INTEGER(*x) + INTEGER(y)))
		    INTEGER(*x) = (long) add;
		  else {
		    *x = int2bignum(INTEGER(*x));
		    mpz_add_ui(BIGNUM(*x), BIGNUM(*x), INTEGER(y));
		  }
		}
		break;
	   case tc_bignum:
		*x = int2bignum(INTEGER(*x));
		mpz_add(BIGNUM(*x), BIGNUM(*x), BIGNUM(y));
		break;
	   case tc_flonum:
		*x = makenumber(INTEGER(*x) + FLONM(y));
		break;
	 }
	 break;
    case tc_bignum:
	 switch (TYPE(y)) {
	   case tc_integer:
	      {
	        SCM tmp = int2bignum(INTEGER(y));
		mpz_add(BIGNUM(*x), BIGNUM(*x), BIGNUM(tmp));
	      }
	      break;
	   case tc_bignum:
		mpz_add(BIGNUM(*x), BIGNUM(*x),  BIGNUM(y));
		break;
	   case tc_flonum:
		*x = makenumber(bignum2double(BIGNUM(*x)) + FLONM(y));
		break;
	 }
	 break;
    case tc_flonum:
	 switch (TYPE(y)) {
	   case tc_integer:
	        FLONM(*x) += INTEGER(y);
		break;
	   case tc_bignum:
		FLONM(*x) += bignum2double(BIGNUM(y));
		break;
	   case tc_flonum:
		FLONM(*x) += FLONM(y);
		break;
	 }
	 break;
  }
}

static void do_substract(SCM *x, SCM y)
{
  switch (TYPE(*x)) {
    case tc_integer:
         switch (TYPE(y)) {
	   case tc_integer:
	        {
		  double add;
		  if (SMALLNUMP(add=(double) INTEGER(*x) - INTEGER(y)))
		    INTEGER(*x) = (long) add;
		  else {
		    *x = int2bignum(INTEGER(*x));
		    mpz_sub_ui(BIGNUM(*x), BIGNUM(*x), INTEGER(y));
		  }
		}
		break;
	   case tc_bignum:
		*x = int2bignum(INTEGER(*x));
		mpz_sub(BIGNUM(*x), BIGNUM(*x), BIGNUM(y));
		break;
	   case tc_flonum:
		*x = makenumber(INTEGER(*x) - FLONM(y));
		break;
	 }
	 break;
    case tc_bignum:
	 switch (TYPE(y)) {
	   case tc_integer:
	      {
	        SCM tmp = int2bignum(INTEGER(y));
		mpz_sub(BIGNUM(*x), BIGNUM(*x), BIGNUM(tmp));
	      }
	      break;
	   case tc_bignum:
		mpz_sub(BIGNUM(*x), BIGNUM(*x),  BIGNUM(y));
		break;
	   case tc_flonum:
		*x = makenumber(bignum2double(BIGNUM(*x)) - FLONM(y));
		break;
	 }
	 break;
    case tc_flonum:
	 switch (TYPE(y)) {
	   case tc_integer:
	        FLONM(*x) -= INTEGER(y);
		break;
	   case tc_bignum:
		FLONM(*x) -= bignum2double(BIGNUM(y));
		break;
	   case tc_flonum:
		FLONM(*x) -= FLONM(y);
		break;
	 }
	 break;
  }
}

static void do_divide(SCM *x, SCM y)
{
  switch (TYPE(*x)) {
    case tc_integer:
         switch (TYPE(y)) {
	   case tc_integer:
	        {
		  double div = (double) INTEGER(*x) / INTEGER(y);
		 
		  if (ISINT(div))
		    INTEGER(*x) = (int) div;
		  else
		    *x = makenumber(div);
		}
		break;
	   case tc_bignum:
		{
		  MP_INT q,  r;

		  mpz_init(&q); mpz_init(&r);
		  *x = int2bignum(INTEGER(*x));
		  mpz_mdivmod(&q, &r, BIGNUM(*x), BIGNUM(y));
		  if (mpz_cmp_ui(&r, 0L) == 0)
		    mpz_set(BIGNUM(*x), &q);
		  else
		    *x=makenumber(bignum2double(BIGNUM(*x)) /
				  bignum2double(BIGNUM(y)));
		  mpz_clear(&q); mpz_clear(&r);
		}
		break;
	   case tc_flonum:
		*x = makenumber(INTEGER(*x) / FLONM(y));
		break;
	 }
	 break;
    case tc_bignum:
	 switch (TYPE(y)) {
	   case tc_integer:
	   case tc_bignum:
	      {
		MP_INT q,  r;	

		mpz_init(&q); mpz_init(&r);
		if (INTEGERP(y)) 
		  y = int2bignum(INTEGER(y));

		mpz_mdivmod(&q, &r, BIGNUM(*x), BIGNUM(y));
		if (mpz_cmp_ui(&r, 0L) == 0)
		  mpz_set(BIGNUM(*x), &q);
		else
		  *x=makenumber(bignum2double(BIGNUM(*x)) /
				bignum2double(BIGNUM(y)));
		mpz_clear(&q); mpz_clear(&r);
	      }
	      break;
	   case tc_flonum:
		*x = makenumber(bignum2double(BIGNUM(*x)) / FLONM(y));
		break;
	 }
	 break;
    case tc_flonum:
	 switch (TYPE(y)) {
	   case tc_integer:
	        FLONM(*x) /= INTEGER(y);
		break;
	   case tc_bignum:
		FLONM(*x) /= bignum2double(BIGNUM(y));
		break;
	   case tc_flonum:
		FLONM(*x) /= FLONM(y);
		break;
	 }
	 break;
  }
}

static SCM do_integer_division(SCM x, SCM y, int quotient)
{
  /* Divide x by y and return its quotient or remainder. y is not 0 */
  if (INTEGERP(x) && INTEGERP(y)) {
    return makeinteger(quotient? INTEGER(x)/INTEGER(y):
		       		 INTEGER(x)%INTEGER(y));
  }
  else {
    /* result is a bignum. Eventually, transform argument to bignum  */
    MP_INT q,  r;
    SCM res   = makebignum("0");
    int exact = 1;

    switch (TYPE(x)) {
      case tc_integer: x = int2bignum(INTEGER(x)); break;
      case tc_bignum:  break;
      case tc_flonum:  if (ISINT(FLONM(x))) {x=double2bignum(FLONM(x)); exact=0;}
      		       else goto Error;
    }
    switch (TYPE(y)) {
      case tc_integer: y = int2bignum(INTEGER(y)); break;
      case tc_bignum:  break;
      case tc_flonum:  if (ISINT(FLONM(y))) {y=double2bignum(FLONM(y)); exact=0;}
      		       else { x = y; goto Error;}
    }

    mpz_init(&q); mpz_init(&r);
    mpz_mdivmod(&q, &r, BIGNUM(x), BIGNUM(y));
    mpz_set(BIGNUM(res), quotient? &q : &r);
    mpz_clear(&q); mpz_clear(&r);
    return exact? res : exact2inexact(res);
  }
Error:
  err("bad number in an integer division", x);
}

double do_compare(SCM x, SCM y)
{
  switch (TYPE(x)) {
    case tc_integer:
         switch (TYPE(y)) {
	   case tc_integer: return (double) (INTEGER(x)-INTEGER(y));
	   case tc_bignum:  return (double) -(mpz_cmp_si(BIGNUM(y), INTEGER(x)));
	   case tc_flonum:  return (double) INTEGER(x) - FLONM(y);
	 }
    case tc_bignum:
	 switch (TYPE(y)) {
	   case tc_integer: return (double) mpz_cmp_si(BIGNUM(x), INTEGER(y));
	   case tc_bignum:  return (double) mpz_cmp(BIGNUM(x), BIGNUM(y));
	   case tc_flonum:  return bignum2double(BIGNUM(x))-FLONM(y);
	 }
    case tc_flonum:
	 switch (TYPE(y)) {
	   case tc_integer: return (FLONM(x) - (double) INTEGER(y));
	   case tc_bignum:  return (FLONM(x) - bignum2double(BIGNUM(y)));
	   case tc_flonum:  return (FLONM(x) - FLONM(y));
	 }
  }
}

/******************************************************************************
 * 
 * Scheme primitives and utilities 
 *
 ******************************************************************************/

SCM makenumber(double x)
{
  SCM z;
  NEWCELL(z,tc_flonum); FLONM(z) = x; return z;
}

SCM makeinteger(long x)
{
  SCM z;
  NEWCELL(z, tc_integer); INTEGER(z) = x; return z;
}

SCM makebignum(char *s)
{
  SCM z;
  NEWCELL(z, tc_bignum); 
  BIGNUM(z) = must_malloc(sizeof(MP_INT));
  mpz_init_set_str(BIGNUM(z), s, 10);
  return z;
}

/******************************************************************************/

/**** Section 6.5 ****/

PRIMITIVE numberp(SCM x)
{
  return NUMBERP(x)? truth : ntruth;
}

PRIMITIVE integerp(SCM x)
{
  return (INTEGERP(x) || BIGNUMP(x)) ? truth : ntruth;
}

PRIMITIVE exactp(SCM x)
{
  if (NNUMBERP(x)) err("exact?: Bad number", x);
  return EXACTP(x) ? truth: ntruth;
}

PRIMITIVE inexactp(SCM x)
{
  if (NNUMBERP(x)) err("inexact?: Bad number", x);
  return NEXACTP(x) ? truth: ntruth;
}


#define Compare(name, operator)					\
PRIMITIVE name(SCM l, SCM env)					\
{								\
  register SCM tmp1, tmp2;					\
								\
  if (NCONSP(l)) err("too few parameters", l);			\
  tmp1 = EVALCAR(l);						\
  if (NNUMBERP(tmp1)) goto Error;				\
								\
  for (l=CDR(l); NNULLP(l); l=CDR(l),tmp1=tmp2) {		\
    tmp2 = EVALCAR(l);						\
    if (NNUMBERP(tmp2)) goto Error;				\
    if (do_compare(tmp1, tmp2) operator 0) return ntruth;	\
  }								\
  return truth;							\
Error:								\
  err("Bad number in a comparison", l);				\
}

Compare(numequal,  !=)
Compare(lessp,     >=)
Compare(greaterp,  <=)
Compare(lessep,    >)
Compare(greaterep, <)


PRIMITIVE zerop(SCM n)
{
  switch (TYPE(n)) {
    case tc_integer: return (INTEGER(n) == 0) 		     ? truth: ntruth;
    case tc_flonum:  return (FLONM(n) == 0.0) 		     ? truth: ntruth;
    case tc_bignum:  return (mpz_cmp_ui(BIGNUM(n), 0L) == 0) ? truth: ntruth;
    default:         err("zero?: bad number", n);
  }
}

PRIMITIVE positivep(SCM n)
{
  switch (TYPE(n)) {
    case tc_integer: return (INTEGER(n) > 0) 		     ? truth: ntruth;
    case tc_flonum:  return (FLONM(n) > 0.0) 		     ? truth: ntruth;
    case tc_bignum:  return (mpz_cmp_ui(BIGNUM(n), 0L) > 0)  ? truth: ntruth;
    default:         err("positive?: bad number", n);
  }
}  

PRIMITIVE negativep(SCM n)
{
  switch (TYPE(n)) {
    case tc_integer: return (INTEGER(n) < 0) 		     ? truth: ntruth;
    case tc_flonum:  return (FLONM(n) < 0.0) 		     ? truth: ntruth;
    case tc_bignum:  return (mpz_cmp_ui(BIGNUM(n), 0L) < 0)  ? truth: ntruth;
    default:         err("negative?: bad number", n);
  }
}

PRIMITIVE oddp(SCM n)
{
  MP_INT q, r;
  int res;

  switch (TYPE(n)) {
    case tc_integer: return (INTEGER(n)%2) 		     ? truth: ntruth;
    case tc_bignum:  mpz_init(&q), mpz_init(&r);
      		     mpz_divmod_ui(&q, &r, BIGNUM(n), 2L);
      		     res = mpz_cmp_ui(&r, 0L);
      		     mpz_clear(&q); mpz_clear(&r);
      		     return (res != 0) ? truth: ntruth;
    default:         err("odd?: bad number", n);
  }
}  

PRIMITIVE evenp(SCM n)
{
  MP_INT q, r;
  int res;

  switch (TYPE(n)) {
    case tc_integer: return (INTEGER(n)%2) 		     ? ntruth: truth;
    case tc_bignum:  mpz_init(&q), mpz_init(&r);
      		     mpz_divmod_ui(&q, &r, BIGNUM(n), 2L);
      		     res = mpz_cmp_ui(&r, 0L);
      		     mpz_clear(&q); mpz_clear(&r);
      		     return (res == 0) ? truth: ntruth;
    default:         err("even?: bad number", n);
  }
}

PRIMITIVE lmax(SCM l, SCM env)
{
  register SCM tmp, max;
  int inexact;

  if (NULLP(l)) err("max: bad number of arguments", NIL);
  
  if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
  inexact = FLONUMP(tmp);
  max = tmp;
  
  for (l=CDR(l); NNULLP(l); l=CDR(l)) {
    if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
    inexact |= FLONUMP(tmp);
    if (do_compare(tmp, max) > 0) max = tmp;
  }
  return (inexact && EXACTP(max)) ? exact2inexact(max) : max;
Error:
  err("max: bad number", tmp);
}

PRIMITIVE lmin(SCM l, SCM env)
{
  register SCM tmp, min;
  int inexact;

  if (NULLP(l)) err("min: bad number of arguments", NIL);
  
  if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
  inexact = FLONUMP(tmp);
  min = tmp;
  
  for (l=CDR(l); NNULLP(l); l=CDR(l)) {
    if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
    inexact |= FLONUMP(tmp);
    if (do_compare(tmp, min) < 0) min = tmp;
  }
  return (inexact && EXACTP(min)) ? exact2inexact(min) : min;
Error:
  err("min: bad number", tmp);
}

PRIMITIVE plus(SCM l, SCM env)
{
  SCM tmp, res = makeinteger(0);
  
  for ( ; CONSP(l); l=CDR(l)) {
    tmp = EVALCAR(l);
    if (NNUMBERP(tmp)) err("+: not a number", tmp);
    do_addition(&res, tmp);
  }
  return res;
}

PRIMITIVE difference(SCM l, SCM env)
{
  SCM tmp, res;

  if (NULLP(l)) err("-: no argumenent given", NIL);
  if (NULLP(CDR(l))) l = cons(makeinteger(0), l);	/* (- x) --> (- 0 x) */

  res = clone(EVALCAR(l));
  if (NNUMBERP(res)) goto Error;
  for (l=CDR(l) ; CONSP(l); l=CDR(l)) {
    tmp = EVALCAR(l);
    if (NNUMBERP(tmp)) goto Error;
    do_substract(&res, tmp);
  }
  return res;
Error:
  err("-: not a number", tmp);
}

PRIMITIVE ltimes(SCM l, SCM env)
{
  SCM tmp, res = makeinteger(1);

  for ( ; CONSP(l); l=CDR(l)) {
    tmp = EVALCAR(l);
    if (NNUMBERP(tmp)) err("*: not a number", tmp);
    do_multiply(&res, tmp);
  }
  return res;
}

PRIMITIVE division(SCM l, SCM env)
{
  SCM tmp, res;

  if (NULLP(l)) err("/: no argumenent given", NIL);
  if (NULLP(CDR(l))) l = cons(makeinteger(1), l);	/* (/ x) --> (/ 1 x) */
  
  tmp = res = clone(EVALCAR(l));
  if (NNUMBERP(res)) goto Error;
  for (l=CDR(l) ; CONSP(l); l=CDR(l)) {
    tmp = EVALCAR(l);
    if (NNUMBERP(tmp) || zerop(tmp) == truth) goto Error;
    do_divide(&res, tmp);
  }
  return res;
Error:
  err("/: not a valid number", tmp);
}

PRIMITIVE labsolute(SCM x)
{
  switch (TYPE(x)) {
    case tc_integer: return (INTEGER(x) < 0) ? makeinteger(-INTEGER(x)) : x;
    case tc_flonum:  return (FLONM(x) < 0.0) ? makenumber(-FLONM(x))    : x;
    case tc_bignum:  if (mpz_cmp_ui(BIGNUM(x), 0L) < 0) {
     			SCM tmp = clone(x);
			mpz_neg(BIGNUM(tmp), BIGNUM(x));
			return tmp;
		      }
      		      return x;
    default:         err("abs: bad number", x);
  }
}  

PRIMITIVE lquotient(SCM n1, SCM n2)
{
  if (NNUMBERP(n1) || NNUMBERP(n2))
    err("quotient: bad arguments", cons(n1, n2));
  if (zerop(n2) == truth) err("quotient: division by 0", NIL);
      
  return do_integer_division(n1, n2, TRUE);
}

PRIMITIVE lremainder(SCM n1, SCM n2)
{
  if (NNUMBERP(n1) || NNUMBERP(n2))
    err("remainder: bad arguments", cons(n1, n2));
  if (zerop(n2) == truth) err("remainder: division by 0", NIL);

  return do_integer_division(n1, n2, FALSE);
}

PRIMITIVE lmodulo(SCM n1, SCM n2)
{
  SCM z;

  if (NNUMBERP(n1) || NNUMBERP(n2))
    err("modulo: bad arguments", cons(n1, n2));
  if (zerop(n2) == truth) err("modulo: division by 0", NIL);
  
  z = do_integer_division(n1, n2, FALSE);
  if (negativep(n1) != negativep(n2))
    do_addition(&z, n2);
  return z;
}


static SCM lgcd2(SCM n1, SCM n2)
{
  return (zerop(n2) == truth) ? n1 : lgcd2(n2, lmodulo(n1, n2));
}

PRIMITIVE lgcd(SCM l, SCM env)
{
  register SCM tmp, res;

  if (NULLP(l)) return makeinteger(0L);
  
  if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
  res = labsolute(tmp);
  
  for (l=CDR(l); NNULLP(l); l=CDR(l)) {
    if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
    res = labsolute(lgcd2(res, tmp));
  }
  return res;
Error:
  err("gcd: bad number", tmp);
}

PRIMITIVE llcm(SCM l, SCM env)
{
  SCM tmp, res, gcd;

  if (NULLP(l)) return makeinteger(1L);
  
  if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
  res = labsolute(clone(tmp));
  
  for (l=CDR(l); NNULLP(l); l=CDR(l)) {
    if (NNUMBERP(tmp=EVALCAR(l))) goto Error;
    if (zerop(tmp)==truth) return tmp;
    gcd = lgcd2(res, tmp);
    do_multiply(&res, tmp);
    do_divide(&res, gcd);
    res =labsolute(res);
  }
  return res;
Error:
  err("lcm: bad number", tmp);
}

PRIMITIVE lfloor(SCM x)
{
  if (NNUMBERP(x)) err("floor: bad number", x);
  if (FLONUMP(x)) return makenumber(floor(FLONM(x)));
  return x;
}

PRIMITIVE lceiling(SCM x)
{
  if (NNUMBERP(x)) err("ceiling: bad number", x);
  if (FLONUMP(x)) return makenumber(ceil(FLONM(x)));
  return x;
}

PRIMITIVE ltruncate(SCM x)
{
  if (NNUMBERP(x)) err("truncate: bad number", x);
  if (FLONUMP(x)) {
    double d = FLONM(x);
    return makenumber(d < 0.0 ? ceil(d) : floor(d));
  }
  return x;
}
  
PRIMITIVE lround(SCM x)
{
  if (NNUMBERP(x)) err("round: bad number", x);
  if (FLONUMP(x)) return makenumber(rint(FLONM(x)));
  return x;
}

#define transcendental(fct)					\
    SCM l##fct(SCM z)						\
    {								\
      if (NNUMBERP(z)) err(#fct ": bad number", z);		\
      else return makenumber(fct(FLONM(exact2inexact(z))));	\
    }

transcendental(exp)
transcendental(log)
transcendental(sin)
transcendental(cos)
transcendental(tan)
transcendental(asin)
transcendental(acos)

PRIMITIVE latan(SCM y, SCM x)
{
  if (NNUMBERP(y)) err("atan: bad  number", y);
  return (x == UNBOUND) ? makenumber(atan(FLONM(exact2inexact(y)))) :
    			  makenumber(atan2(FLONM(exact2inexact(y)),
					   FLONM(exact2inexact(x))));
}

PRIMITIVE lsqrt(SCM z)
{
  switch (TYPE(z)) {
    case tc_integer: {
      	 double d;
	 
	 if (INTEGER(z) < 0) goto Error;
	 d = (double) sqrt((double) INTEGER(z));
	 return ISINT(d) ? makeinteger((long) d): makenumber(d);
       }
    case tc_bignum: {
         MP_INT root, remainder;
	 SCM res;

	 if (mpz_cmp_si(BIGNUM(z), 0L) < 0) goto Error;
	 mpz_init(&root); mpz_init(&remainder);
	 mpz_sqrtrem(&root, &remainder, BIGNUM(z));
	 if (mpz_cmp_si(&remainder, 0L) == 0) {
	   /* Result is an integer */
	   res = makebignum("0");
	   mpz_set(BIGNUM(res), &root);
	 }
	 else {
	   /* Result is a flonum */
	   res = makenumber(bignum2double(&root));
	 }
	 mpz_clear(&root); mpz_clear(&remainder);
	 return res;
       }
    case tc_flonum: {
         if (FLONM(z) < 0.0) goto Error;
	 return makenumber((double) sqrt(FLONM(z)));
       }
    default:
         err("sqrt: bad number", z);
  }
Error:
  err("sqrt: number is negative", z);
}

/*
 (define (exact_expt z1 z2)
  (define (square x) (* x x))
  (cond ((= z2 0) 1)
	((negative? z2) (/ 1 (exact_expt z1 (abs z2))))
        ((even? z2) (square (exact_expt z1 (/ z2 2))))
        (else (* z1 (exact_expt z1 (- z2 1))))))
*/

static SCM exact_expt(SCM z1, SCM z2)
{
  SCM tmp, res;
  
  if (zerop(z2) == truth) return makeinteger(1);
  if (negativep(z2) == truth) { 
    /* (/ 1 (expt z1 (abs z2) */
    res = makeinteger(1);
    do_divide(&res, exact_expt(z1, labsolute(z2)));
    return res;
  }
  if (evenp(z2) == truth) {
    /* (square (expt z1 (/ z2 2)) */
    res = clone(z2);
    do_divide(&res, makeinteger(2));
    res = exact_expt(z1, res);
    do_multiply(&res, res);
    return res;
  }
  /* (* (expt z1 (- z2 1)) z1) */
  res = clone(z2);
  do_substract(&res, makeinteger(1));
  res = exact_expt(z1, res);
  do_multiply(&res, z1);
  return res;
}

PRIMITIVE lexpt(SCM z1, SCM z2)
{
  if (NNUMBERP(z1)) err("expt: bad number", z1);
  if (NNUMBERP(z2)) err("expt: bad number", z2);

  return (EXACTP(z1) && EXACTP(z2)) ?
    		exact_expt(z1, z2)  :
		makenumber((double) pow(FLONM(exact2inexact(z1)),
					FLONM(exact2inexact(z2))));
}

PRIMITIVE exact2inexact(SCM z)
{
  switch (TYPE(z)) {
    case tc_integer: return makenumber((double) INTEGER(z));
    case tc_bignum:  return makenumber(bignum2double(BIGNUM(z)));
    case tc_flonum:  return z;
    default:         err("exact->inexact: bad number", z);
  }
  return z;
}

PRIMITIVE inexact2exact(SCM z)
{
  switch (TYPE(z)) {
    case tc_integer: 
    case tc_bignum:  return z;
    case tc_flonum:  return double2bignum(floor(FLONM(z)+0.5));
    default:         err("inexact->exact: bad number", z);
  }
}


PRIMITIVE string2number(SCM str, SCM base)
{
  if (NSTRINGP(str)) err("string->number: not a string", str);
  return Cstr2number(CHARS(str), (base == UNBOUND)? 10 : integer_value(base));
}

PRIMITIVE number2string(SCM n, SCM base)
{
  char *s, buffer[100];
  SCM z;

  if (NNUMBERP(n))   err("number->string: bad number", n);
  s = number2Cstr(n, (base == UNBOUND)? 10 : integer_value(base), buffer);
  z = makestrg(strlen(s), s);
  if (s != buffer) free(s);
  return z;
}
