/*
 *
 *  g c . c			-- Mark and Sweep Garbage Collector 
 *
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 17-Feb-1993 12:27
 * Last file update: 19-Dec-1993 15:05
 *
 *
 */

#include "stk.h"

/* exported vars */
SCM freelist;
char* stack_start_ptr;
double alloccells;

/* internal vars */
static jmp_buf save_regs_gc_mark;
static long    gc_cells_collected;
static SCM     heap_org, heap_end;
static long    heap_size  = 200000;
static int     gc_verbose = 1;

static struct gc_protected {
  SCM location;
  struct gc_protected *next;
} *protected_registers = NULL;

static void mark_locations(SCM *start, SCM *end);


static void gc_ms_stats_start(void)
{
  gc_cells_collected = 0;
  if (gc_verbose = (VCELL(intern(GC_VERBOSE)) != ntruth)) 
    fprintf(stderr, "[starting GC]\n");
}

static void gc_ms_stats_end(void)
{
  long n;
  if (gc_verbose) {
    n = heap_end - heap_org;;
    fprintf(stderr, "[end of GC (cells used: %ld/%ld)]\n", n-gc_cells_collected, n);
  }
}

static void gc_mark(SCM ptr)
{
Top:
   if NULLP(ptr)     	return;
   if (ptr->gc_mark) 	return;

   ptr->gc_mark = 1;
   switch (ptr->type) {
     case tc_nil: 	  return;
     case tc_cons:	  gc_mark(CAR(ptr));ptr = CDR(ptr); goto Top;
     case tc_flonum:	  return;
     case tc_integer:	  return;
     case tc_bignum:	  return;
     case tc_symbol:      ptr = VCELL(ptr);goto Top;
     case tc_keyword:	  return;  
     case tc_subr_0:	  return;
     case tc_subr_1:	  return;
     case tc_subr_2:	  return;
     case tc_subr_3:	  return;
     case tc_subr_0_or_1: return;
     case tc_subr_1_or_2: return;
     case tc_lsubr:	  return;
     case tc_fsubr:	  return;
     case tc_syntax:	  return;
     case tc_closure:	  gc_mark(ptr->storage_as.closure.code);
			  ptr = ptr->storage_as.closure.env;
			  goto Top;
     case tc_free_cell:	  /* -----> Error */
     case tc_char:	  return;
     case tc_string:	  return;
     case tc_vector:	  {
       			    long j;
			    for(j = 0;j < ptr->storage_as.vector.dim; j++)
			      gc_mark(ptr->storage_as.vector.data[j]);
			    return;
			  }
     case tc_eof:      	  return;
     case tc_undefined:	  return;
     case tc_iport:	  return;
     case tc_oport:	  return;
     case tc_isport:	  return;
     case tc_osport:	  return;
     case tc_boolean:	  return;
     case tc_macro:	  ptr = ptr->storage_as.macro.code; goto Top;
     case tc_localvar:	  ptr = ptr->storage_as.localvar.symbol; goto Top;
     case tc_globalvar:	  ptr = VCELL(ptr); goto Top;
     case tc_cont:	  mark_locations((SCM *)C_STACK(ptr), 
					 (SCM *)(C_STACK(ptr)+C_LEN(ptr)-1));
       			  ptr = C_WIND_STACK(ptr); 
       			  goto Top;
     case tc_env:	  ptr = ptr->storage_as.env.data;
			  goto Top;
#ifdef USE_TK
     case tc_tkcommand:	  return;
#endif
     case tc_quote:	  return;
     case tc_lambda:	  return;
     case tc_if:	  return;
     case tc_setq:	  return;
     case tc_cond:	  return;
     case tc_and:	  return;
     case tc_or:	  return;
     case tc_let:	  return;
     case tc_letstar:	  return;
     case tc_letrec:	  return;
     case tc_begin:	  return;
     case tc_promise: 	  ptr = ptr->storage_as.promise.expr; goto Top;
     case tc_unbound:     return;
   }
   /* if we are here, it's an implementation error. Signal it */
   fprintf(stderr, "INTERNAL ERROR: trying to mark %x (type=%d)\n", 
	           ptr, ptr->type);
}

static void gc_sweep(void)
{
  SCM ptr,end,nfreelist;
  long n;

  end       = heap_end;
  n 	    =  0;
  nfreelist = NIL;

  for (ptr=heap_org; ptr < end; ptr++) {
    if (ptr->gc_mark == 0) {
      switch (ptr->type) {
        case tc_nil: 	     break;
	case tc_cons:	     break;
	case tc_flonum:	     break;
	case tc_integer:     break;
	case tc_bignum:	     mpz_clear(BIGNUM(ptr)); free(BIGNUM(ptr)); break;
	case tc_symbol:      break;
	case tc_keyword:     free(KEYVAL(ptr)); break;
	case tc_subr_0:	     break;
	case tc_subr_1:	     break;
	case tc_subr_2:	     break;
	case tc_subr_3:	     break;
	case tc_subr_0_or_1: break;
	case tc_subr_1_or_2: break;
	case tc_lsubr:	     break;
	case tc_fsubr:	     break;
	case tc_syntax:	     break;
	case tc_closure:     break;
	case tc_free_cell:   break;
	case tc_char:	     break;
	case tc_string:	     free(ptr->storage_as.string.data); break;
	case tc_vector:	     free(ptr->storage_as.vector.data); break;
	case tc_eof:         break;
	case tc_undefined:   break;
	case tc_iport:	     freeport(ptr); break;
	case tc_oport:	     freeport(ptr); break;
	case tc_isport:	     free_string_port(ptr); break;
	case tc_osport:	     free_string_port(ptr); break;
	case tc_boolean:     break;
	case tc_macro:	     break;
	case tc_localvar:    break;
	case tc_globalvar:   break;
	case tc_cont:	     free(ptr->storage_as.cont.data); break;
	case tc_env:	     break;
#ifdef USE_TK
	case tc_tkcommand:   free(ptr->storage_as.tk.data); break;
#endif
	case tc_quote: 	     break;
	case tc_lambda:	     break;
	case tc_if:	     break;
	case tc_setq:	     break;
	case tc_cond:	     break;
	case tc_and:	     break;
	case tc_or:	     break;
	case tc_let:	     break;
	case tc_letstar:     break;
	case tc_letrec:	     break;
	case tc_begin:	     break;
	case tc_promise:     return;
	case tc_unbound:     break;
	default:	    fprintf(stderr,
				    "FATAL ERROR: trying to sweep %x (type=%d)\n",
				    ptr, ptr->type);
      }

      /* Declare this cell free and put it in free list */
      ptr->type = tc_free_cell;
      CDR(ptr)  = nfreelist;
      nfreelist = ptr;
      n        += 1;
    }
    else
      ptr->gc_mark = 0;
  }
  gc_cells_collected = n;
  freelist 	     = nfreelist;
}

static void mark_locations(SCM *start, SCM *end)
{
  register SCM p;
  register long j, n;
  
  if (start > end) {
    SCM *tmp;
    tmp = start; start = end; end = tmp;
  }
  n = end - start;

  if (gc_verbose)
    fprintf(stderr, "[Marking zone <0x%x->0x%x> (%ld words)]\n", start, end, n);
  for(j=0; j<n; j++) {
    p = start[j];
    
    if ((p >= heap_org) && (p < heap_end) &&
	(((((char *)p) - ((char *)heap_org)) % sizeof(struct obj)) == 0) &&
	NTYPEP(p,tc_free_cell))
      gc_mark(p);
  }
}

static void mark_protected(void)
{
  struct gc_protected *reg;
  long j;

  /* Mark protected vars */
  for(reg = protected_registers; reg; reg = reg->next) gc_mark(reg->location);
  
  /* Mark all objects accessible from obarray */
  for (j = 0; j < OBARRAY_SIZE; j++) gc_mark(obarray[j]);
}

void gc_mark_and_sweep(void)
{
  SCM stack_end;	/* The topmost variable allocated on stack */

  gc_ms_stats_start();
  setjmp(save_regs_gc_mark);
  mark_locations((SCM *) save_regs_gc_mark,
		 (SCM *) (((char *) save_regs_gc_mark)+sizeof(save_regs_gc_mark)));
  mark_protected();
  mark_locations((SCM *) stack_start_ptr, (SCM *) &stack_end);

  gc_sweep();
  gc_ms_stats_end();
}

void gc_for_newcell(void)
{
  long flag;

  if (error_context != ERR_FATAL) {
    flag = no_interrupt(1);
    error_context = ERR_FATAL;
    gc_mark_and_sweep();
    error_context = ERR_OK;
    no_interrupt(flag);
    if NNULLP(freelist) return;
  }
  err("Out of storage",NIL);
}


PRIMITIVE lgc(void)
{
  long flag;

  flag      = no_interrupt(1);
  error_context = ERR_FATAL;
  gc_mark_and_sweep();
  error_context = ERR_OK;
  no_interrupt(flag);
  return(UNDEFINED);
}


void gc_protect(SCM location)
{
  struct gc_protected *reg;

  reg = (struct gc_protected *) must_malloc(sizeof(struct gc_protected));
  
  reg->location 	= location;
  reg->next     	= protected_registers;
  protected_registers 	= reg;
}

void init_gc(void)
{
  SCM ptr, next;

  heap_org = (SCM) must_malloc(sizeof(struct obj)*heap_size);
  heap_end = heap_org + heap_size;

  /* Prepare heap space */
  for(ptr = heap_org, next=ptr+1; next < heap_end; ptr=next, next=ptr+1) {
    ptr->type = tc_free_cell;
    CDR(ptr) = next;
  }
  CDR(ptr) = NIL;
  freelist = heap_org;
}
