; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File translate.scm / Copyright (c) 1991 Jonathan Rees / See file COPYING

;;;; Translation from Scheme to Common Lisp

; TRANSLATE translates a single Scheme expression into Common Lisp.

(define (translate form env)
  (with-target-package (program-env-package env)
    (lambda ()
      (translate-to-common-lisp (list form) env))))

; Used by translate and translate-file

(define (translate-to-common-lisp forms env)
  (prognify
   (let recur ((forms forms))
     (if (null? forms)
	 '()
	 (cons (with-uid-reset
		(lambda ()
		  (let-fluid @free-variables '()
		    (lambda ()
		      (let ((node (alpha-top (car forms) env)))
			(generate-top
			 node
			 (generation-env (fluid @free-variables))
			 (not (null? (cdr forms)))))))))
	       (recur (cdr forms)))))))

; Used by SCHEME-COMPILE.

(define (translate-lambda form env)
  (with-uid-reset
   (lambda ()
     (let-fluid @free-variables '()
       (lambda ()
	 (let ((node (alpha-top form env)))
	   (if (lambda? node)
	       (generate-lambda-top
		  node
		  (generation-env (fluid @free-variables)))
	       (error "not a lambda expression" form))))))))

; File transduction

(define (really-translate-file source-file-name
			       translated-file-name
			       program-env)
  (let ((source-code (read-file source-file-name)))
    (compiling-to-file
      translated-file-name
      (program-env-package program-env)
      (lambda (port)
	(display ";  from file " port)
	(display (lisp:namestring (lisp:truename source-file-name)) port)
	(newline port))
      (lambda (port)
	(for-each (lambda (form)
		    (write-flattened form port))
		  (translate-to-common-lisp source-code program-env))))))

; The following generates a file CLOSED.PSO from the information we
; have on how to open-code the built-in procedures.

(define (write-closed-definitions module outfile)
  (compiling-to-file outfile
		     (module-package module)
		     (lambda (port) port)
		     (lambda (port)
		       (write-closed-definitions-1 module port))))

(define (write-closed-definitions-1 module port)
  (let ((package (module-package module))
	(sig (module-signature module))
	(env (module-program-env module)))
    (write-form
      `(lisp:export
	 (lisp:quote ,(map (lambda (name) (change-package name package))
			   (signature-names sig))))
      port)
    (let ((funs '())
	  (defs '()))
      (let ((do-it
	     (lambda (name)
	       (let* ((den (program-env-lookup env name))
		      (info (get-integration den)))
		 (if info
		     (let ((sym (program-variable-location den)))
		       (case (car info)
			 ((val)
			  (write-form `(lisp:locally
					   (lisp:declare (lisp:special ,sym))
					 (lisp:setq ,sym ,(cadr info)))
				      port)
			  (write-form `(schi:set-function-from-value
					    (lisp:quote ,sym))
				      port))
			 ((fun)
			  (if (not (memq name '(car cdr))) ;kludge
			      (set! funs (cons (list sym (cadr info))
					       funs))))
			 ((pred)
			  (write-form
			   (case (if (null? (cddr info))
				    'n
				    (caddr info))
			     ((1)
			      `(lisp:defun ,sym (x)
				 (schi:true? (,(cadr info) x))))
			     ((2)
			      `(lisp:defun ,sym (x y)
				 (schi:true? (,(cadr info) x y))))
			     (else
			      `(lisp:defun ,sym (lisp:&rest x)
				 (schi:true? (lisp:apply #',(cadr info)
							 x)))))
			   port)
			  (set! defs (cons sym defs)))
			 ((subst lambda)
			  (write-form `(lisp:defun ,sym ,@(cdr info)) port)
			  (set! defs (cons sym defs)))
			 ((special) 0) ;don't generate any definition
			 (else
			  (error "peculiar built-in" info)))))))))
	(for-each do-it (signature-names sig))
	(for-each do-it (signature-aux-names sig)))
      (write-form
        `(lisp:mapc (lisp:function schi:set-value-from-function)
		    (lisp:quote ,(reverse defs)))
	port)
      (write-form
        `(lisp:mapc #'(lisp:lambda (z)
			(lisp:let ((our-sym (lisp:car z))
				   (cl-sym (lisp:cadr z)))
			  (lisp:setf (lisp:symbol-function our-sym) 
				     (lisp:symbol-function cl-sym))
			  (schi:set-value-from-function our-sym)))
		    (lisp:quote ,(reverse funs)))
	port))))

; Utilities

(define (with-target-package package thunk)
  (let-fluid @target-package package
    thunk))

(define (compiling-to-file outfile package write-message proc)
  (let-fluid @translating-to-file? #t
    (lambda ()
      (with-target-package package
	(lambda ()
	  (call-with-output-file outfile
	    (lambda (port)
	      (write-file-identification port)
	      (write-message port)
	      (newline port)
	      (display "(SCHI:BEGIN-TRANSLATED-FILE)" port)
	      (newline port)
	      ;; Now do the real work.
	      (proc port)
	      (newline port)
	      outfile)))))))

(define (write-file-identification port)
  (newline)
  (display "Writing ")
  (display (lisp:namestring (lisp:truename port)))
  (display "; -*- Mode: Lisp; Syntax: Common-Lisp; Package: " port)
  (display (lisp:package-name (fluid @target-package)) port) ;Heuristic
  (display "; -*-" port)
  (newline port)
  (newline port)
  (display "; This file was generated by " port)
  (display (translator-version) port)
  (newline port)
  (display ";  running in " port)
  (display (scheme-implementation-version) port)
  (newline port))

(define (write-flattened form port)
  (cond ((not (pair? form))
	 (if (not (or (symbol? form)
		      (number? form)
		      (boolean? form)
		      (string? form)
		      (char? form)))
	     ;; Who knows, it might be important.
	     (write-form form port)))
        ((eq? (car form) 'lisp:quote)
	 )				;do nothing
	((eq? (car form) 'lisp:progn)
	 (for-each (lambda (form)
		     (write-flattened form port))
		   (cdr form)))
	(else
	 (write-form form port))))

(define (write-form form port)
  (write-pretty form port (fluid @target-package)))

; (put 'lisp:defun 'scheme-indent-hook 2)
