; File alpha.scm -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Alpha-conversion

; Contexts

(define (note-context! context node)
  (context node))

(define value-context     set-value-refs!)
(define procedure-context set-proc-refs!)
(define lvalue-context	  set-assigned!)
(define define-context	  (lambda (var) var 'define))
(define top-level-context (lambda (var) var 'top-level))

(define (lose context)  ;Ugh.   (let ((f (lambda () 1))) ((begin (foo) f)))
  context ;lose
  value-context)

(define (alpha-top form s-env)
  (alpha form s-env top-level-context))

; Alphatization of a single scheme expression

(define @where (make-fluid '<top>))

(define (alpha exp s-env context)
  (cond ((symbol? exp)
	 (alpha-variable exp s-env context))
	((or (number? exp) (char? exp) (string? exp) (boolean? exp))
	 (make-constant exp))
	((syntactic-closure? exp)
	 (alpha-syntactic-closure exp s-env context))
	((not (pair? exp))
	 (note "Strange expression" exp)
	 (make-constant `(syntax-error ,exp)))
	((symbol? (car exp))
	 (let ((node-or-alphatizer (binding (car exp) s-env)))
	   (if (node? node-or-alphatizer)
	       (alpha-call exp s-env context)
	       (node-or-alphatizer exp s-env context))))
	(else
	 (alpha-call exp s-env context))))

(define (alpha-variable name s-env context)
  (let ((node-or-alphatizer (binding name s-env)))
    (cond ((node? node-or-alphatizer)
	   (note-context! context node-or-alphatizer)
	   node-or-alphatizer)
	  (else (note "syntactic keyword in losing position" name)
		(make-constant name)))))

(define (alpha-call exp s-env context)
  context ;ignored
  (make-call (alpha (car exp) s-env procedure-context)
	     (map (lambda (arg) (alpha arg s-env value-context))
		  (cdr exp))))

; The primitive special forms.

(define (alpha-quote exp s-env context)
  s-env context				;ignored
  (make-constant (cadr exp)))

(define (alpha-lambda exp s-env context)
  (if (not (eq? context procedure-context))
      ;; Not very accurate.  Improve later.
      (for-each-local set-closed-over!
		      s-env))
  (let ((s-env (rename-vars (proper-listify (cadr exp)) s-env)))
    (make-lambda (new-names (cadr exp) s-env)
		 (alpha-beginify (cddr exp) s-env value-context))))

(define (alpha-letrec exp s-env context)
  (let* ((specs (cadr exp))
	 (vars (map car specs))
	 (s-env (rename-vars vars s-env))
	 (new-vars (new-names vars s-env)))
    (make-letrec new-vars
		 (map (lambda (spec)
			(alpha (cadr spec) s-env value-context))
		      specs)
		 (alpha-beginify (cddr exp) s-env (lose context)))))

(define (alpha-if exp s-env context)
  (let ((test (alpha (cadr exp) s-env value-context))
	(con  (alpha (caddr exp) s-env (lose context)))
	(alt  (alpha (cadddr exp) s-env (lose context))))
    (make-if test con alt)))

(define (alpha-set! exp s-env context)
  context				;ignored
  (let ((lhs (alpha (cadr exp) s-env lvalue-context)))
    (if (variable? lhs)
	(make-set! lhs
		   (alpha (caddr exp) s-env value-context))
	(error "bogus SET!" exp))))

(define (alpha-begin exp s-env context)
  (alpha-beginify (cdr exp) s-env context))

(define (alpha-beginify exp-list s-env context)
  (cond ((null? (cdr exp-list))
	 (alpha (car exp-list) s-env context))
	(else
	 (make-begin
	  (alpha (car exp-list)
		 s-env
		 (if (eq? context top-level-context)
		     context
		     value-context))
	  (alpha-beginify (cdr exp-list)
			  s-env
			  (if (eq? context top-level-context)
			      context
			      (lose context)))))))

(define (alpha-define exp s-env context)
  (cond ((eq? context top-level-context)
	 (let ((var (alpha (definition-form-lhs exp) s-env define-context)))
	   (if (not (global-variable? var))
	       (error "This shouldn't happen" exp))
	   (set-status! var 'defined)
	   (let-fluid @where (variable-name var)
	     (lambda ()
	       (make-define var
			    (alpha (definition-form-rhs exp) s-env value-context))))))
	(else
	 (note "definition in illegal context" exp)
	 (lisp:format t "~&*** context = ~S ***~%" context)
	 (lisp:break "Sux")
	 (make-constant "definition in illegal context"))))

; Definitions (for PROCESS-BODY)
; Rather inadequate, but it's within the Scheme spec.  Fix later.

(define (definition-form? thing s-env names)
  (cond ((pair? thing)
	 (and (symbol? (car thing))
	      (not (memq (car thing) names))   ;Hmm.
	      (eq? (maybe-binding (car thing) s-env)
		   alpha-define)))
	((syntactic-closure? thing)
	 (definition-form? (syntactic-closure-exp thing)
			   (syntactic-closure-s-env thing)
			   (syntactic-closure-free-names thing)))
	(else #f)))

(define (definition-form-lhs form)
  (let ((pat (cadr (definition-form-form form))))
    (if (pair? pat) (car pat) pat)))

(define (definition-form-rhs form)
  (let* ((form (definition-form-form form))
	 (pat (cadr form)))
    (if (pair? pat)
	`(lambda ,(cdr pat) ,@(cddr form))
	(caddr form))))

(define (definition-form-form form)
  (if (pair? form)
      form
      (syntactic-closure-exp form)))

; Syntactic closures:

(define syntactic-closure-rtd
  (make-record-type 'syntactic-closure
		    ;; put exp first, for nicer printing
		    '(exp free-names s-env)))

(define make-syntactic-closure
  (record-constructor syntactic-closure-rtd
		      '(s-env free-names exp)))

(define (make-syntactic-closure-list s-env free-names exps)
  (map (lambda (exp)
         (make-syntactic-closure s-env free-names exp))
       exps))

(define syntactic-closure? (record-predicate syntactic-closure-rtd))

(define syntactic-closure-s-env
  (record-accessor syntactic-closure-rtd 's-env))
(define syntactic-closure-free-names
  (record-accessor syntactic-closure-rtd 'free-names))
(define syntactic-closure-exp
  (record-accessor syntactic-closure-rtd 'exp))


(define (alpha-syntactic-closure sc free-names-s-env context)
  (alpha (syntactic-closure-exp sc)
	 (filter-syntactic-env
	     (syntactic-closure-free-names sc)
	     free-names-s-env
	     (syntactic-closure-s-env sc))
	 context))

(define (filter-syntactic-env names
                              names-syntactic-env
                              else-syntactic-env)
  (bind names
        (map (lambda (name)
	       (binding name names-syntactic-env))
	     names)
	else-syntactic-env))

(define (make-expander proc known-s-env)
  (lambda (exp s-env context)
    (alpha (apply proc s-env (cdr exp)) known-s-env context)))

; Syntactic environments

(define (bind names vals s-env)
  (append (map cons names vals) s-env))

(define (binding name s-env)
  (binding-internal 'binding name s-env))

(define (maybe-binding name s-env)
  (binding-internal 'maybe-binding name s-env))

(define (binding-internal request name s-env)
  ;; Creates binding if none is already there
  (let loop ((e s-env))
    (cond ((procedure? e)		;+++ ought to something faster
	   (e request name))
	  ((eq? name (caar e)) (cdar e))
	  (else (loop (cdr e))))))

(define (bind-later s-env)
  (bind (list nil) (list nil) s-env))

(define (bind-now! names vals s-env)
  (set-cdr! s-env (bind names vals (cdr s-env)))
  s-env)

; Add bindings for the core scheme keywords to a given syntactic environment:

(define (core-scheme-macrology s-env core-struct-var)
  core-struct-var ; currently unused
  (let ((a-list `((begin  ,alpha-begin)
		  (if     ,alpha-if)
		  (lambda ,alpha-lambda)
		  (quote  ,alpha-quote)
		  (set!   ,alpha-set!)
		  (letrec ,alpha-letrec)
		  (define ,alpha-define))))
    (bind (map car a-list) (map cadr a-list) s-env)))

; Global syntactic environments

(define (var-accumulator s-env)
  (binding-internal 'accumulator (list nil) s-env))

(define (accumulated-globals s-env)
  (fluid (var-accumulator s-env)))

(define (accumulating-globals g-env thunk)
  (let-fluid (var-accumulator g-env) '() thunk))

(define (make-global-s-env path)
  (let ((table (make-table))
	(accumulator (make-fluid '())))
    (lambda (request name)
      (case request
	((binding)
	 (let ((probe (table-ref table name)))
	   (or probe
	       (let ((new (make-global-variable name path)))
		 (set-fluid! accumulator (cons new (fluid accumulator)))
		 (table-set! table name new)
		 new))))
	((maybe-binding) (table-ref table name))
	((accumulator) accumulator)
	(else (error "bogus request of global environment" request))))))

; Cache the qualified syntactic environment in the variable that
; names the structure.

(define (get-qualified-s-env struct-var)
  (if (variable? struct-var)
      (let ((sub (variable-substitution struct-var)))
	(if sub
	    (if (and (pair? sub)
		     (eq? (car sub) 'struct))
		;; (STRUCT <s-env>)
		(cadr sub)
		(begin (note "variable won't name a structure" struct-var)
		       (make-global-s-env struct-var)))
	    (let ((g-env (make-global-s-env struct-var)))
	      (set-substitution!
	        struct-var
		`(struct ,g-env ,(lambda () (accumulated-globals g-env))))
	      g-env)))
      (begin (note "attempting to qualify by a non-variable" struct-var)
	     (make-global-s-env struct-var))))

; Utilities:

(define (read-file filename)
  (call-with-input-file filename
    (lambda (i-port)
      (let loop ((l '()))
	(let ((form (read i-port)))
	  (cond ((eof-object? form) (reverse l))
		(else
		 (loop (cons form l)))))))))

(define (note msg node)
  (newline)
  (display "** ")
  (display msg)
  (display ": ")
  (write (let-fluid @where '<note>
	   (lambda ()
	     (schemify-top node))))
  (newline)
  (display "   Location: ")
  (write (fluid @where))
  (newline))

; Code generation utilities:; Unique id's

(define @unique-id (make-fluid 0))

(define (with-uid-reset thunk)
  (let-fluid @unique-id 0 thunk))

(define (generate-uid)
  (let ((uid (fluid @unique-id)))
    (set-fluid! @unique-id (+ uid 1))
    uid))

(define (make-name-from-uid name uid)  ;Best if it's not a legal Scheme symbol.
  (string->symbol
   (string-append (symbol->string name)
		  "\\"
		  (number->string uid '(heur)))))

(define (for-each-local proc s-env)
  (do ((e s-env (cdr e)))
      ((or (not (pair? e))
	   (not (pair? (car e))))  ;kludge for losing procedures
       #t)
    (let ((var (cdr (car e))))
      (if (and (node? var)
	       (local-variable? var))
	  (proc var)))))

(define (rename-vars names s-env)
  (bind names (map make-local-variable names) s-env))

(define (new-names bvl env)
  (map-bvl (lambda (var)
	     (binding var env))
	   bvl))

(define (car-is? thing x)  ;useful for peephole optimizers
  (and (pair? thing)
       (eq? (car thing) x)))

; Macrology stuff

(define (add-macrologies g-env macrologies)
  ;; Cf. MAKE-MACROLOGY
  ;; Lucid can't compile this if the LET is present, so the binding has
  ;; been beta-reduced.
  ;(let ((add-macrology
  ;	 (lambda (s-env macrology name)
  ;	   (macrology s-env (binding name g-env))))) ...)
    (do ((macrologies macrologies (cdr macrologies))
	 (s-env g-env
		((caar macrologies)
		 s-env
		 (binding (cadar macrologies) g-env))))
	((null? macrologies)
	 s-env)))

; A macrology in this formulation is a procedure of two arguments: a
; syntactic environment to be extended, and a node representing a
; structure (i.e. environment) relative to which auxiliary bindings may
; be accessed.

; The following gives a convenient way to create macrologies that must
; reference variables from some structure.

; PROC has the form
;  (lambda (outer-s-env known-s-env add-macrology) ... new-s-env)

(define (make-macrology proc)
  (lambda (s-env struct-var)
    (let ((qualified-s-env		;Qualify references to r^3 vars.
	   (get-qualified-s-env struct-var)))
      (proc s-env
	    qualified-s-env
	    ;; add-macrology:
	    (lambda (s-env macrology name)
	      (macrology s-env (binding name qualified-s-env)))))))
