;;; -*-Scheme-*-
;;;
;;;	$Header: /usr/local/scheme/src/edwin/RCS/rcs.scm,v 1.7 1992/02/12 23:49:22 cph Exp $
;;;
;;;	Copyright (c) 1991-92 Massachusetts Institute of Technology
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;
;;; NOTE: Parts of this program (Edwin) were created by translation
;;; from corresponding parts of GNU Emacs.  Users should be aware that
;;; the GNU GENERAL PUBLIC LICENSE may apply to these parts.  A copy
;;; of that license should have been included along with this file.
;;;

;;;; RCS Interface

(declare (usual-integrations))

(define-variable rcs-ci-default-switches
  "Default switches to pass to \\[rcs-ci]."
  "-u")

(define-variable rcs-co-default-switches
  "Default switches to pass to \\[rcs-co]."
  "-l")

(define rcs-log-buffer-name "*RCS-log*")
(define rcs-message-buffer-name "*RCS*")
(define rcs-scratch-buffer-name " *RCS-scratch*")

(define-command rcs-co
  "Check a file out from RCS and visit it.
With prefix argument, prompt for switches to give to `co'.
Otherwise the value of `rcs-co-default-switches' is used."
  (lambda ()
    (let ((switches
	   (let ((default (ref-variable rcs-co-default-switches)))
	     (if (command-argument)
		 (prompt-for-string "co switches" default 'INSERTED-DEFAULT)
		 default))))
      (list switches (rcs-prompt-for-filename "Check out RCS file"))))
  (lambda (switches filename)
    (with-values (lambda () (rcs-files filename))
      (lambda (working-file rcs-file)
	(if (and (file-exists? working-file)
		 (file-writable? working-file)
		 (or (not (prompt-for-confirmation?
			   (string-append "Overwrite "
					  (->namestring working-file))))
		     (begin
		       (delete-file working-file)
		       false)))
	    (editor-error "abort RCS checkout"))
	(let ((buffer (pathname->buffer filename)))
	  (rcs-modified-check buffer "checkout")
	  (let ((working-file-dir (directory-pathname working-file)))
	    (rcs-command (string-append "Checking out "
					(file-namestring working-file)
					" from RCS")
			 working-file-dir
			 "co "
			 (rcs-default-revision-switch switches
						      "rlq"
						      working-file
						      buffer
						      true)
			 " "
			 (enough-namestring rcs-file working-file-dir)))
	  (if buffer
	      (begin
		(revert-buffer buffer true true)
		(select-buffer buffer))
	      (find-file working-file)))))))

(define-command rcs-unlock
  "Unlock a file that has been has been checked out from RCS."
  (lambda ()
    (list (rcs-prompt-for-filename "Unlock RCS file")))
  (lambda (filename)
    (with-values (lambda () (rcs-files filename))
      (lambda (working-file rcs-file)
	(let ((buffer (pathname->buffer filename)))
	  (rcs-modified-check buffer "unlock")
	  (let ((working-file-name (file-namestring working-file))
		(working-file-dir (directory-pathname working-file)))
	    (let ((rcs-file (enough-namestring rcs-file working-file-dir)))
	      (rcs-command (string-append "Unlocking "
					  working-file-name
					  " from RCS")
			   working-file-dir
			   "rm -f " working-file-name
			   ";rcs -u " rcs-file
			   ";co " rcs-file)))
	  (if buffer
	      (begin
		(revert-buffer buffer true true)
		(select-buffer buffer))))))))

(define-command rcs-ci
  "Check a file in to RCS.
With prefix argument, prompt for switches to give to `ci';
otherwise the value of `rcs-ci-default-switches' is used.
Edit log message at prompt except for the initial revision."
  (lambda ()
    (let ((switches
	   (let ((default (ref-variable rcs-ci-default-switches)))
	     (if (command-argument)
		 (prompt-for-string "ci switches" default 'INSERTED-DEFAULT)
		 default))))
      (list switches (rcs-prompt-for-filename "Check in RCS file"))))
  (lambda (switches filename)
    (with-values (lambda () (rcs-files filename))
      (lambda (working-file rcs-file)
	(let ((log-buffer (find-or-create-buffer rcs-log-buffer-name)))
	  (if (file-exists? rcs-file)
	      (let ((window (current-window)))
		(rcs-init-existing log-buffer)
		(if (window-visible? window)
		    (select-window window)))
	      (let ((working-file-dir (directory-pathname working-file)))
		(let ((rcs-file (enough-namestring rcs-file working-file-dir)))
		  (rcs-command (string-append "Creating RCS file " rcs-file)
			       working-file-dir
			       "rcs -i " rcs-file))))
	  (rcs-ci-internal switches working-file rcs-file log-buffer)
	  (bury-buffer log-buffer))))))

(define (rcs-init-existing log-buffer)
  (pop-up-buffer log-buffer true)
  (set-current-major-mode! (ref-mode-object text))
  (message
   (substitute-command-keys
    "Edit RCS log message, finish with \\[exit-recursive-edit]."))
  (enter-recursive-edit)
  (select-buffer log-buffer)
  (set-current-point! (buffer-end log-buffer))
  (if (let ((char (extract-left-char)))
	(and char
	     (not (char=? char #\newline))))
      (insert-newline)))

(define (rcs-ci-internal switches working-file rcs-file log-buffer)
  (let ((working-file-buffer (pathname->buffer working-file))
	(message-buffer (get-rcs-message-buffer))
	(working-file-dir (directory-pathname working-file)))
    (let ((command
	   (string-append "cd " (->namestring working-file-dir)
			  ";ci "
			  (if (string=? "-k" switches)
			      switches
			      (rcs-default-revision-switch switches
							   "rfluq"
							   working-file
							   working-file-buffer
							   true))
			  " "
			  (enough-namestring rcs-file working-file-dir)))
	  (msg
	   (string-append "Checking in "
			  (file-namestring working-file)
			  " to RCS...")))
      (if working-file-buffer
	  (begin
	    (find-file-noselect working-file true)
	    (save-buffer working-file-buffer false)))
      (message msg)
      (rcs-display message-buffer command)
      (shell-command (buffer-region log-buffer)
		     (buffer-end message-buffer)
		     false
		     false
		     command)
      (bury-buffer message-buffer)
      (if working-file-buffer
	  (revert-buffer working-file-buffer true true))
      (message msg "done"))))

(define (rcs-default-revision-switch switches options working-file buffer
				     branch?)
  (if (or (re-search-string-forward
	   (re-compile-pattern (string-append "-[" options "][0-9]") false)
	   false false switches)
	  (not (or buffer (file-exists? working-file))))
      switches
      (let ((revision
	     (if buffer
		 (rcs-get-revision buffer)
		 (let ((buffer
			(find-or-create-buffer rcs-scratch-buffer-name)))
		   (visit-file buffer working-file)
		   (let ((revision (rcs-get-revision buffer)))
		     (kill-buffer buffer)
		     revision)))))
	(if (< (length revision) 3)
	    switches
	    (let ((index
		   (re-search-string-forward
		    (re-compile-pattern (string-append "-[" options "]") false)
		    false false switches))
		  (revision
		   (apply string-append
			  (cons (number->string (car revision))
				(let loop ((revision (cdr revision)))
				  (if (null?
				       (if branch? (cdr revision) revision))
				      '()
				      (cons* "."
					     (number->string (car revision))
					     (loop (cdr revision)))))))))
	      (if index
		  (string-append (string-head switches index)
				 revision
				 (if (or (= index (string-length switches))
					 (char=? #\space
						 (string-ref switches index)))
				     ""
				     " ")
				 (string-tail switches index))
		  (string-append "-"
				 (substring options 0 1)
				 revision
				 " "
				 switches)))))))

(define (rcs-get-revision buffer)
  (let ((end (buffer-end buffer)))
    (let ((find-keyword
	   (lambda (keyword)
	     (let ((mark
		    (search-forward (string-append "$" keyword ":")
				    (buffer-start buffer)
				    end
				    false)))
	       (and mark
		    (skip-chars-forward " " mark end false)))))
	  (parse-revision
	   (lambda (start)
	     (let ((end (skip-chars-forward "0-9." start end)))
	       (let loop ((start start))
		 (if (mark< start end)
		     (let ((mark (char-search-forward #\. start end)))
		       (if (not mark)
			   (list (string->number (extract-string start end)))
			   (cons (if (mark< start mark)
				     (string->number
				      (extract-string start (mark-1+ mark)))
				     0)
				 (loop mark))))
		     '()))))))
      (cond ((find-keyword "Header")
	     => (lambda (mark)
		  (parse-revision
		   (skip-chars-forward " "
				       (skip-chars-forward "^ " mark end)
				       end))))
	    ((find-keyword "Revision") => parse-revision)
	    (else '())))))

(define (rcs-files filename)
  (let ((pathname (->pathname filename)))
    (if (rcs-file-pathname? pathname)
	(values (rcs->working-file-pathname pathname)
		pathname)
	(values pathname
		(working->rcs-file-pathname pathname)))))

(define (rcs-file-pathname? pathname)
  (let ((type (pathname-type pathname)))
    (if type
	(and (string? type)
	     (string-suffix? ",v" type))
	(let ((name (pathname-name pathname)))
	  (and (string? name)
	       (string-suffix? ",v" name))))))

(define (rcs->working-file-pathname pathname)
  (rcs-guarantee-directory
   "working"
   (let ((pathname
	  (let ((type (pathname-type pathname)))
	    (if type
		(pathname-new-type
		 pathname
		 (string-head type (- (string-length type) 2)))
		(pathname-new-name
		 pathname
		 (let ((name (pathname-name pathname)))
		   (string-head name (- (string-length name) 2))))))))
     (let ((directory (pathname-directory pathname)))
       (if (and (pair? directory)
		(equal? "RCS" (car (last-pair directory))))
	   (pathname-new-directory pathname (except-last-pair directory))
	   pathname)))))

(define (working->rcs-file-pathname pathname)
  (rcs-guarantee-directory
   "RCS"
   (let ((pathname
	  (let ((type (pathname-type pathname)))
	    (if type
		(pathname-new-type pathname (string-append type ",v"))
		(pathname-new-name
		 pathname
		 (let ((name (pathname-name pathname)))
		   (if (not name)
		       (error "illegal working file" pathname))
		   (string-append name ",v")))))))
     (let ((directory (pathname-directory pathname)))
       (if (and (pair? directory)
		(not (equal? "RCS" (car (last-pair directory)))))
	   (pathname-new-directory pathname (append directory '("RCS")))
	   pathname)))))

(define (rcs-guarantee-directory type pathname)
  (let ((directory (directory-pathname pathname)))
    (if (not (file-exists? directory))
	(rcs-create-directory type directory))
    pathname))

(define (rcs-create-directory type directory)
  (let ((name (->namestring directory)))
    (if (prompt-for-confirmation?
	 (string-append type " directory " name " does not exist.  Create"))
	(make-directory directory)
	(editor-error "abort RCS operation"))))

(define (rcs-command message-string working-dir . args)
  (message message-string "...")
  (let ((command
	 (apply string-append
		(cons* "cd "
		       (->namestring working-dir)
		       ";"
		       args)))
	(buffer (get-rcs-message-buffer)))
    (rcs-display buffer command)
    (shell-command false (buffer-end buffer) false false command)
    (bury-buffer buffer))
  (append-message "done"))

(define (rcs-display buffer command)
  (pop-up-buffer buffer false)
  (let ((window (get-buffer-window buffer)))
    (set-window-point! window (buffer-end (window-buffer window)))
    (let ((point (window-point window)))
      (if (not (line-start? point))
	  (insert-newline point))
      (set-window-start-mark! window point true)
      (insert-string command point)
      (insert-newline point))
    (update-screen! (window-screen window) false)))

(define-integrable (get-rcs-message-buffer)
  (find-or-create-buffer rcs-message-buffer-name))

(define (rcs-prompt-for-filename prompt)
  (->namestring
   (let ((default (buffer-pathname (current-buffer))))
     (if default
	 (let ((default-name (file-pathname default)))
	   (let ((pathname
		  (prompt-for-pathname
		   (string-append prompt
				  " (default "
				  (->namestring default-name)
				  ")")
		   default
		   false)))
	     (if (or (pathname-name pathname)
		     (and (pathname-type pathname)
			  (not (eq? 'UNSPECIFIC (pathname-type pathname))))
		     (and (pathname-version pathname)
			  (not (eq? 'UNSPECIFIC (pathname-version pathname)))))
		 pathname
		 (merge-pathnames pathname default-name))))
	 (prompt-for-pathname prompt false false)))))

(define (rcs-modified-check buffer operation)
  (if (and buffer
	   (buffer-modified? buffer)
	   (not (prompt-for-confirmation?
		 (string-append "Buffer "
				(buffer-name buffer)
				" is modified.  Continue anyway"))))
      (editor-error "abort RCS " operation)))