/* -*-C-*-

Copyright (c) 1987, 1988 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

/* $Header: unix.c,v 9.41.1.2 88/12/19 22:31:20 GMT cph Exp $

Contains operating system (Unix) dependent procedures. */

#include <sys/types.h>
#include <sys/times.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <signal.h>
#include <errno.h>

#define SYSTEM_NAME "unix"

#if defined(bsd)
#define HAS_DIR
#include <sys/dir.h>
#include <sys/timeb.h>
#include <sys/time.h>
#include <sgtty.h>
#ifdef vax
#define SYSTEM_VARIANT "bsd (vax)"
#endif
#ifdef celerity
#define SYSTEM_VARIANT "bsd (Celerity)"
#endif    
#ifdef sun
#define SYSTEM_VARIANT "bsd (sun)"
#include <sys/vadvise.h>
#endif
#ifdef pyr
#define SYSTEM_VARIANT "bsd (Pyramid)"
#endif
#ifdef alliant
#define SYSTEM_VARIANT "bsd (Alliant)"
#endif
#ifndef SYSTEM_VARIANT
#define SYSTEM_VARIANT "bsd (unknown)"
#endif

#else
#if defined(nu)
#include <sys/timeb.h>
#include <time.h>
#include <sgtty.h>
#define SYSTEM_VARIANT "nu (lose)"

#else /* hpux, ATT */
#include <time.h>
#include <termio.h>
#include <fcntl.h>
#ifdef system3
#include <mknod.h>
#ifdef hpux
#define HAS_DIR
#include <ndir.h>
/* The hp9000s500 system3 version of ndir defines void! */
#ifdef void
#undef void
#endif
#define SYSTEM_VARIANT "hpux (III)"
#else
#define SYSTEM_VARIANT "AT&T (III)"
#endif
#else	/* Not system 3 below here */
#ifndef spectrum
#include <sys/mknod.h>
#else
#include <sys/sysmacros.h>
#endif
#ifdef hpux
#define HAS_DIR
#include <ndir.h>
#define SYSTEM_VARIANT "hpux (V)"
#else
#define SYSTEM_VARIANT "ATT (V)"
#endif
#endif
#endif
#endif

/* Fixnum multiplication */

#ifdef vax

#define Mul_handled

/* Note that "register" is used here (not "fast") since the
   assembly code requires knowledge of the location of
   the variables and they therefore must be in registers.
   This is a kludge.  It depends on what register variables 
   get assigned to what registers.  It should be entirely 
   coded in assembly language.  -- JINX
*/

Pointer
Mul(Arg1, Arg2)
Pointer Arg1, Arg2;
{
  register long A, B, C;

  Sign_Extend(Arg1, A);
  Sign_Extend(Arg2, B);
  asm("	emul	r11,r10,$0,r10");  /* A is in 11, B in 10 */
  C = A;
  A = B;	/* What is all this shuffling? -- JINX */
  B = C;
  /* B should have high order result, A low order */
  if ((B==0  && (A&(-1<<23)) == 0) ||
      (B==-1 && (A&(-1<<23)) == (-1<<23)))
    return Make_Non_Pointer(TC_FIXNUM, A);
  else return NIL;
}

#endif

/* 68k family code.  Uses hp9000s200 conventions for the new compiler. */

#if defined(hp9000s200) && defined(new_cc)
#define Mul_handled

/* The following constants are hard coded in the assembly language
 * code below.  The code assumes that d0 and d1 are scratch registers 
 * for the compiler. 
 */

#if (NIL != 0) || (TC_FIXNUM != 0x1A)
#include "Error: types changed.  Change assembly language appropriately"
#endif

#ifdef MC68020

static long Fixnum_Range[2] = {SMALLEST_FIXNUM , BIGGEST_FIXNUM};

	asm("	text");
	asm("	global _Mul");
	asm("_Mul:");
	asm("	bfexts	4(%sp){&8:&24},%d0");
	asm("	bfexts	8(%sp){&8:&24},%d1");
	asm("	muls.l	%d1,%d0");
	asm("	bvs.b	result_is_nil");
	asm("	cmp2.l	%d0,_Fixnum_Range");
	asm("	bcs.b	result_is_nil");
	asm("	moveq	&0x1A,%d1");
	asm("	bfins	%d1,%d0{&0:&8}");
	asm("	rts");
	asm("result_is_nil:");
	asm("	clr.l	%d0");
	asm("	rts");
	asm("	data");

#else	/* not MC68020, but 68k family */

	/* 20(sp) = arg0; 24(sp) = arg1 because of movem */

	asm("	text");
	asm("	global _Mul");
	asm("_Mul:");
	asm("	movem.l	%d2-%d5,-(%sp)");
	asm("	clr.b	%d5");
	asm("	tst.b	21(%sp)");
	asm("	slt	20(%sp)");
	asm("	bge.b	coerce_1");
	asm("	moveq	&1,%d5");
	asm("	neg.l	20(%sp)");

	asm("coerce_1:");
	asm("	tst.b	25(%sp)");
	asm("	slt	24(%sp)");
	asm("	bge.b	after_coerce");
	asm("	eori.b	&1,%d5");
	asm("	neg.l	24(%sp)");
	asm("after_coerce:");
	asm("	move.l	20(%sp),%d0");
	asm("	move.l	24(%sp),%d1");
	asm("	move.w	%d0,%d2");
	asm("	mulu	%d1,%d2");
	asm("	move.w	%d1,%d4");
	asm("	swap	%d1");
	asm("	move.w	%d1,%d3");
	asm("	mulu	%d0,%d3");
	asm("	swap	%d0");
	asm("	mulu	%d0,%d4");
	asm("	add.l	%d4,%d3");
	asm("	bcs.b	result_is_nil");
	asm("	mulu	%d0,%d1");
	asm("	bne.b	result_is_nil");
	asm("	swap	%d2");
	asm("	add.w	%d3,%d2");
	asm("	bcs.b	result_is_nil");
	asm("	swap	%d3");
	asm("	tst.w	%d3");
	asm("	bne.b	result_is_nil");
	asm("	cmpi.w	%d2,&0x7F");
	asm("	bgt.b	result_is_nil");
	asm("	swap	%d2");
	asm("	tst.b	%d5");
	asm("	beq.b	sign_is_right");
	asm("	neg.l	%d2");
	asm("sign_is_right:");
	asm("	move.l	%d2,-(%sp)");
	asm("	move.b	&0x1A,(%sp)");
	asm("	move.l	(%sp)+,%d0");
	asm("	movem.l	(%sp)+,%d2-%d5");
	asm("	rts");
	asm("result_is_nil:");
	asm("	clr.l	%d0");
	asm("	movem.l	(%sp)+,%d2-%d5");
	asm("	rts");
	asm("	data");
	
#endif	/* not MC68020 */
#endif  /* hp9000s200 */

#ifndef Mul_handled
/* Portable case */
#include "mul.c"
#endif

#if defined(TCFLSH)	/* hpux, ATT */
#define TIOCFLUSH	TCFLSH
#endif

#ifndef FREAD
#define FREAD		0
#endif

/* OpSys dependent I/O operations. */

/* First flush the buffered input,
   then tell the OS to flush the pending input.
*/

#define flush_input_buffer()						\
{									\
  int flags;								\
									\
  flags = FREAD;							\
  (stdin)->_cnt = 0;							\
  ioctl( fileno( stdin), TIOCFLUSH, &flags);				\
}

/* Binary file I/O Operations */

extern long Load_Data(), Write_Data();
extern Boolean Open_Dump_File(), Close_Dump_File();

static FILE *File_Handle;

Boolean
Open_Dump_File( Name, flag)
     Pointer Name;
     char *flag;
{
  extern FILE *fopen();

  File_Handle = fopen(Scheme_String_To_C_String( Name), flag);
  return (File_Handle != ((FILE *) NULL));
}

Boolean
Close_Dump_File()
{
  extern int fclose();

  return (fclose(File_Handle) == 0);
}

long
Load_Data(Count, To_Where)
     long Count;
     char *To_Where;
{
  extern int fread();

  return (fread(To_Where, sizeof(Pointer), Count, File_Handle));
}

long
Write_Data( Count, From_Where)
     long Count;
     char *From_Where;
{
  extern int fwrite();

  return (fwrite(From_Where, sizeof( Pointer), Count, File_Handle));
}

/* File I/O Operations */

FILE *
OS_file_open( name, output_p)
     char *name;
     Boolean output_p;
{
  return (fopen( name, (output_p ? "w" : "r")));
}

Boolean
OS_file_close( stream)
     FILE *stream;
{
  return ((fclose( stream)) == 0);
}

Boolean
OS_file_eof_p( stream)
     FILE *stream;
{
  return (feof( stream));
}

long
OS_file_length( stream)
     FILE *stream;
{
  long file_descriptor, current_position, result;
  extern long lseek();

  file_descriptor = fileno( stream);
  current_position = lseek( file_descriptor, 0, 1);
  result = lseek( file_descriptor, 0, 2);
  if (current_position != lseek( file_descriptor, current_position, 0))
    Primitive_Error( ERR_EXTERNAL_RETURN);
  return (result);
}

/*
  Three-valued; Lisp may need this smartness someday:

     -1   Does not exist.
      0   Don't know.
      1   Exists.
*/

int
OS_file_existence_test (name)
     char *name;
{
  extern int errno;
  struct stat file_stat;
  int status;

  if ((stat (name, &file_stat)) == 0)
    return (1);
  status = errno;
  if ((status == EACCES) || (status == EIO))
    return (0);
  return (-1);
}

long
OS_file_read_chars( stream, buffer, nchars)
     FILE *stream;
     char *buffer;
     long nchars;
{
  return (fread( buffer, 1, nchars, stream));
}

Boolean
OS_file_write_chars( stream, buffer, nchars)
     FILE *stream;
     char *buffer;
     long nchars;
{
  return (fwrite( buffer, 1, nchars, stream) == nchars);
}

/* Working Directory */

#if defined(hpux) || defined(system3)
char *
getcwd( buffer, length)
     char *buffer;
     int length;
{
  FILE *the_pipe;
  char *finder;

  /* Allocate the buffer if needed. */
  if (buffer == NULL)
    {
      extern char *malloc();

      buffer = malloc( length);
      if (buffer == NULL)
	{
	  fprintf( stderr, "\ngetcwd: unable to allocate result.");
	  return (NULL);
	}
    }

  /* Invoke `pwd' and fill the buffer with its output. */
  the_pipe = popen( "pwd", "r");
  if (the_pipe == NULL)
    {
      fprintf( stderr, "\ngetcwd: popen failed.");
      return (NULL);
    }
  fgets( buffer, length, the_pipe);
  pclose( the_pipe);

  /* Remove extraneous newline. */
  finder = buffer;
  while (true)
    {
      if (*finder == '\n')
	{
	  *finder = '\0';
	  break;
	}
      else if (*finder++ == '\0')
	break;
    }
  return (buffer);
}
#endif

Pointer
OS_working_dir_pathname()
{
#if defined(hpux)
  char *path;
  extern char *getenv();
#else
  char path[(FILE_NAME_LENGTH + 2)];
#endif
  Primitive_0_Args();

#if defined(hpux)
  path = getenv( "HOME");
  if (path == NULL)
#else
#if defined(sun) || defined(bsd)
  if (getwd( path, (FILE_NAME_LENGTH + 2)) == NULL)
#else
/* Brain-damage in hp-ux prevents this from working consistently. */
  if (getcwd( path, (FILE_NAME_LENGTH + 2)) == NULL)
#endif
#endif
    return (NIL);
  else
    return (C_String_To_Scheme_String( path));
}

Boolean
OS_set_working_dir_pathname( name)
     char *name;
{
  return (chdir( name) == 0);
}

/* File System Operations */

Boolean
OS_file_remove( name)
     char *name;
{
  return (unlink( name) == 0);
}

Boolean
OS_file_link_physical( old_name, new_name)
     char *old_name, *new_name;
{
  return (link( old_name, new_name) == 0);
}

Boolean
OS_file_link_symbolic( old_name, new_name)
     char *old_name, *new_name;
{
#if defined(bsd)
  return (symlink( old_name, new_name) == 0);
#else
  return (false);
#endif
}

/* Moves the file from OLD-NAME to NEW-NAME.  Simply reduces to a
   hard-link then a delete file; should be fixed for cross-structure
   rename. */

Boolean
OS_file_rename( old_name, new_name)
     char *old_name, *new_name;
{
  if (link( old_name, new_name) != 0)
    return (false);
  else if (unlink( old_name) != 0)
    return (false);
  else
    return (true);
}

#define file_copy_finish( result)				\
{								\
  if (!OS_file_close( source_file))				\
    Primitive_Error( ERR_EXTERNAL_RETURN);			\
  if (!OS_file_close( destination_file))			\
    Primitive_Error( ERR_EXTERNAL_RETURN);			\
  return (result);						\
}

#define file_copy_1( nchars)					\
{								\
  if (OS_file_read_chars( source_file, buffer, (nchars)) < (nchars)) \
    file_copy_finish( false);					\
  if (!OS_file_write_chars( destination_file, buffer, (nchars))) \
    file_copy_finish( false);					\
}

/* An arbitrary length -- could use `malloc' and compute

   file_copy_buffer_length = ((ulimit( 3, 0) - sbrk( 0)) / 2);

   but that is hairy and might not be easy to port to various unices.
   This should be adequate and hopefully will perform better than
   single-character buffering. */

#define file_copy_buffer_length 8192

Boolean
OS_file_copy( source_name, destination_name)
     char *source_name, *destination_name;
{
  FILE *source_file, *destination_file;
  long source_length, buffer_length;
  char buffer[file_copy_buffer_length];

  source_file = OS_file_open( source_name, false);
  if (source_file == NULL)
    return (false);
  destination_file = OS_file_open( destination_name, true);
  if (destination_file == NULL)
    {
      if (!OS_file_close( source_file))
	Primitive_Error( ERR_EXTERNAL_RETURN);
      return (false);
    }
  source_length = OS_file_length( source_file);
  buffer_length =
    ((source_length < file_copy_buffer_length)
     ? source_length : file_copy_buffer_length);
  while (source_length > buffer_length)
    {
      file_copy_1( buffer_length);
      source_length -= buffer_length;
    }
  file_copy_1( source_length);
  file_copy_finish( true);
}
 
Boolean
OS_directory_make( name)
     char *name;
{
  int old_umask;
  Boolean result;

  old_umask = umask( 0);
#ifdef bsd
  result = (mkdir( name, 511) == 0);
#else
  result = (mknod( name, 0040666, ((dev_t) 0)) == 0);
#endif
  umask( old_umask);
  return (result);
}

#ifndef HAS_DIR

Pointer
OS_directory_open( name)
     char *name;
{
  return (NIL);
}

Pointer
OS_directory_read()
{
  Primitive_Error( ERR_EXTERNAL_RETURN);
}

#else /* has directory library */

static DIR *directory_pointer = NULL;
static struct direct *directory_entry = NULL;

#define read_directory_entry()					\
{								\
  directory_entry = readdir( directory_pointer);		\
  if (directory_entry == NULL)					\
    {								\
      closedir( directory_pointer);				\
      directory_pointer = NULL;					\
      return NIL;						\
    }								\
  else								\
    return (C_String_To_Scheme_String( directory_entry->d_name)); \
}

Pointer
OS_directory_open( name)
     char *name;
{
  if (directory_pointer != NULL)
    Primitive_Error( ERR_EXTERNAL_RETURN);
  directory_pointer = opendir( name);
  if (directory_pointer == NULL)
    return (NIL);
  else
    read_directory_entry();
}

Pointer
OS_directory_read()
{
  if (directory_pointer == NULL)
    Primitive_Error( ERR_EXTERNAL_RETURN);
  read_directory_entry();
}
#endif

/* Terminal hacking. */

static Boolean stdin_is_a_kbd, stdout_is_a_crt, Under_Emacs;
forward int TYI_Immediate(), TYI_Buffered();
forward void OS_Re_Init(), OS_Quit();
forward long System_Clock();
extern int OS_tty_tyi();

int
OS_tty_tyi( Immediate, Interrupted)
     Boolean Immediate, *Interrupted;
{
  int C;

  if (stdin_is_a_kbd)
    C = (Immediate ?
	 TYI_Immediate( Interrupted) :
	 TYI_Buffered( Interrupted));
  else if (!Under_Emacs)
    {
      *Interrupted = false;
      if ((C = getchar()) == EOF)
	Microcode_Termination( TERM_EOF);
    }   
  else
    C = TYI_Buffered( Interrupted);
  return (C);
}

#define define_OS_tty_read_char( name, immediate)		\
char								\
name()								\
{								\
  int chr;							\
								\
  while (true)							\
    {								\
      Boolean Interrupted;					\
								\
      chr = OS_tty_tyi( (immediate), &Interrupted);		\
      if (Interrupted)						\
	{							\
	  if ((IntEnb & IntCode) != 0)				\
	    Primitive_Interrupt();				\
	}							\
      else							\
	return ((char) chr);					\
    }								\
}

define_OS_tty_read_char( OS_tty_read_char, false)
define_OS_tty_read_char( OS_tty_read_char_immediate, true)

void
OS_tty_write_char (chr)
     char chr;
{
  putc (chr, stdout);
  return;
}

Boolean
OS_tty_write_chars (string, string_length)
     char *string;
     long string_length;
{
  Boolean result;

  result = ((fwrite (string, 1, string_length, stdout)) == string_length);
  return (result);
}

void
OS_Flush_Output_Buffer()
{
  fflush (stdout);
}

extern char *getenv(), *tgetstr(), *tgoto();
char *CM, *BC, *UP, *CL, *CE, *term;
int LI, CO;

void
outc( C)
     char C;
{
  putchar( C);
  return;
}

static Boolean Can_Do_Cursor;	/* Initialized below. */

Boolean
OS_Clear_Screen()
{
  if (Can_Do_Cursor)
    tputs (CL, LI, outc);
  else
    OS_tty_write_char ('\f');
}

Boolean
OS_tty_move_cursor( x, y)
     long x, y;
{
  if (Can_Do_Cursor)
    tputs (tgoto (CM, x, y), 1, outc);
  return (Can_Do_Cursor);
}

/* Maybe sometime this should be upgraded to use termcap too. */

void
OS_tty_beep()
{
  OS_tty_write_char (BELL);
  return;
}

void
OS_tty_newline()
{
  OS_tty_write_char ('\n');
  return;
}

/* Not currently implemented. */

Boolean
OS_tty_get_cursor (x, y)
     long *x, *y;
{
  *x = 0;
  *y = 0;
  return (false);
}

long
NColumns ()
{
  return (Can_Do_Cursor ? CO : 79);
}

long
NLines ()
{
  return (Can_Do_Cursor ? LI : 24);
}

Boolean
OS_Clear_To_End_Of_Line ()
{
  if (Can_Do_Cursor)
    tputs (CE, 1, outc);
  return (Can_Do_Cursor);
}

/* GNU Emacs interface hackery */

#define emacs_message(mess)						\
{									\
  printf (mess);							\
  fflush (stdout);							\
}

#define SCHEME_ENTER_INPUT_WAIT		"\033s"
#define SCHEME_EXIT_INPUT_WAIT		"\033f"

Boolean
OS_Under_Emacs ()
{
  return (Under_Emacs);
}

/* System clock */

#if defined(bsd) || defined(nu)

long initial_real_time, initial_real_millitm;
long initial_process_time;

long
System_Clock ()
{
  struct tms buff;

  times (&buff);
  return ((100 * (buff.tms_utime - initial_process_time)) / 60);
}

long
OS_real_time_clock ()
{
  struct timeb time_block;
  long delta_time, delta_millitm;

  ftime (&time_block);
  delta_time = (time_block.time - initial_real_time);
  delta_millitm = (time_block.millitm - initial_real_millitm);
  return ((delta_time * 1000) + delta_millitm);
}

void
Init_System_Clock ()
{
  struct timeb time_block;
  struct tms buff;

  ftime (&time_block);
  initial_real_time = time_block.time;
  initial_real_millitm = time_block.millitm;

  times (&buff);
  initial_process_time = buff.tms_utime;

  return;
}

#define OS_time_mark(delta)	(OS_real_time_clock() + (delta))

#else

#if defined(hpux) || defined(ATT)

/* Get definition of HZ. It is 50 for Bobcats, and 100 for Indigo's. */
/* There is a bug in the version of param.h for the hp9000s500 that we have. */
#ifndef hp9000s500
#include <sys/param.h>
#endif

#ifndef HZ	/* Too bad */
#define HZ	60
#endif

long initial_process_time, initial_real_time;

long
System_Clock ()
{
  struct tms buff;

  times (&buff);
  return ((100 * (buff.tms_utime - initial_process_time)) / HZ);
}

long
OS_real_time_clock ()
{
  struct tms buff;

  return ((((times (&buff)) - initial_real_time) * 1000) / HZ);
}

void
Init_System_Clock ()
{
  struct tms buff;
  long current_real_time;

  initial_real_time = (times (&buff));
  initial_process_time = buff.tms_utime;
  return;
}

#define OS_time_mark(delta)	(OS_real_time_clock() + (delta))

#else

/* Should be fixed some day. */

/* Note: These cannot cause errors because the garbage collector wrapper */
   and other system utilities use them.
*/

long
System_Clock ()
{
  return 0;
}

long
OS_real_time_clock ()
{
  return 0;
}

void
Init_System_Clock ()
{
  return;
}

#define OS_time_mark(delta)	0

#endif
#endif

/* Time and dates. */

#if defined(bsd) || defined(hpux) || defined(nu)

extern struct tm *(localtime());

#define Date_Part(C_Name, Which)	\
int					\
C_Name()				\
{					\
  struct tm *Time;			\
  long The_Time;			\
					\
  time(&The_Time);			\
  Time = localtime(&The_Time);		\
  return (Time->Which);			\
}

#else

#define Date_Part(C_Name, ignore)	\
int					\
C_Name()				\
{					\
  return -1;				\
}

#endif

Date_Part(OS_Current_Year, tm_year);
Date_Part(OS_Current_Month, tm_mon + 1);
Date_Part(OS_Current_Day, tm_mday);
Date_Part(OS_Current_Hour, tm_hour);
Date_Part(OS_Current_Minute, tm_min);
Date_Part(OS_Current_Second, tm_sec);

/* Timers (for timer interrupts) */

#if defined(ITIMER_VIRTUAL)
void
Clear_Int_Timer()
{
  struct itimerval New_Value, Old_Value;

  New_Value.it_value.tv_sec = 0;
  New_Value.it_value.tv_usec = 0;

  /* The following two are not necessary according to the
     documentation, but there seems to be a bug in BSD, at least on
     Suns.
   */

  New_Value.it_interval.tv_sec = 0;
  New_Value.it_interval.tv_usec = 0;
  setitimer(ITIMER_REAL, &New_Value, &Old_Value);
  setitimer(ITIMER_VIRTUAL, &New_Value, &Old_Value);
  return;
}

void
Set_Int_Timer(Days, Centi_Seconds)
     long Days, Centi_Seconds;
{
  struct itimerval New_Value, Old_Value;
  long Which_Timer = ITIMER_VIRTUAL;

  Clear_Int_Timer();
  if (Centi_Seconds < 0)
  {
    Centi_Seconds = -Centi_Seconds;
    Which_Timer = ITIMER_REAL;
  }
  New_Value.it_value.tv_sec =
    (Days * 24 * 60 * 60 * 60) + (Centi_Seconds / 100);
  New_Value.it_value.tv_usec = (Centi_Seconds % 100) * 10000;
  New_Value.it_interval.tv_sec = 0;	/* Turn off after it rings */
  New_Value.it_interval.tv_usec = 0;
  setitimer(Which_Timer, &New_Value, &Old_Value);
  return;
}

#else
void
Clear_Int_Timer()
{
  Primitive_Error( ERR_UNIMPLEMENTED_PRIMITIVE);
}

void
Set_Int_Timer(days, centi_seconds)
     long days, centi_seconds;
{
  Primitive_Error( ERR_UNIMPLEMENTED_PRIMITIVE);
}
#endif

/* Keyboard I/O and interrupts */

#if defined(CBREAK)	/* bsd */

#define Immediate_Declarations()					\
struct sgttyb TTY_Block

#define Immediate_Prolog(only_polling)					\
{									\
  ioctl ((fileno (stdin)), TIOCGETP, (& TTY_Block));			\
  (TTY_Block . sg_flags) |= CBREAK;					\
  ioctl ((fileno (stdin)), TIOCSETN, (& TTY_Block));			\
}

#define Immediate_Epilog()						\
{									\
  (TTY_Block . sg_flags) &= (~ CBREAK);					\
  ioctl ((fileno (stdin)), TIOCSETN, (& TTY_Block));			\
}

#else
#if defined(TCFLSH)	/* hpux, ATT */

#ifndef VINTR
#define VINTR	  	0
#define VQUIT	  	1
#define VEOF		4
#define VMIN		4
#define VTIME		5
#endif

#define Immediate_Declarations()					\
struct termio The_Chars;						\
char Old_VMIN, Old_VTIME;						\
unsigned short lflag

#define Immediate_Prolog(only_polling)					\
{									\
  ioctl(fileno(stdin), TCGETA, &The_Chars);				\
  lflag = The_Chars.c_lflag;						\
  The_Chars.c_lflag &= ~(ICANON | ECHO);				\
  Old_VMIN = The_Chars.c_cc[VMIN];					\
  Old_VTIME = The_Chars.c_cc[VTIME];					\
  if (only_polling)							\
  {									\
    The_Chars.c_cc[VMIN] = (char) 0;					\
    The_Chars.c_cc[VTIME] = (char) 0;					\
  }									\
  else									\
  {									\
    The_Chars.c_cc[VMIN] = (char) 1; /* Min # of chars. */		\
    The_Chars.c_cc[VTIME] = (char) 1; /* Timeout in 1/10 sec. */	\
  }									\
  ioctl(fileno(stdin), TCSETA, &The_Chars);				\
}

#define Immediate_Epilog()						\
{									\
  The_Chars.c_cc[VMIN] = Old_VMIN;					\
  The_Chars.c_cc[VTIME] = Old_VTIME;					\
  The_Chars.c_lflag = lflag;						\
  ioctl(fileno(stdin), TCSETA, &The_Chars);				\
}

#else	/* ??? */
/* No immediate IO */

#define Immediate_Declarations()
#define Immediate_Prolog(only_polling)
#define Immediate_Epilog()

#endif
#endif

/* These are pretty trivial */

#define Buffered_Declarations()
#define Buffered_Prolog()
#define Buffered_Epilog()

/* Keyboard Interrupts */

#define CONTROL_A	'A'
#define CONTROL_B	'B'
#define CONTROL_G	'G'
#define CONTROL_U	'U'
#define CONTROL_X	'X'

#define CONTROL_BIT	0100
#define C_A		(CONTROL_A - CONTROL_BIT)
#define C_G		(CONTROL_G - CONTROL_BIT)
#define DISABLE_EOF	-1

static char Int_Char;

int
OS_Get_Next_Interrupt_Character()
{
  int result;

  if (Int_Char == '\0')
    return -1;
  result = ((int) Int_Char);
  Int_Char = '\0';
  return result;
}

/* OS_Clean_Interrupt_Channel is used to clear the input buffer when a
   character interrupt is received.  On most systems this is not
   currently used, but the Emacs interface under hp-ux needs some
   assistance.  Normally this is used in conjunction with some kind of
   distinguished marker in the input stream which indicates where each
   interrupt occurred.

   The `mode' argument allows the following values: */

/* This mode indicates that the input buffer should be flushed up to
   and including the most recent interrupt marker. */
#define UNTIL_MOST_RECENT_INTERRUPT_CHARACTER	0

/* This mode indicates that all interrupts which match
   `interrupt_char' should be removed from the input buffer.  Any
   other interrupts should be left alone. */
#define MULTIPLE_COPIES_ONLY			1

Boolean
OS_Clean_Interrupt_Channel (mode, interrupt_char)
     int mode, interrupt_char;
{
  if (Under_Emacs && (mode == UNTIL_MOST_RECENT_INTERRUPT_CHARACTER))
    while ((OS_tty_read_char_immediate ()) != '\0')
      ;
  return (true);
}

/* Keyboard Interrupts and I/O synchronization */

#define FIRST_TIME	0
#define INTERRUPT	1
#define REDO		2

/* Why is this only looking at certain interrupts? -- CPH */

#define Interrupt_available						\
(((IntCode & IntEnb) & (INT_Character | INT_Timer)) != 0)

typedef struct { Boolean in_input_wait; jmp_buf storage; } reader_context;
static reader_context real_read_env, *saved_read_env;
  
#define Keyboard_Input_Procedure(Name, decl, prolog, epilog)		\
int									\
Name(Interrupted)							\
Boolean *Interrupted;							\
{									\
  int Which_Way;							\
  int C;								\
  decl;									\
									\
  Which_Way = setjmp(saved_read_env->storage);				\
  while(true)								\
  { switch (Which_Way)							\
    { case FIRST_TIME:							\
      		 prolog;						\
      case REDO:							\
		 saved_read_env->in_input_wait = true;			\
		 if (!Interrupt_available)				\
		 {							\
		   C = getchar();					\
		   saved_read_env->in_input_wait = false;		\
		   epilog;						\
		   if (C == EOF)					\
		     Microcode_Termination(TERM_EOF);			\
		   *Interrupted = false;				\
		   return C;						\
		 }							\
									\
      case INTERRUPT:							\
		 saved_read_env->in_input_wait = false;			\
		 epilog;						\
		 *Interrupted = true;					\
      		 return EOF;						\
      default:	 continue;						\
    }									\
  }									\
}

Keyboard_Input_Procedure(TYI_Immediate,
			 Immediate_Declarations(),
			 Immediate_Prolog(false),
			 Immediate_Epilog())

Keyboard_Input_Procedure(TYI_Buffered,
			 Buffered_Declarations(),
			 Buffered_Prolog(),
			 Buffered_Epilog())


/* Keyboard test will need to be considerably haired up to make it portable.
   See the file `src/keyboard.c' in GNU Emacs for the details.
   Who knows what magic VMS will require to perform this. */

#if defined(FIONREAD) && !defined(hp9000s200)

Boolean
are_there_characters_ready()
{
  long temp;

  if (((stdin)->_cnt) > 0)
    return true;
  if (ioctl( fileno( stdin), FIONREAD, &temp) < 0)
    return false;
  return (temp > 0);
}

#else
#if defined(TCFLSH) && !defined(hp9000s500)

Boolean
are_there_characters_ready()
{
  int result;

  result = (getchar ());
  if (result < 0)
    return (false);
  ungetc (result, stdin);
  return (true);
}

#else
/* Unknown... no characters ready. */

#define are_there_characters_ready() false
#endif
#endif

Boolean
OS_read_char_ready_p( delay)
     long delay;
{
  long limit;
  Boolean result;
  Immediate_Declarations();

  Immediate_Prolog(true);
  limit = OS_time_mark(delay);
  while (true)
    {
      if (stdin_is_a_kbd && (are_there_characters_ready ()))
	{
	  result = true;
	  break;
	}
      if (OS_time_mark(0) >= limit)
	{
	  result = false;
	  break;
	}
    }
  Immediate_Epilog();
  return result;
}

/* Interrupt Handlers. Utility definitions. */

typedef int (*signal_handler)();

#define save_read_context(extra)				\
reader_context next_buffer, *old_env;				\
extra;								\
next_buffer.in_input_wait = false;				\
old_env = saved_read_env;					\
saved_read_env = &next_buffer

#define restore_read_context(extra, action)			\
if (old_env->in_input_wait)					\
{								\
  old_env->in_input_wait = false;				\
  saved_read_env = old_env;					\
  extra;							\
  longjmp(saved_read_env->storage, action);			\
  /*NOTREACHED*/						\
}								\
extra;								\
saved_read_env = old_env;					\
return

#define disable_interrupt(signal_name)				\
signal(signal_name, SIG_IGN)

#define enable_interrupt(signal_name, routine)			\
signal(signal_name, routine)

#define interrupt_start(signal_name)				\
save_read_context(disable_interrupt(signal_name))

#define interrupt_end(signal_name, routine, action)		\
restore_read_context(enable_interrupt(signal_name, routine), action)

/* Abort, Termination, and Timer interrupt handlers. */

int
Control_G(sig)
     int sig;
{
  interrupt_start(sig);

  OS_tty_beep();
  OS_Flush_Output_Buffer();
  IntCode |= INT_Character;
  Int_Char = CONTROL_G;
  New_Compiler_MemTop();
  interrupt_end(sig, Control_G, INTERRUPT);
}

/* Kill Scheme after undoing terminal garbage */
/* OS_Quit is called by Microcode_Termination */

int
Kill_Me(sig)
     int sig;
{
  Microcode_Termination(TERM_SIGNAL);
  /*NOTREACHED*/
}

int
Timer_Interrupt(sig)
     int sig;
{
  interrupt_start(sig);
  IntCode |= INT_Timer;
  New_Compiler_MemTop();
  interrupt_end(sig, Timer_Interrupt, INTERRUPT);
}

/* Temporary suspension interrupt handler. */

int Scheme_Process_Id;

#ifdef SIGTSTP

/* Assumes there is sigsetmask */

#define NO_SIGNALS_ALLOWED -1
#define TSTP_MASK ~(1 << (SIGTSTP - 1))
#define BOGUS_SIGNAL 0

/* sig should only be SIGTSTP or BOGUS_SIGNAL in the following */

int
Suspend_Me(sig)
     int sig;
{
  int saved_mask = sigsetmask(NO_SIGNALS_ALLOWED);
  interrupt_start(SIGTSTP);

  OS_Quit();
  sigsetmask(saved_mask & TSTP_MASK);
  kill(Scheme_Process_Id, SIGTSTP);
  sigsetmask(NO_SIGNALS_ALLOWED);
  OS_Re_Init();
  sigsetmask(saved_mask);
  interrupt_end(SIGTSTP, Suspend_Me, REDO);
}

Boolean
Restartable_Exit()
{
  Suspend_Me(BOGUS_SIGNAL);
  return true;
}

#else

Boolean
Restartable_Exit()
{ 
#if false
  fprintf(stderr, "\Restartable_Exit: unimplemented.");
#endif
  return false;
}

#endif

/* Interactive interrupt handler: Utility procedure. */

#define C_STRING_LENGTH 256

void
Examine_Memory()
{
  Pointer *Where;
  char input_string[10];
  int free;
  Boolean interrupted;

  interrupted = false;
  printf("Enter location to examine (0x prefix for hex) : ");
  OS_Flush_Output_Buffer();

  /* Considerably haired up to go through standard (safe) interface.
     Taken from debug.c */
  
  if (interrupted)
    return;
  for (free = 0; free < C_STRING_LENGTH; free++)
  {
    input_string[free] = OS_tty_tyi(false, &interrupted);
    if (interrupted)
      return;
    if (input_string[free] == '\n')
    {
      input_string[free] = '\0';
      break;
    }
  }

  /* Check to see if address is in Hex (0x prefix). */
  if ((input_string[0] == '0') && (input_string[1] == 'x')) 
    sscanf(input_string + 2, "%x", &Where);
  else
    sscanf(input_string, "%d", &Where);
  Print_Expression(*Where, "Contents");
  OS_tty_newline();
  return;
}

/* Interactive interrupt handler. */

int
Ask_Me(sig)
     int sig;
{
  char command;
  Boolean Interrupted;
  Boolean Really_Interrupted;
  interrupt_start(sig);

  Really_Interrupted = false;
  if (!Under_Emacs)
    {
      OS_tty_beep();
      OS_tty_newline();
    }
 Loop:
  if (!Under_Emacs)
    {
      printf("Interrupt character (? for help): ");
      OS_Flush_Output_Buffer();
    }
  do
  {
    flush_input_buffer();
    command = OS_tty_tyi(true, &Interrupted);
    if (Interrupted && (IntCode != INT_Timer))
      goto exit_gracefully;
    Really_Interrupted = Interrupted || Really_Interrupted;
  } while (Interrupted);
  switch (command)
    {
    case 'B':
    case 'b':
      Int_Char = CONTROL_B; break;
    case 'E':
    case 'e':
      if (!Under_Emacs)
	{
	  OS_tty_newline();
	  Examine_Memory();
	}
      goto exit_gracefully;

    case 'D':
    case 'd':
      if (!Under_Emacs)
	{
	  OS_tty_newline();
	  Handle_Debug_Flags();
	}
      goto exit_gracefully;
    case 'T':
    case 't':
      if (!Under_Emacs)
	{
	  OS_tty_newline(); 
	  Back_Trace();
	}
      goto exit_gracefully;

    case 'G':
    case 'g':
      Int_Char = CONTROL_G; break;

    case 'U':
    case 'u':
      Int_Char = CONTROL_U; break;

    case 'X':
    case 'x':
      Int_Char = CONTROL_X; break;

    case 'Z':
    case 'z':
      if (!Under_Emacs)
	{
	  OS_tty_newline();
	}
      Restartable_Exit();
      goto exit_gracefully;
      
    case 'Q':
    case 'q':
      if (!Under_Emacs)
	{
	  OS_tty_newline();
	}
      Microcode_Termination(TERM_HALT);

    case '\f':
      if (!Under_Emacs)
	{
	  OS_Clear_Screen();
	}
      goto exit_gracefully;

    case 'R':
    case 'r':
    {
      forward Boolean confirm();
      forward void recover_from_trap();

      if (!confirm("Do you really want to reset? [Y or N] "))
      {
	goto exit_gracefully;
      }
      recover_from_trap(-1);
    }

    case 'H':
    case 'h':
      if (!Under_Emacs)
	{
	  OS_tty_newline();
	  printf(
"The interrupt character is ^G (cntrl-G).  Unless redefined, when\n");
	  printf(
"typed, the running (Scheme) program will be aborted and the top level\n");
	  printf(
"read-eval-print loop will resume control.\n");
	  printf(
"The quit character is ^A.  When typed, it offers various options.\n");
	  printf(
"Type ^A? for a list of the options.\n");
	}
      goto exit_gracefully;

    case 'I':
    case 'i':
      if (!Under_Emacs)
	{
	  printf("Ignored.\n");
	} 

exit_gracefully:
      interrupt_end(sig, Ask_Me,
		    (Really_Interrupted ? INTERRUPT : REDO));

    default:
      if (!Under_Emacs)
	{
	  OS_tty_newline();
	  printf("B: Enter a breakpoint loop.\n");
	  printf("D: Debugging: change interpreter flags.\n");
	  printf("E: Examine memory location.\n");
	  printf("X: Abort to current REP loop.\n");
	  printf("G: Goto to top level read-eval-print (REP) loop.\n");
	  printf("H: Print simple information on interrupts.\n");
	  printf("I: Ignore interrupt request.\n");
	  printf("U: Up to previous (lower numbered) REP loop.\n");
	  printf("Q: Quit instantly, killing Scheme.\n");
	  printf("T: Stack trace.\n");
	  printf("R: Hard reset.  Very dangerous.\n");
	  printf("Z: Quit instantly, suspending Scheme.\n");
	  printf("^L: Clear the screen.\n");
	}
      goto Loop;
  }
  IntCode |= INT_Character;
  New_Compiler_MemTop();
  interrupt_end(sig, Ask_Me, INTERRUPT);
}

/* Trap handlers. */

/* Can it happen while we are not inside a primitive?
   Otherwise we can find out in what primitive it happened in from
   the expression register.
 */

int
Floating_Trap(sig, code)
     int sig, code;
{
  disable_interrupt(sig);

#if false
  fprintf(stderr, "\nFloating trap: code = %d\n", code);
  Primitive_Error(ERR_FLOATING_OVERFLOW);
#endif
  enable_interrupt(sig, Floating_Trap);
  return;
}

void
recover_from_trap(sig)
     int sig;
{
  IntCode = INT_GC;
  IntEnb = INT_Mask;
  if (Free < MemTop)
    Free = MemTop;
  History = Make_Dummy_History();
  Initialize_Stack();
 Will_Push(CONTINUATION_SIZE + (STACK_ENV_EXTRA_SLOTS + 2));
  Store_Return(RC_END_OF_COMPUTATION);
  Store_Expression(NIL);
  Save_Cont();
  Push(Make_Non_Pointer(TC_FIXNUM, sig));
  Push(Get_Fixed_Obj_Slot(Trap_Handler));
  Push(STACK_FRAME_HEADER + 1);
 Pushed();
  longjmp(*Back_To_Eval, PRIM_APPLY);
  /* The following comment is by courtesy of LINT, your friendly sponsor. */
  /*NOTREACHED*/
}

char
trap_read_option(Interrupted_ptr)
     Boolean *Interrupted_ptr;
{
  char option;
  Boolean Interrupted;

  OS_Flush_Output_Buffer();
  if (Under_Emacs)
    emacs_message(SCHEME_ENTER_INPUT_WAIT);
  flush_input_buffer();
  option = OS_tty_tyi(true, &Interrupted);
  if (Under_Emacs)
    emacs_message(SCHEME_EXIT_INPUT_WAIT);
  if (!Interrupted && !Under_Emacs && option != '\n')
    OS_tty_write_char(option);
  if (Interrupted_ptr != ((Boolean *) NULL))
    *Interrupted_ptr = Interrupted;
  return option;
}

Boolean
confirm(string)
     char *string;
{
  char answer;

  printf("%s", string);
  answer = trap_read_option((Boolean *) NULL);
  return ((answer == 'y') || (answer == 'Y'));
}

int
Hardware_Trap(sig, code)
     int sig, code;
{
  void Trap_Common();

  Trap_Common("The hardware", sig, code, Hardware_Trap);
  /*NOTREACHED*/
}

int
Software_Trap(sig, code)
     int sig, code;
{
  void Trap_Common();

  Trap_Common("System software", sig, code, Software_Trap);
  /*NOTREACHED*/
}

void
Trap_Common(message, sig, code, handler)
     char *message;
     int sig, code;
     signal_handler handler;
{
  char command, *name;
  Boolean Interrupted;
  char *find_signal_name();

  disable_interrupt(sig);

  name = find_signal_name(sig);
  printf("\n%s has trapped! signal = %d (%s); code = %d",
	 message, sig, name, code);
  printf("\nThe system may not be able to recover.");
  command = '\0';

try_again:
  do
  {
    if (command != '\n')
      printf("\nChoose an action [Y = proceed; N = kill Scheme; D = dump core] ");
    command = trap_read_option(&Interrupted);
  } while (Interrupted ||
	   ((command != 'N') && (command != 'n') &&
	    (command != 'Y') && (command != 'y') &&
	    (command != 'D') && (command != 'd')));
  printf("\n");
  enable_interrupt(sig, handler);
  if ((command == 'D') || (command == 'd'))
  {
    if (!confirm("Do you really want to dump core? [Y or N] "))
      goto try_again;
    OS_Quit();
    signal(SIGQUIT, SIG_DFL);
    kill(Scheme_Process_Id, SIGQUIT);
    /*NOTREACHED*/
  }
  if ((command == 'N') || (command == 'n'))
  {
    if (!confirm("Do you really want to kill Scheme? [Y or N] "))
      goto try_again;
    Microcode_Termination(TERM_EXIT);
    /*NOTREACHED*/
  }
  recover_from_trap(sig);
  /*NOTREACHED*/
}

/* Signal handler descriptors */

typedef struct { int the_signal;
		 char *the_name;
		 Boolean do_it_always;
		 signal_handler handler;
	       } signal_state;

/* The only signals which are always assumed to be there are
   SIGINT and SIGQUIT.
 */

static signal_state scheme_signal_handlers[] =
{ { SIGINT,	"SIGINT",	true,	((signal_handler) Control_G)},
  { SIGQUIT,	"SIGQUIT",	false,	((signal_handler) Ask_Me)},
#ifdef SIGTERM
  { SIGTERM,	"SIGTERM",	true,	((signal_handler) Kill_Me)},
#endif
#ifdef SIGTSTP
  { SIGTSTP,	"SIGTSTP",	false,	((signal_handler) Suspend_Me)},
#endif
#ifdef SIGALRM
  { SIGALRM,	"SIGALRM",	true,	((signal_handler) Timer_Interrupt)},
#endif
#ifdef SIGVTALRM
  { SIGVTALRM,	"SIGVTALRM",	true,	((signal_handler) Timer_Interrupt)},
#endif
#ifdef SIGFPE
  { SIGFPE,	"SIGFPE",	true,	((signal_handler) Floating_Trap)},
#endif
#ifdef SIGILL
  { SIGILL,	"SIGILL",	false,	((signal_handler) Hardware_Trap)},
#endif
#ifdef SIGBUS
  { SIGBUS,	"SIGBUS",	false,	((signal_handler) Hardware_Trap)},
#endif
#ifdef SIGSEGV
  { SIGSEGV,	"SIGSEGV",	false,	((signal_handler) Hardware_Trap)},
#endif

#ifdef SIGTRAP
  { SIGTRAP,	"SIGTRAP",	false,	((signal_handler) Hardware_Trap)},
#endif
#ifdef SIGHUP
  { SIGHUP,	"SIGHUP",	false,	((signal_handler) Kill_Me)},
#endif
#ifdef SIGPIPE
  { SIGPIPE,	"SIGPIPE",	false,	((signal_handler) Kill_Me)},
#endif
#ifdef SIGPWR
  { SIGPWR,	"SIGPWR",	false,	((signal_handler) Kill_Me)},
#endif
#ifdef SIGIOT
  { SIGIOT,	"SIGIOT",	false,	((signal_handler) Software_Trap)},
#endif
#ifdef SIGEMT
  { SIGEMT,	"SIGEMT",	false,	((signal_handler) Software_Trap)},
#endif
#ifdef SIGSYS
  { SIGSYS,	"SIGSYS",	false,	((signal_handler) Software_Trap)},
#endif
#ifdef SIGUSR1
  { SIGUSR1,	"SIGUSR1",	false,	((signal_handler) Software_Trap)},
#endif
#ifdef SIGUSR2
  { SIGUSR2,	"SIGUSR2",	false,	((signal_handler) Software_Trap)},
#endif
#ifdef SIGPROF
  { SIGPROF,	"SIGPROF",	false,	((signal_handler) Software_Trap)},
#endif
};

/* Missing HPUX signals:
   SIGKILL, SIGCLD, SIGIO, SIGWINDOW
*/

/* Missing bsd signals:
   SIGKILL, SIGURG, SIGSTOP, SIGCONT, SIGCHLD,
   SIGTTIN, SIGTTOU, SIGIO, SIGXCPU, SIGXFSZ
*/

#define NHANDLERS sizeof(scheme_signal_handlers)/sizeof(signal_state)

static signal_state outside_signal_handlers[NHANDLERS];

void
hack_signals(source, dest, do_all, save_state)
     signal_state *source, *dest;
     Boolean do_all, save_state;
{
  signal_handler old_handler;
  fast int i;
  fast signal_state *from, *to;

  /* The following is total paranoia in case a signal which undoes all
     comes in while we are not yet done setting them all up.
   */
  if (save_state)
    for (i = 0, from = source, to = dest;
	 i < NHANDLERS;
	 i++, *from++, *to++)
    {
      to->the_signal = from->the_signal;
      to->the_name = from->the_name;
      to->do_it_always = from->do_it_always;
      to->handler = ((signal_handler) SIG_DFL);
    }
  for (i = 0, from = source, to = dest;
       i < NHANDLERS;
       i++, *from++, *to++)
  {
    if (from->do_it_always || do_all)
    {
      old_handler =
	((signal_handler) signal(from->the_signal, from->handler));
      if (old_handler == ((signal_handler) -1))
	old_handler = ((signal_handler) SIG_DFL);
    }
    else
      old_handler = ((signal_handler) SIG_DFL);
    if (save_state)
      to->handler = old_handler;
  }
  return;
}

char *
find_signal_name(sig)
     int sig;
{
  fast int i;

  for (i = 0; i < NHANDLERS; i++)
  {
    if (scheme_signal_handlers[i].the_signal == sig)
      return (scheme_signal_handlers[i].the_name);
  }

  return "unknown";
}

#define TERMCAP_BUFFER_SIZE 1024

#if defined(HPUX) && !defined(system3)
#define Break_Terminal_Connection()	setpgrp()
#else
#define Break_Terminal_Connection()
#endif

void
OS_Init(ignore)
     Boolean ignore;
{
  char termcaps[TERMCAP_BUFFER_SIZE];
  static char tcb[TERMCAP_BUFFER_SIZE];
  char *tcp = &tcb[0];

  OS_Name = SYSTEM_NAME;
  OS_Variant = SYSTEM_VARIANT;

  printf("MIT Scheme, %s [%s] version\n", OS_Name, OS_Variant);

  Init_System_Clock();

  real_read_env.in_input_wait = false;
  saved_read_env = &real_read_env;

  /* Find process information */

  Under_Emacs =
    Parse_Option("-emacs", Saved_argc, Saved_argv, true) != NOT_THERE;

  Scheme_Process_Id = getpid();

  stdin_is_a_kbd = isatty( fileno( stdin));
  stdout_is_a_crt = isatty( fileno( stdout));

  /* The ultimate in C style -- by Jinx */

  if (Under_Emacs					||
      (!stdout_is_a_crt)				||
      ((term = getenv("TERM")) == NULL) 		||
      (tgetent(termcaps, term) <= 0)			||
      ((CM = tgetstr("cm", &tcp)) == NULL))
    Can_Do_Cursor = false;
  else /* Find terminal information */
    {
      LI = tgetnum("li");
      CO = tgetnum("co");
      UP = tgetstr("up", &tcp);
      CL = tgetstr("cl", &tcp);
      CE = tgetstr("ce", &tcp);
      BC = tgetflag("bs") ? "\b" : tgetstr("bc", &tcp);
      Can_Do_Cursor = true;
    }
  Int_Char = 0;

  if (stdout_is_a_crt && stdin_is_a_kbd && !Under_Emacs)
    printf("^AH (CTRL-A, then H) shows help on interrupt keys.\n");

  if (!stdin_is_a_kbd && !stdout_is_a_crt && !Under_Emacs)
    Break_Terminal_Connection();
  
  OS_Flush_Output_Buffer();

  /* Swap in Scheme IO */
  OS_Re_Init();
}

/* Terminal parameter hacking. */

static char Orig_Interrupt, Orig_Quit, Orig_EOF;

#if defined(bsd) || defined(nu)

static long Orig_flags;

#define hack_crt(old, new)						\
{									\
  int crt_pgrp;								\
									\
  ioctl(fileno(stdout), TIOCGPGRP, &crt_pgrp);				\
  if (getpgrp(Scheme_Process_Id) == crt_pgrp)				\
  {									\
    struct sgttyb sg;							\
									\
    ioctl(fileno(stdout), TIOCGETP, &sg);				\
    Orig_flags = old;							\
    sg.sg_flags = new;							\
    ioctl(fileno(stdout), TIOCSETN, &sg);				\
  }									\
}

#define hack_kbd(Ni, Oi, Nq, Oq, Ne, Oe)				\
{									\
  int crt_pgrp;								\
									\
  ioctl(fileno(stdin), TIOCGPGRP, &crt_pgrp);				\
  if (getpgrp(Scheme_Process_Id) == crt_pgrp)				\
    basic_hack_kbd(Ni, Oi, Nq, Oq, Ne, Oe);				\
}

#else
#if defined(TCSETA) /* hpux, ATT */

/* Make hpux/system V look like bsd so hack_kbd works */

#define tchars termio
#define TIOCGETC TCGETA
#define TIOCSETC TCSETA
#define t_intrc c_cc[VINTR]
#define t_quitc c_cc[VQUIT]
#define t_eofc  c_cc[VEOF]
#define hack_crt(old, new)

#define hack_kbd(Ni, Oi, Nq, Oq, Ne, Oe)				\
  basic_hack_kbd(Ni, Oi, Nq, Oq, Ne, Oe)

#else /* ??? */

#define hack_crt(old, new)
#define hack_kbd(Ni, Oi, Nq, Oq, Ne, Oe)

#endif
#endif

/* This swaps interrupt characters */

#define basic_hack_kbd(Nintr, Ointr, Nquit, Oquit, NEOF, OEOF)		\
{									\
  struct tchars Terminal_Chars;						\
									\
  ioctl(fileno(stdin), TIOCGETC, &Terminal_Chars);			\
  Orig_Interrupt = Ointr;						\
  Orig_Quit = Oquit;							\
  Orig_EOF = OEOF;							\
  Terminal_Chars.t_intrc = Nintr;					\
  Terminal_Chars.t_quitc = Nquit;					\
  Terminal_Chars.t_eofc  = NEOF;					\
  ioctl(fileno(stdin), TIOCSETC, &Terminal_Chars);			\
}

/* These procedures swap the terminal and signals from outside to scheme,
   and viceversa.
*/

void
OS_Re_Init()
{
  if (stdin_is_a_kbd)
    hack_kbd(C_G, Terminal_Chars.t_intrc,
	     C_A, Terminal_Chars.t_quitc,
	     DISABLE_EOF, Terminal_Chars.t_eofc);

  hack_signals(scheme_signal_handlers,
	       outside_signal_handlers,
	       (stdin_is_a_kbd || Under_Emacs),
	       true);

  if (stdout_is_a_crt)
    hack_crt(sg.sg_flags, (sg.sg_flags & (~XTABS)));
#ifdef sun
  vadvise(VA_ANOM);		/* Anomolous paging, don't try to guess. */
#endif  
  return;
}

void
OS_Quit()
{
  OS_Flush_Output_Buffer();
  if (stdout_is_a_crt)
    hack_crt(Orig_flags, Orig_flags);
  
  hack_signals(outside_signal_handlers,
	       scheme_signal_handlers,
	       (stdin_is_a_kbd || Under_Emacs),
	       false);
	       
  if (stdin_is_a_kbd)
    hack_kbd(Orig_Interrupt, Orig_Interrupt,
	     Orig_Quit, Orig_Quit,
	     Orig_EOF, Orig_EOF);
  return;
}
