/* -*-C-*-

Copyright (c) 1987 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

/* $Header: Xgraph.c,v 1.1 87/06/16 04:14:05 GMT cph Rel $ */

/************************************************************************
 * Athena graphics primitives.
 *
 * The following routines are the graphics primitives for CScheme on the VAX
 * when using the X Window.  The routines
 * mostly use the same primitive procedure slots as in the 68000 version.
 * Any discrepancies are resolved in "graphics.scm" with the definition
 * for the graphics-package. 
 *
 * The X interface supports multiple windows.
 * The properties of each window are described in a structure 
 * (SchemeWindowInfo) bound to the respective window. Graphics operations
 * are permitted to distinguish between draw and fill parameters.
 *
 * as of 30.01.87 support for penplot was eliminated.
 * the code is still prepared to dispatch on display type, but
 * only the X interface is present.
 *
 * Written by James Anderson
 ************************************************************************/

#include "scheme.h"
#include "primitive.h"
#include "flonum.h"
#include "Xgraph.h"
#include "math.h"
#include "ctype.h"

/* manifest constants */
#define YES			    1
#define NO			    0

/* additional display types can be added here for switching */
#define XTERM			    1
#define MaxActiveDisplays           4

#define ESC			'\033'

#define RedYIQFactor		0.30 /* for color to intensity */
#define GreenYIQFactor		0.59
#define BlueYIQFactor		0.11


/* macros */
#define CheckGraphicsStatus()	        			\
if (currentDisplay == NULL) {					\
  fprintf(stderr,"You have not yet initialized graphics.\n");	\
    }

#define CheckXStatus()						\
if (currentWindow == 0) { return NIL; }

#define CheckDrawFunction(FVal)					\
((FVal < 0 || FVal > 16) ? -1 : FVal)

#define CanonicalLinePattern(lp)                                \
((lp >= 0 && lp <= 9)?linePatterns[lp]:lp)
 
#define If_Error(Err)						\
        {if (Err == 1)                                          \
        Primitive_Error(ERR_ARG_1_WRONG_TYPE);                  \
	if (Err == 2)                                           \
	  Primitive_Error(ERR_ARG_2_WRONG_TYPE);		\
       }

#define ConditionalXFlush()					\
if (flushMode == YES) { XFlush(); }

#define W2DC_NoTranslate(w,w1,w2,d1,d2) W2DC(w,w1,w2,d1,d2)

/* static storage */
/* flags */
static int first_timeX = YES;
static int flushMode = YES;
/* descriptors */
static XEvent currentEvent;
static int terminalType = 0;
static int currentDisplayId = -1;
static Display *currentDisplay = NULL;
static Window currentWindow = 0;
static Window terminalWindow = 0;
static XAssocTable *schemeXAssocTable = NULL;
static SchemeWindowInfo *currentWindowInfo = NULL;
static struct {
  Display *display;
  Window window;
  SchemeWindowInfo *windowInfo;
} activeDisplay[MaxActiveDisplays];

static long linePatterns[10] = { 0xffff, /* solid line */
				 0x1111, /* widely spaced dots */
				 0xf0f0, /* long dashes */
				 0xcaca, /* short dash two dots */
				 0xeddb, /* medium dash, short dash */
				 0xf57a, /* long dash two dots */
				 0xeeee, /* medium dashes */
				 0xdb6d, /* short dashes */
				 0xd6b5, /* short dash, one dot */
				 0xaaaa /* closely spaced dots */
				 };
static Cursor arrowCursor;
static Cursor crossCursor;
static Cursor nullCursor;
#define arrow_width 16
#define arrow_height 16
#define arrow_x_hot 0
#define arrow_y_hot 0
static short arrow_bits[] = {
   0x003f, 0x001f, 0x000f, 0x001f,
   0x003b, 0x0071, 0x00e0, 0x0040,
   0x0000, 0x0000, 0x0000, 0x0000,
   0x0000, 0x0000, 0x0000, 0x0000};
#define cross_width 16
#define cross_height 16
#define cross_x_hot 4
#define cross_y_hot 4
static short cross_bits[] = {
   0x0010, 0x0010, 0x0010, 0x0010,
   0x01ef, 0x0010, 0x0010, 0x0010,
   0x0010, 0x0000, 0x0000, 0x0000,
   0x0000, 0x0000, 0x0000, 0x0000};
#define null_width 16
#define null_height 16
static short null_bits[] = {
   0x0000, 0x0000, 0x0000, 0x0000,
   0x0000, 0x0000, 0x0000, 0x0000,
   0x0000, 0x0000, 0x0000, 0x0000,
   0x0000, 0x0000, 0x0000, 0x0000};


/* (GRAPHICS-INITIALIZE)
   Determines terminal type;
   establishes connection to server;
*/
Define_Primitive(Prim_Graphics_Initialize, 1, "GRAPHICS-INITIALIZE")
{
  float left, right, bottom, top;
  int strcmp();
  Window window;
  SchemeWindowInfo *windowInfo;
  int SchemeXError();
  int dummy;
  char displayName[64];
  int displayId;
  char *getenv();
  Pointer GetStringValue();
  Primitive_1_Arg();

/* fetch display name */
  switch (Type_Code(Arg1)) {
  case TC_UNINTERNED_SYMBOL:
  case TC_INTERNED_SYMBOL:
    Arg1 = GetStringValue(Arg1);
  case TC_CHARACTER_STRING:
    strncpy(displayName,Scheme_String_To_C_String(Arg1),64);
    break;
  default:
    displayName[0] = '\0';
    break;
  }
  terminalType = GetTerminalType(displayName);

  switch(terminalType) {
  case XTERM:
    if (first_timeX == YES || displayName[0]) {
      if ((displayId = GetDisplayId()) < 0)
	return NIL;
      currentDisplayId = displayId;
      currentDisplay = XOpenDisplay(((displayName[0])?displayName:NULL)); 
      if (currentDisplay == NULL)
	return NIL;
      if (first_timeX == YES) {	/* allocate assoc table only once, */
				/* locate terminal window, and set */
				/* error handler */
	if ((schemeXAssocTable =(XAssocTable *) XCreateAssocTable(24)) == NULL)
	  return NIL;
	XQueryMouse(RootWindow,&dummy,&dummy,&terminalWindow);
	XErrorHandler(SchemeXError);
	nullCursor = XCreateCursor(null_width,null_height,
				   null_bits,null_bits,0,0,
				   WhitePixel,BlackPixel,GXnoop);
	crossCursor = XCreateCursor(cross_width,cross_height,
				    cross_bits,cross_bits,
				    cross_x_hot,cross_y_hot,
				    WhitePixel,BlackPixel,GXxor);
	arrowCursor = XCreateCursor(arrow_width,arrow_height,
				    arrow_bits,arrow_bits,
				    arrow_x_hot,arrow_y_hot,
				    WhitePixel,BlackPixel,GXxor);

	first_timeX = NO;
      }
      activeDisplay[currentDisplayId].display = currentDisplay;
      currentWindow = activeDisplay[currentDisplayId].window = 0;
      currentWindowInfo = activeDisplay[currentDisplayId].windowInfo = NULL;
    }
    else if (currentWindow != 0) {
      XRaiseWindow(currentWindow);
      XClear(currentWindow);
      XFlush();
    }
    break;
  default:
    fprintf(stderr,"\nCan't initialize graphics for display %s.\n",
	    getenv("TERM"));
    Primitive_Error(ERR_EXTERNAL_RETURN);
  }
  
  return C_Integer_To_Scheme_Integer(currentDisplayId);
}

/* (GRAPHICS-DONE)
   When finished user will call this routine to finish things up and
   restore the terminal to normal operating conditions.
   Cycle through activeDisplay table and close all displays.
*/
Define_Primitive(Prim_Graphics_Done, 0, "GRAPHICS-DONE")
{
  int displayIndex;

  for (displayIndex = 0; displayIndex < MaxActiveDisplays; displayIndex ++) {
    if (activeDisplay[displayIndex].display != NULL)
      switch(activeDisplay[displayIndex].display->dtype) {
      case XTERM:
	XCloseDisplay(activeDisplay[displayIndex].display);
	break;
      default:
	return NIL;
      }
    activeDisplay[displayIndex].display = NULL;
    activeDisplay[displayIndex].window = 0;
    activeDisplay[displayIndex].windowInfo = NULL;
  }
  first_timeX = YES;
  currentDisplayId = -1;
  currentDisplay = NULL;
  currentWindow = 0;
  currentWindowInfo = NULL;
  return TRUTH;
}

/* (GRAPHICS_SET_DISPLAY! DISPLAY-ID)
   set display parameters from table.
 */
Define_Primitive(Prim_Graphics_Set_Display, 1, "GRAPHICS-SET-DISPLAY!")
{
  int displayId;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  if (ToInteger(Arg1,&displayId) == NIL)
    return(NIL);
  if (0 <= displayId && displayId < MaxActiveDisplays)
    if (activeDisplay[displayId].display != NULL) {
      currentDisplayId = displayId;
      currentDisplay = activeDisplay[displayId].display;
      currentWindow = activeDisplay[displayId].window;
      currentWindowInfo = activeDisplay[displayId].windowInfo;
      switch(currentDisplay->dtype) {
      default:
	terminalType = XTERM;
	XSetDisplay(currentDisplay);
	break;
      }
      return(TRUTH);
    }
  return NIL;
}

/* (GRAPHICS-OPEN-WINDOW X Y W H)
   opens a new window and returns cell with window handle
 */
Define_Primitive(Prim_Graphics_Open_Window, 4, "GRAPHICS-OPEN-WINDOW")
{
  int x, y, w, h;
  char geometry[64];
  char tg[32];
  Window window;
  fast Pointer Arg4;
  Primitive_3_Args();
  Arg4 = Stack_Pointer[3];

  CheckGraphicsStatus();
  strcpy(geometry,"=");
  if (ToInteger(Arg3,&w) != NIL) {
    sprintf(tg,"%d",w);
    strcat(geometry,tg);
  }
  if (ToInteger(Arg4,&h) != NIL) {
    sprintf(tg,"x%d",h);
    strcat(geometry,tg);
  }
  if (ToInteger(Arg1,&x) != NIL) {
    if (x >= 0)
      sprintf(tg,"+%d",x);
    else sprintf(tg,"%d",x);
    strcat(geometry,tg);
    if (ToInteger(Arg2,&y) != NIL) {
      if (y >=0)
	sprintf(tg,"+%d",y);
      else sprintf(tg,"%d",y);
      strcat(geometry,tg);
    }
  }
  switch (terminalType) {
  case XTERM:
    if (strcmp(geometry,"=") == 0)
      geometry[0] = '\0';
    if ((window = MakeSchemeWindow(geometry)) == 0)
      return NIL;
    return C_Integer_To_Scheme_Integer(window);
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-WINDOW!)
   sets current window as passed and raises it.
 */
Define_Primitive(Prim_Graphics_Set_window, 1, "GRAPHICS-SET-WINDOW!")
{
  unsigned long int window;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  if (ToInteger(Arg1,&window) == NIL)
    return(NIL);

  switch (terminalType) {
  case XTERM:
    Scheme_Integer_To_C_Integer(Arg1,&window);
    if (SetCurrentWindow(window) != 0) {
      return TRUTH;
    }
    return NIL;
  default:
    return NIL;
  }
}

/* (GRAPHICS-CLOSE-WINDOW)
   closes current window
 */
Define_Primitive(Prim_Graphics_Close_Window, 0, "GRAPHICS-CLOSE-WINDOW")
{
  Primitive_1_Arg();

  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (currentWindow) {
      XDeleteAssoc(schemeXAssocTable,currentWindow);
      XDestroyWindow(currentWindow);
      activeDisplay[currentDisplayId].window = currentWindow = 0;
      free(currentWindowInfo);
      activeDisplay[currentDisplayId].windowInfo = currentWindowInfo = NULL;
    }
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-CONFIGURE-WINDOW WINDOW-ID WINDOW-ID X Y W H)
   modifies position or size of window
   if an argument is NIL, the value is taken from the record of the
   window configuration, rather than from a window query. this makes
   it possible to constrain window configuration.
 */
Define_Primitive(Prim_Graphics_Configure_Window, 5, "GRAPHICS-CONFIGURE-WINDOW")
{
  int x, y, w, h;
  Window window;
  fast Pointer Arg4;
  fast Pointer Arg5;
  Primitive_3_Args();
  Arg4 = Stack_Pointer[3];
  Arg5 = Stack_Pointer[4];

  CheckGraphicsStatus();
  switch(terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = currentWindow;
    if (Arg2 == NIL)
      x = currentWindowInfo->info.x;
    else if (ToInteger(Arg2,&x) == NIL)
      Primitive_Error(ERR_ARG_2_WRONG_TYPE);
    if (Arg3 == NIL)
      y = currentWindowInfo->info.y;
    else if (ToInteger(Arg3,&y) == NIL)
      Primitive_Error(ERR_ARG_3_WRONG_TYPE);
    if (Arg4 == NIL)
      w = currentWindowInfo->info.width;
    else if (ToInteger(Arg4,&w) == NIL)
      Primitive_Error(ERR_ARG_4_WRONG_TYPE);
    if (Arg5 == NIL)
      h = currentWindowInfo->info.height;
    else if (ToInteger(Arg5,&h) == NIL)
      Primitive_Error(ERR_ARG_5_WRONG_TYPE);
    XConfigureWindow(window,x,y,w,h);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}


/* (GRAPHICS-MAP-WINDOW WINDOW-ID)
   maps window onto display
 */
Define_Primitive(Prim_Graphics_Map_Window, 1, "GRAPHICS-MAP-WINDOW")
{
  Window window;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = currentWindow;
    XMapWindow(window);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}


/* (GRAPHICS-UNMAP-WINDOW WINDOW-ID)
   unmaps window from display
 */
Define_Primitive(Prim_Graphics_Unmap_Window, 1, "GRAPHICS-UNMAP-WINDOW")
{
  Window window;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = currentWindow;
    XUnmapWindow(window);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}
 
/* (GRAPHICS-TEXT)
   rotates the monitor window to the top
 */
Define_Primitive(Prim_Graphics_Text, 0, "GRAPHICS-TEXT")
{
  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    XRaiseWindow(terminalWindow);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-GRAPHICS)
   rotates the current graphics window to the top
 */
Define_Primitive(Prim_Graphics_Graphics, 0, "GRAPHICS-GRAPHICS")
{
  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    XRaiseWindow(currentWindow);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-FLUSH! FLUSH-MODE)
   set flag
    (T) immeditately flush
    (NIL) to permit buffering
*/
Define_Primitive(Prim_Graphics_Set_Flush, 1, "GRAPHICS-SET-FLUSH!")
{
  Primitive_1_Arg();

  Touch_In_Primitive(Arg1,Arg1);
  if (Arg1 == NIL)
    flushMode = NO;
  else
    flushMode = YES;
  return TRUTH;
}

/* (GRAPHICS-FLUSH)
   force X to flush buffers
   check status quietly to allow use before initialization
 */
Define_Primitive(Prim_Graphics_Flush, 0, "GRAPHICS-FLUSH")
{
  if (currentDisplay == NULL) /* no error message */
    return NIL;
  switch (terminalType) {
  case XTERM:
    XFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-CLEAR)
   clear the current window;
*/
Define_Primitive(Prim_Graphics_Clear, 0, "GRAPHICS-CLEAR")
{
  float left, right, top, bottom;

  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    XClear(currentWindow);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-PIXFILL [ PIXEL-ID | NIL ])
 * fill current window with specified color
 */
Define_Primitive(Prim_Graphics_PixFill, 1, "GRAPHICS-PIXFILL")
{
  int fillColor;
  Primitive_1_Arg();

  if (Type_Code(Arg1) != TC_NULL && ToInteger(Arg1,&fillColor) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();
  if (Type_Code(Arg1) == TC_NULL)
    fillColor = currentWindowInfo->pnColor;

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    XPixFill(currentWindow,0,0,
	    currentWindowInfo->info.width,currentWindowInfo->info.height,
	    fillColor,0,currentWindowInfo->fillFunction,AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-TILEFILL FILL-PIXMAP)
 * tile current window with specified tile pattern
 */
Define_Primitive(Prim_Graphics_TileFill, 1, "GRAPHICS-TILEFILL")
{
  int fillPixmap;
  Primitive_1_Arg();

  if (Type_Code(Arg1) != TC_NULL && ToInteger(Arg1,&fillPixmap) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();
  if (Type_Code(Arg1) == TC_NULL)
    fillPixmap = currentWindowInfo->fillPat;

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    XTileFill(currentWindow,0,0,
	    currentWindowInfo->info.width,currentWindowInfo->info.height,
	    fillPixmap,0,currentWindowInfo->fillFunction,AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-FUNCTION! DISPLAY-FUNCTION)
 * set draw and fill display functions as passed
 */
Define_Primitive(Prim_Graphics_Set_Function, 1, "GRAPHICS-SET-FUNCTION!")
{
  int function;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&function) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();
  if (CheckDrawFunction(function) < 0)
    return NIL;
  currentWindowInfo->fillFunction =
    currentWindowInfo->pnFunction = function;
  
  switch (terminalType) {
  case XTERM:
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-FILL-FUNCTION! DISPLAY-FUNCTION)
 */
Define_Primitive(Prim_Graphics_Set_Fill_Function, 1, "GRAPHICS-SET-FILL-FUNCTION!")
{
  int function;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&function) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();
  if (CheckDrawFunction(function) < 0)
    return NIL;
  currentWindowInfo->fillFunction = function;
  
  switch (terminalType) {
  case XTERM:
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-DRAW-FUNCTION! DISPLAY_FUNCTION)
 */
Define_Primitive(Prim_Graphics_Set_Draw_Function, 1, "GRAPHICS-SET-DRAW-FUNCTION!")
{
  int function;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&function) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();
  if (CheckDrawFunction(function) < 0)
    return NIL;
  currentWindowInfo->pnFunction = function;
  
  switch (terminalType) {
  case XTERM:
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-POLY-OPERATION FRAME-FILL-MODE)
 * set flag for filling and / or framing polygons
 */
Define_Primitive(Prim_Graphics_Set_Poly_Operation, 1, "GRAPHICS-SET-POLY-OPERATION!")
{
  int polyOp;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&polyOp) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();

  if (polyOp != PolyFill && polyOp != PolyFrame && polyOp != PolyFrameFill)
    return NIL;
  currentWindowInfo->polyOp = polyOp;
  
  switch (terminalType) {
  case XTERM:
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-DRAW-PATTERN! PATTERN)
   Used to change the style of the line to dashes or dots and 
   dashes or whatever.  Simulates penplot LINE procedure by generating a
   pattern and saving it in the current window.
*/
Define_Primitive(Prim_Graphics_Set_Line, 1, "GRAPHICS-SET-DRAW-PATTERN!")
{
  int linePattern;
  Primitive_1_Arg();

/* check and fetch args */
  if (ToInteger(Arg1,&linePattern) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();

/* for x generate line type */
  switch (terminalType) {
  case XTERM:
    linePattern = CanonicalLinePattern(linePattern);
    currentWindowInfo->pnPat = XMakePattern(linePattern,16,2);
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-COLOR! PIXEL-ID)
 * set the draw and fill color as passed
 */
Define_Primitive(Prim_Graphics_Set_Color, 1, "GRAPHICS-SET-COLOR!")
{
  int pixel;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&pixel) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();

  currentWindowInfo->pnColor = pixel;
  switch (terminalType) {
  case XTERM:
    if (DisplayCells() > 2) {
      currentWindowInfo->fillPat = XMakeTile(pixel);
    }
    else {
      if (pixel == BlackPixel)
	currentWindowInfo->fillPat = BlackPixmap;
      else currentWindowInfo->fillPat = WhitePixmap;
    }
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-DRAW-COLOR PIXEL-ID)
 */
Define_Primitive(Prim_Graphics_Set_Draw_Color, 1, "GRAPHICS-SET-DRAW-COLOR!")
{
  int pixel;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&pixel) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();

  currentWindowInfo->pnColor = pixel;
  switch (terminalType) {
  case XTERM:
    return TRUTH;
  default:
    return NIL;
  }
}
/* (GRAPHICS-SET-FILL-COLOR PIXEL-ID)
 */
Define_Primitive(Prim_Graphics_Set_Fill_Color, 1, "GRAPHICS-SET-FILL-COLOR!")
{
  int pixel;
  Primitive_1_Arg();

  if (ToInteger(Arg1,&pixel) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();

  currentWindowInfo->fillPat = pixel;
  switch (terminalType) {
  case XTERM:
    return TRUTH;
  default:
    return NIL;
  }
}

/* set function for graphics operations, generic and specific */

/* (GRAPHICS-SET-PEN! PEN)
 * Select pen.  pen color set from passed pixel-id.
 * Uses the penplot PEN routine.  On a real vt125, the pen are just different
 * intensities, and on a vs100, this has no effect.
 */
Define_Primitive(Prim_Graphics_Set_Pen, 1, "GRAPHICS-SET-PEN!")
{
  long npen;
  Primitive_1_Arg();

/* fetch and check args */
  if (ToInteger(Arg1,&npen) == NIL)
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  CheckGraphicsStatus();
  npen = Get_Integer(Arg1);

/* set pen pattern */
  switch (terminalType) {
  case XTERM:
    switch(npen) {
    case -1:
      currentWindowInfo->pnColor = currentWindowInfo->fgColor;
      break;
    case 0:
      currentWindowInfo->pnColor = currentWindowInfo->bgColor;
      break;
    default:
      currentWindowInfo->pnColor = npen;
      break;
    }
    break;
  default:
    return NIL;
  }
  return TRUTH;
}

/* (GRAPHICS-DEFINE-PEN (PEN-NUMBER PEN-HUE PEN-INTENSITY
			 PEN-SATURATION))
   This procedure will change the characteristics of any of the four
   pens.  Will require four arguments, the pen number, the color as
   specified on a color wheel, the percent light intensity, and the
   percent saturation.
   For X this is sensitive only to the first two arguements. the second
   is taken to be a pixel id.
*/
Define_Primitive(Prim_Graphics_Define_Pen, 1, "GRAPHICS-DEFINE-PEN")
{ int args[4], i = 0;
  Pointer temp;
  Primitive_1_Arg();

/* extract arguments from list */
   while (Type_Code(Arg1) == TC_LIST && i < 4) {
     temp = Vector_Ref(Arg1, CONS_CAR);
     if (Type_Code(temp) != TC_FIXNUM)
       Primitive_Error(ERR_ARG_1_WRONG_TYPE);
     args[i++] = (int) Get_Integer(temp);
     Arg1 = Vector_Ref(Arg1, CONS_CDR);
   }
  if (i != 4)
    Primitive_Error(ERR_WRONG_NUMBER_OF_ARGUMENTS); 
  CheckGraphicsStatus();

  currentWindowInfo->pnColor = ((args[1] != 0) ? args[1] :
				currentWindowInfo->fgColor);
  switch (terminalType) {
  case XTERM:
    args[0] = CanonicalLinePattern(args[0]);
    currentWindowInfo->pnPat = XMakePattern(args[0],16,2);
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-SET-CURSOR! CURSOR-ID)
 * modifies the cursor bound to the window
 */
Define_Primitive(Prim_Graphics_Set_Cursor,1,"GRAPHICS-SET-CURSOR!")
{
  Cursor cursor;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (Arg1 == NIL)
      cursor = 0;
    if (ToInteger(Arg1,&cursor) == NIL)
      Primitive_Error(ERR_ARG_1_WRONG_TYPE);
    if (cursor)
      XDefineCursor(currentWindow,cursor);
    else XDefineCursor(currentWindow,nullCursor);
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-GET-DRAW-PATTERN PATTERN-NAME)
 * return index for named pattern
 */
Define_Primitive(Prim_Graphics_Get_Draw_Pattern,1,"GRAPHICS-GET-DRAW-PATTERN")
{
  char patternName[32];
  int nameLength;
  char *ToLower();
  Primitive_1_Arg();

  /* no status check */
  if (Type_Code(Arg1) != TC_UNINTERNED_SYMBOL &&
      Type_Code(Arg1) != TC_CHARACTER_STRING &&
      Type_Code(Arg1) != TC_INTERNED_SYMBOL) 
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  switch (Type_Code(Arg1)) {
  case TC_UNINTERNED_SYMBOL:
  case TC_INTERNED_SYMBOL:
    Arg1 = GetStringValue(Arg1);
  case TC_CHARACTER_STRING:
    strncpy(patternName,Scheme_String_To_C_String(Arg1),32);
    ToLower(patternName);
    nameLength = strlen(patternName);
    if (strncmp(patternName,"solid-line",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,0));
    if (strncmp(patternName,"widely-spaced-dots",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,1));
    if (strncmp(patternName,"wide-spaced-dots",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,1));
    if (strncmp(patternName,"long-dashes",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,2));
    if (strncmp(patternName,"short-dash-two-dots",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,3));
    if (strncmp(patternName,"medium-dash-short-dash",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,4));
    if (strncmp(patternName,"long-dash-two-dots",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,5));
    if (strncmp(patternName,"medium-dashes",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,6));
    if (strncmp(patternName,"short-dashes",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,7));
    if (strncmp(patternName,"short-dash-one-dot",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,8));
    if (strncmp(patternName,"closely-spaced-dots",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,9));
    if (strncmp(patternName,"close-spaced-dots",nameLength) == 0)
      return(Make_Non_Pointer(TC_FIXNUM,9));
    return(Make_Non_Pointer(TC_FIXNUM,0));
  }
}
/* (GRAPHICS-GET-DRAW-COLOR [ (RED-VALUE GREEN-VALUE BLUE-VALUE) |
 *  				COLOR-NAME ] )
 * For X generate and return a pixel value for given list of color values
 * or the given name.
 * For Penplot these generate color id from names "red", "green", "blue".
 * There are four pen numbers.  Pen 0 is the background.
 * Pen 1 is blue, pen 2 is red, and pen 3 is green.  Pen 3
 * is also the pen used for normal input to the terminal. 
 * For X, library routines resolve  color name. in addition, 0 is
 * "bgColor", -1 is "fgColor".
 *
 * COLOR-NAME may be a string or it may be a symbol. 
 */
Define_Primitive(Prim_Graphics_Get_Draw_Color,1,"GRAPHICS-GET-DRAW-COLOR")
{
  int pixel;
  char colorName[16];
  Pointer GetStringValue();
  Pointer temp;
  int args[3];
  int i = 0;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  if (Type_Code(Arg1) != TC_LIST &&
      Type_Code(Arg1) != TC_UNINTERNED_SYMBOL &&
      Type_Code(Arg1) != TC_CHARACTER_STRING &&
      Type_Code(Arg1) != TC_INTERNED_SYMBOL) 
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  switch (Type_Code(Arg1)) {
  case TC_LIST:
    while (Type_Code(Arg1) == TC_LIST) {
      temp = Vector_Ref(Arg1, CONS_CAR);
      if (Type_Code(temp) == TC_FIXNUM) {
	args[i++] = Get_Integer(temp);
      }
      else Primitive_Error(ERR_ARG_1_WRONG_TYPE);
      Arg1 = Vector_Ref(Arg1, CONS_CDR);
    }
    if (i != 3)
      Primitive_Error(ERR_WRONG_NUMBER_OF_ARGUMENTS);
    pixel = GetRGBColor(args[0], args[1], args[2]);
    break;
  case TC_UNINTERNED_SYMBOL:
  case TC_INTERNED_SYMBOL:
    Arg1 = GetStringValue(Arg1);
  case TC_CHARACTER_STRING:
    strncpy(colorName,Scheme_String_To_C_String(Arg1),16);
    pixel = GetNamedColor(colorName);
  }
  return ((pixel == NIL) ? NIL : C_Integer_To_Scheme_Integer(pixel));
}

/* (GRAPHICS-GET-FILL-COLOR [ (RED-VALUE GREEN-VALUE BLUE-VALUE) |
   				COLOR-NAME ] )
 * For Penplot these are nop's
 * For X generate and return a tile value for given list of color values
 * or the given name.
 * For Penplot these generate color id from names "red", "green", "blue".
 * For X, library routines resolve  color name. 
 * COLOR-NAME may be a string or it may be a symbol. 
 */
Define_Primitive(Prim_Graphics_Get_Fill_Color,1,"GRAPHICS-GET-FILL-COLOR")
{
  int pixel;
  char colorName[16];
  Pointer GetStringValue();
  Pointer temp;
  int args[3];
  int i = 0;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  if (Type_Code(Arg1) != TC_LIST &&
      Type_Code(Arg1) != TC_UNINTERNED_SYMBOL &&
      Type_Code(Arg1) != TC_CHARACTER_STRING &&
      Type_Code(Arg1) != TC_INTERNED_SYMBOL) 
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  switch (Type_Code(Arg1)) {
  case TC_LIST:
    while (Type_Code(Arg1) == TC_LIST) {
      temp = Vector_Ref(Arg1, CONS_CAR);
      if (Type_Code(temp) == TC_FIXNUM) {
	args[i++] = Get_Integer(temp);
      }
      else Primitive_Error(ERR_ARG_1_WRONG_TYPE);
      Arg1 = Vector_Ref(Arg1, CONS_CDR);
    }
    if (i != 3)
      Primitive_Error(ERR_WRONG_NUMBER_OF_ARGUMENTS);
    pixel = GetRGBTile(args[0],args[1],args[2]);
    break;
  case TC_UNINTERNED_SYMBOL:
  case TC_INTERNED_SYMBOL:
    Arg1 = GetStringValue(Arg1);
  case TC_CHARACTER_STRING:
    strncpy(colorName,Scheme_String_To_C_String(Arg1),16);
    pixel = GetNamedTile(colorName);
  }
  return ((pixel == NIL) ? NIL : C_Integer_To_Scheme_Integer(pixel));
}

/* (GRAPHICS-GET-CURSOR [ (BITS-0 ... BITS-16) |COLOR-NAME ] MODE )
 * For X generate and return a cursor for given list bits or the given name.
 * COLOR-NAME name like "NULL", "ARROW", or "CROSS" for internally
 * defined cursors, or any name which appears in "/use/include/X/cursors". 
 */
Define_Primitive(Prim_Graphics_Get_Cursor,6,"GRAPHICS-GET-CURSOR")
{

  char cursorName[96];
  char fileName[96];
  Pointer GetStringValue();
  Pointer temp;
  short *bitmapbits;
  int func;
  int x, y, w, h;
  int targ;
  int hwc;			/* half word count */
  int xhot, yhot;
  int i = 0;
  Cursor cursor;
  Pointer Arg4;
  Pointer Arg5;
  Pointer Arg6;
  Primitive_3_Args();
  Arg4 = Stack_Pointer[3];
  Arg5 = Stack_Pointer[4];
  Arg6 = Stack_Pointer[5];
  
  CheckGraphicsStatus();
  if (Arg2 == NIL)
    func = GXcopy;
  else if (ToInteger(Arg2,&func) == NIL)
    Primitive_Error(ERR_ARG_2_WRONG_TYPE);
  if (Arg3 == NIL)
    w = 16;
  else if (ToInteger(Arg3,&w) == NIL)
    Primitive_Error(ERR_ARG_3_WRONG_TYPE);
  if (Arg4 == NIL)
    h = 16;
  else if (ToInteger(Arg4,&h) == NIL)
    Primitive_Error(ERR_ARG_4_WRONG_TYPE);
  if (Arg5 == NIL)
    x = 0;
  else if (ToInteger(Arg5,&x) == NIL)
    Primitive_Error(ERR_ARG_5_WRONG_TYPE);
  if (Arg6 == NIL)
    y = 0;
  else if (ToInteger(Arg6,&y) == NIL)
    Primitive_Error(ERR_ARG_6_WRONG_TYPE);
  if (Type_Code(Arg1) != TC_LIST &&
      Type_Code(Arg1) != TC_UNINTERNED_SYMBOL &&
      Type_Code(Arg1) != TC_CHARACTER_STRING &&
      Type_Code(Arg1) != TC_INTERNED_SYMBOL) 
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  switch (Type_Code(Arg1)) {
  case TC_LIST:
    hwc = BitmapSize(w,h) / sizeof(short);
    bitmapbits =(short *) calloc(hwc,sizeof(short));
    while (Type_Code(Arg1) == TC_LIST) {
      temp = Vector_Ref(Arg1, CONS_CAR);
      if (ToInteger(temp,&targ) == NIL)
	Primitive_Error(ERR_ARG_1_WRONG_TYPE);
      bitmapbits[i++] = targ;
      Arg1 = Vector_Ref(Arg1, CONS_CDR);
    }
    if (i != hwc)
      Primitive_Error(ERR_WRONG_NUMBER_OF_ARGUMENTS);
    break;
  case TC_UNINTERNED_SYMBOL:
  case TC_INTERNED_SYMBOL:
    Arg1 = GetStringValue(Arg1);
  case TC_CHARACTER_STRING:
    strncpy(cursorName,Scheme_String_To_C_String(Arg1),96);
    ToLower(cursorName);
    if (strcmp(cursorName,"null") == 0)
      return C_Integer_To_Scheme_Integer(nullCursor);
    else if (strcmp(cursorName,"arrow") == 0)
      return C_Integer_To_Scheme_Integer(arrowCursor);
    else if (strcmp(cursorName,"cross") == 0)
      return C_Integer_To_Scheme_Integer(crossCursor);
    if (cursorName[0] != '/' && cursorName[0] != '.')
      strcpy(fileName,"/usr/include/X/cursors/");
    else fileName[0] = '\0';
    strncat(fileName,cursorName,96-strlen(cursorName));
    fileName[95] = '\0';
    if (XReadBitmapFile(fileName,&w,&h,&bitmapbits,&xhot,&yhot) <= 0)
      return NIL;
    if (xhot >= 0) x = xhot;
    if (yhot >= 0) y = yhot;
  }

  cursor = XCreateCursor(w,h,bitmapbits,bitmapbits,x,y,
			 currentWindowInfo->fgColor,currentWindowInfo->bgColor,
			 func);
  free(bitmapbits);
  return ((cursor == 0) ? NIL : C_Integer_To_Scheme_Integer(cursor));
}

/* (GRAPHICS-GET-POSITION)
      Returns in cons cell the current pen position on the graphics
      screen.  Uses penlot WHERE which returns two floating point
      numbers.
*/
Define_Primitive(Prim_Graphics_Get_Position, 0, "GRAPHICS-GET-POSITION")
{ float pos1, pos2;
  Pointer *Cons_Cell;
  double floor();

/* retrieve and check args */
  CheckGraphicsStatus();

/* make storage */
  Cons_Cell = Free;
  Primitive_GC_If_Needed(2);
  Free += 2;

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    pos1 = currentWindowInfo->pnLoc.x;
    pos2 = currentWindowInfo->pnLoc.y;
    break;
  default:
    Free -= 2;
    return NIL;
  }
  Store_Reduced_Flonum_Result((double) pos1, Cons_Cell[CONS_CAR]);
  Store_Reduced_Flonum_Result((double) pos2, Cons_Cell[CONS_CDR]);
  return Make_Pointer(TC_LIST, Cons_Cell); 
}

/* (GRAPHICS-GET-WINDOW)
      Returns id of current window
*/
Define_Primitive(Prim_Graphics_Get_Window, 0, "GRAPHICS-GET-WINDOW")
{
  Primitive_0_Args();

  CheckGraphicsStatus();

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    return C_Integer_To_Scheme_Integer(currentWindow);
  default:
    return NIL;
  }
}

/* (GRAPHICS-GET-WINDOW-SIZE WINDOW-ID)
      Returns in cons cell the size of the current graphics
      screen. defaults to current window for NIL, rootwindow for 0.
*/
Define_Primitive(Prim_Graphics_Get_Window_Size, 1, "GRAPHICS-GET-WINDOW-SIZE")
{ float pos1, pos2;
  Pointer *Cons_Cell;
  double floor();
  Window window;
  WindowInfo tmpInfo;
  Primitive_1_Arg();

/* retrieve and check args */
  CheckGraphicsStatus();

/* make storage */
  Cons_Cell = Free;
  Primitive_GC_If_Needed(2);
  Free += 2;

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL) {
      XQueryWindow(currentWindow,&(currentWindowInfo->info));
      pos1 =(double) currentWindowInfo->info.width;
      pos2 =(double) currentWindowInfo->info.height;
    }
    else {
      if (window == 0)
	window = RootWindow;
      XQueryWindow(window,&(tmpInfo));
      pos1 =(double) tmpInfo.width;
      pos2 =(double) tmpInfo.height;
    }
    break;
  default:
    Free -= 2;
    return NIL;
  }
  Store_Reduced_Flonum_Result((double) pos1, Cons_Cell[CONS_CAR]);
  Store_Reduced_Flonum_Result((double) pos2, Cons_Cell[CONS_CDR]);
  return Make_Pointer(TC_LIST, Cons_Cell); 
}

/* (GRAPHICS-GET-WINDOW-POSITION WINDOW-ID)
      Returns in cons cell the position of the current graphics
      screen. defaults to current window for NIL, rootwindow for 0.
*/
Define_Primitive(Prim_Graphics_Get_Window_Position, 1, "GRAPHICS-GET-WINDOW-POSITION")
{ float pos1, pos2;
  Pointer *Cons_Cell;
  double floor();
  Window window;
  WindowInfo tmpInfo;
  Primitive_1_Arg();

/* retrieve and check args */
  CheckGraphicsStatus();

/* make storage */
  Cons_Cell = Free;
  Primitive_GC_If_Needed(2);
  Free += 2;

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL) {
      XQueryWindow(currentWindow,&(currentWindowInfo->info));
      pos1 =(double) currentWindowInfo->info.x;
      pos2 =(double) currentWindowInfo->info.y;
    }
    else {
      if (window == 0)
	window = RootWindow;
      XQueryWindow(window,&(tmpInfo));
      pos1 =(double) tmpInfo.x;
      pos2 =(double) tmpInfo.y;
    }
    break;
  default:
    Free -= 2;
    return NIL;
  }
  Store_Reduced_Flonum_Result((double) pos1, Cons_Cell[CONS_CAR]);
  Store_Reduced_Flonum_Result((double) pos2, Cons_Cell[CONS_CDR]);
  return Make_Pointer(TC_LIST, Cons_Cell); 
}

/* (GRAPHICS-GET-MOUSE-POSITION WINDOW)
      Returns in list the current mouse position on the graphics
      screen, the button status, and the highest level window.
      For Penplot, uses penlot WHERE which returns two floating point numbers.
      For X, uses XQueryMouseButtons.
*/
Define_Primitive(Prim_Graphics_Get_Mouse_Position, 1, "GRAPHICS-GET-MOUSE-POSITION")
{ float pos1, pos2;
  Pointer *Cons_Cell;
  double floor();
  int x, y;
  Window subw, window;
  short state;
  Primitive_1_Arg();

/* retrieve and check args */
  CheckGraphicsStatus();

/* make storage */
  Cons_Cell = Free;
  Primitive_GC_If_Needed(2);
  Free += 2;

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = currentWindow;
    if (window == 0)
      window = RootWindow;
    XQueryMouseButtons(window,&x,&y,&subw,&state);
    if (window == RootWindow && subw != 0) {
      window = subw;
      XInterpretLocator(window,&x,&y,&subw,(x<<16)|y);
    }
    DC2W(currentWindowInfo,x,y,&pos1,&pos2);
    break;
  default:
    Free -= 2;
    return NIL;
  }
  Store_Reduced_Flonum_Result((double) pos1, Cons_Cell[CONS_CAR]);
  Store_Reduced_Flonum_Result((double) pos2, Cons_Cell[CONS_CDR]);
  return Make_Pointer(TC_LIST, Cons_Cell); 
}

/* (GRAPHICS-GET-MOUSE-BUTTONS WINDOW)
 */
Define_Primitive(Prim_Graphics_Get_Mouse_Buttons, 1, "GRAPHICS-GET-MOUSE-BUTTONS")
{
  int x, y;
  Window subw, window;
  short state;
  Primitive_1_Arg();

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = RootWindow;
    XQueryMouseButtons(window,&x,&y,&subw,&state);
    state = ButtonState(state);
    return Make_Non_Pointer(TC_FIXNUM,state);
  default:
    return NIL;
  }
}

/* (GRAPHICS-GET-MOUSE-WINDOW WINDOW)
 * determines the window in which the mouse is located.
 * if WINDOW is not specified, it returns a subwindow or NIL
 * if window not specified then the top window will be returned
 */

Define_Primitive(Prim_Graphics_Get_Mouse_Window, 1, "GRAPHICS-GET-MOUSE-WINDOW")
{
  int x, y;
  Window subw, window;
  short state;
  Primitive_1_Arg();

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = currentWindow;
    if (window == 0)
      window = RootWindow;
    XQueryMouseButtons(window,&x,&y,&subw,&state);
    if (subw != 0)
      window = subw;
    return C_Integer_To_Scheme_Integer(window);
  default:
    return NIL;
  }
}

/* (GRAPHICS-GET-EVENT WINDOW-ID TYPE BLOCK)
 * returns event type. BLOCK causes a call which would block 
 * to return TRUTH instead
 */
Define_Primitive(Prim_Graphics_Get_Event,3,"GRAPHICS-GET-EVENT")
{
  Window window;
  long type;
  Pointer x;
  Pointer y;

  Primitive_3_Args();

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (Arg1 == NIL)
      window = currentWindow;
    else {
      if (ToInteger(Arg1,&window) == NIL)
	Primitive_Error(ERR_ARG_1_WRONG_TYPE);
      if (window == 0)
	window = RootWindow;
    }
    if (Arg2 == NIL)
      type = ~NoEvent;
    else if (ToInteger(Arg2,&type) == NIL)
      Primitive_Error(ERR_ARG_2_WRONG_TYPE);
    if (Arg3 == NIL) {
      if (XPending() == 0)
	return(TRUTH);
    }

    switch (window) {
    case -1:
      XMaskEvent(type,&currentEvent);
      break;
    default:
      XWindowEvent(window,type,&currentEvent);
      break;
    }
  }

  return C_Integer_To_Scheme_Integer(currentEvent.type);
}

/* (GRAPHICS-GET-EVENT-DETAIL FIELD)
 * determines the desired detail from the current event
 * the field is identified as:
 *  'T' type,
 *  'W' window,
 *  'x', 'y' locations,
 *  's' subwindow,
 *  'L' locator
 *  'w', 'h' exposure sizes
 *  'k' key
 *  'b' buttons
 * if the field is not present for the current type, nil is returned
 */
Define_Primitive(Prim_Graphics_Get_Event_Detail, 1, "GRAPHICS-GET-EVENT-DETAIL")
{
  long type;
  long field;
  int x, y;
  char *cs;
  Primitive_1_Arg();

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&field) == NIL)
      Primitive_Error(ERR_ARG_2_WRONG_TYPE);
    switch (field) {
    case 0:
    case 'T':
      return C_Integer_To_Scheme_Integer(currentEvent.type);
      break;
    case 1:
    case 'W':
      return C_Integer_To_Scheme_Integer(currentEvent.window);
      break;
    case 'x':
    case 'X':
      switch (currentEvent.type) {
      case ExposeWindow:
      case ExposeRegion:
	x = ((XExposeEvent *)&currentEvent)->x;
	break;
      case ExposeCopy:
      case UnmapWindow:
      case FocusChange:
	return NIL;
	break;
      default:
	x = ((XKeyEvent *)&currentEvent)->x;
	break;
      }
      return C_Integer_To_Scheme_Integer(x);
    case 'y':
    case 'Y':
      switch (currentEvent.type) {
      case ExposeWindow:
      case ExposeRegion:
	y = ((XExposeEvent *)&currentEvent)->y;
	break;
      case ExposeCopy:
      case UnmapWindow:
      case FocusChange:
	return NIL;
	break;
      default:
	y = ((XKeyEvent *)&currentEvent)->y;
	break;
      }
      return C_Integer_To_Scheme_Integer(y);
    case 's':
    case 'S':
      return C_Integer_To_Scheme_Integer(currentEvent.subwindow);
      break;
    case 'L':
      switch (currentEvent.type) {
      case KeyPressed:
      case KeyReleased:
      case ButtonPressed:
      case ButtonReleased:
      case EnterWindow:
      case LeaveWindow:
	return C_Integer_To_Scheme_Integer(currentEvent.pad_l4);
	break;
      default:
	return NIL;
	break;
      }
    case 'w':
      switch (currentEvent.type) {
      case ExposeWindow:
      case ExposeRegion:
	x = ((XExposeEvent *)&currentEvent)->width;
	break;
      default:
	return NIL;
	break;
      }
      return C_Integer_To_Scheme_Integer(x);
    case 'h':
      switch (currentEvent.type) {
      case ExposeWindow:
      case ExposeRegion:
	y = ((XExposeEvent *)&currentEvent)->height;
	break;
      default:
	return NIL;
	break;
      }
      return C_Integer_To_Scheme_Integer(y);
    case 'k':
      switch (currentEvent.type) {
      case KeyPressed:
      case KeyReleased:
	cs =(char *) XLookupMapping(&currentEvent,&y);
	if (y <= 0)
	  return NIL;
	x = *cs;
	return C_Integer_To_Scheme_Integer(x);
      default:
	return NIL;
      }
      break;
    case 'b':
      switch (currentEvent.type) {
      case ButtonPressed:
      case ButtonReleased:
	x = ((XButtonEvent *)&currentEvent)->detail;
	x = (x >> 8) & 0x3;
	return C_Integer_To_Scheme_Integer(x);
      default:
	return NIL;
      }
      break;
    default:
      return NIL;
    }
    break;
  default:
    return NIL;
  }
}

/* (GRAPHICS-GET-EVENT-POSITION)
      Returns in list the current mouse position on the graphics
      screen, the button status, and the highest level window.
      For Penplot, uses penlot WHERE which returns two floating point numbers.
      For X, uses QueryMouseButtons.
*/
Define_Primitive(Prim_Graphics_Get_Event_Position, 0, "GRAPHICS-GET-EVENT-POSITION")
{ float pos1, pos2;
  Pointer *Cons_Cell;
  double floor();
  int x, y;

/* retrieve and check args */
  CheckGraphicsStatus();

/* make storage */
  Cons_Cell = Free;
  Primitive_GC_If_Needed(2);
  Free += 2;

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    x = ((XKeyOrButtonEvent *)&currentEvent)->x;
    y = ((XKeyOrButtonEvent *)&currentEvent)->y;
    DC2W(currentWindowInfo,x,y,&pos1,&pos2);
    break;
  default:
    Free -= 2;
    return NIL;
  }
  Store_Reduced_Flonum_Result((double) pos1, Cons_Cell[CONS_CAR]);
  Store_Reduced_Flonum_Result((double) pos2, Cons_Cell[CONS_CDR]);
  return Make_Pointer(TC_LIST, Cons_Cell); 
}
/* (GRAPHICS-GET-EVENT-WINDOW)
 * returns window id of last event
 */
Define_Primitive(Prim_Graphics_Get_Event_Window, 0, "GRAPHICS-GET-EVENT-WINDOW")
{
  int x, y;

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    return C_Integer_To_Scheme_Integer(currentEvent.window);
  default:
    return NIL;
  }
}

/* (GRAPHICS-PUT-EVENT WINDOW-ID TYPE ARGS)
 * dispatches a toolkit event
 */
Define_Primitive(Prim_Graphics_Put_Event,3,"GRAPHICS-PUT-EVENT")
{
  Window window;
  long type;
  XEvent nextEvent;

  Primitive_3_Args();

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      Primitive_Error(ERR_ARG_1_WRONG_TYPE);
    if (ToInteger(Arg2,&type) == NIL)
      Primitive_Error(ERR_ARG_2_WRONG_TYPE);
    return(TRUTH);
  default:
    return NIL;
  }
}

/* (GRAPHICS-SELECT-EVENT WINDOW-ID TYPE)
 * sets desired events
 */
Define_Primitive(Prim_Graphics_Select_Event,2,"GRAPHICS-SELECT-EVENT")
{
  long type;
  Window window;
  Primitive_2_Args();

/* retrieve and check args */
  CheckGraphicsStatus();

/* set value */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (ToInteger(Arg1,&window) == NIL)
      window = currentWindow;
    if (window == 0)
      window = RootWindow;
    if (ToInteger(Arg2,&type) == NIL)
      type = ~NoEvent;
    XSelectInput(window,type);
    return TRUTH;
    break;
  default:
    return NIL;
  }  
}

/* (GRAPHICS-GET-FUNCTION FUNCTION-NAME)
 * return function identifier for the given function
 */
Define_Primitive(Prim_Graphics_Get_Function,1,"GRAPHICS-GET-FUNCTION")
{
 char displayFunction[32];
 int function;
 int GetDisplayFunction();
 Primitive_1_Arg();

/* no status check */
  if (Type_Code(Arg1) != TC_UNINTERNED_SYMBOL &&
      Type_Code(Arg1) != TC_CHARACTER_STRING &&
      Type_Code(Arg1) != TC_INTERNED_SYMBOL) 
    Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  switch (Type_Code(Arg1)) {
  case TC_UNINTERNED_SYMBOL:
  case TC_INTERNED_SYMBOL:
    Arg1 = GetStringValue(Arg1);
  case TC_CHARACTER_STRING:
    strncpy(displayFunction,Scheme_String_To_C_String(Arg1),32);
    function = GetDisplayFunction(displayFunction);
    break;
  }
  return ((function == -1) ? NIL : C_Integer_To_Scheme_Integer(function));
}

/* (GRAPHICS-MOVE ARG1 ARG2)
   move to the position indicated by ARG1 and ARG2.  Both must be Scheme
   FIXNUMs or FLONUMs. uses move_ or
   retrieves description of current window and sets pen location.
*/
Define_Primitive(Prim_Graphics_Move, 2, "GRAPHICS-MOVE")
{
  float pos1, pos2;
  int Error_Number;
  WorldPoint *SetWorldPoint();
  Primitive_2_Args();

/* check args */
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1, &pos2, Arg1, Arg2);
  If_Error(Error_Number);

/* set position in window */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    SetWorldPoint(&(currentWindowInfo->pnLoc),pos1,pos2);
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-MOVE-RELATIVE ARG1 ARG2)
   move by displacement indicated by ARG1 and ARG2.  Both must be Scheme
   FIXNUMs or FLONUMs. uses move_ or retrieves description of current window
   and sets pen location.
*/
Define_Primitive(Prim_Graphics_Move_Relative, 2, "GRAPHICS-MOVE-RELATIVE")
{
  float pos1, pos2;
  float pos1a, pos2a;
  int Error_Number;
  WorldPoint *SetWorldPoint();
  Primitive_2_Args();

/* check args */
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1, &pos2, Arg1, Arg2);
  If_Error(Error_Number);

/* set position in window */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    SetWorldPoint(&(currentWindowInfo->pnLoc),
		  currentWindowInfo->pnLoc.x + pos1,
		  currentWindowInfo->pnLoc.y + pos2);
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-DRAW ARG1 ARG2 ARG3)
   sets pen-down and uses current line type to draw to position indicated
   by ARG1 and ARG2.
   For Penplot ARG3 is either 0 or 1 which indicates whether the line
   should be drawn in the current foreground color or background color,
   respectively.
   For X, ARG3 is pnFunction. ConditionalXFlush to force buffer to be drawn, so that each line
   is drawn when wanted.
*/
Define_Primitive(Prim_Graphics_Draw, 3, "GRAPHICS-DRAW")
{
  float pos1, pos2;
  int Error_Number;
  int pnMode;
  Vertex vlist[2];

/* check args */
  Primitive_3_Args();
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1, &pos2, Arg1, Arg2);
  If_Error(Error_Number);

/* determine color, draw line, update pen location */
  if (Type_Code(Arg3) == TC_FIXNUM)
    pnMode = (int) Get_Integer(Arg3);
  else pnMode = currentWindowInfo->pnFunction;

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    W2DC(currentWindowInfo,
	 currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,
	 &(vlist[0].x),&(vlist[0].y));
    W2DC(currentWindowInfo,pos1,pos2,
	 &(vlist[1].x),&(vlist[1].y));
    vlist[0].flags = vlist[1].flags = 0;
    XDrawPatterned(currentWindow,&(vlist[0]),2,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		   currentWindowInfo->pnPat,pnMode,AllPlanes);
    SetWorldPoint(&(currentWindowInfo->pnLoc),pos1,pos2);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-DRAW-RELATIVE ARG1 ARG2 ARG3)
   sets pen-down and uses current line type to draw to displacement indicated
   by ARG1 and ARG2. ARG3 is either 0 or 1 which indicates whether the line
   should be drawn in the current foreground color or background color,
   respectively. ConditionalXFlush to force buffer to be drawn, so that each line
   is drawn when wanted.
*/
Define_Primitive(Prim_Graphics_Draw_Relative, 3, "GRAPHICS-DRAW-RELATIVE")
{
  float pos1, pos2;
  float pos1a, pos2a;
  int Error_Number;
  int pnMode;
  Vertex vlist[2];
  Primitive_3_Args();


/* check args */
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1, &pos2, Arg1, Arg2);
  If_Error(Error_Number);

/* determine color, draw line, update pen location */
  if (Type_Code(Arg3) == TC_FIXNUM)
    pnMode = (long) Get_Integer(Arg3);
  else pnMode = currentWindowInfo->pnFunction;

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    W2DC(currentWindowInfo,
	 currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,
	 &(vlist[0].x),&(vlist[0].y));
    pos1 += currentWindowInfo->pnLoc.x;
    pos2 += currentWindowInfo->pnLoc.y;
    W2DC(currentWindowInfo,pos1,pos2,
	 &(vlist[1].x),&(vlist[1].y));
    vlist[0].flags = vlist[1].flags = 0;
    XDrawPatterned(currentWindow,&(vlist[0]),2,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		   currentWindowInfo->pnPat,pnMode,AllPlanes);
    SetWorldPoint(&(currentWindowInfo->pnLoc),pos1,pos2);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-RECTANGLE ARG1 ARG2 ARG3)
   This routine draws a rectangle on the screen at the current point
   to point ARG1, ARG2. the mode is optionally set by ARG3 to foreground or
   background. fill is acording to current mode.
 */
Define_Primitive(Prim_Graphics_Rectangle, 3, "GRAPHICS-RECTANGLE")
{
  float pos1, pos2;
  float pos1a, pos2a;
  int Error_Number;
  int pnMode, fillMode;
  Vertex vlist[5];
  Primitive_3_Args();

/* check args, get current window */
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1,&pos2,Arg1,Arg2);
  If_Error(Error_Number);

/* determine color, draw line, update pen location */
  if (Type_Code(Arg3) == TC_FIXNUM)
    pnMode = fillMode = (long) Get_Integer(Arg3);
  else {
    pnMode = currentWindowInfo->pnFunction;
    fillMode = currentWindowInfo->fillFunction;
  }

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    W2DC(currentWindowInfo,
	 currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,
	 &(vlist[0].x),&(vlist[0].y));
    W2DC(currentWindowInfo,pos1,pos2,
	 &(vlist[2].x),&(vlist[2].y));
    vlist[0].flags = VertexStartClosed;
    vlist[1].flags = vlist[2].flags = vlist[3].flags = 0;
    vlist[4].flags = VertexEndClosed;
    vlist[4].x = vlist[1].x = vlist[0].x;
    vlist[4].y = vlist[3].y = vlist[0].y;
    vlist[1].y = vlist[2].y;
    vlist[3].x = vlist[2].x;
    if (currentWindowInfo->polyOp & PolyFill)
      XDrawTiled(currentWindow,vlist,5,
		  currentWindowInfo->fillPat,fillMode,AllPlanes);
    if (currentWindowInfo->polyOp & PolyFrame)
      XDrawPatterned(currentWindow,vlist,5,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		   currentWindowInfo->pnPat,pnMode,AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-RECTANGLE-RELATIVE ARG1 ARG2 ARG3)
   This routine draws a rectangle on the screen at the current point
   of size ARG1, ARG2. the mode is optionally set by ARG3 to foreground or
   background. fill is acording to current mode.
 */
Define_Primitive(Prim_Graphics_Rectangle_Relative, 3, "GRAPHICS-RECTANGLE-RELATIVE")
{
  float pos1, pos2;
  float pos1a, pos2a;
  int Error_Number;
  int pnMode, fillMode;
  Vertex vlist[5];
  Primitive_3_Args();

/* check args, get current window */
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1,&pos2,Arg1,Arg2);
  If_Error(Error_Number);

/* determine color, draw line, update pen location */
  if (Type_Code(Arg3) == TC_FIXNUM)
    pnMode = fillMode = (long) Get_Integer(Arg3);
  else {
    pnMode = currentWindowInfo->pnFunction;
    fillMode = currentWindowInfo->fillFunction;
  }

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    W2DC(currentWindowInfo,
	 currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,
	 &(vlist[0].x),&(vlist[0].y));
    pos1 += currentWindowInfo->pnLoc.x;
    pos2 += currentWindowInfo->pnLoc.y;
    W2DC(currentWindowInfo,pos1,pos2,
	 &(vlist[2].x),&(vlist[2].y));
    vlist[0].flags = VertexStartClosed;
    vlist[1].flags = vlist[2].flags = vlist[3].flags = 0;
    vlist[4].flags = VertexEndClosed;
    vlist[4].x = vlist[1].x = vlist[0].x;
    vlist[4].y = vlist[3].y = vlist[0].y;
    vlist[1].y = vlist[2].y;
    vlist[3].x = vlist[2].x;
    if (currentWindowInfo->polyOp & PolyFill)
      XDrawTiled(currentWindow,vlist,5,
		  currentWindowInfo->fillPat,fillMode,AllPlanes);
    if (currentWindowInfo->polyOp & PolyFrame)
      XDrawPatterned(currentWindow,vlist,5,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		   currentWindowInfo->pnPat,pnMode,AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-POLY ARG1 ARG2)
   This routine draws a polyline on the screen at the current point
   of size using the point list ARG. the mode is optionally set by ARG3 to
   foreground or background. fill is acording to current mode.
 */
Define_Primitive(Prim_Graphics_Poly, 2, "GRAPHICS-POLY")
{
  float pos1, pos2;
  float pos1a, pos2a;
  int Error_Number;
  int pnMode, fillMode;
  Vertex vlist[256];
  int vertexCount;
  Primitive_2_Args();

/* check args, get current window */
  CheckGraphicsStatus();

/* determine color, draw line, update pen location */
  Arg_1_Type(TC_LIST);
  if (Type_Code(Arg2) == TC_FIXNUM)
    pnMode = fillMode = (long) Get_Integer(Arg2);
  else {
    pnMode = currentWindowInfo->pnFunction;
    fillMode = currentWindowInfo->fillFunction;
  }
/* get poly vector */
  vertexCount = 0;
  while (Type_Code(Arg1) == TC_LIST) {
    Error_Number = Get_Position_From_Point(&pos1,&pos2,
					   Vector_Ref(Arg1, CONS_CAR));
    If_Error(Error_Number);
    W2DC(currentWindowInfo,pos1,pos2,
	 &(vlist[vertexCount].x),&(vlist[vertexCount].y));
    vlist[vertexCount].flags = 0;
    vertexCount ++;
    Arg1 = Vector_Ref(Arg1, CONS_CDR);
  }
  if ((vlist[vertexCount-1].x != vlist[0].x) ||
      (vlist[vertexCount-1].y != vlist[0].y)) {
    vlist[vertexCount].x = vlist[0].x;
    vlist[vertexCount].y = vlist[0].y;
    vlist[vertexCount].flags = 0;
    vertexCount ++;
  }

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (currentWindowInfo->polyOp & PolyFill)
      XDrawTiled(currentWindow,vlist,vertexCount,
		  currentWindowInfo->fillPat,fillMode,AllPlanes);
    if (currentWindowInfo->polyOp & PolyFrame)
      XDrawPatterned(currentWindow,vlist,vertexCount,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		   currentWindowInfo->pnPat,pnMode,AllPlanes);
    SetWorldPoint(&(currentWindowInfo->pnLoc),pos1,pos2);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-POLY-RELATIVE ARG1 ARG2)
   This routine draws a polyline on the screen at the current point
   of size using the point list ARG. the mode is optionally set by ARG3 to
   foreground or background. fill is acording to current mode.
 */
Define_Primitive(Prim_Graphics_Poly_Relative, 2, "GRAPHICS-POLY-RELATIVE")
{
  float pos1, pos2;
  float pos1a, pos2a;
  int Error_Number;
  int pnMode, fillMode;
  Vertex vlist[256];
  int vertexCount;
  Primitive_2_Args();

/* check args, get current window */
  CheckGraphicsStatus();

/* determine color, draw line, update pen location */
  Arg_1_Type(TC_LIST);
  if (Type_Code(Arg2) == TC_FIXNUM)
    pnMode = fillMode = (long) Get_Integer(Arg2);
  else {
    pnMode = currentWindowInfo->pnFunction;
    fillMode = currentWindowInfo->fillFunction;
  }
/* get poly vector */
  vertexCount = 0;
  while (Type_Code(Arg1) == TC_LIST) {
    Error_Number = Get_Position_From_Point(&pos1,&pos2,
					   Vector_Ref(Arg1, CONS_CAR));
    If_Error(Error_Number);
    if (!(vertexCount)) {
      W2DC(currentWindowInfo,pos1,pos2,
	   &(vlist[vertexCount].x),&(vlist[vertexCount].y));
      vlist[vertexCount].flags = 0;
    }
    else {
      W2DC_NoTranslate(currentWindowInfo,pos1,pos2,
	   &(vlist[vertexCount].x),&(vlist[vertexCount].y));
      vlist[vertexCount].flags = VertexRelative;
    }
    vertexCount ++;
    Arg1 = Vector_Ref(Arg1, CONS_CDR);
  }
  if ((vlist[vertexCount-1].x != vlist[0].x) ||
      (vlist[vertexCount-1].y != vlist[0].y)) {
    vlist[vertexCount].x = vlist[0].x;
    vlist[vertexCount].y = vlist[0].y;
    vlist[vertexCount].flags = 0;
    vertexCount ++;
  }

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (currentWindowInfo->polyOp & PolyFill)
      XDrawTiled(currentWindow,vlist,vertexCount,
		  currentWindowInfo->fillPat,fillMode,AllPlanes);
    if (currentWindowInfo->polyOp & PolyFrame)
      XDrawPatterned(currentWindow,vlist,vertexCount,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		   currentWindowInfo->pnPat,pnMode,AllPlanes);
    SetWorldPoint(&(currentWindowInfo->pnLoc),pos1,pos2);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-ARC ARG1 ARG2 ARG3 ARG4)
   This routine draw an arc centered at the current point, of radius ARG1,
   from angle ARG2 to angle ARG3. ARG4 is as with draw.
*/
Define_Primitive(Prim_Graphics_Arc, 4, "GRAPHICS-ARC")
{
  Vertex vlist[8];
  float wv1x, wv2x, wv3x, wv1y, wv2y, wv3y;
  short dcx, dcy;
  float sangle, eangle, mangle;
  float radius;
  long lradius;
  int pnMode;
  int vindex;
  int Error_Number;
  fast Pointer Arg4;
  Primitive_3_Args();
  Arg4 = Stack_Pointer[3];

  CheckGraphicsStatus();
  Error_Number = Get_Position(&sangle,&eangle,Arg2,Arg3);
  If_Error(Error_Number);
  if (Type_Code(Arg1) == TC_FIXNUM) {
    Sign_Extend(Arg1,lradius);
    radius =(float) lradius;
  }
  else if (Type_Code(Arg1) == TC_BIG_FLONUM)
    radius = Get_Float(Arg1);
  else Primitive_Error(ERR_ARG_1_WRONG_TYPE);
  if (Type_Code(Arg4) == TC_FIXNUM)
    pnMode = (int) Get_Integer(Arg4);
  else {
    pnMode = currentWindowInfo->pnFunction;
  }

/* determine start and end angles */
  mangle = (sangle + eangle) / 2.0;
  wv1x =((float) cos(sangle) * radius) + currentWindowInfo->pnLoc.x;
  wv1y =((float) sin(sangle) * radius) + currentWindowInfo->pnLoc.y;
  wv2x =((float) cos(mangle) * radius) + currentWindowInfo->pnLoc.x;
  wv2y =((float) sin(mangle) * radius) + currentWindowInfo->pnLoc.y;
  wv3x =((float) cos(eangle) * radius) + currentWindowInfo->pnLoc.x;
  wv3y =((float) sin(eangle) * radius) + currentWindowInfo->pnLoc.y;

/* generate default circle */
  GetCircle(vlist,currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,
	    radius);
  vlist[0].flags |= VertexDontDraw;
  vlist[1].flags |= VertexDontDraw;
  vlist[2].flags |= VertexDontDraw;
  vlist[3].flags |= VertexDontDraw;
  vlist[4].flags |= VertexDontDraw;
  W2DC(currentWindowInfo,wv1x,wv1y,&dcx,&dcy);
  AddPointCircle(vlist,dcx,dcy);
  W2DC(currentWindowInfo,wv2x,wv2y,&dcx,&dcy);
  AddPointCircle(vlist,dcx,dcy);
  W2DC(currentWindowInfo,wv3x,wv3y,&dcx,&dcy);
  AddPointCircle(vlist,dcx,dcy);
  
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    XDrawPatterned(currentWindow,vlist,8,
		   currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		   currentWindowInfo->pnColor,currentWindowInfo->pnPat,
		   pnMode,AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-CIRCLE ARG1 ARG2)
   This routine draw a circle of radius ARG1 centered at the current point.
   ARG2 is as with draw.
 */
Define_Primitive(Prim_Graphics_Circle, 2, "GRAPHICS-CIRCLE")
{
  float radius;
  long lradius;
  Vertex vlist[5];
  int pnMode;
  int fillMode;

  Primitive_2_Args();
  CheckGraphicsStatus();
  if (Type_Code(Arg1) == TC_FIXNUM) {
    Sign_Extend(Arg1,lradius);
    radius =(float) lradius;
  }
  else if (Type_Code(Arg1) == TC_BIG_FLONUM)
    radius = Get_Float(Arg1);
  else Primitive_Error(ERR_ARG_1_WRONG_TYPE);

  GetCircle(vlist,currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,
	    radius);

  if (Type_Code(Arg2) == TC_FIXNUM)
    fillMode = pnMode = (int) Get_Integer(Arg2);
  else {
    pnMode = currentWindowInfo->pnFunction;
    fillMode = currentWindowInfo->fillFunction;
  }

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (currentWindowInfo->polyOp & PolyFill)
      XDrawTiled(currentWindow,vlist,5,
		  currentWindowInfo->fillPat,fillMode,AllPlanes);
    if (currentWindowInfo->polyOp & PolyFrame)
      XDrawPatterned(currentWindow,vlist,5,
		     currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
		     currentWindowInfo->pnColor,currentWindowInfo->bgColor,
		     currentWindowInfo->pnPat,pnMode,
		     AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-PIXEL ARG1 ARG2 ARG3)
      This routine plots one point on the screen at ARG1, ARG2.  Again,
      ARG1 and ARG2 must FIXNUMs or FLONUMs.  ARG3 is the same as in 
      GRAPHICS_DRAW.
*/
Define_Primitive(Prim_Graphics_Pixel, 3, "GRAPHICS-PIXEL")
{
  float pos1, pos2;
  int Error_Number;
  int pnMode;
  short xloc, yloc;
  Primitive_3_Args();

/* check and fetch args, locate current window description */
  CheckGraphicsStatus();
  Error_Number = Get_Position(&pos1, &pos2, Arg1, Arg2);
  If_Error(Error_Number);

  if (Type_Code(Arg3) == TC_FIXNUM)
    pnMode = (int) Get_Integer(Arg3);
  else {
    pnMode = currentWindowInfo->pnFunction;
  }
  
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    SetWorldPoint(&(currentWindowInfo->pnLoc),pos1,pos2);
    W2DC(currentWindowInfo,pos1,pos2,&xloc,&yloc);
    XLine(currentWindow,xloc,yloc,xloc,yloc,
	  currentWindowInfo->pnSize.x,currentWindowInfo->pnSize.y,
	  currentWindowInfo->pnPat,pnMode,AllPlanes);
    ConditionalXFlush();
    return TRUTH;
  default:
    return NIL;
  }
}

/* (GRAPHICS-LABEL STRING)
      Prints a string label at the current pen position.  The label is
      written according to the current letter type as defined by 
      GRAPHICS_LETTER.
*/
Define_Primitive(Prim_Graphics_Label, 1, "GRAPHICS-LABEL")
{
  char user_label[100];
  int length;
  short x, y;
  int dclength;
  Primitive_1_Arg();

/* get args and window */
  CheckGraphicsStatus();
  Arg_1_Type(TC_CHARACTER_STRING);
  strncpy(user_label,Scheme_String_To_C_String(Arg1),100);
  length = (int) Get_Integer(Vector_Ref(Arg1, STRING_LENGTH));

/* produce text */
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (currentWindowInfo->txFont != 0) {
      W2DC(currentWindowInfo,
	   currentWindowInfo->pnLoc.x,currentWindowInfo->pnLoc.y,&x,&y);
      XText(currentWindow,x,y,user_label,length,currentWindowInfo->txFont->id,
	    currentWindowInfo->fgColor,currentWindowInfo->bgColor);
      dclength = XStringWidth(user_label,currentWindowInfo->txFont,0,0);
      x += dclength;
      DC2W(currentWindowInfo,x,y,
	   &(currentWindowInfo->pnLoc.x),&(currentWindowInfo->pnLoc.y));
      ConditionalXFlush();
    }
    return TRUTH;
  default:
    return NIL;
  }
}
/* (GRAPHICS-SET-LETTER ARG-LIST)
   for vt125:
      This routine will change the way in which letters are drawn for the
   penplot LABEL routine.  The default values are HEIGHT = 3, ASPECT = .7,
   ROTATE = 0, and SLANT = 0.
   for XTERM:
      no effect;
*/
Define_Primitive(Prim_Graphics_Set_Letter, 1, "GRAPHICS-SET-LETTER")
{
  float args[3];
  int i = 0;
  Pointer temp;
  Window window;
  Primitive_1_Arg();

  CheckGraphicsStatus();
  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    Primitive_Error(ERR_UNIMPLEMENTED_PRIMITIVE);
    /*NOTREACHED*/
  default:
    return NIL;
  }
}

/* (GRAPHICS-SCL3)
      One must call this routine to set up for three dimensional graphics.
      After calling this, the user can use GRAPHICS_SPLOT3 to draw in the 
      three dimensional region set up here.
*/
Define_Primitive(Prim_Graphics_Scl3, 2, "GRAPHICS-SCL3")
{ float args[8];
  long ibox;
  int i = 0;
  Pointer temp;
  Primitive_2_Args();

  CheckGraphicsStatus(); 
  Arg_1_Type(TC_LIST);
  Arg_2_Type(TC_FIXNUM);
  while (Type_Code(Arg1) == TC_LIST)
  { temp = Vector_Ref(Arg1, CONS_CAR);
    if (Type_Code(temp) == TC_FIXNUM)
    { long intemp;
      Sign_Extend(temp, intemp);
      args[i++] = (float) intemp;
    }
    else if (Type_Code(temp) == TC_BIG_FLONUM)
      args[i++] = (float) Get_Float(temp);
    else Primitive_Error(ERR_ARG_1_WRONG_TYPE);
    Arg1 = Vector_Ref(Arg1, CONS_CDR);
  }
  if (i != 9)
    Primitive_Error(ERR_WRONG_NUMBER_OF_ARGUMENTS);
  ibox = (long) Get_Integer(Arg2);

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    Primitive_Error(ERR_UNIMPLEMENTED_PRIMITIVE);
    /*NOTREACHED*/
  default:
    return NIL;
  }
}

/* (GRAPHICS-SPLOT3 LIST_OF_COORDS PEN)
      This procedure allows one to draw in a three dimensional region as
      defined by the arguments given to SCL3.
*/
Define_Primitive(Prim_Graphics_Splot3, 2, "GRAPHICS-SPLOT3")
{ float args[2];
  int i = 0;
  long ipen;
  Pointer temp;
  Primitive_2_Args();

  CheckGraphicsStatus();
  Arg_1_Type(TC_LIST);
  Arg_2_Type(TC_FIXNUM);
  while (Type_Code(Arg1) == TC_LIST)
  { temp = Vector_Ref(Arg1, CONS_CAR);
    if (Type_Code(temp) == TC_FIXNUM)
    { long intemp;
      Sign_Extend(temp, intemp);
      args[i++] = (float) intemp;
    }
    else if (Type_Code(temp) == TC_BIG_FLONUM)
      args[i++] = (float) Get_Float(temp);
    else Primitive_Error(ERR_ARG_1_WRONG_TYPE);
    Arg1 = Vector_Ref(Arg1, CONS_CDR);
  }
  if (i != 3)
   Primitive_Error(ERR_WRONG_NUMBER_OF_ARGUMENTS);
  ipen = (long) Get_Integer(Arg2);

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    Primitive_Error(ERR_UNIMPLEMENTED_PRIMITIVE);
    /*NOTREACHED*/
  default:
    return NIL;
  }
}

/************************************************************************
 * utility routines
 */
/* force lower case
 */
char *ToLower(cs)
     char *cs;
{
  char *ts = cs;
  while (*cs) {
    if (isupper(*cs))
      *cs = tolower(*cs);
    cs ++;
  }
  return (ts);
}

/* coerce a symbol to string 
 */
Pointer GetStringValue(arg1)
     Pointer arg1;
{
  if (Type_Code(arg1) == TC_UNINTERNED_SYMBOL ||
      Type_Code(arg1) == TC_INTERNED_SYMBOL)
    arg1 = Fast_Vector_Ref(arg1,SYMBOL_NAME);
  return(arg1);
}
/* force to integer
 */
int ToInteger(arg,value)
     Pointer arg;
     int *value;
{
  if (Type_Code(arg) == TC_FIXNUM)
    *value = Get_Integer(arg);
  else if (Type_Code(arg) == TC_BIG_FIXNUM)
    Scheme_Integer_To_C_Integer(arg,value);
  else return NIL;
  return TRUTH;
}
  
/* extract position from two args and convert to float
 */
Get_Position(ppos1, ppos2, arg1, arg2)
float *ppos1, *ppos2;
fast Pointer arg1, arg2;
{
  long temp;
  Pointer tempPtr;

  if (Type_Code(arg1) == TC_FIXNUM)
  { Sign_Extend(arg1, temp);
    *ppos1 = (float) temp;
  }
  else if (Type_Code(arg1) != TC_BIG_FLONUM) return(1);
  else *ppos1 = (float) Get_Float(arg1);
  if (Type_Code(arg2) == TC_FIXNUM)
  { Sign_Extend(arg2, temp);
    *ppos2 = (float) temp;
  }
  else if (Type_Code(arg2) != TC_BIG_FLONUM) return(2);
  else *ppos2 = (float) Get_Float(arg2);
  return(0);
}

Get_Position_From_Point(ppos1, ppos2, arg1)
     float *ppos1, *ppos2;
     fast Pointer arg1;
{				/* accept coordinates from point objects */
				/* allow a "*point*" string before coords */
				/* and dotted pair for last coord */
  Pointer temp;
  long longtemp;

  if (Type_Code(arg1) != TC_LIST)
    return(1);
  temp = Vector_Ref(arg1, CONS_CAR);
  if (Type_Code(temp) == TC_UNINTERNED_SYMBOL ||
      Type_Code(temp) == TC_INTERNED_SYMBOL ||
      Type_Code(temp) == TC_CHARACTER_STRING) {
    arg1 = Vector_Ref(arg1, CONS_CDR);
    if (Type_Code(arg1) != TC_LIST)
      return(1);
    temp = Vector_Ref(arg1, CONS_CAR);
  }
  if (Type_Code(temp) == TC_FIXNUM) {
    Sign_Extend(temp, longtemp);
    *ppos1 = (float) longtemp;
  }
  else if (Type_Code(temp) != TC_BIG_FLONUM) return(1);
  else *ppos1 = (float) Get_Float(temp);
  arg1 = Vector_Ref(arg1, CONS_CDR);
  if (Type_Code(arg1) != TC_LIST)
    temp = arg1;
  else temp = Vector_Ref(arg1, CONS_CAR);
  if (Type_Code(temp) == TC_FIXNUM)
  { Sign_Extend(temp, longtemp);
    *ppos2 = (float) longtemp;
  }
  else if (Type_Code(temp) != TC_BIG_FLONUM) return(2);
  else *ppos2 = (float) Get_Float(temp);
  return(0);
}
    

W2DC(windowInfo,wx,wy,dx,dy)	/* calculate device coordinates */
     SchemeWindowInfo *windowInfo;
     float wx, wy;
     short *dx, *dy;
{
  *dx =(short) wx;
  *dy =(short) wy;
  return;
}

DC2W(windowInfo,dx,dy,wx,wy)	/* calculate device coordinates */
     SchemeWindowInfo *windowInfo;
     short dx, dy;
     float *wx, *wy;
{
  *wx =(float) dx;
  *wy =(float) dy;
  return;
}

GetTerminalType(displayName)
     char *displayName;
{
  char *term;
  char *getenv();

  if (displayName && displayName[0]) {
    ToLower(displayName);
    if (index(displayName,':') != NULL)
      return(XTERM);
  }
  term = getenv("TERM");	/* dispatch on terminal type */
  if (strncmp(term, "xterm", 5) == 0 ||
      strcmp(term, "vs100") == 0 ||
      strcmp(term, "vx100") == 0)
    return(XTERM);
  return(0);
}

GetDisplayId()
{
  int di;

  for (di = 0; di < MaxActiveDisplays; di ++) {
    if (activeDisplay[di].display == NULL)
      return(di);
  }
  return (-1);
}

Window MakeSchemeWindow(geometry)	/* create new window and description */
     char geometry[];
{
  Window window;
  SchemeWindowInfo *windowInfo;
  DevicePoint *SetDevicePoint();
  WorldPoint *SetWorldPoint();
  OpaqueFrame frame;

  windowInfo =(SchemeWindowInfo *) malloc(sizeof (SchemeWindowInfo));
  frame.border = WhitePixmap;
  frame.background = BlackPixmap;
  frame.bdrwidth = 2;
  window = XCreate("X-Scheme",Saved_argv[0],
		   geometry,"=640x480+0+0",&(frame),40,30);
  windowInfo->device = window;
  SetWorldPoint(&(windowInfo->pnLoc),0.0,0.0);
  SetDevicePoint(&(windowInfo->pnSize),1,1);
  /* this nest would not work with the new x release
   *  windowInfo->pnPat = XMakePattern(((short)0xffff),16,1);
   */
  windowInfo->pnPat = SolidLine;
  windowInfo->pnFunction =
    windowInfo->fillFunction = GXcopy;
  windowInfo->polyOp = PolyFrame;
  windowInfo->fillPat = WhitePixmap;
  if ((windowInfo->txFont = XOpenFont("6x10")) != 0) {
    if (windowInfo->txFont->fixedwidth == 0)
      windowInfo->txFont->widths = XFontWidths("6x10");
    else windowInfo->txFont->widths = NULL;
  }
  windowInfo->pnColor = WhitePixel;
  windowInfo->fgColor = WhitePixel;
  windowInfo->bgColor = BlackPixel;
  XDefineCursor(window,crossCursor);
  XMakeAssoc(schemeXAssocTable,window,windowInfo);
  XMapWindow(window);
  XQueryWindow(window,&(windowInfo->info));
  ConditionalXFlush();
  currentWindow = window;
  currentWindowInfo = windowInfo;
  return(window);
}

/* X error handler
 */
SchemeXError(display, errorEvent)
     Display *display;
     XErrorEvent *errorEvent;
{
  fprintf(stderr,"%s\nserial\terror\trequest\tfunction\twindow.\n\
%ld\t%d\t%d\t%d\t%d\n",
	  XErrDescrip(errorEvent->error_code),
	  errorEvent->serial,errorEvent->error_code,errorEvent->request_code,
	  errorEvent->func,errorEvent->window);
  return;
}

Window SetCurrentWindow(window)
     Window window;
{
  SchemeWindowInfo *windowInfo;

  if ((windowInfo =(SchemeWindowInfo *)
       XLookUpAssoc(schemeXAssocTable,window)) == NULL)
    return 0;
  activeDisplay[currentDisplayId].window = currentWindow = window;
  activeDisplay[currentDisplayId].windowInfo = currentWindowInfo = windowInfo;
  XRaiseWindow(currentWindow);

  return(window);
}

WorldPoint *SetWorldPoint(point,x,y)
  WorldPoint *point;
  float x, y;
{
  point->x = x;
  point->y = y;
  return(point);
}

DevicePoint *SetDevicePoint(point,x,y)
  DevicePoint *point;
  short x, y;
{
  point->x = x;
  point->y = y;
  return(point);
}

SetPenColor(windowInfo,mode)
     SchemeWindowInfo *windowInfo;
     long mode;
{
  return;
}

/************************************************************************
 * geometric utilities
 */
/* create vector of five points at four sectors of circle */
GetCircle(vlist,wcx,wcy,wr)
     Vertex vlist[];
     float wcx, wcy;
     float wr;
{
  float wvx, wvy;

  vlist[0].flags = VertexCurved | VertexStartClosed;
  vlist[1].flags = vlist[2].flags = vlist[3].flags = VertexCurved;
  vlist[4].flags = VertexCurved | VertexEndClosed;

  wvx = wcx - wr;
  wvy = wcy;
  W2DC(currentWindowInfo,wvx,wvy,&(vlist[0].x),&(vlist[0].y));
  wvx += (2 * wr);
  W2DC(currentWindowInfo,wvx,wvy,&(vlist[2].x),&(vlist[2].y));
  wvx = wcx;
  wvy += wr;
  W2DC(currentWindowInfo,wvx,wvy,&(vlist[1].x),&(vlist[1].y));
  wvy -= (2 * wr);
  W2DC(currentWindowInfo,wvx,wvy,&(vlist[3].x),&(vlist[3].y));
  vlist[4].x = vlist[0].x;
  vlist[4].y = vlist[0].y;
  return;
}

/* sort new point into circle vector */
/* as written this won't work becuse previous sorts change contents */
AddPointCircle(vlist,dcx,dcy)
     Vertex vlist[];
     short dcx, dcy;
{
  if (dcy >= vlist[0].y) {	/* in top half */
    if (dcx >= vlist[1].x) {	/* upper right half */
      ShiftVector(vlist,1,8);
      vlist[1].x = dcx;
      vlist[1].y = dcy;
      vlist[1].flags = VertexCurved;
    }
    else {			/* upper left half */
      ShiftVector(vlist,2,8);
      vlist[2].x = dcx;
      vlist[2].y = dcy;
      vlist[2].flags = VertexCurved;
    }
  }
  else {			/* bottom half */
    if (dcx >= vlist[3].x) {	/* lower right half */
      ShiftVector(vlist,4,8);
      vlist[4].x = dcx;
      vlist[4].y = dcy;
      vlist[4].flags = VertexCurved;
    }
    else {			/* lower left half */
      ShiftVector(vlist,5,8);
      vlist[5].x = dcx;
      vlist[5].y = dcy;
      vlist[5].flags = VertexCurved;
    }
  }
  return;
}

ShiftVector(vlist,start,size)
     Vertex vlist[];
     int start, size;
{
  for (; size > start; size --) {
    vlist[size].x = vlist[start].x;
    vlist[size].y = vlist[start].y;
    vlist[size].flags = vlist[start].flags;
  }
  return;
}

/************************************************************************
 * generate pixel values for named or value-specified colors
 */
GetNamedColor(colorName)
     char *colorName;
{
  Color xactColor;
  Color hardColor;
  int pixel;
  int code;

  ToLower(colorName);
  switch (terminalType) {
  case XTERM:
    if (strncmp(colorName,"foreground",strlen(colorName)) == 0 ||
	strncmp(colorName,"fgcolor",strlen(colorName)) == 0)
      return(currentWindowInfo->fgColor);
    else if (strncmp(colorName,"background",strlen(colorName)) == 0 ||
	strncmp(colorName,"bgcolor",strlen(colorName)) == 0)
      return(currentWindowInfo->bgColor);
    else if (strncmp(colorName,"black",strlen(colorName)) == 0)
	return(BlackPixel);
    else if (strncmp(colorName,"white",strlen(colorName)) == 0)
	return(WhitePixel);
    else if (DisplayCells() > 2) {
      if ((code = XGetColor(colorName,&hardColor,&xactColor)) == 0)
	return (currentWindowInfo->fgColor);
      else return(hardColor.pixel);
    }
    else return(currentWindowInfo->fgColor);
  default:
    return NIL;
  }
}

GetRGBColor(RValue, GValue, BValue)
     int RValue, GValue, BValue;
{
  Color hardColor;

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    hardColor.red = RValue;
    hardColor.green = GValue;
    hardColor.blue = BValue;
    if (DisplayCells() > 2)
      if (XGetHardwareColor(&hardColor) == 0)
	return(currentWindowInfo->fgColor);
      else
	return(hardColor.pixel);
    else return(currentWindowInfo->fgColor);
  default:
    return NIL;
  }
}

Pixmap GetRGBTile(RValue, GValue, BValue)
     int RValue, GValue, BValue;
{
  Color hardColor;
  Pixmap pixmap;
  int value;

  switch (terminalType) {
  case XTERM:
    CheckXStatus();
    if (DisplayCells() > 2) {
      hardColor.red = RValue;
      hardColor.green = GValue;
      hardColor.blue = BValue;
      if (XGetHardwareColor(&hardColor) == 0)
	return(currentWindowInfo->fgColor);
      else return(XMakeTile(hardColor.pixel));
    }
    else {
      value = ((float)RValue * RedYIQFactor) +
	((float) GValue * GreenYIQFactor) +
	  ((float) BValue * BlueYIQFactor);
      value = (value * 255) / 65535;
      return(MakeDitherPixmap(value));
    }
  default:
    return NIL;
  }
}

Pixmap GetNamedTile(colorName)
     char *colorName;
{
  Pixmap pixmap;
  Color hardColor;
  int value;
  int code;

  ToLower(colorName);
  switch (terminalType) {
  case XTERM:
    if (strncmp(colorName,"foreground",strlen(colorName)) == 0 ||
	strncmp(colorName,"fgcolor",strlen(colorName)) == 0)
      return(currentWindowInfo->fgColor);
    else if (strncmp(colorName,"background",strlen(colorName)) == 0 ||
	strncmp(colorName,"bgcolor",strlen(colorName)) == 0)
      return(currentWindowInfo->bgColor);
    else if (strncmp(colorName,"black",strlen(colorName)) == 0)
      return(BlackPixmap);
    else if (strncmp(colorName,"white",strlen(colorName)) == 0)
      return(WhitePixmap);
    else {
      XParseColor(colorName,&hardColor);
      if (DisplayCells() > 2) {
	if ((code = XGetHardwareColor(&hardColor)) == 0)
	  return (currentWindowInfo->fgColor);
	else return(XMakeTile(hardColor.pixel));
      }
      else {
	value = ((float)hardColor.red * RedYIQFactor) +
	  ((float) hardColor.green * GreenYIQFactor) +
	    ((float) hardColor.blue * BlueYIQFactor);
	value = (value * 255) / 65535;
	return(MakeDitherPixmap(value));
      }
    }
  default:
    return NIL;
  }
}

GetDisplayFunction(functionName)
     char *functionName;
{
  ToLower(functionName);
  if (strncmp(functionName,"x_",2) == 0)
    functionName += 2;
  if (strncmp(functionName,"gx",2) == 0)
    functionName += 2;
  switch (terminalType) {
  case XTERM:
    if (strcmp(functionName,"clear") == 0) return(GXclear);
    else if (strcmp(functionName,"and") == 0) return(GXand);
    else if (strcmp(functionName,"andreverse") == 0) return(GXandReverse);
    else if (strcmp(functionName,"copy") == 0) return(GXcopy);
    else if (strcmp(functionName,"andinverted") == 0) return(GXandInverted);
    else if (strcmp(functionName,"noop") == 0) return(GXnoop);
    else if (strcmp(functionName,"xor") == 0) return(GXxor);
    else if (strcmp(functionName,"or") == 0) return(GXor);
    else if (strcmp(functionName,"nor") == 0) return(GXnor);
    else if (strcmp(functionName,"equiv") == 0) return(GXequiv);
    else if (strcmp(functionName,"invert") == 0) return(GXinvert);
    else if (strcmp(functionName,"orreverse") == 0) return(GXorReverse);
    else if (strcmp(functionName,"copyinverted") == 0) return(GXcopyInverted);
    else if (strcmp(functionName,"orinverted") == 0) return(GXorInverted);
    else if (strcmp(functionName,"nand") == 0) return(GXnand);
    else if (strcmp(functionName,"set") == 0) return(GXset);
    else return(GXcopy);
  default:
    return NIL;
  }
}

/************************************************************************
 * macXlib.pattern
 * create and maintain tiles for tone fill.
 * SetDitherMatrix:
 *  generates NxN dither matrix, recursively, based on 2x2 and
 *  4*[Dm(N-1xN-1)]+i*[I(N-1xN-1)]
 *  starts with yo, xo == 0, and size the full size. does not work for
 *  non power of two sizes.
 * note that NxN matrix actually allows for NxN + 1 intensity values
 */
#ifndef NULL
#include <stdio.h>
#endif
#define MAXDMS 64
#ifndef RootWindow
#include <X/Xlib.h>
#endif
int debug;

static int ditherMatrix[MAXDMS][MAXDMS];
static int dmSize = 0;
static int tileSize = 0;
static int maxIntensity = 0;
static int *tileTable = 0;

/************************************************************************
 * generate dither matrix
 */
static SetDitherMatrix(dm,size,yo,xo) /* recursively generate dither matrix */
				      /* based on 2x2 and */
				      /* 4x[Dm(N-1xN-1)] + i*[I(N-1xN-1)] */
     int dm[MAXDMS][MAXDMS];
     int size, yo, xo;
{
  int offset;

  if (size <= 1)		/* just in case */
    dm[yo][xo] = 0;
  else if (size == 2) {
    dm[yo][xo] = 0;
    dm[yo][xo+1] = 2;
    dm[yo+1][xo] = 3;
    dm[yo+1][xo+1] = 1;
  }
  else {
    offset = size/2;
    SetDitherMatrix(dm,offset,yo,xo);
    TimesDitherMatrix(dm,offset,yo,xo,4);
    SetDitherMatrix(dm,offset,yo,xo+offset);
    TimesDitherMatrix(dm,offset,yo,xo+offset,4);
    PlusDitherMatrix(dm,offset,yo,xo+offset,2);
    SetDitherMatrix(dm,offset,yo+offset,xo);
    TimesDitherMatrix(dm,offset,yo+offset,xo,4);
    PlusDitherMatrix(dm,offset,yo+offset,xo,3);
    SetDitherMatrix(dm,offset,yo+offset,xo+offset);
    TimesDitherMatrix(dm,offset,yo+offset,xo+offset,4);
    PlusDitherMatrix(dm,offset,yo+offset,xo+offset,1);
  }

  return;
}

static TimesDitherMatrix(dm,size,yo,xo,factor)
     int dm[MAXDMS][MAXDMS];
     int size, yo, xo, factor;
{
  int y, x;

  for (y = yo; y < yo+size; y ++)
    for (x = xo; x < xo+size; x ++)
      dm[y][x] *= factor;
  return;
}

static PlusDitherMatrix(dm,size,yo,xo,delta)
     int dm[MAXDMS][MAXDMS];
     int size, yo, xo, delta;
{
  int y, x;

  for (y = yo; y < yo+size; y ++)
    for (x = xo; x < xo+size; x ++)
      dm[y][x] += delta;
  return;
}

/************************************************************************
 * generate base dither matrix with range 0 -> (size*size)-1
 * and normalize it to the maximum value specified.
 * depend on X to determine the allowable tile size.
 * save parameters for tile and dither sizes.
 * create array to store tiles.
 */
NormalizeDither(intensity,size)	/* generate size*size dither matrix and */
				/* normalize dither to given maximum */
     int intensity;
     int *size;
{
  int x, y;
  static int defaultSize;

  if (size == 0) {
    size = &defaultSize;
    defaultSize = 16;
  }
  XQueryTileShape(*size,*size,size,size);
  dmSize = tileSize = *size;
  dmSize = ((dmSize <= 2) ? 2 :	/* force to multiple of two */
	    ((dmSize <= 4) ? 4 :
	     ((dmSize <= 8) ? 8 :
	      ((dmSize <= 16) ? 16 :
	       ((dmSize <= 32) ? 32 : MAXDMS)))));
  SetDitherMatrix(ditherMatrix,dmSize,0,0);
  maxIntensity = (dmSize * dmSize) - 1;
  for (y = 0; y < dmSize; y ++) {
    for (x = 0; x < dmSize; x ++) {
      ditherMatrix[y][x] = (ditherMatrix[y][x] * (intensity-1)) / maxIntensity;
    }
  }

  maxIntensity = intensity;
  tileTable = (int *) malloc(sizeof(int) * (maxIntensity+1));
  for (y = 0; y <= maxIntensity; y ++)
    tileTable[y] = 0;
  return;
}

Pixmap MakeDitherPixmap(intensity) /* make tone bitmap for given intensity */
				/* dither matrix at least as large as tile */
     int intensity;
{
  static x, y;
  static unsigned short buffer[(MAXDMS * MAXDMS) / 16] ;
  Bitmap tb;
  Pixmap tp;
  int bufferIndex;
  int defaultSize = 16;

  if (tileTable == 0)		/* set up default configuration */
    NormalizeDither(255,&defaultSize);
  intensity = ((intensity < 0) ? 0 :
	       ((intensity  > maxIntensity) ? maxIntensity : intensity));
  if (tileTable[intensity])
    return(tileTable[intensity]);
  for (y = 0; y < tileSize; y ++) {
    for (x = 0; x < tileSize; x ++) {
      bufferIndex = (y*((tileSize+15)/16))+(x/16);
      if ((x % 16) == 0)
	buffer[bufferIndex] = 0;
      if (ditherMatrix[y][x] < intensity) {
	buffer[bufferIndex] |= (0x0001 << (x % 16));
      }
    }
  }
  if (debug) {
    for (y = 0; y < tileSize; y ++) {
      fprintf(stderr,"\n");
      for (x = 0; x < tileSize; x ++) {
	bufferIndex = (y*((tileSize+15)/16))+(x/16);
	if ((x % 16) == 0)
	  fprintf(stderr,"%04x.",buffer[bufferIndex]);
      }
    }
  }
  if ((tb = XStoreBitmap(tileSize,tileSize,buffer)) == 0)
    return(0);
  tp = XMakePixmap(tb,BlackPixel,WhitePixel);
  tileTable[intensity] = tp;
  XFreeBitmap(tb);
  return(tp);
}
