;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus"))


;;;; Mutables: Abstract lookup tables.

;;; A mutable procedure is a procedure which invokes a lookup table to
;;; map between one set of objects and another.  Mutable procedures
;;; have related MUTATORS which apply a modifying function to a given
;;; entry.  Mutable procedures return a special `not defined' token if
;;; no value has been assigned for a given input of the mutable
;;; procedure.  Values are assigned by mutating this `not defined'
;;; entry using the MUTATOR procedure.  We could use a mutable
;;; procedure REFERENCE-COUNT in the following fashion:
;;;
;;;   ;; Define the mutable procedure
;;;   (DEFINE REFERENCE-COUNT (MAKE-MUTABLE))
;;;   ;; The reference count of FOO is undefined:
;;;   (REFERENCE-COUNT 'FOO) ==> (NOT DEFINED)
;;;   (DEFINED? (REFERENCE-COUNT 'FOO)) ==> #F
;;;   ;; Set the reference count for FOO to zero, by mutating
;;;   ;;  whatever is there to `0'.
;;;   ((MUTATOR REFERENCE-COUNT) 'FOO (lambda (ignore) 0))
;;;   ;; Now we get a value returned:
;;;   (REFERENCE-COUNT 'FOO) ==> 0
;;;   ;; Which we can further mutate:
;;;   ((MUTATOR REFERENCE-COUNT) 'FOO 1+)
;;;   (REFERENCE-COUNT 'FOO) ==> 1
;;; This file describes a form of mutable procedures which uses a
;;; lookup table abstraction from extended RRRS.  Other sorts of
;;; mutable procedures can be defined by just defining a MUTATOR
;;; (which is a mutable procedure too) for an arbitrary function.


;;;; Undefinitions

(define %%undefined-token%% '(NOT DEFINED))

(definline (%undefined) %%undefined-token%%)
(definline (undefined? x) (eq? x %%undefined-token%%))
(definline (defined? x) (not (eq? x %%undefined-token%%)))

(definline (apply-if-defined proc arg)
  (if (undefined? arg) arg (proc arg)))


;;;; Mutators and Enumerators

;;; Mutable procedures have MUTATORS defined for them;
;;; Enumerable procedures have ENUMERATORS defined form them;
;;; Both of these are enumerable and mutable procedures.

;;; MUTATOR:
(define mutators (make-empty-lookup))
(definline (mutator x) (do-lookup x mutators (%undefined)))
(define (mutate-mutator! x f) (mutate-lookup! x mutators f (%undefined)))
(define (set-mutator! x m) (modify-lookup! x mutators m))
(define (for-mutators fcn) (over-lookup fcn mutators))

;;; ENUMERATOR:
(define enumerators (make-empty-lookup))
(define (enumerator x) (do-lookup x enumerators (%undefined)))
(define (mutate-enumerator! x f) (mutate-lookup! x enumerators f (%undefined)))
(define (set-enumerator! x e) (modify-lookup! x enumerators e))
(define (for-enumerators fcn) (over-lookup fcn enumerators))

;;; Bootstrapping:
(set-mutator! mutator mutate-mutator!)
(set-mutator! enumerator mutate-enumerator!)
(set-enumerator! enumerator for-enumerators)
(set-enumerator! mutator for-mutators)


;;;; Constructing mutable procedures

(define (make-mutable)
  (let ((lookup (make-empty-lookup)))
    ;; Lookup is a lookup table storing mappings, it is reference by
    ;; the mutator, enumerator, and mutable procedure itself:
    (define (mutable x) (do-lookup x lookup (%undefined)))
    (define (mutate! x f) (mutate-lookup! x lookup f (%undefined)))
    (define (for-all fcn) (over-lookup fcn lookup))
    ;; Set up the mutator and enumerator:
    (set-mutator! mutable mutate!)
    (set-enumerator! mutable for-all)
    ;; Finally, return the mutable procedure which accesses the lookup:
    mutable))

;;; Mutable procedures are procedures have an mutator.
(definline (mutable? p) (defined? (mutator p)))
;;; Enumerable procedures are procedures have an enumerator.
(definline (enumerable? p) (defined? (enumerator p)))


;;;; Useful functions.

;;; This uses the mutator to SET a value for a modifiable procedure.
(definline (modifier mutable)
  (lambda (x v) ((mutator mutable) x (lambda (ignore) v))))

(define (mutable-size mutable)
  (let ((count 0))
    ((enumerator mutable)
     (lambda (ignore ignore) (set! count (1+ count))))))

;;; This returns a mutating procedure which adds an element to a
;;; stored list (if it is not already there) or just returns a list of
;;; the element if the stored list is undefined.
(definline (pusher x)
  (lambda (v) (if (undefined? v) (list x)
		  (if (memq x v) v (cons x v)))))
;;; This returns a list mutator which does a side-effecting delete.
(definline (dirty-deletor x)
  (lambda (v) (if (undefined? v) v (delq! x v))))
;;; This returns a list mutator which does a functional delete.
(definline (deletor x)
  (lambda (v) (if (undefined? v) v (delq x v))))

;;; This copies a list of mutable properties from one object to another.
(define (copy-properties from to properties)
  (define (copy-property prop)
    (let ((value (prop from)))
      (if (defined? value) ((modifier prop) to value))))
  (for-each copy-property properties))


;;;; Invertible mutators.

;;; We can take a mutable mapping and construct an invertible version
;;; of it which stores the inputs for which the procedure has a given
;;; value.  To do this we define a mutable to store inverses in and
;;; define a new mutable procedure which mutates the primary mutable
;;; and the inverse.

;;; This is where we store the inverse of a mutable.  The inverse of
;;; an inverse is the original procedure, and we can detect which
;;; procedure is original by seeing if it has an INVERSE-PRIMARY.
;;; Note that this notion of inversion is sloppy.  The inverse of a
;;; procedure P actually maps from values of P to lists of inputs to
;;; P; P, however, does not map from lists of inputs to values.

;;; These are the properties of invertibles.
(define inverse (make-mutable))
(define inverse-primary (make-mutable))
(begin ((modifier inverse) inverse inverse))

;;; Invertible procedures are those which have defined inverses:
(definline (invertible? p) (defined? (inverse p)))

(define (construct-invertible primary)
  ;; We first construct a mutable procedure to store inverses:
  (let ((the-inverse (make-mutable)))	
    ;; We bind the primary and inverse mutators locally, assuming
    ;; that no one will change them: 
    (let ((mutate-inverse! (mutator the-inverse))
          (mutate-primary! (mutator primary)))
      ;; The invertible procedure simply calls the primary.
      (define (invertible x) (primary x))
      ;; The mutator for the invertible procedure has to mutate both
      ;; the primary and the table of inverses.
      (define (mutate-invertible! x mutation)
	(define (primary-mutator v)
	  ;; In mutating the inverse, we have to both delete the
	  ;; old entry for the value and place a new entry.
	  (if (defined? v) (mutate-inverse! v (dirty-deletor x)))
	  (let ((new-value (mutation v)))
	    (mutate-inverse! new-value (pusher x))
	    new-value))
	(mutate-primary! x primary-mutator))
      ;; Set up mutators and enumerators appropriately:
      ((modifier mutator) invertible mutate-invertible!)
      ((modifier enumerator) invertible (enumerator primary))
      ;; Set up the proeprties of inverses (in both directions):
      ((modifier inverse-primary) invertible primary)
      ((modifier inverse) invertible the-inverse)
      ((modifier inverse) the-inverse invertible)
      ;; Finally, return the invertible procedure.
      invertible)))

;;; This just makes an invertible.
(define (make-invertible) (construct-invertible (make-mutable)))


;;;; Simple caching

;;; This implements procedures which cache their values.  The method
;;; of caching is (suprise!) a mutable procedure which is called to
;;; to access or modify cached values.

;;; This is the cache where a memoized function stores its information.
(define memoizer-cache (make-invertible))
;;; This is the function the memoizer uses for non-cached values.
(define memoizer-function (make-invertible))
;;; And this is the memoizer a CACHE is defined for.
(define cache-memoizer (inverse memoizer-cache))

(define (generate-memoizer cache fcn)
  (let ((mutate! (mutator cache)))
    (define (memoize x)
      (let ((result #F))		; This is the value we should return.
	(define (generate-value v)
	  (if (defined? v)
	      ;; If the current value is defined, just return it:
	      (begin (set! result v) v)
	      ;; Otherwise,  compute it and return it:
	      (begin (set! result (fcn x)) result)))
	(mutate! x generate-value)
	result))
    ;; Update the properties of memoizers:
    ((modifier memoizer-cache) memoize cache)
    ((modifier memoizer-function) memoize fcn)
    memoize))

(definline (memoizer? p) (defined? (memoizer-cache p)))

(define memoizer-properties-to-copy-from-fcn ())
(define memoizer-properties-to-copy-from-cache
  (list inverse))

;;; This makes a mutable to use as a cache for a function and
;;; generates the appropriate memoizer:
(define (generate-simple-cache fcn)
  (let* ((result-cache (make-invertible))
	 (memoizer (generate-memoizer result-cache fcn)))
    (copy-properties fcn memoizer memoizer-properties-to-copy-from-fcn)
    (copy-properties result-cache memoizer memoizer-properties-to-copy-from-cache)
    memoizer))
(define simple-cache (generate-simple-cache generate-simple-cache))

