;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable"))

;;; This provides an abstraction for procedure calls which are time
;;; sliced over multiple execution windows.  A task is a procedure
;;; which returns a continuation (a procedure and arguments) for
;;; resuming itself.  Tasks thus have to be written in a special style
;;; to support time slicing; when they are willing to suspend
;;; themselves they must RETURN a pair of a procedure and a set of
;;; arguments.  They are resumed by calling the procedure on the given
;;; arguments.

;;; Tasks also have a `state description' which is used to describe
;;; their progress;  this state description is modified by functions
;;; called inside of the executing task.


;;;; Tracing engine events.

(define trace-task-creation #T)


;;;; Creating tasks.

;;; Tasks are procedures whose execution is divided over time.  A
;;; tasks `state' is a procedure a set list of arguments.  When a task
;;; is run (advanced), the procedure is called on the
;;; set of arguments.  At some point, the procedure returns either
;;; FALSE (#F), indicating that the procedure completed, or a new
;;; state (procedure and arguments) for continuing the task at a later
;;; time.  Each task also has a state description, which is a
;;; procedure which describes (to the terminal) the tasks current
;;; status.  This state description is updated by the executing task.

;;;; Tasks

;;; A task has a current state (proc . args), a state description (a
;;; documenting procedure), a copy of the tasks original state, a
;;; unique integer index, a set of arbitrary properties, and a
;;; satisfaction cache for use by TYPICALs indexer.
(define-structure task
  (state state-description initial-application index type-cache)
  (let ((appl (task-initial-application task)))
    (sequence (display  "#[Task ") (display  (task-index task))
	      (display ": ") (print-procedure (car appl))
	      (for-each (lambda (x) (write-char #\Space) (printout-printer x))
			(cdr appl))
	      (display "]"))))

(define task-count 0)
(define *all-tasks* ())

;;; This creates a task which beings by applying FCN to ARGS.  Tasks
;;; are numbered in order of creation and stored on the list
;;; *ALL-TASKS*.  The procedure TASK-CALL is a version of TASK-APPLY
;;; which takes a rest argument of the args to FCN.
(define (task-apply fcn args)
  (let ((task (cons-task (cons fcn args)
			 ;; The initialized task description.
			 (lambda () (printout "Applying " ($procedure fcn) " to " args))
			 ;; Auxiliary information.
			 (cons fcn args) task-count (make-empty-type-cache 200))))
    (set! *all-tasks* (cons task *all-tasks*))
    (set! task-count (1+ task-count))
    (if trace-task-creation (message $NL "+++ Created task " task))
    task))
(define (task-call fcn . args) (task-apply fcn args))
(define cached-task-call (canonical-cache task-call))

;;; We can get a task by its number and extract the initial state and
;;; arguments of the task.
(define (task-numbered n) (list-ref *all-tasks* (- task-count n 1)))
(define (task-procedure task) (first (task-initial-application task)))
(define (task-arguments task) (rest (task-initial-application task)))

;;; This is a procedure for returning a task: if ID is an integer, it
;;; is taken as a task index; if ID is a procedure, it is used in a
;;; cached call to TASK-CALL.
(define (task id . args)
  (cond ((procedure? id) (apply cached-task-call id args))
	((and (integer? id) (< id task-count))
	 (list-ref *all-tasks* (- task-count id 1)))
	(ELSE (error "Can't coerce to a task..."))))


;;;; The Engine

;;; The ENGINE facility is inspired by TI's ENGINE facility in
;;; PC-SCHEME.  However, due to the lack of efficient continuation
;;; passing in C-SCHEME, we can't do things as elegantly as ENGINEs
;;; does; c'est la vie et c'est la guerre...

;;; These are fluidly bound by ADVANCE-TASK.
(define current-task #F) ; This is the current task.
(define task-abort)	 ; This is the continuation for aborting ADVANCE-TASK.

;;; This may be fluidly bound by procedures which call ADVANCE-TASK.
;;;  By its default definition, a task call is just like a regular
;;;  procedure call.
(define (surrender?) #F) ; This determines whether ENGINE execution
			 ; should pause....


;;; This procedure aborts if there is a need to surrender.  It is
;;; useful in internal contexts which may not really return to ADVANCE-TASK.
(define (potential-rout . ignore)
  (if (surrender?) (task-abort #F)))

;;; This executes a task, updating the task's state description if it
;;; suspends or finishes.
(define (ADVANCE-task task)
  (let ((state (task-state task)))
    (if state
	(call-with-current-continuation
	 (lambda (abort-continuation)
	   (fluid-let ((current-task task) (task-abort abort-continuation))
	     (set-task-state! task (apply (car state) (cdr state)))))))))

;;; These procedures update the task's state description.
(define (progress-reporter! proc)
  (set-task-state-description! current-task proc))
(define (progress-report! . printout-args)
  (set-task-state-description!
   current-task (lambda () (apply printout printout-args))))

;;; This is the glue that connects the components of a procedure
;;; running as a task.  This calls the fluidly-bound SURRENDER?
;;; procedure to determine if it should just return a state
;;; description (thus suspending) or go on with its computation.
(define (go fcn . args)
  (if (surrender?) (cons fcn args)
      (sequence (set-task-state! current-task (cons fcn args))
		(apply fcn args))))

;;; This returns SCHEME FALSE to indicate the task is finished and
;;; sets a final state description for the task.
(define (finished! . state-description)
  (apply progress-report! state-description)
  #F)

;;; This advances a whole list of tasks.
(define (ADVANCE-tasks tasks) (for-each advance-task tasks))

;;; This advances all the defined tasks for some number of times.
(define (advance-all-tasks how-many-times)
  (if (= how-many-times 0) #F
      (sequence (advance-tasks *all-tasks*)
		(advance-all-tasks (-1+ how-many-times)))))


;;;; Noisy task execution.

;;; These functions are for advancing tasks while printing information
;;; about task execution; in some future incarnation, these might use
;;; hairier I/0 facilities.

(define (noisy-advance-task task)
  (if (task-state task)			; Only work for running tasks.
      (let* ((start (systime)))
	;; We keep track of how long tasks take; we should also track
	;; space usage and such, but there aren't general facilities
	;; for it in RRRS (or even in C-Scheme at the moment).
	(message $NL ">>> Activating task " task)
	(message $NL ">>> State: " ($call (task-state-description task)))
	(advance-task task)		; Actually advance the task.
	(cond ((task-state task)
	       ;; If the task is still running, say `Suspended' and stats:
	       (message $NL "<<< Suspended after "
			($count (/ (- (systime) start) 100) "second") ": "
			($call (task-state-description task))))
	      (ELSE ; If the task has finished, say `Finished' and stats:
	       (message $NL "<<< Task completed after "
			($count (/ (- (systime) start) 100) "second") ".")
	       (message $NL "<<< Finished: " ($call (task-state-description task))))))))

(define (noisy-advance-tasks tasks) (for-each noisy-advance-task tasks))

;;; This is the most often called top-level loop.
(define (noisy-advance-all-tasks how-many-times)
  (define (advance-all n)
    (message $NL "--------------------------------------------------------------------------------")
    (message $NL "<< Advancing all tasks on cycle " n ">>")
    (noisy-advance-tasks *all-tasks*))
  (for-range advance-all 1 how-many-times))


;;;; Resource limits.

;;; This constraints task execution to a particular time limit.  It
;;; fluidly binds the definition of SURRENDER? to check the active
;;; interval and update it if it fails.  This suffers from the problem
;;; that an interval begins where the last interval ended, unless
;;; SURRENDER? is called (kludgily) when the task is entered.
(define (with-time-per-task allotment proc . args)
  (let ((limit 0))
    (define (overtime?)
      (if (> (systime) limit)
	  (sequence (set! limit (+ (systime) allotment)) #T)
	  #F))
    (fluid-let ((surrender? overtime?))
      (set! limit (+ (systime) allotment))
      (apply proc args))))


;;;; Examples.

;;; This is a `tasked' procedure which applies a mapping procedure to
;;; a list of objects, offering to surrender after each application.
(define (slow-mapper procedure l)
  (define (mapper l progress)
    (define (report-progress)
      (printout "Have mapped " ($procedure procedure) " over "
		($count progress "element") "."))
    (cond ((null? l)
	   (finished! "Mapped " ($procedure procedure) " over all "
		      ($count progress "element") "."))
	  (ELSE (procedure (car l))
		(progress-reporter! report-progress)
		(go mapper (cdr l) (1+ progress)))))
  (go mapper l 0))
;;; This is to make description stuff work even if definitions are compiled.
(name-procedure! slow-mapper 'slow-mapper)
;;; This does the actual TASK-CALL.
(define (slow-map procedure l)
  (if (null? l) procedure (task-call slow-mapper procedure l)))
(procedure! slow-map 'SLOW-MAP
	    "Creates a task mapping a procedure over a list.")


