;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/utility/engine")
	 (integrate-external "/u/kwh/programs/utility/tuple")
	 (integrate-external "/u/kwh/programs/typical/kernel"))
(declare (integrate-operator cache-vector cache-value
			     set-cache-vector! set-cache-value!))


;;;; Type caches.
;;; Type caches store satisfaction information for objects; many
;;; TYPICAL objects (tasks, tuples, types) have special slots for
;;; their type caches.  For everything else, we use a mutable
;;; procedure to store the type cache.

;;; Storing random type caches.
(define type-cache-store (make-mutable))
(define mutate-type-cache! (mutator type-cache-store))

;;; A type cache is just a cons of two bit strings.
(define (make-type-cache ignore)
  (cons (make-empty-bit-string (* 2 type-count))
	(make-empty-bit-string (* 2 type-count))))
;;; And this conses a type cache for random objects.
(define get-type-cache (generate-memoizer type-cache-store make-type-cache))

;;; This actually gets type caches, from record structures if possible
;;; and from random storage if not.
(definline (type-cache x)
  (cond ((tuple? x) (tuple-type-cache x))
	((type-description? x) (type-description-type-cache x))
	((task? x) (task-type-cache x))
	((pair? x)
	 (let ((result (type-cache-store x)))
	   (if (defined? result) result
	       (cons (make-empty-bit-string (* 2 type-count))
		     (make-empty-bit-string (* 2 type-count))))))
	(ELSE (get-type-cache x))))


;;; Functions for accessing type caches.

(define cache-vector car)
(define cache-value  cdr)
(define set-cache-vector! set-car!)
(define set-cache-value!  set-cdr!)

(define (cached? cache index) 
  ;; Determines if a cache knows about a type.
  (check-bit (cache-vector cache) index))

(define (cache-contents cache index) 
  ;; The cached value for a type.
  (check-bit (cache-value cache) index))

(define (set-cache! cache index flag)
  (set-cache-vector! cache (bit-string-modify! (cache-vector cache) index #T))
  (set-cache-value! cache (bit-string-modify! (cache-value cache) index flag))
  flag)

;;; This determines what bits NOT to reset when resetting a cache.
(define cache-reset-mask (make-empty-bit-string 10))

(define (reset-cache! cache)
  (set-cache-vector! cache (bit-string-and (cache-vector cache) cache-reset-mask))
  (set-cache-value!  cache (bit-string-and (cache-value  cache) cache-reset-mask))
  cache)


;;;; Computing and caching defaults.

;;; The value of CACHING-RESULT? is true when a satisfaction cache
;;; is being used.  If it is set to false during a computation, the
;;; result of the compuation is not cached.
(define caching-result? #T)

(define (check-type-with-cache object type cache)
  (let ((index (td-id type)))
    (if (cached? cache index) (cache-contents cache index)
	(fluid-let ((caching-result? #T))
	  (let ((default ((td-predicate type) object)))
	    (if (and caching-result? (not (undefined? default)))
		(set-cache! cache index default))
	    default)))))

(define (has-type? object type)
  ;; This is slower that IN? or SATISFIES? in most cases.
  (check-type-with-cache object type (type-cache object)))


(define (maptypes-under procedure object under-type)
  (let ((inner-satisfies? satisfies?)
	(cache (type-cache object))
	(marks (make-empty-bit-string type-count)))
    (define (seer x type)
      (if (eq? x object)
	  (check-type-with-cache x type cache)
	  (inner-satisfies? x type)))
    (define (push-nodes nodes stack)
      (if (null? nodes) (topological-map stack)
	  (let ((head (first nodes)))
	    (if (check-bit marks (td-id head))
		(push-nodes (rest nodes) stack)
		(push-nodes (rest nodes) (cons head stack))))))
    (define (topological-map stack)
      (if (null? stack) object
	  (let ((head (first stack)) (tail (rest stack)))
	    (cond ((pair? head)
		   (procedure (car head))
		   (topological-map tail))
		  ((defined-true? (check-type-with-cache object head cache))
		   (bit-string-modify! marks (td-id head) #T)
		   (set-cdr! stack ())
		   (push-nodes (td-specializations head) (cons stack tail)))
		  (ELSE (topological-map (rest stack)))))))
    (fluid-let ((satisfies? seer))
      (topological-map (list under-type)))))
(definline (maptypes procedure object)
  (maptypes-under procedure object lattice-top))

(define (types-of object under-type)
  (let ((fringe ()))
    (define (maybe-add-to-fringe x)
      (if (every? fringe (lambda (f) (not (<<? f x))))
	  (set! fringe (cons x fringe))))
    (maptypes-under maybe-add-to-fringe object under-type)
    fringe))
(define (is-a thing) (types-of thing lattice-top))



;;;; Mutable collections.

(define declarable-types (generated-collection simple-types))

(define (make-declarable-type beneath)
  (let ((index 0))
    (define (check-membership-with-cache object)
      (let ((cache (type-cache object)))
	(if (check-bit (cache-vector cache) index)
	    (check-bit (cache-value  cache) index)
	    (%undefined))))
    (define (add-to-collection! x flag)
      (let ((cache (type-cache x)))
	(set-cache-vector! cache (bit-string-modify! (cache-vector cache) index #T))
	(set-cache-value! cache (bit-string-modify! (cache-value cache) index flag))))
    (let ((type (simple-type check-membership-with-cache beneath)))
      (set! index (td-id type))
      (set! cache-reset-mask (bit-string-modify! cache-reset-mask index #T))
      ((modifier satisfaction-modifier) type add-to-collection!)
;     (name-type! type (list "Declarable subtype of " beneath))
      type)))
(define declarable-type
  (collection-generator declarable-types make-declarable-type))

