;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

(declare (usual-integrations)
	 (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/utility/tuple")
	 (integrate-external "/u/kwh/programs/typical/kernel"))

;;;; Representing pairs and relations.

;;; Pairs have a left and a right and pair-types constrain this.
(define pairs (type "Pairs" tuple-product lisp-objects lisp-objects))
(definline (left x) (car (tuple-elements x)))
(definline (right x) (cadr (tuple-elements x)))
;;; A fast `pair' checker.
(define (doubleton? t)
  (and (tuple? t) (= (length (tuple-elements t)) 2)))
(declare-function! left  pairs lisp-objects 'LEFT)
(declare-function! right pairs lisp-objects 'RIGHT)

;;; Meta-types for pairs.
(define relations (subtypes-of pairs))
(define (left-constraints type)
  (mapping-constraint car (mapping-constraint tuple-elements type)))
(define (right-constraints tuple-type)
  (if (false? tuple-type) #F
      (let ((type (mapping-constraint tuple-elements tuple-type)))
	(let ((cdr-constraint (mapping-constraint cdr type)))
	  (and cdr-constraint (mapping-constraint car cdr-constraint))))))

(declare-function! left-constraints relations types 'LEFT-CONSTRAINTS)
(declare-function! right-constraints relations types 'RIGHT-CONSTRAINTS)
(add-property! right-constraints relations "Right Constraints:")
(add-property! left-constraints  relations "Left Constraints:")

(define (relation-space r)
  (tuple-product (left-constraints r) (right-constraints r)))
(declare-function! relation-space relations (tuple-product types types)
		  "RELATION-SPACE")


;;;; EQ Pairs.

(definline (pair-eq? p)
  (and (doubleton? p) (eq? (left p) (right p))))
(define eq-pairs (type "EQ Pairs" simple-type pair-eq? pairs))

;;; Fillin in examples of EQ pairs.
(define (eq-pair-generator r)
  (let ((left (left-constraints r)) (right (right-constraints r)))
    ;; Declare instance generators.
    (declare-instance-generator! left)
    (declare-instance-generator! right)
    ;; Generate from their intersection.
    (make-combiner r (lambda (x) (tuple x x)) (<AND> left right))))
(add-daemon! (use-generator eq-pair-generator)
	     (<AND> (subtypes-of eq-pairs) instance-sources))
(add-daemon! (inhibitor (use-generator random-tuple-generator))
	     (subtypes-of eq-pairs))


;;;; Representing relational inverses:

(define (twister pair) (tuple (right pair) (left pair)))
(declare-function! twister pairs pairs)

(define (inverse-relation r)
  (<AND> (tuple-product (right-constraints r) (left-constraints r))
	 (image-constraint twister r)))

;;; Generating examples of twisted classes.

(define (twister-generator class)
  (make-combiner class twister (mapping-constraint twister class)))

(define (twisted-type? type) (not (eq? (mapping-constraint twister type) pairs)))
(define twisted-types (type "Twisted Types" simple-type twisted-type? types))
(add-daemon! (use-generator twister-generator) twisted-types)


;;;; Left and right equal pairs 

(define pair-pairs (type "PAIR-PAIRS" tuple-product pairs pairs))
;;; A fast test for `pair pairs'.
(define (pair-pair? x)
  (and (doubleton? x) (doubleton? (left x)) (doubleton? (right x))))

(define (right-equal? x)
  (and (pair-pair? x) (eq? (right (left x)) (right (right x)))))
(define right-equals (type "RIGHT-EQUALS" simple-type right-equal? pair-pairs))

(define (left-equal?  x)
  (and (pair-pair? x) (eq? (left (left x)) (left (right x)))))
(define left-equals (type "LEFT-EQUALS" simple-type left-equal? pair-pairs))


;;;; Determinism of relations.

(define right-deterministic
  (type "Right Deterministic" empirical-class relations
	(lambda (r) (<AND> (tuple-product r r) right-equals))
	(lambda (r) (<AND> (tuple-product r r) left-equals))))
(define left-deterministic
  (type "Left Deterministic" empirical-class relations
	(lambda (r) (<AND> (tuple-product r r) left-equals))
	(lambda (r) (<AND> (tuple-product r r) right-equals))))

(define one-to-many
  (type "One To Many" <AND> left-deterministic (complement right-deterministic)))
(define one-to-one
  (type "One To One" <AND> right-deterministic left-deterministic))
(define many-to-one
  (type "Many To One" <AND> right-deterministic (complement left-deterministic)))
(define many-to-many
  (type "Many To Many" <AND> relations
	(complement left-deterministic) (complement right-deterministic)))


;;;; Pairings
;;;    are relations whose right and left sides may overlap.
(define <<?-rel
  (type "<<?" simple-type
	(lambda (x) (and (doubleton? x) (<<? (left x) (right x))))
	(tuple-product types types)))
(define >>?-rel
  (type ">>?" simple-type
	(lambda (x) (and (doubleton? x) (<<? (right x) (left x))))
	(tuple-product types types)))
(define disjoint-types
  (type "Disjoint Types" simple-type
	(lambda (x) (and (doubleton? x) (disjoint? (left x) (right x))))
	(tuple-product types types)))

(define pairings
  (type "Pairings" <OR>
	(image-constraint tuple-type->type-tuple <<?-rel)
 	(image-constraint tuple-type->type-tuple >>?-rel)))


;;;; Help with experiments (Triangle Generators)

;;; The procedures below are called triangle generators because they
;;; guess at three elements in a pair-square and construct the whole
;;; square from that.

(define (left-triangle-generator type)
  (define (combine-to-make-left-equals left right-top right-bottom)
    (tuple (tuple left right-top) (tuple left right-bottom)))
  (let ((tops (left-constraints type))
	(bottoms (right-constraints type)))
    (let ((lefts (<AND> (left-constraints tops) (left-constraints bottoms)))
	  (right-tops (right-constraints tops))
	  (right-bottoms (right-constraints bottoms)))
      (make-combiner type combine-to-make-left-equals
		     lefts right-tops right-bottoms))))

(define (right-triangle-generator type)
  (define (combine-to-make-right-equals right left-top left-bottom)
    (tuple (tuple left-top right) (tuple left-bottom right)))
  (let ((tops (left-constraints type))
	(bottoms (right-constraints type)))
    (let ((rights (<AND> (right-constraints tops) (right-constraints bottoms)))
	  (left-tops (left-constraints tops))
	  (left-bottoms (left-constraints bottoms)))
      (make-combiner type combine-to-make-right-equals
		     rights left-tops left-bottoms))))


(define (br-corner-generator type)
  (define (generate-bottom-right-corner top bottom-left)
    (tuple top (tuple bottom-left (right top))))
  (make-combiner type generate-bottom-right-corner
		 (left-constraints type) (left-constraints (right-constraints type))))
(define (bl-corner-generator type)
  (define (generate-bottom-left-corner top bottom-right)
    (tuple top (tuple (left top) bottom-right)))
  (make-combiner type generate-bottom-left-corner
		 (left-constraints type)
		 (right-constraints (right-constraints type))))
(define (tl-corner-generator type)
  (define (generate-top-left-corner bottom top-right)
    (tuple (tuple (left bottom) top-right) bottom))
  (make-combiner type generate-top-left-corner
		 (right-constraints type)
		 (right-constraints (left-constraints type))))
(define (tr-corner-generator type)
  (define (generate-top-right-corner bottom top-left)
    (tuple (tuple top-left (right bottom)) bottom))
  (make-combiner type generate-top-right-corner
		 (right-constraints type)
		 (left-constraints (left-constraints type))))

;;;; Generation daemons

;;; Don't use random tuple generation.

(add-daemon! (inhibitor (use-generator random-tuple-generator))
	     (<AND> instance-sources
		    (<OR> (subtypes-of left-equals) (subtypes-of right-equals))))

;;; Try triangle generation methods.
(add-daemon! (use-generator left-triangle-generator)
	     (<AND> instance-sources (subtypes-of left-equals)))
(add-daemon! (use-generator right-triangle-generator)
	     (<AND> instance-sources (subtypes-of right-equals)))

;;; Try corner generation methods:
(add-daemon! (use-generator br-corner-generator)
	     (<AND> instance-sources (subtypes-of right-equals)))
(add-daemon! (use-generator tr-corner-generator)
	     (<AND> instance-sources (subtypes-of right-equals)))
(add-daemon! (use-generator bl-corner-generator)
	     (<AND> instance-sources (subtypes-of left-equals)))
(add-daemon! (use-generator tl-corner-generator)
	     (<AND> instance-sources (subtypes-of left-equals)))
