#include "scheme.h"
#include "prims.h"
#include <dpsclient/dpsclient.h>
#include <dpsclient/dpsops.h>
#include <math.h>
#include <objc/error.h>
#include <signal.h>
#include "schwraps.h"

#define ATOMIC {int _atomic_mask = sigsetmask(~0);
#define END_ATOMIC sigsetmask(_atomic_mask);}
#define LEAVE_ATOMIC sigsetmask(_atomic_mask)

#define MAX_NAME_LENGTH 1024

#define GIN_FD 3
#define GOUT_FD 1
#define INIT_CONTEXTS 20

static FILE *gin, *gout, *console;
static DPSContext *contexts = NULL;
static unsigned contexts_malloced;
static unsigned contexts_in_use = 0;
static int optimistic, noSpace, inContextWait=0;

static float line_style_pats[8][6] = {{}, {8}, {3}, {8,3,2,3},
                                      {6,2,2,2,2,2}, {11,5},{7,3,3,3},
                                      {5,2,3,2,3,2}};
static int line_style_sizes[8] = {0,1,1,4,6,2,4,6};
static float line_style_offsets[8] = {0,0,0,0,0,0,0,0};

static void
DEFUN (schgraph_error, (message),
       char *message)
{
  fprintf(console, "Scheme graphics: %s\n", message);
  signal_error_from_primitive(ERR_IO_ERROR);
}

static void
DEFUN (MyTextProc, (ctxt, buf, count),
       DPSContext ctxt AND
       const char *buf AND
       long unsigned int count)
{
  fwrite(buf, 1, count, console);
}

static void
DEFUN (handler, (context, errorInfo),
       DPSContext context AND
       NXHandler *errorInfo)
{
  if(errorInfo->code == dps_err_ps){
    fprintf(console, "Scheme graphics: ");
    DPSPrintError(console, (DPSBinObjSeq) errorInfo->data2);
  }
  else{
    fprintf(console,
            "Scheme graphics: non postscript DPS error; code: %d\n",
            errorInfo->code);
    }
  if(errorInfo->data1 != NULL){
    DPSContext ctxt = (DPSContext) errorInfo->data1;
    int device;
    DPSDestroyContext(ctxt);
    for(device = 0; device < contexts_malloced; device++)
      if(contexts[device] == ctxt){
        contexts[device] = NULL;
        contexts_in_use--;
        fprintf(gout, "\033Gc%d\033", device);
        if(context == ctxt)
          signal_error_from_primitive(ERR_IO_ERROR);
        else
          return;
      }
  }
}

#define HANDLER NX_HANDLER handler(context, &NXLocalHandler); NX_ENDHANDLER

static unsigned int
DEFUN (arg_device, (arg_number),
       unsigned int arg_number)
{
  unsigned int result = arg_index_integer(arg_number, contexts_malloced);
  if(contexts[result] == NULL)
    error_bad_range_arg(arg_number);
  return(result);
}

static DPSContext
DEFUN (arg_context, (arg_number),
       unsigned int arg_number)
{
  DPSContext result =
    contexts[arg_index_integer(arg_number, contexts_malloced)];
  if(result == NULL)
    error_bad_range_arg(arg_number);
  return(result);
}

static void
DEFUN_VOID (initialize_schgraph)
{
  extern void *calloc();
  extern char *getenv();
  optimistic = (getenv("SCHGRAPH_OPTIMISTIC") != NULL);
  noSpace = (getenv("SCHGRAPH_NO_SPACE") != NULL);
  if((console = fopen("/dev/console", "w")) == NULL)
    signal_error_from_primitive(ERR_IO_ERROR);
  setvbuf(console, NULL, _IONBF, 0);
  if((gin = fdopen(GIN_FD, "r")) == NULL)
    schgraph_error("error opening graphics input");
  if((gout = fdopen(GOUT_FD, "w")) == NULL)
    schgraph_error("error opening graphics output");
  setvbuf(gout, NULL, _IONBF, 0);
  if((contexts = calloc(sizeof(DPSContext), INIT_CONTEXTS)) == NULL)
    schgraph_error("error allocating initial contexts array");
  contexts_malloced = INIT_CONTEXTS;
  DPSSetTextBackstop(MyTextProc);
  DPSSetErrorBackstop(DPSDefaultErrorProc);
}

static void
DEFUN (adjustCoordinates, (context, old_xl, old_yb, old_xr, old_yt,
                           new_xl, new_yb, new_xr, new_yt),
       DPSContext context AND
       float old_xl AND
       float old_yb AND
       float old_xr AND
       float old_yt AND
       float new_xl AND
       float new_yb AND
       float new_xr AND
       float new_yt)
{
  float yscale = (new_yt - new_yb) / (old_yt - old_yb);
  float xscale = (new_xr - new_xl) / (old_xr - old_xl);
  float avgscale = sqrt(fabs(xscale * yscale));
  PSWadjustCoordinates(context, old_xl, old_yb, old_xr, old_yt, new_xl,
                       new_yb, new_xr, new_yt, xscale, yscale, avgscale);
}

DEFINE_PRIMITIVE ("SCHGRAPH-OPEN", Prim_schgraph_open, 2, 2,
 "(SCHGRAPH-OPEN WIDTH HEIGHT)\n\
Return a window of the exact non_negative integer dimensions in pixels.")
{
  PRIMITIVE_HEADER (2);
  {
    extern void *realloc();
    long width = UNSIGNED_FIXNUM_ARG(1);
    long height = UNSIGNED_FIXNUM_ARG(2);
    DPSContext context;
    long window, device;
    char hostName[MAX_NAME_LENGTH], serverName[MAX_NAME_LENGTH];
    int c;
    extern int errno;
    if(contexts == NULL)
      initialize_schgraph();
    if(contexts_in_use == contexts_malloced){
      int i;
      contexts_malloced *= 2;
      if((contexts = realloc(contexts, sizeof(DPSContext)*contexts_malloced))
         == NULL)
        schgraph_error("error extending contexts array");
      for(i = contexts_in_use; i < contexts_malloced; i++)
        contexts[i] = NULL;
    }
    contexts_in_use++;
    for(device = 0; contexts[device] != NULL; device++)
      ;
    fprintf(gout, "\033Go%d %d %d\033",  device, width, height);
    ATOMIC
    errno = 0;
    if((c=fscanf(gin, "%d", &window)) != 1){
      LEAVE_ATOMIC;
      fprintf(console, "Scheme graphics: fscanf returned %d, errno=%d\n",
              c, errno);
      schgraph_error("error reading window number from Schematik");
    }
    errno = 0;
    if((c=getc(gin)) != '\n'){
      LEAVE_ATOMIC;
      fprintf(console, "Scheme graphics: received char %#o, errno=%d\n",
              c, errno);
      schgraph_error("didn't find expected newline from Schematik");
    }
    errno = 0;
    if(fgets(hostName, MAX_NAME_LENGTH, gin) == NULL){
      LEAVE_ATOMIC;
      fprintf(console, "Scheme graphics: errno=%d\n",
              errno);
      schgraph_error("error or eof getting host name from Schematik");
    }
    if(hostName[c=strlen(hostName)-1] != '\n'){
      LEAVE_ATOMIC;
      schgraph_error("host name from Schematik is too long");
    }
    hostName[c] = '\0';
    errno = 0;
    if(fgets(serverName, MAX_NAME_LENGTH, gin) == NULL){
      LEAVE_ATOMIC;
      fprintf(console, "Scheme graphics: errno=%d\n",
              errno);
      schgraph_error("error or eof getting server name from Schematik");
    }
    if(serverName[c=strlen(serverName)-1] != '\n'){
      LEAVE_ATOMIC;
      schgraph_error("server name from Schematik is too long");
    }
    serverName[c] = '\0';
    END_ATOMIC
    NX_DURING
    contexts[device] =
      context = DPSCreateContext(*hostName ? hostName : NULL,
                                 *serverName ? serverName : NULL,
                                 MyTextProc, DPSDefaultErrorProc);
    PSWinitialize(context, window);
    adjustCoordinates(context, 0.0, 0.0, (float) width, (float) height,
                      -1.0, -1.0, 1.0, 1.0);
    HANDLER
    PRIMITIVE_RETURN (LONG_TO_UNSIGNED_FIXNUM(device));
  }
}

DEFINE_PRIMITIVE ("SCHGRAPH-CLOSE", Prim_schgraph_close, 1, 1,
 "(SCHGRAPH-CLOSE WINDOW)\n\
Close a window specified by id number.")
{
  PRIMITIVE_HEADER (1);
  {
    long device = arg_device(1);
    DPSContext context = contexts[device];
    NX_DURING
    DPSDestroyContext(context);
    fprintf(gout, "\033Gc%d\033", device);
    contexts[device] = NULL;
    contexts_in_use--;
    HANDLER
  }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-FLUSH", Prim_schgraph_flush, 1, 1,
 "(SCHGRAPH-FLUSH WINDOW)\n\
Flush a window specified by id number.")
{
  PRIMITIVE_HEADER (1);
  {
    long device = arg_device(1);
    DPSContext context = contexts[device];
    NX_DURING
    DPSFlushContext(context);
    if(!optimistic)
        DPSWaitContext(context);
    fprintf(gout, "\033Gf%d 3\033", device);
    HANDLER
  }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-PRINT", Prim_schgraph_print, 1, 1,
 "(SCHGRAPH-PRINT WINDOW)\n\
Print a window specified by id number.")
{
  PRIMITIVE_HEADER (1);
  {
    long device = arg_device(1);
    DPSContext context = contexts[device];
    NX_DURING
    DPSFlushContext(context);
    if(!optimistic)
        DPSWaitContext(context);
    fprintf(gout, "\033Gp%d\033", device);
    HANDLER
  }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-CLEAR", Prim_schgraph_clear, 1, 1,
 "(SCHGRAPH-CLEAR WINDOW)\n\
Clear a window.")
{
  PRIMITIVE_HEADER (1);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWclear(context);
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-DRAW-LINE", Prim_schgraph_draw_line, 5, 5,
 "(SCHGRAPH-DRAW-LINE WINDOW X0 Y0 X1 Y1)\n\
Draw a line from (x0,y0) to (x1,y1) in window.")
{
  PRIMITIVE_HEADER (5);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWdrawLine(context, arg_real_number(2), arg_real_number(3),
                arg_real_number(4), arg_real_number(5));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-DRAW-POINT", Prim_schgraph_draw_point, 3, 3,
 "(SCHGRAPH-DRAW-POINT WINDOW X Y)\n\
Draw a point at (x,y) in window.")
{
  PRIMITIVE_HEADER (3);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWdrawPoint(context, arg_real_number(2), arg_real_number(3));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-DRAW-POSTSCRIPT", Prim_schgraph_draw_postscript,
                  2, 2,
 "(SCHGRAPH-DRAW-POSTSCRIPT WINDOW STRING)\n\
Emit arbitrary postscript to window.")
{
  PRIMITIVE_HEADER (2);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    DPSWritePostScript(context, STRING_ARG(2), STRING_LENGTH(ARG_REF(2)));
    if(!noSpace)
      DPSWritePostScript(context, " ", 1);
    HANDLER
  }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-DRAW-TEXT", Prim_schgraph_draw_text, 4, 4,
 "(SCHGRAPH-DRAW-TEXT WINDOW X Y STRING)\n\
Draw the text string at (x,y) in  window.")
{
  PRIMITIVE_HEADER (4);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWdrawText(context,
                arg_real_number(2), arg_real_number(3), STRING_ARG(4));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-RESET-CLIP-RECTANGLE",
                  Prim_schgraph_reset_clip_rectangle, 1, 1,
 "(SCHGRAPH-RESET-CLIP-RECTANGLE WINDOW)\n\
Clip only to boundaries of window.")
{
  PRIMITIVE_HEADER (1);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    DPSinitclip(context);
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-CLIP-RECTANGLE",
                  Prim_schgraph_set_clip_rectangle, 5, 5,
 "(SCHGRAPH-SET-CLIP-RECTANGLE WINDOW XL YB XR YT)\n\
Clip to rectangle with corners (xl,yb) and (xr,yt) in window.")
{
  PRIMITIVE_HEADER (5);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetClipRectangle(context,
                        arg_real_number(2), arg_real_number(3),
                        arg_real_number(4), arg_real_number(5));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-FOREGROUND-HSB",
                  Prim_schgraph_set_foreground_hsb, 4, 4,
 "(SCHGRAPH-SET-FOREGROUND-HSB WINDOW H S B)\n\
Sets the foreground hue/saturation/brightness for window; 0 <= H,S,B <= 1.")
{
  PRIMITIVE_HEADER (4);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetForegroundHSB(context, arg_real_in_range(2, 0.0, 1.0),
                        arg_real_in_range(3, 0.0, 1.0),
                        arg_real_in_range(4, 0.0, 1.0));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-FOREGROUND-RGB",
                  Prim_schgraph_set_foreground_rgb, 4, 4,
 "(SCHGRAPH-SET-FOREGROUND-RGB WINDOW R G B)\n\
Sets the foreground red/green/blue for window; 0 <= R,G,B <= 1.")
{
  PRIMITIVE_HEADER (4);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetForegroundRGB(context, arg_real_in_range(2, 0.0, 1.0),
                        arg_real_in_range(3, 0.0, 1.0),
                        arg_real_in_range(4, 0.0, 1.0));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-FOREGROUND-GRAY",
                  Prim_schgraph_set_foregound_gray, 2, 2,
 "(SCHGRAPH-SET-FOREGROUND-GRAY WINDOW GRAY)\n\
Sets the foreground gray for window; 0 <= gray <= 1.")
{
  PRIMITIVE_HEADER (2);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetForegroundGray(context, arg_real_in_range(2, 0.0, 1.0));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-BACKGROUND-HSB",
                  Prim_schgraph_set_background_hsb, 4, 4,
 "(SCHGRAPH-SET-BACKGROUND-HSB WINDOW H S B)\n\
Sets the background hue/saturation/brightness for window; 0 <= H,S,B <= 1.")
{
  PRIMITIVE_HEADER (4);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetBackgroundHSB(context, arg_real_in_range(2, 0.0, 1.0),
                        arg_real_in_range(3, 0.0, 1.0),
                        arg_real_in_range(4, 0.0, 1.0));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-BACKGROUND-RGB",
                  Prim_schgraph_set_background_rgb, 4, 4,
 "(SCHGRAPH-SET-BACKGROUND-RGB WINDOW R G B)\n\
Sets the background red/green/blue for window; 0 <= R,G,B <= 1.")
{
  PRIMITIVE_HEADER (4);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetBackgroundRGB(context, arg_real_in_range(2, 0.0, 1.0),
                        arg_real_in_range(3, 0.0, 1.0),
                        arg_real_in_range(4, 0.0, 1.0));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-BACKGROUND-GRAY",
                  Prim_schgraph_set_backgound_gray, 2, 2,
 "(SCHGRAPH-SET-BACKGROUND-GRAY WINDOW GRAY)\n\
Sets the background gray for window; 0 <= gray <= 1.")
{
  PRIMITIVE_HEADER (2);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    PSWsetBackgroundGray(context, arg_real_in_range(2, 0.0, 1.0));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-DRAWING-MODE",
                  Prim_schgraph_set_drawing_mode, 2, 2,
 "(SCHGRAPH-SET-DRAWING-MODE WINDOW MODE)\n\
Sets the drawing mode for window; only 0,3,7,15 aproximately work.")
{
  PRIMITIVE_HEADER (2);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    if(arg_index_integer(2, 16) == 0)
      PSWbackground(context);
    else
      PSWforeground(context);
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-SET-LINE-STYLE",
                  Prim_schgraph_set_line_style, 2, 2,
 "(SCHGRAPH-SET-LINE-STYLE WINDOW STYLE)\n\
Sets the line style for window; only looks good if in device coordinates.")
{
  PRIMITIVE_HEADER (2);
  {
    int style = arg_index_integer(2, 8);
    DPSContext context = arg_context(1);
    NX_DURING
    DPSsetdash(context, line_style_pats[style],
               line_style_sizes[style], line_style_offsets[style]);
    HANDLER
  }
  PRIMITIVE_RETURN (UNSPECIFIC);
}

DEFINE_PRIMITIVE ("SCHGRAPH-ADJUST-COORDINATES",
                  Prim_schgraph_adjust_coordinates, 9, 9,
 "(SCHGRAPH-SET-ADJUST-COORDINATES WINDOW MODE OLD-XL OLD-YB OLD-XR OLD-YT NEW-XL NEW-YB NEW-XR NEW-YT)\n\
Changes the coordinate limits of window from (old-xl,old-yb)-(old-xr,old-yt) to (new-xl,new-yb)-(new-xr,new-yt)")
{
  PRIMITIVE_HEADER (9);
  {
    DPSContext context = arg_context(1);
    NX_DURING
    adjustCoordinates(context,
                      arg_real_number(2), arg_real_number(3),
                      arg_real_number(4), arg_real_number(5),
                      arg_real_number(6), arg_real_number(7),
                      arg_real_number(8), arg_real_number(9));
    HANDLER
    }
  PRIMITIVE_RETURN (UNSPECIFIC);
}
