/* UNIX specific extensions */


#include "os.h"
#include "mem.h"
#include "strings.h"


/*---------------------------------------------------------------------------*/

/*
 * This file contains C procedures that are to be made available to the
 * Scheme world.  A special naming convention is used for the procedure
 * names so that any Scheme identifier can be written out (including
 * identifiers with '#' characters).  The procedure 'c_id_to_symbol' in
 * 'strings.c' implements the naming convention.
 *
 * The arguments and results of these procedures are of type 'SCM_obj'.
 * Macros and procedures exist to convert from Scheme data representation
 * to C data representation (see 'gambit.h').
 *
 */


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#ifdef unix_extensions

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>

#ifdef sun
#define pid_t int
#endif


SCM_obj X23X23unixDsystem( str )
SCM_obj str;
{ SCM_obj result;
  char *mark = local_mark();              /* put mark on local C heap    */
  char *command = string_to_c_str( str ); /* convert Scheme string to C  */
  if (command == NULL)
    result = SCM_false;
  else
    result = SCM_int_to_obj( system( command ) );
  local_release( mark );                  /* get rid of converted string */
  return result;
}


/* defined in os_unix.c and run.c */

#define MAX_NB_FDS 32
extern OS_FILE file[MAX_NB_OPEN_FILES];


SCM_obj X23X23unixDpipeDopen( path, args )
SCM_obj path, args;
{ char *exec_args[100], *mark;
  long i;
  SCM_obj lst = args;
  int j = 1;
  pid_t child_pid;
  int child_to_parent[2], parent_to_child[2];

  for (i=0; i<(long)MAX_NB_OPEN_FILES-1; i++)
    if ((file[i] == -1) && (file[i+1] == -1)) break; /* need 2 slots */
  if (i == (long)MAX_NB_OPEN_FILES-1) return (long)SCM_false;
  mark = local_mark();
  exec_args[0] = os_expand_filename( string_to_c_str(path) );
  if (exec_args[0] == NULL) goto err3;

  while (lst != SCM_null)
  { char *s = string_to_c_str(
                *(SCM_obj *)(lst-SCM_type_PAIR+PAIR_CAR*sizeof(SCM_obj)) );
    exec_args[j++] = s;
    if ((s == NULL) || (j == 100)) goto err3;
    lst = *(SCM_obj *)(lst-SCM_type_PAIR+PAIR_CDR*sizeof(SCM_obj));
  }
  exec_args[j] = NULL;

  if (pipe( child_to_parent ) != 0) goto err3;
  if (pipe( parent_to_child ) != 0) goto err2;

  if ((child_to_parent[0] < 0) || (child_to_parent[0] >= MAX_NB_FDS) ||
      (child_to_parent[1] < 0) || (child_to_parent[1] >= MAX_NB_FDS) ||
      (parent_to_child[0] < 0) || (parent_to_child[0] >= MAX_NB_FDS) ||
      (parent_to_child[1] < 0) || (parent_to_child[1] >= MAX_NB_FDS))
    goto err1;

  child_pid = vfork();
  if (child_pid == -1)
    goto err1;
  else if (child_pid == 0)
  { setpgrp();
    if ((dup2( parent_to_child[0], 0 ) == 0) && /* setup stdin */
        (dup2( child_to_parent[1], 1 ) == 1) && /* setup stdout */
        (dup2( child_to_parent[1], 2 ) == 2) && /* setup stderr */
        (close( child_to_parent[0] ) == 0) &&
        (close( child_to_parent[1] ) == 0) &&
        (close( parent_to_child[0] ) == 0) &&
        (close( parent_to_child[1] ) == 0))
      execv( exec_args[0], exec_args );
    _exit(0);
  }

  if ((close( child_to_parent[1] ) != 0) ||
      (close( parent_to_child[0] ) != 0)) goto err1;

  file[i]   = child_to_parent[0];
  file[i+1] = parent_to_child[1];

  local_release( mark );
  return SCM_int_to_obj(i);

  err1:
  close( parent_to_child[0] );
  close( parent_to_child[1] );

  err2:
  close( child_to_parent[0] );
  close( child_to_parent[1] );

  err3:
  local_release( mark );
  return (long)SCM_false;
}


SCM_obj X23X23unixDpipeDclose( ind )
SCM_obj ind;
{ long i = SCM_obj_to_int(ind);
  if ((i>=3) && (i<(long)MAX_NB_OPEN_FILES-1))
  { OS_FILE f1 = file[i], f2 = file[i+1];
    int ok1, ok2;
    file[i] = -1;
    file[i+1] = -1;
    ok1 = ((f1 != -1) && (close( f1 ) == 0));
    ok2 = ((f2 != -1) && (close( f2 ) == 0));
    if (ok1 && ok2)
      return (long)SCM_true;
  }
  return (long)SCM_false;
}


/* Sample use of pipes:


(define (##open-pipe path . args)
  (let ((descr (##unix-pipe-open path args)))
    (if descr
      (##make-port descr path 1
        ##os-file-read
        (lambda (descr s i j) (##os-file-write (##fixnum.+ descr 1) s i j))
        ##os-file-read-ready
        ##unix-pipe-close
        (##make-string 64 #\space)
        (##make-string 1 #\space))
      #f)))

(define (pwd)  (run "/bin/pwd"))
(define (date) (run "/bin/date"))

(define (run . args)
  (let ((p (##apply ##open-pipe args)))
    (if p
      (let loop ((l '()))
        (let ((x (read-char p)))
          (if (or (eof-object? x) (char=? x #\newline))
            (begin
              (close-input-port p)
              (list->string (reverse l)))
            (loop (cons x l)))))
      #f)))

*/

#endif

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#ifdef x_extensions


#include <X11/Xlib.h>
#include <X11/Xutil.h>


#define X_MAX_NB_WINDOWS 20
#define X_MAX_NB_COLORS 2

static int x_initialized = 0;

static Display *x_display;
static int x_screen;
static int x_depth;
static int x_black, x_white;
static Visual *x_visual;
static int x_nb_colors;

static struct {
  int in_use;
  int width, height;
  Window wind;
  GC color[X_MAX_NB_COLORS];
  XFontStruct *font;
  } x_wind[X_MAX_NB_WINDOWS];


static int x_check_wind( w )
int w;
{
  return ((!x_initialized) ||
          (w < 0) || (w >= X_MAX_NB_WINDOWS) ||
          (!x_wind[w].in_use));
}


static void x_clear_window( w )
int w;
{
  XFillRectangle( x_display, x_wind[w].wind, x_wind[w].color[0],
                  0, 0, x_wind[w].width, x_wind[w].height );
  XFlush( x_display );
}


static int x_open_window( title, width, height )
char *title;
int width, height;
{
  int w;
  XEvent report;

  if (!x_initialized)
  {
    int i;

    /* connect to X server */

    if ((x_display = XOpenDisplay(NULL)) == NULL)
    {
      fprintf( stderr, "X: Can't connect to X server %s\n",
                       XDisplayName(NULL) );
      return -1;
    }

    x_screen = DefaultScreen( x_display );

    x_black = BlackPixel( x_display, x_screen );
    x_white = WhitePixel( x_display, x_screen );

    if ((x_black < 0) || (x_black > 1) ||
        (x_white < 0) || (x_white > 1))
    { fprintf( stderr, "X: Black and white must be 0 or 1\n" ); return -1; }

    x_visual = DefaultVisual( x_display, x_screen );

    x_depth = DefaultDepth( x_display, x_screen );

    for (i=0; i<X_MAX_NB_WINDOWS; i++) x_wind[i].in_use = 0;

    x_initialized = 1;
  }

  for (w=0; w<X_MAX_NB_WINDOWS; w++)
    if (!x_wind[w].in_use) break;
  if (w==X_MAX_NB_WINDOWS) return -1;

  /* get font */

  if ((x_wind[w].font=XLoadQueryFont(x_display,"6x10")) == NULL)
    if ((x_wind[w].font=XLoadQueryFont(x_display,"fixed")) == NULL)
    { fprintf( stderr, "X: Can't find font 6x10 or fixed\n" ); return -1; }

  /* create the window */

  x_wind[w].in_use = 1;
  x_wind[w].width  = width;
  x_wind[w].height = height;

  x_wind[w].wind =
    XCreateSimpleWindow( x_display,
                         RootWindow( x_display, x_screen ),
                         0, 0,
                         width, height,
                         0, x_white, x_black );

  if (x_wind[w].wind == NULL)
  { fprintf( stderr, "X: Can't create window\n" ); return -1; }

  XSetStandardProperties( x_display, x_wind[w].wind, title, title,
                          NULL, NULL, 0, NULL );

  /* set graphic context */

  {
    int i;
    XGCValues values;

    x_nb_colors = 2;

    for (i=0; i<x_nb_colors; i++)
    {
      values.foreground = (i==0) ? x_black : x_white;
      values.background = x_black;

      x_wind[w].color[i] =
        XCreateGC( x_display,
                   RootWindow( x_display, x_screen ),
                   (GCForeground | GCBackground),
                   &values );

      XSetFont( x_display, x_wind[w].color[i], x_wind[w].font->fid );
    }
  }

  /* display window */

  XMapWindow( x_display, x_wind[w].wind );
  XFlush( x_display );

  /* wait until window appears */

  XSelectInput( x_display, x_wind[w].wind, ExposureMask );
  XWindowEvent( x_display, x_wind[w].wind, ExposureMask, &report );

  x_clear_window( w );

  return w;
}


static void x_close_window( w )
int w;
{
  int i;
  XDestroyWindow( x_display, x_wind[w].wind );
  XFlush( x_display );
  x_wind[w].in_use = 0;
  for (i=0; i<X_MAX_NB_WINDOWS; i++)
    if (x_wind[i].in_use) return;
  XCloseDisplay( x_display );
  x_initialized = 0;
}


static void x_draw_string( w, col, x, y, str, center )
int w, col, x, y;
char *str;
int center;
{
  int len, width, xx, yy;

  len = 0; while (str[len] != '\0') len++;
  width = XTextWidth( x_wind[w].font, str, len );
  yy = x_wind[w].height - (y /* - x_wind[w].font->max_bounds.ascent/2 */);
  switch (center)
  {
    case 0: xx = x;           break;
    case 1: xx = x - width/2; break;
    case 2: xx = x - width;   break;
  }
  XDrawString( x_display, x_wind[w].wind, x_wind[w].color[col],
               xx, yy, str, len );
  XFlush( x_display );
}


static void x_draw_line( w, col, x1, y1, x2, y2 )
int w, col, x1, y1, x2, y2;
{
  XDrawLine( x_display, x_wind[w].wind, x_wind[w].color[col],
             x1, x_wind[w].height-1-y1, x2, x_wind[w].height-1-y2 );
  XFlush( x_display );
}


static void x_draw_rectangle( w, col, x, y, width, height )
int w, col, x, y, width, height;
{
  XFillRectangle( x_display, x_wind[w].wind, x_wind[w].color[col],
                  x, x_wind[w].height-y-height, width, height );
  XFlush( x_display );
}


SCM_obj X23X23xDopenDwindow( title, width, height )
SCM_obj title, width, height;
{
  SCM_obj result;
  char *mark = local_mark();
  char *t = string_to_c_str( title );
  if (t == NULL)
    result = SCM_false;
  else
  {
    int w = x_open_window( t,
                           SCM_obj_to_int( width ),
                           SCM_obj_to_int( height ) );
    if (w == -1)
      result = SCM_false;
    else
      result = SCM_int_to_obj( w );
  }
  local_release( mark );
  return result;
}


SCM_obj X23X23xDcloseDwindow( win )
SCM_obj win;
{
  int w = SCM_obj_to_int( win );
  if (x_check_wind( w )) return SCM_false;
  x_close_window( w );
  return SCM_true;
}


SCM_obj X23X23xDclearDwindow( win )
SCM_obj win;
{
  int w = SCM_obj_to_int( win );
  if (x_check_wind( w )) return SCM_false;
  x_clear_window( w );
  return SCM_true;
}


SCM_obj X23X23xDdrawDstring( win, col, x, y, str )
SCM_obj win, col, x, y, str;
{
  SCM_obj result;
  char *mark = local_mark();
  char *s = string_to_c_str( str );
  if (s == NULL)
    result = SCM_false;
  else
  {
    int w = SCM_obj_to_int( win );
    if (x_check_wind( w ))
      result = SCM_false;
    else
    {
      x_draw_string( w,
                     SCM_obj_to_int( col ),
                     SCM_obj_to_int( x ),
                     SCM_obj_to_int( y ),
                     s,
                     0 );
      result = SCM_true;
    }
  }
  local_release( mark );
  return result;
}


SCM_obj X23X23xDdrawDline( win, col, x1, y1, x2, y2 )
SCM_obj win, col, x1, y1, x2, y2;
{
  int w = SCM_obj_to_int( win );
  if (x_check_wind( w )) return SCM_false;
  x_draw_line( w,
               SCM_obj_to_int( col ),
               SCM_obj_to_int( x1 ),
               SCM_obj_to_int( y1 ),
               SCM_obj_to_int( x2 ),
               SCM_obj_to_int( y2 ) );
  return SCM_true;
}


SCM_obj X23X23xDdrawDrectangle( win, col, x, y, width, height )
SCM_obj win, col, x, y, width, height;
{
  int w = SCM_obj_to_int( win );
  if (x_check_wind( w )) return SCM_false;
  x_draw_rectangle( w,
                    SCM_obj_to_int( col ),
                    SCM_obj_to_int( x ),
                    SCM_obj_to_int( y ),
                    SCM_obj_to_int( width ),
                    SCM_obj_to_int( height ) );
  return SCM_true;
}


#endif


/*---------------------------------------------------------------------------*/


void ext_init()
{
#ifdef unix_extensions
  DEFINE_C_PROC(X23X23unixDsystem);
  DEFINE_C_PROC(X23X23unixDpipeDopen);
  DEFINE_C_PROC(X23X23unixDpipeDclose);
#endif
#ifdef x_extensions
  DEFINE_C_PROC(X23X23xDopenDwindow);
  DEFINE_C_PROC(X23X23xDcloseDwindow);
  DEFINE_C_PROC(X23X23xDclearDwindow);
  DEFINE_C_PROC(X23X23xDdrawDstring);
  DEFINE_C_PROC(X23X23xDdrawDline);
  DEFINE_C_PROC(X23X23xDdrawDrectangle);
#endif
}


/*---------------------------------------------------------------------------*/
