/* The generational, incremental garbage collector.
 *
 *
 * There are a couple of minor problems that need to be fixed:
 *
 *  o  the return value of ExpandHeap() is sometimes ignored.
 *  o  the percentage displayed at the end of a GC run is sometimes wrong.
 *  o  running out of memory when expanding the heap shouldn't be handled
 *     as a fatal error.  Instead, the garbage collector should clean up
 *     and then invoke UncatchableError() to return control to the Scheme
 *     program.
 */

#include <sys/types.h>
#ifdef HAS_MPROTECT
#  include <sys/mman.h>
#endif
#ifdef SYSCONF_PAGESIZE
#  define link FOO
#  include <unistd.h>
#  undef link
#  ifdef _SC_PAGE_SIZE   /* It has the wrong name in HP-UX */
#    define _SC_PAGESIZE _SC_PAGE_SIZE
#  endif
#endif
#ifdef SIGSEGV_SIGINFO
#  include <siginfo.h>
#  include <ucontext.h>
#endif

/* The following variables may be set from outside the collector to
 * fine-tune some used parameters.
 */

int tuneable_forward_region = 5;   /* fraction of heap pages that are tried
				    * to allocate as forward region when
				    * collecting.
				    */
int tuneable_force_total = 35;     /* % newly allocated during collection
				    * to force total collection
				    */
int tuneable_newly_expand = 25;    /* % of heap newly allocated during
				    * a total collection to force heap
				    * expansion.
				    */
int tuneable_force_expand = 20;    /* % stable to force heap expansion
				    */

/* ------------------------------------------------------------------------

defined in object.h:

typedef unsigned char gcspace_t;  // type used for space and type arrays
typedef unsigned gcptr_t;         // type used for pointers

   ------------------------------------------------------------------------ */

static int percent = 0;

static int inc_collection = 0;

static int incomplete_msg = 0;

static int logical_pages, spanning_pages, physical_pages;

/* pagebase is #defined in object.h if ARRAY_BROKEN is not defined. */

#ifdef ARRAY_BROKEN
  unsigned pagebase;
#endif

static unsigned firstpage, lastpage;

gcspace_t *space;
static gcspace_t *type, *pmap;
static unsigned *link;

static unsigned current_pages, forwarded_pages;
static unsigned protected_pages, allocated_pages;

static unsigned bytes_per_pp, pp_shift;  /* bytes per physical page */
static unsigned hp_per_pp;        /* number of heap pages per physical page */
static unsigned pp_mask;          /* ANDed with a vitual address gives
				   * base address of physical page
				   */
static unsigned hp_per_pp_mask;   /* ANDed with heap page number gives
				   * first page number in the physical
				   * page the heap page belongs to.
				   */
#define SAME_PHYSPAGE(a,b) (((a) & pp_mask) == ((b) & pp_mask))

gcspace_t current_space; /* has to be exported because IS_ALIVE depends on it */

static gcspace_t forward_space, previous_space;
static unsigned current_freepage, current_free;
static unsigned forward_freepage, forward_free;
static unsigned last_forward_freepage;

static Object *current_freep, *forward_freep;

static int scanning = 0; /* set to true if scanning a
			  * physical page is in progress */
static Object *scanpointer;
static Object *scanfirst, *scanlast;
#define IN_SCANREGION(addr) ((Object*)(addr) >= scanfirst \
			     && (Object*)(addr) <= scanlast)
#define IS_SCANNED(addr) ((Object *)(addr) < scanpointer)
#define MAXRESCAN 10
static unsigned rescan[MAXRESCAN];
static int rescanpages = 0;
static int allscan = 0;

static unsigned stable_queue, stable_tail; /* head and tail of the queue
					    * of stable pages */

#define DIRTYENTRIES 20
struct dirty_rec {
    unsigned pages[DIRTYENTRIES];
    struct dirty_rec *next;
};

static struct dirty_rec *dirtylist, *dirtyhead;
static int dirtyentries;

static int ScanCluster ();
static int Scanner ();
static void TerminateGC ();

/*****************************************************************************/

/* PAGEBYTES is defined in object.h */

#define PAGEWORDS      (PAGEBYTES / sizeof (Object))
#define HEAPPAGEMASK   ~(PAGEBYTES-1)

#ifdef POINTER_CONSTANT_HIGH_BITS
#  define GC_PCH_BITS POINTER_CONSTANT_HIGH_BITS
#else
#  define GC_PCH_BITS 0
#endif

#ifdef ALIGN_8BYTE
#  define MAX_OBJECTWORDS       (PAGEWORDS - 1)
#  define NEEDED_PAGES(size)    (((size) + PAGEWORDS) / PAGEWORDS)
#else
#  define MAX_OBJECTWORDS       PAGEWORDS
#  define NEEDED_PAGES(size)    (((size) + PAGEWORDS - 1) / PAGEWORDS)
#endif

#define MAKE_HEADER(words,type)		((type) << VALBITS | (words))
#define HEADER_TO_TYPE(header)		((unsigned)(header) >> VALBITS)
#define HEADER_TO_WORDS(header)		((unsigned)(header) & VALMASK)

/* some conversion stuff. PHYSPAGE converts a logical page number into the
 * start address of the physical page the logical page lies on.
 * If ARRAY_BROKEN is defined, page numbering will start at 0 for the
 * first heap page. Not that this will introduce some extra overhead.
 * Note that PAGE_TO_ADDR(0) == 0 if ARRAY_BROKEN is not defined...
 */

#define OBJ_TO_PPADDR(obj) ((gcptr_t)POINTER((Object)(obj)) & pp_mask)
#define PTR_TO_PPADDR(ptr) ((gcptr_t)(ptr) & pp_mask)
#define ADDR_TO_PAGE(addr) ((((addr) & HEAPPAGEMASK) / PAGEBYTES) - pagebase)
#define PAGE_TO_ADDR(page) (((page) + pagebase) * PAGEBYTES)
#define PHYSPAGE(page)     ((((page) + pagebase) * PAGEBYTES) & pp_mask)

#define UNALLOCATED_PAGE   (gcspace_t)(-2)
#define FREE_PAGE          1

#define OBJECTPAGE         0
#define CONTPAGE           1

#define HEAPPERCENT(x)  (((x) * 100) / logical_pages)

#define IS_CLUSTER(a,b) (SAME_PHYSPAGE (PAGE_TO_ADDR ((a)), \
					PAGE_TO_ADDR ((b))) || \
			 (type[(a)&hp_per_pp_mask] == OBJECTPAGE && \
			  type[((b)&hp_per_pp_mask)+hp_per_pp] == OBJECTPAGE))

/* check whether the (physical) page starting at address addr is protected
 * or not. SET_PROTECT and SET_UNPROTECT are used to set or clear the flag
 * for the page starting at address addr in the pmap array. The job of
 * protecting a page (by calling mprotect) is done in PROTECT/UNPROTECT.
 */

#define PMAP(addr)           *(pmap+(((addr) - PAGE_TO_ADDR(0)) >> pp_shift))

#define IS_PROTECTED(addr)   ( PMAP (addr) )
#define SET_PROTECT(addr)    { PMAP (addr) = 1; protected_pages++; }
#define SET_UNPROTECT(addr)  { PMAP (addr) = 0; protected_pages--; }

#ifdef HAS_MPROTECT
#  ifndef PROT_RW
#    define PROT_RW   (PROT_READ | PROT_WRITE)
#  endif
#  ifndef PROT_NONE
#    define PROT_NONE 0
#  endif
#  define MPROTECT(addr,len,prot) { if (inc_collection) \
                                        mprotect ((caddr_t)(addr), (len), \
						  (prot)); }
#else
#  define PROT_RW
#  define PROT_NONE
#  define MPROTECT(addr,len,prot)
#endif

#define PROTECT(addr)   { if (!IS_PROTECTED (addr)) {                         \
			      if (!scanning) {                                \
			          SET_PROTECT (addr);                         \
				  MPROTECT ((addr), bytes_per_pp, PROT_NONE); \
			      } else                                          \
                                  AddDirty ((addr));                          \
			  } }

#define UNPROTECT(addr) { if (IS_PROTECTED (addr)) {                          \
			      SET_UNPROTECT (addr);                           \
			      MPROTECT ((addr), bytes_per_pp, PROT_RW);       \
			  } }

/*****************************************************************************/

/* the following functions maintain a linked list to remember pages that
 * are "endangered" while scanning goes on. The list elements are arrays,
 * each one containing some page addresses. If an array is filled, a new
 * one is appended to the list (dynamically).
 * An address is not added to the list if the most recently added entry
 * is the same address. It is not necessary to add an address if it is in
 * the list anywhere, but searching would be too time-consuming.
 */

static void SetupDirtyList () {
    dirtylist = (struct dirty_rec *) malloc (sizeof (struct dirty_rec));
    if (dirtylist == (struct dirty_rec *)0)
	Fatal_Error ("SetupDirtyList: unable to allocate memory");
    dirtyhead = dirtylist;
    dirtyentries = 0;
}

static void AddDirty (addr) unsigned addr; {
    struct dirty_rec *p;

    if (dirtylist->pages[(dirtyentries-1) % DIRTYENTRIES] == addr)
	return;
    else
	dirtylist->pages[dirtyentries++ % DIRTYENTRIES] = addr;

    if (dirtyentries % DIRTYENTRIES == 0) {
	p = (struct dirty_rec *) malloc (sizeof (struct dirty_rec));
	if (p == (struct dirty_rec *)0)
	    Fatal_Error ("AddDirty: unable to allocate memory");
	dirtylist->next = p;
	dirtylist = p;
	p->next = (struct dirty_rec *)0;
    }
}

static void ReprotectDirty () {
    int i;

    dirtylist = dirtyhead;
    while (dirtylist) {
	for (i = 0; i < DIRTYENTRIES && dirtyentries--; i++)
	    PROTECT (dirtylist->pages[i]);
	dirtylist = dirtylist->next;
    }

    dirtyentries = 0;
    dirtylist = dirtyhead;
    dirtylist->next = (struct dirty_rec *)0;
}


/* register a page which has been promoted into the scan region by the
 * Visit function. If that page has not been scanned yet, return, else
 * remember the page to be scanned later. If there is not enough space
 * to remember pages, set a flag to rescan the whole scan region.
 */

static void RegisterPage (page) unsigned page; {
    if (allscan)
	return;

    if (IS_SCANNED (PAGE_TO_ADDR (page))) {
	if (rescanpages < MAXRESCAN)
	    rescan[rescanpages++] = page;
	else
	    allscan = 1;
    }
}

/* determine a physical page cluster. Search backward until the beginning
 * of the cluster is found, then forward until the length of the cluster
 * is determined. The first parameter is the address of the first physical
 * page in the cluster, the second one is the length in physical pages.
 * Note that these parameters are value-result parameters !
 */

static void DetermineCluster (addr, len) gcptr_t *addr; int *len; {
    gcptr_t addr1;

    *len = 1;
    while (type[ADDR_TO_PAGE (*addr)] != OBJECTPAGE) {
	*addr -= bytes_per_pp;
	(*len)++;
    }
    addr1 = *addr + ((*len) << pp_shift);

    while (type[ADDR_TO_PAGE (addr1)] != OBJECTPAGE) {
	addr1 += bytes_per_pp;
	(*len)++;
    }
}


/* the following two functions are used to protect or unprotect a page
 * cluster. The first parameter is the address of the first page of the
 * cluster, the second one is the length in physical pages. If the length
 * is 0, DetermineCluster is called to set length accordingly.
 */

static void ProtectCluster (addr, len) gcptr_t addr; {
    if (!len) DetermineCluster (&addr, &len);
    if (len > 1) {
	while (len) {
	    if (!IS_PROTECTED (addr)) {
		MPROTECT (addr, len << pp_shift, PROT_NONE);
		break;
	    }
	    len--;
	    addr += bytes_per_pp;
	}
	while (len--) {
	    if (!IS_PROTECTED (addr)) SET_PROTECT (addr);
	    addr += bytes_per_pp;
	}
    } else {
	if (!IS_PROTECTED (addr)) {
	    MPROTECT (addr, bytes_per_pp, PROT_NONE);
	    SET_PROTECT (addr);
	}
    }
}


static void UnprotectCluster (addr, len) gcptr_t addr; {
    if (!len) DetermineCluster (&addr, &len);
    MPROTECT (addr, len << pp_shift, PROT_RW);
    while (len--) {
	if (IS_PROTECTED (addr)) SET_UNPROTECT (addr);
	addr += bytes_per_pp;
    }
}


/* add one page to the stable set queue */

static void AddQueue (page) unsigned page; {

    if (stable_queue != (unsigned)(-1))
	link[stable_tail] = page;
    else
	stable_queue = page;
    link[page] = (unsigned)(-1);
    stable_tail = page;
}


/* the following function promotes all heap pages in the stable set queue
 * into current space. After this, there are no more forwarded pages in the
 * heap.
 */

static void PromoteStableQueue () {
    Object *p;
    int pcount, size;
    unsigned start;
    
    while (stable_queue != (unsigned)(-1)) {
	p = PAGE_TO_OBJ (stable_queue);
#ifdef ALIGN_8BYTE
	p++;
#endif
	size = HEADER_TO_WORDS (*p);
	pcount = NEEDED_PAGES (size);
	
	start = stable_queue;
	while (pcount--)
	    space[start++] = current_space;
	stable_queue = link[stable_queue];
    }
    current_pages = allocated_pages;
    forwarded_pages = 0;
}

/* calculate the logarithm (base 2) for arguments == 2**n
 */

static Logbase2 (psize) unsigned psize; {
    int shift = 0;

    if (psize & 0xffff0000) shift += 16;
    if (psize & 0xff00ff00) shift += 8;
    if (psize & 0xf0f0f0f0) shift += 4;
    if (psize & 0xcccccccc) shift += 2;
    if (psize & 0xaaaaaaaa) shift += 1;

    return (shift);
}

/* return next heap page number, wrap around at the end of the heap. */

static int next (page) unsigned page; {
    return ((page < lastpage) ? page+1 : firstpage);
}

/*****************************************************************************/

#ifdef MPROTECT_MMAP

static char *heapmalloc (s) {
    char *ret = mmap (0, s, PROT_READ|PROT_WRITE, MAP_ANON, -1, 0);

    if (ret == (char*)-1)
	ret = 0;

    return ret;
}

#else

#  define heapmalloc(size)  (char *)malloc ((size))

#endif

/*
 * make a heap of size kilobytes. It is divided into heappages of
 * PAGEBYTES byte and is aligned at a physical page boundary. The
 * heapsize is rounded up to the nearest multiple of the physical
 * pagesize.
 */

Make_Heap (size) {
    unsigned heapsize = size * 1024;
    char *heap;
    unsigned i;
    
#ifdef HAS_MPROTECT
    InstallHandler ();
#endif
    
    /* calculate number of logical heappages and of used physical pages.
     * First, round up to the nearest multiple of the physical pagesize,
     * then calculate the resulting number of heap pages.
     */
    
#ifdef SYSCONF_PAGESIZE
    if ((bytes_per_pp = sysconf(_SC_PAGESIZE)) == -1)
	Fatal_Error ("sysconf(_SC_PAGESIZE) failed; can't get pagesize");
#else
#ifdef GETPAGESIZE
    bytes_per_pp = getpagesize();
#else
#   ifdef HAS_MPROTECT
#       include "mprotect requires getpagesize or sysconf_pagesize"
#   else
        bytes_per_pp = 4096;
#   endif
#endif
#endif
    physical_pages = (heapsize+bytes_per_pp-1)/bytes_per_pp;
    hp_per_pp = bytes_per_pp / PAGEBYTES;
    hp_per_pp_mask = ~(hp_per_pp - 1);
    logical_pages = spanning_pages = physical_pages * hp_per_pp;
    pp_mask = ~(bytes_per_pp-1);
    pp_shift = Logbase2 (bytes_per_pp);

    heap = heapmalloc (logical_pages*PAGEBYTES+bytes_per_pp-1);

    if (heap == NULL)
	Fatal_Error ("cannot allocate heap (%u KBytes)", size);
    
    if ((((gcptr_t)heap+heapsize-1) & ~VALMASK) != GC_PCH_BITS)
	Fatal_Error ("heap too large, %u KBytes max",
		     (VALMASK - ((gcptr_t)heap & ~GC_PCH_BITS) >> 10));
    
    /* Align heap at a memory page boundary */
    
    if ((gcptr_t)heap & (bytes_per_pp-1))
	heap = (char*)(((gcptr_t)heap+bytes_per_pp) & ~(bytes_per_pp-1));

#ifdef ARRAY_BROKEN    
    pagebase = ((gcptr_t)POINTER ((Object)(heap))) / PAGEBYTES;
#endif
    firstpage = OBJ_TO_PAGE ((Object)heap);
    lastpage = firstpage+logical_pages-1;
    
    space = (gcspace_t *)malloc (logical_pages*sizeof (gcspace_t));
    type = (gcspace_t *)malloc ((logical_pages + 1)*sizeof (gcspace_t));
    pmap = (gcspace_t *)malloc (physical_pages*sizeof (gcspace_t));
    link = (unsigned *)malloc (logical_pages*sizeof (unsigned));
    if (!space || !type || !pmap || !link) {
	free (heap);
	if (space) free ((char*)space);
	if (type) free ((char*)type);
	if (pmap) free ((char*)pmap);
	if (link) free ((char*)link);
	Fatal_Error ("cannot allocate heap maps");
    }

    bzero ((char *)pmap, physical_pages*sizeof (gcspace_t));
    space -= firstpage; /* to index the arrays with the heap page number */
    type -= firstpage;
    type[lastpage+1] = OBJECTPAGE;
    link -= firstpage;
#ifndef ARRAY_BROKEN
    pmap -= (PAGE_TO_ADDR (firstpage) >> pp_shift);
#endif

    for (i = firstpage; i <= lastpage; i++)
	space[i] = FREE_PAGE;

    allocated_pages = 0;
    forwarded_pages = 0;
    current_pages = 0;
    protected_pages = 0;
    stable_queue = (unsigned)(-1);
    SetupDirtyList ();

    current_space = forward_space = previous_space = 3;
    current_freepage = firstpage; current_free = 0;
}


/*
 * increment the heap by 1024 kB.
 */

static int ExpandHeap () {
    int increment = (1024 * 1024 + bytes_per_pp - 1) / bytes_per_pp;
    int incpages = increment * hp_per_pp;
    unsigned heapinc = incpages * PAGEBYTES;
    unsigned new_first, inc_first;
    unsigned new_last, inc_last;
    int new_logpages, new_physpages;
    int new_spanpages;
    gcptr_t addr;
    gcspace_t *new_space, *new_type, *new_pmap;
    unsigned *new_link, i;
    char *heap;
#ifdef ARRAY_BROKEN
    unsigned new_pagebase, offset;
    unsigned new_firstpage, new_lastpage;
#else
#   define offset 0
#endif
    
    heap = heapmalloc (heapinc+bytes_per_pp-1);

    if (heap == NULL) {
	if (Truep (Var_Get (V_Garbage_Collect_Notifyp))) {
 	    Format (Standard_Output_Port, "[Heap expansion failed]~%",
 		    25, 0, (Object *)0);
	    (void)fflush (stdout);
	    incomplete_msg = 0;
	}
	return (0);
    }
    
    if ((((gcptr_t)heap+heapinc-1) & ~VALMASK) != GC_PCH_BITS) {
	if (Truep (Var_Get (V_Garbage_Collect_Notifyp))) {
 	    Object a = Make_Fixnum ((VALMASK -
 				     ((gcptr_t)heap & ~GC_PCH_BITS)) >> 10);
 	    
 	    Format (Standard_Output_Port, "[Heap expansion failed, max ~sK]~%",
 		    34, 1, &a);
	    (void)fflush (stdout);
	    incomplete_msg = 0;
	}
	return (0);
    }
    
    /* Align heap at a memory page boundary */
    
    if ((gcptr_t)heap & (bytes_per_pp-1))
	heap = (char*)(((gcptr_t)heap+bytes_per_pp) & ~(bytes_per_pp-1));
    
    new_first = firstpage;
    new_last = lastpage;

#ifdef ARRAY_BROKEN
    new_pagebase = ((gcptr_t)POINTER ((Object)(heap))) / PAGEBYTES;
    inc_first = 0; /* = OBJ_TO_PAGE ((Object)heap) - new_pagebase */

    new_firstpage = (pagebase > new_pagebase)
	? new_pagebase : pagebase;

    new_lastpage = (pagebase > new_pagebase)
	? pagebase + lastpage
	: new_pagebase + incpages - 1;

    offset = pagebase - new_firstpage;
#else
    inc_first = OBJ_TO_PAGE ((Object)heap);
#endif

    inc_last = inc_first+incpages-1;
    if (inc_last > lastpage)
	new_last = inc_last;
    if (inc_first < firstpage)
	new_first = inc_first;
    new_logpages = logical_pages+incpages;
#ifdef ARRAY_BROKEN
    new_spanpages = new_lastpage-new_firstpage+1;
    new_last = new_spanpages-1;
#else
    new_spanpages = new_last-new_first+1;
#endif
    new_physpages = new_spanpages / hp_per_pp;
    
    new_space = (gcspace_t *)malloc (new_spanpages*sizeof (gcspace_t));
    new_type = (gcspace_t *)malloc ((new_spanpages + 1)*sizeof (gcspace_t));
    new_pmap = (gcspace_t *)malloc (new_physpages*sizeof (gcspace_t));
    new_link = (unsigned *)malloc (new_spanpages*sizeof (unsigned));
    if (!new_space || !new_type || !new_pmap || !new_link) {
	free (heap);
	if (new_space) free ((char*)new_space);
	if (new_type) free ((char*)new_type);
	if (new_pmap) free ((char*)new_pmap);
	if (new_link) free ((char*)new_link);
	if (Truep (Var_Get (V_Garbage_Collect_Notifyp))) {
 	    Format (Standard_Output_Port, "[Heap expansion failed]~%",
 		    25, 0, (Object *)0);
	    (void)fflush (stdout);
	    incomplete_msg = 0;
	}
	return (0);
    }

    /* new_first will be 0 if ARRAY_BROKEN is defined. */
    
    new_space -= new_first;
    new_type -= new_first;
    new_link -= new_first;
    bzero ((char*)new_pmap, new_physpages * sizeof (gcspace_t));
#ifndef ARRAY_BROKEN
    new_pmap -= (PHYSPAGE (new_first) >> pp_shift);
#endif

    for (i = firstpage; i <= lastpage; i++) {
	new_link[i + offset] = link[i] + offset;
	new_type[i + offset] = type[i];
    }
    
    for (addr = PAGE_TO_ADDR (firstpage); addr < PAGE_TO_ADDR (lastpage);
	 addr += bytes_per_pp)

	*(new_pmap+(((addr - PAGE_TO_ADDR(0)) >> pp_shift) + offset)) =
	    IS_PROTECTED (addr);

#ifdef ARRAY_BROKEN
    for (i = 0; i < new_spanpages; i++) new_space[i] = UNALLOCATED_PAGE;
    for (i = firstpage; i <= lastpage; i++) new_space[i+offset] = space[i];
    offset = offset ? 0 : new_pagebase - pagebase;
    for (i = offset; i < offset + inc_last; i++) new_space[i] = FREE_PAGE;
    new_type[new_spanpages] = OBJECTPAGE;
#else
    for (i = new_first; i < firstpage; i++) new_space[i] = UNALLOCATED_PAGE;
    for (i = firstpage; i <= lastpage; i++) new_space[i] = space[i];
    for (i = lastpage+1; i < new_last; i++) new_space[i] = UNALLOCATED_PAGE;
    for (i = inc_first; i <= inc_last; i++) new_space[i] = FREE_PAGE;
    new_type[new_last+1] = OBJECTPAGE;
#endif
    
    current_freepage += offset;
    forward_freepage += offset;
    last_forward_freepage += offset;

    free ((char*)(link+firstpage));
    free ((char*)(type+firstpage));
    free ((char*)(space+firstpage));
#ifndef ARRAY_BROKEN
    free ((char*)(pmap+(PAGE_TO_ADDR (firstpage) >> pp_shift)));
#else
    free ((char*)pmap);
#endif

    link = new_link;
    type = new_type;
    space = new_space;
    pmap = new_pmap;

    firstpage = new_first;
    lastpage = new_last;
    logical_pages = new_logpages;
    spanning_pages = new_spanpages;
    physical_pages = new_physpages;
    
    if (Truep (Var_Get (V_Garbage_Collect_Notifyp))) {
 	Object a = Make_Fixnum ((logical_pages * PAGEBYTES) >> 10);
 	
 	Format (Standard_Output_Port, "[Heap expanded to ~sK]~%", 26, 1, &a);
	(void)fflush (stdout);
	incomplete_msg = 0;
    }
    
    return (1);
}


/* allocate new logical heappages. npg is the number of pages to allocate.
 * If there is not enough space left, the heap will be expanded if possible.
 * The new page is allocated in current space.
 */

static int ProtectedInRegion (start, npages) unsigned start; {
    gcptr_t beginpage = PHYSPAGE (start);
    gcptr_t endpage = PHYSPAGE (start+npages-1);

    do {
	if (IS_PROTECTED (beginpage))
	    return (1);
	beginpage += bytes_per_pp;
    } while (beginpage <= endpage);

    return (0);
}

static void AllocPage (npg) {
    unsigned first_freepage;    /* first free heap page */
    int cont_free;              /* contiguous free pages */
    unsigned n, p;
    
    if (current_space != forward_space) {
	Scanner (1);
	if (!protected_pages)
	    TerminateGC ();
    } else {
	if (inc_collection) {
	    if (allocated_pages+npg >= logical_pages/3)
		P_Collect_Incremental ();
	} else {
	    if (allocated_pages+npg >= logical_pages/2)
		P_Collect ();
	}
    }
    
    /* now look for a cluster of npg free pages. cont_free counts the
     * number of free pages found, first_freepage is the number of the
     * first free heap page in the cluster.
     */
    
    for (p = spanning_pages, cont_free = 0; p; p--) {
	if (space[current_freepage] < previous_space
	    && !STABLE (current_freepage)) {
	    if (!(cont_free++)) {
		if (IS_CLUSTER (current_freepage, current_freepage+npg-1))
		    first_freepage = current_freepage;
		else {
		    current_freepage = next (current_freepage -
					     current_freepage % hp_per_pp +
					     hp_per_pp-1);
		    cont_free = 0;
		    continue;
		}
	    }
	    
	    if (cont_free == npg) {
		space[first_freepage] = current_space;
		type[first_freepage] = OBJECTPAGE;
		for (n = 1; n < npg; n++) {
		    space[first_freepage+n] = current_space;
		    type[first_freepage+n] = CONTPAGE;
		}
		current_freep = PAGE_TO_OBJ (first_freepage);
		current_free = npg*PAGEWORDS;
		current_pages += npg;
		allocated_pages += npg;
		current_freepage = next (first_freepage+npg-1);
		if (ProtectedInRegion (first_freepage, npg))
		    ScanCluster (PHYSPAGE (first_freepage));
		return;
	    } else {
		current_freepage = next (current_freepage);
		if (current_freepage == firstpage) cont_free = 0;
	    }
	} else {
	    current_freepage = next (current_freepage);
	    cont_free = 0;
	}
    }
    
    /* no space available, try to expand heap */
    
    if (ExpandHeap ()) {
	AllocPage (npg);
	return;
    }
    
    Fatal_Error ("Unable to allocate %u bytes in heap", npg*PAGEBYTES);
    
    /*NOTREACHED*/
}


/* allocate an object in the heap. size is the size of the new object
 * in bytes, type describes the object's type (see object.h), and konst
 * determines whether the object is immutable.
 */

Object Alloc_Object (size, type, konst) {
    register Object obj;
    register s = ((size+3) >> 2) + 1; /* size in words */
    int big = 0;

    if (GC_Debug) {
	if (inc_collection)
	    P_Collect_Incremental ();
	else
	    P_Collect ();
    }
    
    /* if there is not enough space left on the current page, discard
     * the left space and allocate a new page. Space is discarded by
     * writing a T_Freespace object.
     */
    
    if (s > current_free) {
	if (current_free) {
	    *current_freep = MAKE_HEADER (current_free, T_Freespace);
	    current_free = 0;
	}
	
	/* If we are about to allocate an object bigger than one heap page,
	 * set a flag. The space behind big objects is discarded, see below.
	 */
	
#ifdef ALIGN_8BYTE
	if (s < PAGEWORDS-1)
	    AllocPage (1);
	else {
	    AllocPage ((s+PAGEWORDS)/PAGEWORDS);
	    big = 1;
	}
	*current_freep++ = MAKE_HEADER (1, T_Align_8Byte);
	current_free--;
#else
	if (s < PAGEWORDS)
	    AllocPage (1);
	else {
	    AllocPage ((s+PAGEWORDS-1)/PAGEWORDS);
	    big = 1;
	}
#endif
    }
    
    /* now write a header for the object into the heap and update the
     * pointer to the next free location and the counter of free words
     * in the current heappage.
     */
    
    *current_freep++ = MAKE_HEADER (s, type);
    SET (obj, type, current_freep);
    if (big)
	current_freep = (Object*)0, current_free = 0;
    else
	current_freep += (s-1), current_free -= s;
#ifdef ALIGN_8BYTE
    if (!((gcptr_t)current_freep & 7) && current_free) {
	*current_freep++ = MAKE_HEADER (1, T_Align_8Byte);
	current_free--;
    }
#endif
    if (type == T_Control_Point)
	CONTROL(obj)->reloc = 0;
    
    if (konst) SETCONST (obj);
    return (obj);
}

Object Alloc_Const_Object (size, type) {
    return (Alloc_Object (size, type, 1));
}


/* allocate a page in forward space. If there is no space left, the heap
 * is expanded. The argument prevents allocation of a heap page which lies
 * on the same physical page the referenced object lies on.
 */

static void AllocForwardPage (bad) Object bad; {
    Object *badaddr = (Object *)POINTER (bad);
    int whole_heap = spanning_pages;
    unsigned tpage;
    
    while (whole_heap--) {
	if (space[forward_freepage] < previous_space
	    && !STABLE (forward_freepage)
	    && !SAME_PHYSPAGE ((gcptr_t)badaddr,
		    PAGE_TO_ADDR (forward_freepage))
	    && !IN_SCANREGION (PAGE_TO_ADDR (forward_freepage))) {
	    
	    allocated_pages++;
	    forwarded_pages++;
	    space[forward_freepage] = forward_space;
	    type[forward_freepage] = OBJECTPAGE;
	    forward_freep = PAGE_TO_OBJ (forward_freepage);
	    forward_free = PAGEWORDS;
	    AddQueue (forward_freepage);
	    
	    tpage = last_forward_freepage;
	    last_forward_freepage = next (forward_freepage);
	    forward_freepage = tpage;
	    return;
	} else {
	    forward_freepage = next (forward_freepage);
	}
    }
    
    if (ExpandHeap ()) {
	AllocForwardPage (bad);
	return;
    }
    
    Fatal_Error ("unable to allocate forward page in %u KBytes heap",
		 (logical_pages * PAGEBYTES) >> 10);
    
    /*NOTREACHED*/
}


/* Visit an object and move it into forward space.  The forwarded
 * object must be protected because it is to be scanned later.
 */

Visit (cp) register Object *cp; {
    register unsigned page = OBJ_TO_PAGE (*cp);
    register Object *obj_ptr = (Object *)POINTER (*cp);
    int tag = TYPE (*cp);
    int objwords, objpages, pcount;
    gcptr_t ffreep, pageaddr = 0;
    short outside;
    
    /* if the Visit function is called via the REVIVE_OBJ macro and we are
     * not inside an incremental collection, exit immediately.
     */

    if (current_space == forward_space)
	return;

    if (page < firstpage || page > lastpage || STABLE (page)
	|| space[page] == current_space  || space[page] == UNALLOCATED_PAGE
	|| !Types[tag].haspointer)
	return;

    if (space[page] != previous_space) {


	char buf[100];
	sprintf (buf, "Visit: object not in prev space at 0x%x ('%s')",
		     obj_ptr, Types[tag].name);
	Panic (buf);
    }
    
    if (!IN_SCANREGION (obj_ptr) && IS_PROTECTED ((gcptr_t)obj_ptr)) {
	pageaddr = OBJ_TO_PPADDR (*cp);
	UNPROTECT (pageaddr);
    }
    
    if (WAS_FORWARDED (*cp)) {
	if (pageaddr != 0)
	    PROTECT (pageaddr);
	*cp = MAKEOBJ (tag, *obj_ptr);
	return;
    }

    ffreep = PTR_TO_PPADDR (forward_freep);    
    outside = !IN_SCANREGION (forward_freep);
    objwords = HEADER_TO_WORDS (*(obj_ptr - 1));
    if (objwords >= forward_free) {
#ifdef ALIGN_8BYTE
	if (objwords >= PAGEWORDS - 1) {
	    objpages = (objwords + PAGEWORDS) / PAGEWORDS;
#else
	if (objwords >= PAGEWORDS) {
	    objpages = (objwords + PAGEWORDS - 1) / PAGEWORDS;
#endif
	    forwarded_pages += objpages;
	    for (pcount = 0; pcount < objpages; pcount++)
		space[page + pcount] = forward_space;
	    AddQueue (page);
	    if (IN_SCANREGION (PAGE_TO_ADDR (page)))
		RegisterPage (page);
	    else
		ProtectCluster (PHYSPAGE (page), 0);
	    
	    if (pageaddr != 0)
		PROTECT (pageaddr);
		
	    return;
	}
	    
	if (forward_free) {
	    if (outside && IS_PROTECTED (ffreep)
		&& !SAME_PHYSPAGE ((gcptr_t)obj_ptr, ffreep)) {
		
		UNPROTECT (ffreep);
		*forward_freep = MAKE_HEADER (forward_free, T_Freespace);
		forward_free = 0;
		PROTECT (ffreep);
	    } else {
		*forward_freep = MAKE_HEADER (forward_free, T_Freespace);
		forward_free = 0;
	    }
	}
	    
	AllocForwardPage (*cp);
	outside = !IN_SCANREGION (forward_freep);
	ffreep = PTR_TO_PPADDR (forward_freep); /* re-set ffreep ! */
#ifdef ALIGN_8BYTE
	if (outside && IS_PROTECTED (ffreep))
	    UNPROTECT (ffreep);
	*forward_freep++ = MAKE_HEADER (1, T_Align_8Byte);
	forward_free--;
	goto do_forward;
#endif
    }
	
    if (outside && IS_PROTECTED (ffreep))
	UNPROTECT (ffreep);

#ifdef ALIGN_8BYTE
do_forward:
#endif
    if (tag == T_Control_Point)
	CONTROL (POINTER (*cp))->reloc = (forward_freep + 1) - obj_ptr;
    *forward_freep++ = MAKE_HEADER (objwords, tag);
    bcopy ((char*)obj_ptr, (char*)forward_freep, (objwords-1)*sizeof (int));
    SET (*obj_ptr, T_Broken_Heart, forward_freep);
    *cp = MAKEOBJ (tag, forward_freep);
    forward_freep += (objwords - 1);
    forward_free -= objwords;
	
#ifdef ALIGN_8BYTE
    if (!((gcptr_t)forward_freep & 7) && forward_free) {
	*forward_freep++ = MAKE_HEADER (1, T_Align_8Byte);
	forward_free--;
    }
#endif
	
    if (outside)
	PROTECT (ffreep);
	
    if (pageaddr != 0)
	PROTECT (pageaddr);
	
    return;
}
    
    
/* Scan a page and visit all objects referenced by objects lying on the
 * page. This will possibly forward the referenced objects.
 */
    
static void ScanPage (currentp, nextcp) Object *currentp, *nextcp; {
    Object *cp = currentp;
    int len, m, n, t;
    
    while (cp < nextcp && (cp != forward_freep || forward_free == 0)) {
	t = HEADER_TO_TYPE (*cp);
	len = HEADER_TO_WORDS (*cp);
	cp++;
	
	/* cp now points to the real Scheme object in the heap. t denotes
	 * the type of the object, len its length inclusive header in
	 * words.
	 */
	
	if (Types[t].visit != NOFUNC)
	    switch (t) {
	    case T_Symbol:
		Visit (&SYMBOL((Object)cp)->next);
		Visit (&SYMBOL((Object)cp)->name);
		Visit (&SYMBOL((Object)cp)->value);
		Visit (&SYMBOL((Object)cp)->plist);
		break;
		
	    case T_Pair:
	    case T_Environment:
		Visit (&PAIR((Object)cp)->car);
		Visit (&PAIR((Object)cp)->cdr);
		break;
		
	    case T_Vector:
		for (n = 0, m = VECTOR((Object)cp)->size; n < m; n++ )
		    Visit (&VECTOR((Object)cp)->data[n]);
		break;
		
	    case T_Compound:
		Visit (&COMPOUND((Object)cp)->closure);
		Visit (&COMPOUND((Object)cp)->env);
		Visit (&COMPOUND((Object)cp)->name);
		break;
		
	    case T_Control_Point:
		(CONTROL((Object)cp)->delta) += CONTROL((Object)cp)->reloc;
		
#ifdef USE_ALLOCA
		Visit_GC_List (CONTROL((Object)cp)->gclist, CONTROL((Object)cp)->delta);
#else
		Visit (&CONTROL((Object)cp)->gcsave);
#endif
		Visit_Wind (CONTROL((Object)cp)->firstwind,
			    (CONTROL((Object)cp)->delta) );
		
		Visit (&CONTROL((Object)cp)->env);
		break;
		
	    case T_Promise:
		Visit (&PROMISE((Object)cp)->env);
		Visit (&PROMISE((Object)cp)->thunk);
		break;
		
	    case T_Port:
		Visit (&PORT((Object)cp)->name);
		break;
		
	    case T_Autoload:
		Visit (&AUTOLOAD((Object)cp)->files);
		Visit (&AUTOLOAD((Object)cp)->env);
		break;
		
	    case T_Macro:
		Visit (&MACRO((Object)cp)->body);
		Visit (&MACRO((Object)cp)->name);
		break;
	    default: {
		Object o;
		SET(o, t, cp);
		(*Types[t].visit) (&o, Visit);
	    }
	    }		
	cp += (len - 1);
    }
}
    
    
/* rescan all pages remembered by the RegisterPage function. */
    
static void RescanPages () {
    register Object *cp;
    register int i;
    int pages = rescanpages;
	
    rescanpages = 0;
    for (i = 0; i < pages; i++) {
	cp = PAGE_TO_OBJ (rescan[i]);
#ifdef ALIGN_8BYTE
	ScanPage (cp + 1, cp + PAGEWORDS);
#else
	ScanPage (cp, cp + PAGEWORDS);
#endif
    }
}
    
static int ScanCluster (addr) gcptr_t addr; {
    register unsigned page, lastpage;
    int npages = 0;
	
    scanning = 1;
    DetermineCluster (&addr, &npages);
    scanfirst = (Object *)addr;
    scanlast = (Object *)(addr + (npages << pp_shift) - sizeof (Object));
    UnprotectCluster (scanfirst, npages);

 rescan_cluster:
    lastpage = ADDR_TO_PAGE ((gcptr_t)scanlast);
    for (page = ADDR_TO_PAGE ((gcptr_t)scanfirst); page <= lastpage; page++) {
	if (STABLE (page) && type[page] == OBJECTPAGE) {
	    scanpointer = PAGE_TO_OBJ (page);
#ifdef ALIGN_8BYTE
	    ScanPage (scanpointer + 1, scanpointer + PAGEWORDS);
#endif
	    ScanPage (scanpointer, scanpointer + PAGEWORDS);
	}
    }

    while (rescanpages) {
	if (allscan) {
	    allscan = 0;
	    goto rescan_cluster;
	} else
	    RescanPages ();
    }

    scanfirst = (Object *)0;
    scanlast = (Object *)0;
    scanning = 0;
    ReprotectDirty ();

    return (npages); /* return number of scanned pages */
}


static int Scanner (npages) {
    register gcptr_t addr, lastaddr;
    int spages;
    int scanned = 0;

    while (npages > 0 && protected_pages) {
	lastaddr = PAGE_TO_ADDR (lastpage);
	for (addr = PAGE_TO_ADDR(firstpage); addr < lastaddr && npages > 0;
	     addr += bytes_per_pp) {

	    if (IS_PROTECTED (addr)) {
		if (space[ADDR_TO_PAGE (addr)] == UNALLOCATED_PAGE)
		    Panic ("Scanner: found incorrect heap page");
		spages = ScanCluster (addr);
		scanned += spages;
		npages -= spages;
	    }
	}
    }

    scanfirst = (Object *)0;
    scanlast = scanfirst;

    return (scanned);
}

#ifdef HAS_MPROTECT
/* the following function handles a page fault. If the fault was caused
 * by the mutator and incremental colletion is enabled, this will result
 * in scanning the physical page the fault occured on.
 */

#ifdef SIGSEGV_SIGCONTEXT

static void PagefaultHandler (sig, code, scp) struct sigcontext *scp; {
    char *addr = (char *)(scp->sc_badvaddr);

#else
#ifdef SIGSEGV_AIX

static void PagefaultHandler (sig, code, scp) struct sigcontext *scp; {
    char *addr = scp->sc_jmpbuf.jmp_context.except[3];

#else
#ifdef SIGSEGV_SIGINFO

static void PagefaultHandler (sig, sip, ucp) siginfo_t *sip; ucontext_t *ucp; {
    char *addr;

#else
#ifdef SIGSEGV_ARG4

static void PagefaultHandler (sig, code, scp, addr) struct sigcontext *scp;
    char *addr; {

#else
#  include "HAS_MPROTECT defined, but missing SIGSEGV_xxx"
#endif
#endif
#endif
#endif

    unsigned page;
    gcptr_t ppage;
    char *errmsg = 0;

#ifdef SIGSEGV_AIX
    if (scp->sc_jmpbuf.jmp_context.except[0] != addr)
	Panic("except");
#endif

#ifdef SIGSEGV_SIGINFO
    if (sip == 0)
	Fatal_Error ("SIGSEGV handler got called with zero siginfo_t");
    addr = sip->si_addr;
#endif

    ppage = PTR_TO_PPADDR(addr);
    page = ADDR_TO_PAGE((gcptr_t)addr);

    if (!inc_collection)
	errmsg = "SIGSEGV signal received";
    else if (current_space == forward_space)
	errmsg = "SIGSEGV signal received while not garbage collecting";
    else if (page < firstpage || page > lastpage)
	errmsg = "SIGSEV signal received; address outside of heap";
    if (errmsg) {
	fprintf (stderr, "\n[%s]\n", errmsg);
	abort ();
    }

    GC_In_Progress = 1;
    ScanCluster (ppage);
    GC_In_Progress = 0;
#ifdef SIGSEGV_AIX
    InstallHandler ();
#endif
    return;
}

InstallHandler () {
#ifdef SIGSEGV_SIGINFO
    struct sigaction sact;
    sigset_t mask;

    sact.sa_handler = PagefaultHandler;
    sigemptyset (&mask);
    sact.sa_mask = mask;
    sact.sa_flags = SA_SIGINFO;
    if (sigaction(SIGSEGV, &sact, 0) == -1) {
	perror("sigaction"); exit (1);
    }
#else
    (void)signal (SIGSEGV, PagefaultHandler);
#endif
}
#endif


static void TerminateGC () {
    int save_force_total;

    forward_space = current_space;
    previous_space = current_space;

    if (protected_pages)
	Panic ("TerminateGC: protected pages after collection");

    allocated_pages = current_pages + forwarded_pages;
    current_pages = 0;

    if (forward_free) {
	*forward_freep = MAKE_HEADER (forward_free, T_Freespace);
	forward_free = 0;
    }
    forward_freep = (Object *)0;

    Call_After_GC();
    GC_In_Progress = 0;
    Enable_Interrupts;

    if (Truep (Var_Get (V_Garbage_Collect_Notifyp)) && !GC_Debug) {
 	int foo = percent - HEAPPERCENT (allocated_pages);
 	Object bar = Make_Fixnum (foo);

	if (!incomplete_msg)
	    Format (Standard_Output_Port, "[", 1, 0, (Object *)0);

	if (foo >= 0)
	    Format (Standard_Output_Port, "~s% reclaimed]~%", 16, 1, &bar);
	else
	    Format (Standard_Output_Port, "finished]~%", 11, 0, (Object *)0);
	(void)fflush (stdout);
	incomplete_msg = 0;
    }

    if (HEAPPERCENT (allocated_pages) >= tuneable_force_total) {
	PromoteStableQueue ();
	save_force_total = tuneable_force_total;
	tuneable_force_total = 100;
	if (inc_collection)
	    P_Collect_Incremental ();
	else
	    P_Collect ();
	tuneable_force_total = save_force_total;
	if (HEAPPERCENT (allocated_pages) >= tuneable_newly_expand)
	    ExpandHeap ();
    }
}


static void Finish_Collection () {
    register gcptr_t addr;

    do {
	for (addr = PAGE_TO_ADDR(firstpage);
	     addr < PAGE_TO_ADDR(lastpage);
	     addr += bytes_per_pp) {

	    if (IS_PROTECTED (addr)) {
		ScanCluster (addr);
		if (protected_pages == 0) TerminateGC ();
	    }
	}
    } while (protected_pages);

    return;
}


static void General_Collect (initiate) {
    unsigned fpage, free_fpages, i;
    unsigned page;
    unsigned fregion_pages;

    if (!Interpreter_Initialized)
	Fatal_Error ("Out of heap space (increase heap size)");

    if (current_space != forward_space && !inc_collection) {
	Format (Standard_Output_Port, "GC while GC in progress~%",
		25, 0, (Object*)0);
	return;
    }

    /* Call all user-registered functions to be executed just before GC. */

    Disable_Interrupts;
    GC_In_Progress = 1;
    Call_Before_GC();
    percent = HEAPPERCENT (allocated_pages);

    if (Truep (Var_Get (V_Garbage_Collect_Notifyp)) && !GC_Debug) {
	if (initiate) {
	    Format (Standard_Output_Port, "[Garbage collecting...]~%",
		    25, 0, (Object *)0);
	    incomplete_msg = 0;
	} else {
	    Format (Standard_Output_Port, "[Garbage collecting... ",
		    23, 0, (Object *)0);
	    incomplete_msg = 1;
	}
        (void)fflush (stdout);
    }

    if (GC_Debug) {
	printf ("."); (void)fflush (stdout);
    }

    /* discard any remaining portion of the current heap page */

    if (current_free) {
        *current_freep = MAKE_HEADER (current_free, T_Freespace);
        current_free = 0;
    }

    /* partition regions for forwarded and newly-allocated objects. Then
     * advance the current free pointer so that - if possible - there will
     * be RESERVEDPAGES free heap pages in the forward region.
     */

    forward_freepage = current_freepage;
    last_forward_freepage = forward_freepage;

    current_freep = PAGE_TO_OBJ (current_freepage);
    forward_freep = current_freep;

    fpage = forward_freepage;
    free_fpages = 0;
    fregion_pages = logical_pages / tuneable_forward_region;

    for (i = 0; free_fpages <= fregion_pages && i < spanning_pages; i++) {
        if (space[fpage] != current_space && !STABLE (fpage))
	    free_fpages++;
        fpage = next (fpage);
    }
    current_freep = (Object *)PHYSPAGE (fpage);
    current_freepage = OBJ_TO_PAGE (current_freep);

    /* advance spaces. Then forward all objects directly accessible
     * via the global GC lists and the WIND list.
     */

    current_pages = 0;
    forward_space = current_space + 1;
    current_space = current_space + 2;

    Visit_GC_List (Global_GC_Obj, 0);
    Visit_GC_List (GC_List, 0);
    Visit_Wind (First_Wind, 0);

    /* If collecting in a non-incremental manner, scan all heap pages which
     * have been protected, else check whether to expand the heap because
     * the stable set has grown too big.
     */
 
    page = stable_queue;
    while (page != (unsigned)(-1)) {
        ProtectCluster (PHYSPAGE (page), 0);
        page = link[page];
    }

    if (!initiate) {
	Finish_Collection ();
    } else 
	if (HEAPPERCENT (forwarded_pages) > tuneable_force_expand)
            ExpandHeap();

    GC_In_Progress = 0;
    return;
}


Object P_Collect_Incremental () {
    /* if already collecting, scan a few pages and return */

    if (!inc_collection) {
	if (current_space == forward_space)
	    Primitive_Error ("incremental garbage collection not enabled");
	else {
	    inc_collection = 1;
	    Finish_Collection ();
	    inc_collection = 0;
	    return (True);
	}
    } else {
	if (current_space != forward_space) {
	    Scanner (1);
	    GC_In_Progress = 0;
	    if (protected_pages == 0)
		TerminateGC ();
	    return (protected_pages ? False : True);
	} else {
	    General_Collect (1);
	    return (False);
	}
    }
}


Object P_Collect () {
    /* Check the inc_collection flag. If an incremental GC is in
     * progress and the flag has been changed to false, finish
     * the collection.
     */

    if (!inc_collection && current_space != forward_space) {
	inc_collection = 1;
	Finish_Collection ();
	inc_collection = 0;
	return (Void);
    }

    if (current_space != forward_space) {
	Finish_Collection ();
	return (Void);
    } else {
	General_Collect (0);
	return (Void);
    }
}

Object Internal_GC_Status (strat, flags) {
    Object list, cell;
    GC_Node;

    list = Cons (Sym_Generational_GC, Null);
    GC_Link (list);
    switch (strat) {
    default:            /* query or stop-and-copy */
#ifdef HAS_MPROTECT
	if (inc_collection) {
	    cell = Cons (Sym_Incremental_GC, Null);
	    (void)P_Setcdr (list, cell);
	}
#endif
	break;
    case GC_STRAT_GEN:
	if (flags == GC_FLAGS_INCR) {
#ifdef HAS_MPROTECT
	    inc_collection = 1;
	    cell = Cons (Sym_Incremental_GC, Null);
	    (void)P_Setcdr (list, cell);
#endif
	} else inc_collection = 0;
	break;
    }
    GC_Unlink;
    return (list);
}
