;;; Interpreter for Stack Development

;;; now represent environments on the stack.
;;; representation specs now in interp5reps.txt

;;; ****************************************************************

;;; environments

(define extend-env 
  (lambda (names vals old-env)
    (make-extended-env names vals old-env)))

(define extend-rec
  (lambda (decls env)
    (make-extended-rec-env decls env)))

;; now env is an environment pointer, to be interpreted relative to
;; the stack found in vk.

;;; here we use stack-ref rather than stack-pop because we want this
;;; to be done non-destructively.

(define apply-env
  (lambda (ep var-sym vk)
    (if (memq 'apply-env debug-points)
      (begin
        (printf "apply-env: var-sym = ~s ep = ~s~%"
          var-sym ep)
        (debug-print vk ep)))
    (if (zero? ep)
      (apply-initial-env var-sym vk)
      (case (stack-ref vk ep)
        ((proc-value-cont)
         (let ((formals (stack-ref vk (- ep 1)))
               (bp      (stack-ref vk (- ep 2)))
               (old-ep (stack-ref vk (- ep 3))))
           (letrec 
             ((loop
                (lambda (formals i)
                  (cond 
                    ((null? formals)
                     (apply-env old-ep var-sym vk))
                    ((eq? var-sym (car formals))
                     ;; found it at position i
                     (if (memq 'apply-env debug-points)
                       (printf
                         "apply-env: variable found at ~s value = ~s val-env = ~s~%"
                         bp
                         (stack-ref vk (+ bp (* 2 i)))
                         (stack-ref vk (+ bp (* 2 i) 1))))
                     (apply-valcont vk
                       (stack-ref vk (+ bp (* 2 i)))
                       (stack-ref vk (+ bp (* 2 i) 1))))
                    (else
                      (loop (cdr formals) (+ 1 i)))))))
             (loop formals 1))))
        ((extend-rec)
         (let ((decls   (stack-ref vk (- ep 1)))
               (old-ep (stack-ref vk (- ep 2))))
           (letrec
             ((loop 
                (lambda (decls)
                  (cond 
                    ((null? decls)
                     (apply-env old-ep var-sym vk))
                    ((eq? var-sym (rec-decl->name (car decls)))
                     (apply-valcont vk (car decls) ep)) 
                    (else
                      (loop (cdr decls)))))))
             (loop decls))))
        (else
          (error 'apply-env
            "unknown environment frame ~s"
            (stack-ref vk ep)))))))

(define debug-apply-env #f)

(define ribassoc
  (lambda (var-sym names vals vk)
    (letrec 
      ((loop
         (lambda (names vals)
           (cond 
             ((null? names)
              (error 'apply-env "unbound identifier ~s" var-sym))
             ((eq? var-sym (car names))
              (apply-valcont vk (caar vals) (cdar vals)))
             (else
               (loop (cdr names) (cdr vals)))))))
      (loop names vals))))

;;; ****************************************************************

;;; The Interpreter Proper

(define eval-exp
  (lambda (exp env vk)
    (record-case exp
      (lit-exp (constant) (apply-valcont vk constant '*dummy*))
      (var-exp (id) (apply-env env id vk))

      (letrec-exp (decls body)          ; letrec instead of proc
        ; (eval-exp body (extend-rec decls env) vk)
        (let ((vk (make-extend-rec decls env vk)))
          (eval-exp body (stack->sp vk) vk)))
          
      (if-exp (test-exp exp1 exp2)
        (eval-exp test-exp env
          (make-test-value-cont exp1 exp2 env vk)))
      (app-exp (rator rands)
        (eval-rands rands env 
          (make-all-args-cont rator env vk)))
      (else (error 'eval-exp "Bad abstract syntax: ~s" exp)))))

(define eval-rands
  (lambda (rands env ak)
    (if (null? rands)
      (apply-ak ak '())
      (eval-exp (car rands) env
        (make-first-arg-cont rands env ak)))))

;; args is now a base pointer

(define apply-proc
  (lambda (proc proc-env args vk)
    (let ((bp args))
    (record-case proc

      (prim-1 (unary-op)
        (let ((arg1 (stack-ref vk (+ bp 2)))
              (vk   (stack-set-sp (- bp 2) vk))) ; pop down to next continuation
          (apply-valcont vk
            (apply-unary-primop unary-op arg1)
            '*dummy*)))

      (prim-2 (binary-op)
        (let ((arg1 (stack-ref vk (+ bp 2)))
              (arg2 (stack-ref vk (+ bp 4)))
              (vk   (stack-set-sp (- bp 2) vk)))
          (apply-valcont vk
            (apply-binary-primop binary-op arg1 arg2)
            '*dummy*)))

      (rec-decl (name formals body)
        (let ((env proc-env))
          (let ((vk (make-proc-value-cont formals args env vk)))
            (eval-exp body (stack->sp vk) vk))))

      (else (error 'apply-proc "Bad Procedure: ~s" proc))))))

;;; ****************************************************************

;;; Value Continuations

(define make-init-cont
  (lambda ()
    (let ((s (make-stack 200 1 0)))
      (stack-push 'final-valcont s))))

(define make-test-value-cont
  (lambda (exp1 exp2 env vk)
    (stack-push 'test-value exp1 exp2 env vk)))
    
(define make-first-arg-cont
  (lambda (rands env ak)
    (let ((bp (stack->bp ak)))          
      ;; push bp explicitly
      (stack-push 'first-arg-cont rands env bp ak))))

(define make-proc-value-cont 
  (lambda (formals args env vk)         ; args is really bp
    (stack-push 'proc-value-cont formals args env vk)))

(define make-extend-rec
  (lambda (decls env vk)
    (stack-push 'extend-rec decls env vk)))

(define make-proc-cont
  (lambda (args vk)
    (stack-push 'proc-cont args vk)))

(define apply-valcont
  (lambda (vk val val-env)
    (if (memq 'apply-valcont debug-points)
      (begin
        (printf "apply-valcont: val = ~s val-env = ~s~%"
          val val-env)
        (debug-print vk)))
    (stack-pop 1 vk
      (lambda (tag s)
        (case tag
          ((final-valcont) 
           val)
          ((test-value)
           (stack-pop 3 s
             (lambda (exp1 exp2 env vk)
               (if (true-value? val)
                 (eval-exp exp1 env vk)
                 (eval-exp exp2 env vk)))))
          ((first-arg-cont)
           (stack-pop 3 s
             (lambda (rands env bp ak)
               (let ((first-arg val)(first-arg-env val-env))
                 (eval-rands (cdr rands) env
                   (make-other-args-cont first-arg first-arg-env 
                                        ; and reset the bp explicitly
                     (stack-set-bp bp ak)))))))
          ((proc-value-cont)
           (stack-pop 3 s
             (lambda (formals args env vk)
               (if (number? val)
                 (apply-valcont
                   (stack-set-sp (- args 2) vk) ; pop off everything
                   val val-env)
                 (error 'apply-proc
                   "Procedure can't return non-number ~s"
                   v)))))
          ((extend-rec)
           (stack-pop 2 s
             (lambda (decls env vk)
               (apply-valcont vk val val-env))))
          ((proc-cont)
           (stack-pop 1 s
             (lambda (args vk)
               (let ((proc val) (proc-env val-env))
                 (apply-proc proc proc-env args vk)))))
          (else
            (error 'apply-valcont
              "unknown continuation tag ~s"
              tag)))))))
               
;;; Args Continuations

(define make-all-args-cont
  (lambda (rator env vk)
    (let ((s (stack-push rator env vk))) ; push rator and env
      (stack-set-bp (stack->sp s) s))))  ; then set bp = sp


(define make-other-args-cont
  (lambda (first-arg first-arg-env ak)
    (stack-push first-arg first-arg-env ak)))

;; only external call to apply-ak has args = '()
;; internal loop just walks down to bp.


(define apply-ak
  (lambda (ak args)  
    (let ((bp (stack->bp ak)))
      ;; the rator and env are sitting at the bp, so need to get them
      ;; explicitly. 
      (let ((rator (stack-ref ak bp))
            (env (stack-ref ak (- bp 1))))
        (apply-env env rator
          (make-proc-cont bp ak))))))

;;; *****************************************************************

;;; The Initial Environment

(define build-init-env
  (lambda (entries)
    (cons                               ; new!
      (map car entries)
      (map (lambda (entry)
             (let ((op (cadr entry))
                   (arity (caddr entry)))
               (cond
                 ((= arity 1) (cons (make-prim-1 op) '*dummy*))
                 ((= arity 2) (cons (make-prim-2 op) '*dummy*))
                 (else (error 'build-init-env
                         "unknown arity in entry ~s" entry)))))
           entries))))

(define apply-initial-env
  (let ((init-env (make-init-env)))
    (lambda (var-sym vk)
      (ribassoc var-sym (car init-env) (cdr init-env) vk))))

;;; ****************************************************************

;;; toplevel

(define run
  (lambda (string)
    (eval-exp (scan&parse string) 0 (make-init-cont))))

(define debug-points '())
