;;; -*- Mode:Scheme; Base:10 -*- 

(declare (usual-integrations))

(define (compile exp target linkage)
  (cond ((self-evaluating? exp) (compile-self-evaluating exp target linkage))
	((quoted? exp) (compile-quoted exp target linkage))
	((variable? exp) (compile-variable exp target linkage))
	((assignment? exp) (compile-assignment exp target linkage))
	((definition? exp) (compile-definition exp target linkage))
	((if? exp) (compile-if exp target linkage))
	((lambda? exp) (compile-lambda exp target linkage))
	((begin? exp) (compile-sequence (begin-actions exp) target linkage))
	((cond? exp) (compile (COND->IF exp) target linkage))
	((let? exp) (compile (LET->combination exp) target linkage))
	((application? exp) (compile-application exp target linkage))
	(else
	 (error "Unknown expression type -- COMPILE" exp))))

(define (compile-self-evaluating exp target linkage)
  (preserving '(continue)
	      (make-instruction-sequence empty-set (list target)
	       `((assign ,target ',exp)))
	      (compile-linkage linkage)))

(define (compile-quoted exp target linkage)
  (preserving '(continue)
	      (make-instruction-sequence empty-set (list target)
	       `((assign ,target ',(text-of-quotation exp))))
	      (compile-linkage linkage)))

(define (compile-variable exp target linkage)
  (preserving '(continue)
	      (make-instruction-sequence '(env) (list target)
	       `((assign ,target (lookup-variable-value ',exp (fetch env)))))
	      (compile-linkage linkage)))

(define (compile-if exp target linkage)
  (let ((true-branch (make-label 'true-branch))
	(false-branch (make-label 'false-branch))		     
	(after-if (make-label 'after-if)))
    (let ((consequent-linkage
	   (if (eq? linkage 'next) after-if linkage)))
      (let ((p-code (compile (if-predicate exp) 'val 'next))
	    (c-code (compile (if-consequent exp)
			     target
			     consequent-linkage))
	    (a-code (compile (if-alternative exp) target linkage)))
	(preserving '(env continue)
		    p-code
		    (append-instruction-sequences
		     (make-instruction-sequence
		      '(val) empty-set
		      `((branch (false? (fetch val)) ,false-branch)))
		     (parallel-instruction-sequences
		      (append-instruction-sequences true-branch c-code)
		      (append-instruction-sequences false-branch a-code))
		     after-if))))))

(define (compile-application exp target linkage)
  (let ((proc-code (compile (operator exp) 'proc 'next))
	(operand-codes
	 (map (lambda (operand) (compile operand 'val 'next))
	      (operands exp))))
    (preserving '(env continue)
		proc-code
		(preserving '(proc continue)
			    (construct-arglist operand-codes)
			    (compile-procedure-call target linkage)))))

(define (construct-arglist operand-codes)
  (let ((operand-codes (reverse operand-codes)))
    (if (null? operand-codes)
	(make-instruction-sequence empty-set '(argl)
	 `((assign argl '())))
	(let ((code-to-get-last-arg
	       (append-instruction-sequences
		(car operand-codes)
		(make-instruction-sequence '(val) '(argl)
		 '((assign argl (list (fetch val))))))))
	  (if (null? (cdr operand-codes))
	      code-to-get-last-arg
	      (preserving '(env)
			  code-to-get-last-arg
			  (code-to-get-rest-args (cdr operand-codes))))))))

(define (code-to-get-rest-args operand-codes)
  (let ((code-for-next-arg
	 (preserving '(argl)
		     (car operand-codes)
		     (make-instruction-sequence '(val argl) '(argl)
		      `((assign argl (cons (fetch val) (fetch argl))))))))
    (if (null? (cdr operand-codes))
	code-for-next-arg
	(preserving '(env)
		    code-for-next-arg
		    (code-to-get-rest-args (cdr operand-codes))))))


(define (compile-procedure-call target linkage)
  (let ((primitive-branch (make-label 'primitive-branch))
	(compound-branch (make-label 'compound-branch))
	(after-call (make-label 'after-call)))
    (let ((compound-linkage (if (eq? linkage 'next) after-call linkage)))
    (append-instruction-sequences
     (make-instruction-sequence '(proc) empty-set
      `((branch (primitive-procedure? (fetch proc)) ,primitive-branch)))
     (parallel-instruction-sequences
      (append-instruction-sequences
       compound-branch
       (compound-application target compound-linkage))
      (append-instruction-sequences
       primitive-branch
       (make-instruction-sequence '(proc argl) (list target)
	`((assign ,target
		  (apply-primitive-procedure (fetch proc) (fetch argl)))))
       (compile-linkage linkage)))
     after-call))))


(define (compound-application target linkage)
  ;; linkage is never next because compile-procedure-call replaces
  ;; next with a new tag, after-call
  (if (eq? linkage 'next)
      (error "Unexpected linkage = next -- COMPOUND-APPLICATION"))
  (cond ((and (eq? target 'val) (not (eq? linkage 'return)))
	 (make-instruction-sequence '(proc) all-regs
	   `((assign continue ,linkage)
	     (assign val (compiled-procedure-entry (fetch proc)))
	     (goto (fetch val)))))

	((and (not (eq? target 'val)) (not (eq? linkage 'return)))
	 (let ((proc-return (make-label 'proc-return)))
	   (make-instruction-sequence '(proc) all-regs
	    `((assign continue ,proc-return)
	      (assign val (compiled-procedure-entry (fetch proc)))
	      (goto (fetch val))
	      ,proc-return
	      (assign ,target (fetch val))
	      (goto ,linkage)))))

	((and (eq? target 'val) (eq? linkage 'return))
	 (make-instruction-sequence '(proc continue) all-regs
	  '((assign val (compiled-procedure-entry (fetch proc)))
	    (goto (fetch val)))))

	((and (not (eq? target 'val)) (eq? linkage 'return))
	 (error "(target != val) and (linkage = return) -- COMPOUND-APPLICATION"
		target)

	 #|
	 ;; Convention of returned values in VAL prevents this case from happening. 
	 (let ((proc-return (make-label 'proc-return)))
	   (preserving '(continue)
	    (make-instruction-sequence '(proc) all-regs
	     `((assign continue ,proc-return)
	       (assign val (compiled-procedure-entry (fetch proc)))
	       (goto (fetch val))
	       ,proc-return
	       (assign ,target (fetch val))))
	    (make-instruction-sequence '(continue) all-regs
	     `((goto (fetch continue))))))
	 |#
	 )))

(define (compile-assignment exp target linkage)
  (let ((var (assignment-variable exp))
	(get-value-code (compile (assignment-value exp) 'val 'next)))
    (preserving '(env continue)
     get-value-code
     (preserving '(continue)
      (make-instruction-sequence
       '(env val) (list target)
       `((perform (set-variable-value! ',var (fetch val) (fetch env)))
	 (assign ,target ',the-unspecified-value)))
      (compile-linkage linkage)))))

(define (compile-definition exp target linkage)
  (let ((var (definition-variable exp))
	(get-value-code (compile (definition-value exp) 'val 'next)))
    (preserving '(env continue)
     get-value-code
     (preserving '(continue)
      (make-instruction-sequence '(env val) (list target)
       `((perform (define-variable! ',var (fetch val) (fetch env)))
	 (assign ,target ',the-unspecified-value)))
      (compile-linkage linkage)))))


(define (compile-sequence seq target linkage)
  (if (last-exp? seq)
      (compile (first-exp seq) target linkage)
      (preserving '(env continue)
		  (compile (first-exp seq) target 'next)
		  (compile-sequence (rest-exps seq) target linkage))))

(define (compile-lambda exp target linkage)
  (if (eq? linkage 'next)
      (let ((after-lambda (make-label 'after-lambda)))
        (append-instruction-sequences
         (compile-lambda-2 exp  target after-lambda)
         after-lambda))
      (compile-lambda-2 exp target linkage)))

(define (compile-lambda-2 exp  target linkage)
  (let ((proc-entry (make-label 'entry)))
    (tack-on-instruction-sequence
     (append-instruction-sequences
      (make-instruction-sequence '(env) (list target)
       `((assign ,target (make-compiled-procedure ,proc-entry (fetch env)))))
      (compile-linkage linkage))
     (compile-lambda-body exp  proc-entry))))

(define (compile-lambda-body exp proc-entry)
  (let ((formals (lambda-parameters exp)))
    (preserving '(continue)
     (make-instruction-sequence
      '(env proc argl) '(env)
      `(,proc-entry
	(assign env (compiled-procedure-env (fetch proc)))
	(assign env
		(extend-environment ',formals (fetch argl) (fetch env)))))
     (compile-sequence (lambda-body exp) 'val 'return))))

