;;; Controller code for explicit-control evaluator with NON-STRICT
;;; parallel evaluation.
;;; Based on the regular explicit-control evaluator from Abelson and Sussman.
;;; R.S.Nikhil, MIT, November 27, 1989

;;; ****************************************************************
;;; ****************************************************************
;;; Processors have nine registers:
;;;   EXP, ENV, VAL, FUN, UNEV, ARGL, CONTINUE,
;;;   PROGRAM-COUNTER, STACK
;;; ****************************************************************
;;; New instructions:

;;;    (fork <starting-pc> <stack-contents>)
;;; Forks a new processor with program-counter and stack initialized
;;; with given values.

;;;    (get-I-cell <register> <I-cell>)
;;; If <I-cell> is ``empty'', place this processor on the waiting list of the
;;; I-cell, so that this instruction will be retried later; if ``full'', simply
;;; fetch the value into the given register.

;;;    (set-I-cell!  <I-cell> <value>)
;;; Store the value into the I-cell, and awaken any processors in the waiting list
;;; of the I-cell.

;;; ****************************************************************
;;; ****************************************************************
;;; Here is the controller code.

(define non-strict-controller-code '(

;;; ****************************************************************
;;; Top-level evaluation of an expression
;;; Assume: stack: (exp env)
;;; Effect: stop after storing eval(exp,env) in VAL register, and recording
;;; the result.

TOP-LEVEL-EVAL
  (assign continue RECORD-RESULT)
  (goto EVAL-DISPATCH)

RECORD-RESULT
  (record-value (fetch val))
  (stop)

;;; ****************************************************************
;;; The regular EVAL function.
;;; Assume: EXP:e, ENV:E, CONTINUE:L
;;; Effect: at L; stack unchanged; VAL:  [[ e ]]E

EVAL-DISPATCH
  (branch (self-evaluating? (fetch exp)) EV-SELF-EVAL)
  (branch (symbol? (fetch exp))          EV-SYMBOL)
  (branch (define? (fetch exp))          EV-DEFINE)
  (branch (quoted? (fetch exp))          EV-QUOTE)
  (branch (lambda? (fetch exp))          EV-LAMBDA)
  (branch (let? (fetch exp))             EV-LET)
  (branch (if? (fetch exp))              EV-IF)
  (branch (cond? (fetch exp))            EV-COND)
  (branch (application? (fetch exp))     EV-APPLICATION)

  (perform (user-print 'unknown-expression-type-error))
  (perform (user-print (fetch exp)))
  (stop)

;;; ****************************************************************
;;; Numbers, strings, etc.
EV-SELF-EVAL
  (assign val (fetch exp))
  (goto (fetch continue))

;;; ****************************************************************
;;; Looking up symbols in the environment.

EV-SYMBOL
  (assign val (lookup (fetch exp) (fetch env)))
  (get-I-cell val (fetch val))                  ; wait for it
  (goto (fetch continue))

;;; ****************************************************************
;;; Definitions: (DEFINE x e) or (DEFINE (x ..args..) ..es..)

EV-DEFINE
  ; first convert (DEFINE (x ...) ..es..) to (DEFINE x e) form, if nec'y
  (assign exp (desugar-define exp))

  (assign unev (make-I-cell))          ; for value of e
  (assign exp  (cdr (fetch exp)))
  (assign val  (car (fetch exp)))      ; x from (DEFINE x e)
  (perform (define-var-value! (fetch val) (fetch unev) (fetch env)))

  (assign exp (cdr (fetch exp)))
  (assign exp (car (fetch exp)))       ; e from (DEFINE x e)
  (save unev)
  (save env)
  (save exp)                           ; STACK: e,Env,I-cell,..
  (fork FORKED-EVAL (fetch stack))
  (restore exp)                        ; discard top
  (restore exp)                        ;  three things on
  (restore exp)                        ; stack
  (goto (fetch continue))

;;; ****************************************************************
EV-QUOTE
  (assign val (cdr (fetch exp)))
  (assign val (car (fetch val)))
  (goto (fetch continue))

;;; ****************************************************************
EV-LAMBDA
  (assign val (make-procedure (fetch exp) (fetch env)))
  (goto (fetch continue))

;;; ****************************************************************
EV-LET
  (assign exp (desugar-let (fetch exp)))
  (goto EV-APPLICATION)

;;; ****************************************************************
EV-APPLICATION
  (save continue)

  (assign unev (cdr (fetch exp)))    ; (e2 ... eN)
  (assign exp  (car (fetch exp)))    ; e1
  (assign argl '())

FORK-ARGS-LOOP
  (branch (null? (fetch unev)) EVAL-OP)

;;; --- Fork an argument; VAL is used as a temporary
  (assign val  (make-I-cell))          ; I-cell for this arg
  (save val)
  (assign argl (cons (fetch val) (fetch argl)))
  (save env)
  (assign val (car (fetch unev)))      ; eJ
  (assign unev (cdr (fetch unev)))     ; (eJ+1 ... eN)
  (save val)                           ; STACK: eJ,env,I-cell,...
  (fork FORKED-EVAL (fetch stack))     ; fork it

  (restore val)                        ; discard
  (restore val)                        ;    exp, env, I-cell
  (restore val)                        ; from stack
  (goto FORK-ARGS-LOOP)

;;; All args forked, go to eval operator
EVAL-OP
  (save argl)

  (assign continue EVAL-OP-DONE)
  (goto EVAL-DISPATCH)

EVAL-OP-DONE
  (assign fun (fetch val))
  (restore argl)
  (goto APPLY-DISPATCH)

;;; ****************************************************************
;;; Forked expression-evaluation begins here.
;;; Assume: stack: exp,env,I-cell, ...
;;; Effect: eval(exp,env), store into I-cell, stop.

FORKED-EVAL
  (restore exp)
  (restore env)
  (assign continue SET-VALUE)
  (goto EVAL-DISPATCH)
SET-VALUE
  (restore exp)                           ; the I-cell
  (set-I-cell! (fetch exp) (fetch val))
  (stop)

;;; ****************************************************************
;;; The apply function
;;; Assumes: FUN: a procedure; ARGL: reversed list of arg I-cells ; STACK: L,...
;;; Effect: PC:L; STACK: ...; VAL: value of application of proc to args

APPLY-DISPATCH
  (branch (primitive-procedure? (fetch fun))      PRIMITIVE-APPLY)
  (branch (compound-procedure? (fetch fun))       COMPOUND-APPLY)

UNKNOWN-PROCEDURE-TYPE-ERROR
  (perform (user-print 'unknown-procedure-type-error))
  (perform (user-print (fetch fun)))
  (stop)

;;; ****************************************************************

PRIMITIVE-APPLY
  (branch (prim-car?      (fetch fun)) APPLY-CAR)
  (branch (prim-cdr?      (fetch fun)) APPLY-CDR)
  (branch (prim-cons?     (fetch fun)) APPLY-CONS)

  (branch (prim-pair??    (fetch fun)) APPLY-PAIR?)
  (branch (prim-null??    (fetch fun)) APPLY-NULL?)
  (branch (prim-not?      (fetch fun)) APPLY-NOT)
  (branch (prim-truncate? (fetch fun)) APPLY-TRUNCATE)
  (branch (prim-floor?    (fetch fun)) APPLY-FLOOR)

  (branch (prim-eq??      (fetch fun)) APPLY-EQ?)
  (branch (prim-and?      (fetch fun)) APPLY-AND)
  (branch (prim-or?       (fetch fun)) APPLY-OR)
  (branch (prim-+?        (fetch fun)) APPLY-PLUS)
  (branch (prim--?        (fetch fun)) APPLY-MINUS)
  (branch (prim-*?        (fetch fun)) APPLY-TIMES)
  (branch (prim-/?        (fetch fun)) APPLY-DIVIDE)
  (branch (prim-<?        (fetch fun)) APPLY-LESS-THAN)
  (branch (prim->?        (fetch fun)) APPLY-GREATER-THAN)
  (branch (prim-=?        (fetch fun)) APPLY-=)
  (branch (prim-max?      (fetch fun)) APPLY-MAX)
  (branch (prim-min?      (fetch fun)) APPLY-MIN)

UNKNOWN-PRIMITIVE-ERROR
  (perform (user-print 'unknown-primitive-error))
  (perform (user-print (fetch fun)))
  (stop)

;;; ----------------
APPLY-CAR
  ; get and wait for the cons cell
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  ; get and wait for the CAR
  (assign val (car (fetch val)))
  (get-I-cell val (fetch val))

  (restore continue)
  (goto (fetch continue))

APPLY-CDR
  ; get and wait for the cons cell
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  ; get and wait for the CDR
  (assign val (cdr (fetch val)))
  (get-I-cell val (fetch val))

  (restore continue)
  (goto (fetch continue))

APPLY-CONS
  ; return the CONS of the two I-cells in the frame
  ; i.e., don't wait for either car or cdr to be computed.
  (assign val (car (fetch argl)))
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))
  (assign val (cons (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

;;; ----------------
;;; Monadic functions

APPLY-PAIR?
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  (assign val (pair? (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-NULL?
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  (assign val (null? (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-NOT
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  (assign val (not (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-TRUNCATE
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  (assign val (truncate (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-FLOOR
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  (assign val (floor (fetch val)))
  (restore continue)
  (goto (fetch continue))

;;; ----------------
;;; Dyadic functions

APPLY-EQ?
  (assign continue APPLY-EQ?-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-EQ?-GOT-ARGS
  (assign val (eq? (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-AND
  (assign continue APPLY-AND-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-AND-GOT-ARGS
  (assign val (and (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-OR
  (assign continue APPLY-OR-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-OR-GOT-ARGS
  (assign val (or (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-PLUS
  (assign continue APPLY-PLUS-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-PLUS-GOT-ARGS
  (assign val (+ (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-MINUS
  (assign continue APPLY-MINUS-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-MINUS-GOT-ARGS
  (assign val (- (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-TIMES
  (assign continue APPLY-TIMES-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-TIMES-GOT-ARGS
  (assign val (* (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-DIVIDE
  (assign continue APPLY-DIVIDE-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-DIVIDE-GOT-ARGS
  (assign val (/ (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-LESS-THAN
  (assign continue APPLY-LESS-THAN-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-LESS-THAN-GOT-ARGS
  (assign val (< (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-GREATER-THAN
  (assign continue APPLY-GREATER-THAN-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-GREATER-THAN-GOT-ARGS
  (assign val (> (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-=
  (assign continue APPLY-=-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-=-GOT-ARGS
  (assign val (= (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-MAX
  (assign continue APPLY-MAX-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-MAX-GOT-ARGS
  (assign val (max (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

APPLY-MIN
  (assign continue APPLY-MIN-GOT-ARGS)
  (goto WAIT-FOR-TWO-ARGS)
APPLY-MIN-GOT-ARGS
  (assign val (min (fetch exp) (fetch val)))
  (restore continue)
  (goto (fetch continue))

WAIT-FOR-TWO-ARGS
  (assign exp (car (fetch argl)))
  (get-I-cell val (fetch exp))
  (assign exp (cdr (fetch argl)))
  (assign exp (car (fetch exp)))
  (get-I-cell exp (fetch exp))
  (goto (fetch continue))

;;; ****************************************************************
COMPOUND-APPLY
  (assign env (make-bindings (fetch fun) (fetch argl)))
;                                        Warning: argl is in rev order.
  (assign unev (procedure-body (fetch fun)))

;;; ****************************************************************
;;; Eval-sequence is the same as sequential case

EVAL-SEQUENCE
  (assign exp (car (fetch unev)))
  (assign val (cdr (fetch unev)))
  (branch (null? (fetch val)) LAST-EXP)
  (save unev)
  (save env)
  (assign continue EVAL-SEQUENCE-CONTINUE)
  (goto EVAL-DISPATCH)

EVAL-SEQUENCE-CONTINUE
  (restore env)
  (restore unev)
  (assign unev (cdr (fetch unev)))
  (goto EVAL-SEQUENCE)

LAST-EXP
  (restore continue)
  (goto EVAL-DISPATCH)

;;; ****************************************************************
;;; Evaluation of conditionals is the same as before

EV-IF
  (save continue)
  (assign continue EV-IF-DECIDE)
  (assign unev (cdr (fetch exp)))
  (assign exp (car (fetch unev)))
  (assign unev (cdr (fetch unev)))
  (save env)
  (save unev)
  (goto EVAL-DISPATCH)

EV-IF-DECIDE
  (restore unev)
  (restore env)
  (restore continue)
  (branch (null? (fetch val)) EV-ELSE-PART)
EV-THEN-PART
  (assign exp (car (fetch unev)))
  (goto EVAL-DISPATCH)
EV-ELSE-PART
  (assign exp (cdr (fetch unev)))
  (assign exp (car (fetch exp)))
  (goto EVAL-DISPATCH)

;;; ----------------
;;; (cond ((p1 ..e1s..) (p2 ..e2s..) ... (else ..eNs..)))
EV-COND
  (save continue)
  (assign continue EVCOND-DECIDE)
  (assign unev (cdr (fetch exp)))

EVCOND-PRED
  (branch (null? (fetch unev)) EVCOND-RETURN-NIL)
  (assign exp (car (fetch unev)))
  (branch (else-clause? (fetch exp)) EVCOND-ELSE-CLAUSE)
  (save env)
  (save unev)
  (assign exp (car (fetch exp)))    ; predicate
  (goto EVAL-DISPATCH)

EVCOND-DECIDE
  (restore unev)
  (restore env)
  (branch (fetch val) EVCOND-TRUE-PREDICATE)
  (assign unev (cdr (fetch unev)))
  (goto EVCOND-PRED)

EVCOND-TRUE-PREDICATE
  (assign exp (car (fetch unev)))
EVCOND-ELSE-CLAUSE
  (assign unev (cdr (fetch exp)))
  (goto EVAL-SEQUENCE)

EVCOND-RETURN-NIL
  (restore continue)
  (assign val '())
  (goto (fetch continue))

;;; ****************************************************************
;;; End of controller sequence

))

;;; ----------------------------------------------------------------
;;; A particular non-strict multiprocess simulator called NS-EVAL

(define ns-eval
  (let ((mp (make-multiprocessor non-strict-controller-code)))
    (lambda (e) (load-and-go mp e))))

;;; ----------------------------------------------------------------
;;; Having compiled the controller code into the multiprocessor,
;;; redefine it to '() to save space.  This is purely an efficiency hack.

(define non-strict-controller-code '())

;;; ****************************************************************
