;;; Auxiliary definitions for multiprocess simulators.
;;; 6.001, R.S.Nikhil, MIT, November 24, 1988
;;;        revised November 19, 1989

;;; ****************************************************************
(define (user-print v)
  (if (compound-procedure? v)
      (print-compound-procedure v)
      (print v)))

;;; ****************************************************************
;;; Expression types (for EVAL-DISPATCH)

(define (self-evaluating? exp) (or (number? exp) (null? exp)))

(define (has-type? exp type)
  (if (pair? exp)
      (eq? (car exp) type)
      #f))

(define (define? exp) (has-type? exp 'define))
(define (quoted? exp) (has-type? exp 'quote))
(define (lambda? exp) (has-type? exp 'lambda))
(define (let?    exp) (has-type? exp 'let))
(define (letrec? exp) (has-type? exp 'letrec))
(define (if?     exp) (has-type? exp 'if))
(define (cond?   exp) (has-type? exp 'cond))
(define (else-clause? clause) (eq? (car clause) 'else))

(define (application? exp) (pair? exp))
(define (last-operand? exp) (null? (cdr exp)))

;;; ****************************************************************
;;; Compound procedures

(define (make-procedure lambda-exp env)
  (list 'procedure (cadr lambda-exp) (cddr lambda-exp) env))

(define (compound-procedure? proc) (has-type? proc 'procedure))

(define (parameters            proc) (cadr proc))
(define (procedure-body proc)        (caddr proc))
(define (procedure-environment proc) (cadddr proc))

(define (print-compound-procedure proc)
  (print (list 'procedure
               (parameters proc)
               (procedure-body proc)
               '<procedure-env>)))

;;; ****************************************************************
;;; Primitive procedures
;;;     Representation: (PRIMITIVE <opname>)

(define (primitive-procedure? proc) (has-type? proc 'primitive))

(define prim-car?      (lambda (proc) (eq? (cadr proc) 'car)))
(define prim-cdr?      (lambda (proc) (eq? (cadr proc) 'cdr)))
(define prim-cons?     (lambda (proc) (eq? (cadr proc) 'cons)))

(define prim-pair??    (lambda (proc) (eq? (cadr proc) 'pair?)))
(define prim-null??    (lambda (proc) (eq? (cadr proc) 'null?)))
(define prim-not?      (lambda (proc) (eq? (cadr proc) 'not)))
(define prim-truncate? (lambda (proc) (eq? (cadr proc) 'truncate)))
(define prim-floor?    (lambda (proc) (eq? (cadr proc) 'floor)))

(define prim-eq??      (lambda (proc) (eq? (cadr proc) 'eq?)))
(define prim-and?      (lambda (proc) (eq? (cadr proc) 'and)))
(define prim-or?       (lambda (proc) (eq? (cadr proc) 'or)))
(define prim-+?        (lambda (proc) (eq? (cadr proc) '+)))
(define prim--?        (lambda (proc) (eq? (cadr proc) '-)))
(define prim-*?        (lambda (proc) (eq? (cadr proc) '*)))
(define prim-/?        (lambda (proc) (eq? (cadr proc) '/)))
(define prim-<?        (lambda (proc) (eq? (cadr proc) '<)))
(define prim->?        (lambda (proc) (eq? (cadr proc) '>)))
(define prim-=?        (lambda (proc) (eq? (cadr proc) '=)))
(define prim-max?      (lambda (proc) (eq? (cadr proc) 'max)))
(define prim-min?      (lambda (proc) (eq? (cadr proc) 'min)))

;;; ****************************************************************
;;; Environments
;;;   Env = list of frames
;;;   Frame = list of bindings
;;;   Binding = (var . value)

(define (make-frame vars vals)
  (cond ((and (null? vars) (null? vals)) '())
	((null? vars)
	   (error "MAKE-FRAME: too many actuals" vals))
	((null? vals)
	   (error "MAKE-FRAME: too many formals" vars))
	(else (cons (cons (car vars) (car vals))
		    (make-frame (cdr vars) (cdr vals))))))

(define (make-bindings proc args)             ; Warning: args are in rev order
  (let ((env (procedure-environment proc))
	(vars (parameters proc)))
    (cons (make-frame vars (reverse args))
	  env)))

(define (lookup var env)
  (if (null? env)
      (error "LOOKUP: unbound variable" var)
      (let ((b (assq var (car env))))
	(if b
	    (cdr b)
	    (lookup var (cdr env))))))

(define (set-var-value! var val env)
  (if (null? env)
      (error "SET-VAR-VALUE!: unbound variable" var)
      (let ((b (assq var (car env))))
	(if b
	    (set-cdr! b val)
	    (set-var-value! var val (cdr env))))))

(define (define-var-value! var val env)
  (set-car! env
	    (cons (cons var val)
		  (car env))))

(define *the-global-environment*
  '( ( (nil       . ())
       (car       . (primitive car))
       (cdr       . (primitive cdr))
       (cons      . (primitive cons))

       (pair?     . (primitive pair?))
       (null?     . (primitive null?))
       (not       . (primitive not))
       (truncate  . (primitive truncate))
       (floor     . (primitive floor))

       (eq?       . (primitive eq?))
       (and       . (primitive and))
       (or        . (primitive or))
       (+         . (primitive +))
       (-         . (primitive -))
       (*         . (primitive *))
       (/         . (primitive /))
       (<         . (primitive <))
       (>         . (primitive >))
       (=         . (primitive =))
       (max       . (primitive max))
       (min       . (primitive min))
     )
   )
)

;;; ****************************************************************
;;; ``I-structure cells'' for non-strict interpreter.

(define (make-I-cell) (cons 'I-cell (cons 'empty '())))

(define (is-I-cell? x) (has-type? x 'I-cell))

(define (I-cell-flag-set? I-cell) (eq? (cadr I-cell) 'full))

(define (set-I-cell-flag! I-cell) (set-car! (cdr I-cell) 'full))

(define (I-cell-value I-cell) (cddr I-cell))

(define (set-I-cell-value! I-cell v)
  (if (eq? (cadr I-cell) 'empty)
      (set-cdr! (cdr I-cell) v)
      (error "SET-I-CELL-VALUE!: multiple writes : " I-cell v)))

(define (I-cell-waiting-list I-cell) (cddr I-cell))

(define (add-to-I-cell-waiting-list I-cell p)
  (set-cdr! (cdr I-cell) (cons p (cddr I-cell))))

;;; ****************************************************************
;;; Desugaring LETs into LAMBDA applications
;;; and DEFINES of functions into DEFINES of symbols

(define (desugar-let exp)             ;   (let ((x e) .. (y e)) ..bs..)
    (cons (cons 'lambda
                (cons (mapcar car (cadr exp))
                      (cddr exp)))
          (mapcar cadr (cadr exp))))

(define (desugar-define def)
  (if (pair? (cadr def))                 ; (DEFINE (f ..xs..) ..bs..)
      (list (car def)                    ; DEFINE
            (caadr def)                  ; f
	    (cons 'lambda
		  (cons (cdadr def)      ; (..xs..)
			(cddr def))))    ;  ..bs..
      def))

;;; ****************************************************************
