;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;; A simple ordered set facility.  Items kept in these sets must
;;; have an order function: these are supplied for integers and
;;; osets themselves.  Items are kept in sorted lists, smallest
;;; first.  Could be re-done with binary search trees.
;;; See integer-order-function for how order functions are supposed to
;;; work.

;;; Constructor will default to make a set that orders integers.

(define-struct oset
  (item-list '())
  (order-fn integer-order-function)
  (cardinality 0))

(define (oset-empty? oset) (null? (oset-item-list oset)))

;;; Example of how the order function is supposed to work.

(define (integer-order-function a b)
  (cond ((< a b) 'correct-order)
	((> a b) 'wrong-order)
	(else 'equal)))

;;; Destructively insert an item into a set
;;; Returns the item if it wasn't there already, else #f.

(define (oset-insert! item set)
  (let ((pair (oset-insert-2! item set)))
    (if (car pair) (cdr pair) #f)))

;;; Returns a pair whose car is #f if nothing is inserted
;;; and whose cdr is a pointer to the item either found or inserted
;;; into the set (so is eq to a member of the set).

(define (oset-insert-2! item set)
  (let ((ilist (oset-item-list set))
	(odf (oset-order-fn set))
	(order 'bogus!))
    (cond ((or (null? ilist)
	       (eq? 'correct-order
		    (begin (set! order (odf item (car ilist)))
			   order)))
	   (begin (set-slot! (oset-item-list set) (cons item ilist))
		  (set-slot! (oset-cardinality set)
			(1+ (oset-cardinality set)))
		  (cons #t item)))
	  ((eq? 'equal order) (cons #f (car ilist))) ; item already there
	  (else (oset-insert-aux-2 item set ilist (cdr ilist) odf)))))

;;; Ilist isn't null, and item goes somewhere after the car of ilist.

(define (oset-insert-aux-2 item set ilist ilist-cdr odf)
  (let ((order 'bogus))
    (cond ((or (null? ilist-cdr)
	       (eq? 'correct-order
		    (begin (set! order (odf item (car ilist-cdr)))
			   order)))
	   (begin (set-cdr! ilist (cons item ilist-cdr))
		  (set-slot! (oset-cardinality set)
			     (1+ (oset-cardinality set)))
		  (cons #t item)))
	  ((eq? 'equal order) (cons #f (car ilist-cdr)));; already there
	  (else (oset-insert-aux-2 item set (cdr ilist) (cdr ilist-cdr) odf)
		))))

;;; Insert a list of items into an oset. returns the SET.

(define (oset-insert-list! list oset)
  (for-each (lambda (x) (oset-insert! x oset)) list)
  oset)

;;; It's easy to define a generic order function on osets if they
;;; have the same order function
;;; making for easy osets of osets.

(define (oset-order-function oset-a oset-b)
  (cond ((not (eq? (oset-order-fn oset-a)
		   (oset-order-fn oset-b)))
	 (error "incompatible types of sets: oset-order-function"))
	((< (oset-cardinality oset-a) (oset-cardinality oset-b))
	 'correct-order)
	((> (oset-cardinality oset-a) (oset-cardinality oset-b))
	 'wrong-order)
	;; same cardinality, same type, so march down the lists...
	(else (oset-order-aux (oset-item-list oset-a)
			      (oset-item-list oset-b)
			      (oset-order-fn oset-a)))))

(define (oset-order-aux ilista ilistb odf)
  (if (null? ilista)
      'equal
      (let ((item-order (odf (car ilista) (car ilistb))))
	(if (eq? 'equal item-order)
	    (oset-order-aux (cdr ilista)
			    (cdr ilistb)
			    odf)
	    item-order))))


(define (oset-comparable? oseta osetb)
       (eq? 'equal (oset-order-function oseta osetb)))


;;; Yields a list of disjoint subsets whose union is the set.  For
;;; each subset the value of selection-fn applied to the members is
;;; the same in the sense of eqv.

(define (oset-select-subsets set selection-fn)
  (let ((r-ilist (reverse (oset-item-list set)))
	(alist '()))
    (for-each (lambda (item)
		(let* ((key (selection-fn item))
		       (found-association (assv key alist)))
		  (if found-association 
		      (set-cdr! found-association 
				(cons item (cdr found-association)))
		      (push (cons key (list item)) alist))))
	      r-ilist)
    (map (lambda (x) 
	   (make-oset 'item-list (cdr x)
		      'cardinality (-1+ (length x))
		      'order-fn (oset-order-fn set)))
	 alist)))


(define (oset-for-each procedure set)
  (for-each procedure (oset-item-list set)))

(define (oset-memq elt set)
  (memq elt (oset-item-list set)))

(define (oset-copy oset)
   (make-oset
      'item-list (copy-list (oset-item-list oset))
      'order-fn (oset-order-fn oset)
      'cardinality (oset-cardinality oset)))	
	
(define (oset-union oset1 oset2)
  (assert (eqv? (oset-order-fn oset1) (oset-order-fn oset2))
	  "Mismatched order functions in oset union.")
  (oset-insert-list! (oset-item-list oset1)
		     (oset-copy oset2)))
		
(define (oset-delete item oset)
  (let ((new-oset (oset-copy oset)))
    (if (oset-memq item oset)
	(begin
	 (set-slot! (oset-cardinality new-oset)
		    (- (oset-cardinality oset) 1))
	 (set-slot! (oset-item-list new-oset)
		    (delete! item (oset-item-list new-oset)))))
    new-oset))			

(define (oset-empty! oset)
  (set-slot! (oset-cardinality oset) 0)
  (set-slot! (oset-item-list oset) '()))

(define (copy-list the-list)
  (reverse! (reverse the-list)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; test:


(comment-out
 (load "c-scheme")
 (load "oset")
 (integer-order-function 1 2)
 (define fred (make-oset))
 (oset-item-list fred)
 (oset-insert! 3 fred)
 (oset-insert-2! 4 fred) 
 (oset-insert-list! '(5 6 7 7) fred)
 (oset-insert-list! '(10 11) fred)
 (oset-insert! 1100 fred)
 (define ned (make-oset))
 (define mary (make-oset 'order-fn oset-order-function))
 (oset-insert! ned mary)
 (oset-insert! ned mary)
 (oset-insert! fred mary)
 (oset-insert! fred mary)
 (map oset-item-list (oset-item-list mary))
 (map oset-item-list  (oset-select-subsets fred (lambda (x) (> x 5))))
 (map oset-item-list  (oset-select-subsets fred even?))
 (oset-for-each (lambda (x) (display x)) fred)
 (oset-memq 5 fred)
 (oset-memq 99 fred)
 (define freddy (oset-copy fred))
 (oset-item-list freddy)
 (define al (car (oset-select-subsets fred even?)))
 (define hal (cadr (oset-select-subsets fred even?)))
 (oset-item-list (oset-union al hal))
 (oset-item-list fred)
 (oset-item-list (oset-delete 1100 fred))
 (oset-empty! freddy)
 (oset-item-list freddy)
)


;;; PC scheme requires a control-Z at the end of each source file: 
