;;;             Copyright (C) 1989, by William M. Wells III
;;;                         All Rights Reserved
;;;     Permission is granted for unrestricted non-commercial use.

(declare (usual-integrations))

;;;
;;; (Changed for lalr(1).)
;;;
;;; lr(1) items.
;;; These are going to be represented by structs:
;;; after-dot is a integer which indexes the symbol in the
;;; production which follows the dot
;;; that comes after the dot.
;;;
;;; look-aheads is an oset of grammar symbols.
;;; The item data structure
;;; essentially stands for the set of lr(1) items which are the same
;;; except for each having one lookahead symbol from the set look-aheads.
;;;
;;; look-ahead-dependers is an oset of items to whom
;;; lalr(1) lookaheads
;;; propagate from this item.

(define-struct item
  production
  after-dot
  (look-aheads (make-oset 'order-fn g-symbol-order-function))
  (look-ahead-dependers
   (make-oset 'order-fn item-order-function)))

;;; A handy predicate.

(define (dot-at-right-end? item)
        (equal? (-1+ (length (item-production item))) (item-after-dot item)))

;;; Get the symbol after the dot -- 'the-bogus-symbol if dot is flushright.

(define (symbol-after-dot item)
  (if (dot-at-right-end? item)
      'the-bogus-symbol
      (list-ref (item-production item) (1+ (item-after-dot item)))))

;;; Make an item with the dot moved one to the right, or false if
;;; dot gets past the end.
;;; Since this is used during lr(0) set construction, it only
;;; deals with production and after-dot slots, the others
;;; are filled in as '() by default.

(define (advance-dot item)
  (if (dot-at-right-end? item) #f
      (make-item 'production (item-production item)
		 'after-dot (1+ (item-after-dot item)))))

;;; Make an item which has the dot at the left end of the rhs.

(define (new-item production)
  (make-item 'production production
	     'after-dot 1))

;;; For osets of items:
;;; this is used during lr(0) sets of items construction.  Only the
;;; production and after dot fields are tested, since these characterize
;;; lr(0) items.

(define (item-order-function ia ib)
  (cond ((< (production-index (item-production ia))
	    (production-index (item-production ib)))
	 'correct-order)
	((> (production-index (item-production ia))
	    (production-index (item-production ib)))
	 'wrong-order)
	((< (item-after-dot ia)
	    (item-after-dot ib))
	 'correct-order)
	((> (item-after-dot ia)
	    (item-after-dot ib))
	 'wrong-order)

	(else 'equal)))


;;; This only prints the lr(0) parts and the lookaheads.


(define (item-print item)
  (display (g-symbol-print-name (lhs (item-production item))))
  (display " -> ")
  (do ((ncdr (rhs (item-production item)) (cdr ncdr))
       (i 1 (1+ i)))
      ((null? ncdr)
       (begin (if (equal? (item-after-dot item) i) (display ". "))
	      (if (not (oset-empty? (item-look-aheads item)))
		  (begin
		    (display ",")
		    (oset-for-each
		     (lambda (gs)
		       (display (g-symbol-print-name gs))
		       (display " "))
		     (item-look-aheads item))))))
    (if (equal? (item-after-dot item) i) (display ". "))
    (display (g-symbol-print-name (car ncdr))) (display " ")
    ))



(define (item-list-print item-list)
  (for-each
   (lambda (item)
     (newline)
     (item-print item))
   item-list))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; test:
(comment-out
 (load "item")
 (define fred (new-item (car productions)))
 (item-print fred)
 (define ned (advance-dot fred))
 (item-print ned)
 (item-order-function ned ned)
 (item-order-function ned fred)
 (item-order-function fred ned)
 (symbol-after-dot fred)
 (dot-at-right-end? fred)
 (dot-at-right-end? ned))

;;; PC scheme requires a control-Z at the end of each source file: 
