;;; Posted to comp.lang.scheme by Bryan O'Sullivan <bosullvn@serd.cscs.ch>
;;; on Mon, 25 Oct 1993.

; A quick Scheme implementation of simple pattern matching using the
; R4RS macro extensions.
;
; As allowing Haskell-style function declarations with patterns would
; seem to require a fairly considerable retooling of Scheme, my pattern
; matching macros introduce a new special form, PATTERN-CASE.  I have
; included some very simple examples at the end which give an idea of
; how pattern matching can be applied in Scheme.
;
; Incidentally, the examples are adapted from Henson's functional
; languages book, and from the Hudak & Fasel Haskell tutorial paper.



;; -------->8 begin pattern-case.scm
;; pattern-case.scm
;; Simple pattern-matching case special form.  Uses the R4RS macro
;; system.
;;
;; Bryan O'Sullivan <bosullvn@serd.cscs.ch> 10.93

(require 'macro)
(require 'repl)
(repl:top-level macro:eval)

;; PATTERN-CASE syntax:
;;   (pattern-case EXPR
;; 		   ((PATTERN BODY)
;; 	 	    ...))
;; where EXPR is a list or sexp, PATTERN one of
;;	()			- matches the empty list
;;	(BINDING1 : BINDING2)	- binds BINDING1 to (car EXPR) and BINDING2
;;				  to (cdr EXPR)
;;	BINDING			- binds BINDING to EXPR
;;	else			- this always gets executed if no preceding
;;				  match has succeeded
;; and BODY a chunk of code to be executed if the particular match
;; succeeds.
;;
;; Semantics: much the same as for CASE; the value returned is the
;; value returned by the code for the first matching pattern, or for
;; the optional ELSE clause, which matches anything not already
;; matched, but binds nothing.

(define-syntax pattern-case
  (syntax-rules ()
		((pattern-case expr body)
		 (let ((result (pattern-single-case expr body)))
		   (if (pair? result)
		       (cdr result))))
		((pattern-case expr body1 body2 ...)
		 (let ((result (pattern-single-case expr body1)))
		   (if (pair? result)
		       (cdr result)
		       (pattern-case expr body2 ...))))))

;; PATTERN-SINGLE-CASE syntax:
;;    (pattern-single-case EXPR BODY)
;; The rules for EXPR and BODY are as above.  We have to use a colon
;; for separating the CAR and CDR of a pattern, since the macro system
;; barfs if we use a period.

(define-syntax pattern-single-case
  (syntax-rules (: else)
		((single-case expr (else body))
		 (cons #t body))
		((single-case expr (() body))
		 (if (null? expr)
		     (cons #t (begin body))))
		((single-case expr
			      ((binding1 : binding2) body))
		 (if (pair? expr)
		     (let ((binding1 (car expr))
			   (binding2 (cdr expr)))
		       (cons #t (begin body)))))
		((single-case pat (binding body))
		 (let ((binding pat))
		   (cons #t (begin body))))))

(provide 'pattern-case)

-------->8 pattern-examples.scm
;; pattern-examples.scm
;; Some quick hack example usages of simple pattern matching.
;;
;; Bryan O'Sullivan <bosullvn@serd.cscs.ch> 10.93

(require 'pattern-case)

;; INSERT syntax:
;;   (insert ELEMENT LIST)
;; where ELEMENT has the ordering <= defined on it, and LIST is a list
;; with a similar constraint.
;;
;; Semantics: inserts ELEMENT into LIST, which should be ordered.

(define (insert a l)
  (pattern-case l
		(()
		 (list a))
		((x : xs)
		 (if (<= a x)
		     (append (list a x) xs)
		     (append (list x) (insert a xs))))))

;; INSERT-SORT syntax:
;;   (insert-sort FROM TO)
;; where FROM and TO are lists which both have the <= ordering defined
;; on them; TO should already be ordered.

(define (insert-sort from to)
  (pattern-case from
		(()
		 to)
		((x : xs)
		 (insert-sort xs (insert x to)))))

;; MAKE-FILTER syntax:
;;   (make-filter PREDICATE RHS)
;; where PREDICATE is the predicate to apply, and RHS is the second
;; argument of the predicate.
;;
;; Semantics: returns a function which takes one argument; the
;; function is a predicate, and the argument should be the intended
;; first argument to the predicate.

(define (make-filter pred right)
  (lambda (left)
    (pred left right)))

;; PICK syntax:
;;   (pick FROM FILTER)
;; where FROM is a list, and FILTER is a filter as returned by
;; MAKE-FILTER.
;;
;; Semantics: returns the elements of LIST (in a new list) for which
;; FILTER evaluates to #t.  Ordering is preserved.

(define (pick from-list filter)
  (define (pick-acc from want? acc)
    (pattern-case from
		  (() acc)
		  ((x : xs)
		   (pick-acc xs want?
			     (if (want? x)
				 (append acc (list x))
				 acc)))))
  (pick-acc from-list filter '()))

;; QUICKSORT syntax:
;;   (quicksort LIST)
;; where LIST is a list which has < and >= defined on it.
;;
;; Semantics: returns a sorted list.

(define (quicksort l)
  (pattern-case l (() '())
		((x : xs)
		 (append
		  (quicksort (pick xs (make-filter < x)))
		  (append (list x)
			  (quicksort (pick xs (make-filter >= x))))))))

;;; *EOF*
