;;;; -*- Scheme -*-
;;;; $Header: /home/panda/pg/bevan/progs/scheme/misc/RCS/loop-language.scm,v 1.1 91/05/18 17:29:16 bevan Exp $

;;;+file-summary
;;; An interpreter for the language LOOP.
;;; This is based on the Algol code in :-
;;;   ALGOL 68 as a metalanguage for denotational semantics
;;;   F.G. Pagan
;;;   The Computer Journal 22(1):63-66
;;; which in turn uses the denotational definition of LOOP in :-
;;;   The Denotational Semantics of Programming Languages
;;;   R.D. Tennent
;;;   CACM 19(8):437-53
;;; The earliest reference for LOOP I have is :-
;;;   The complexity of LOOP programs
;;;   A.R. Mayer and D.M. Ritchie
;;;   Proceedings of the 22nd ACM National Conference pp. 465-469
;;;-

;;; Syntactic Domains
;;;   P : Prog   (programs)
;;;   C : Cmd    (commands)
;;;   E : Exp    (expressions)
;;;   V : Var    (variables)

;;; Abstract Syntax
;;;   P ::= read V; C; write E
;;;   C ::= C1 ; C2 | V := E | to E do C
;;;   E ::= 0 | V | succ E

(define P
  (lambda (variable command expression)
    (list 'P variable command expression)))
(define P? (lambda (p) (eq? (car p) 'P)))
(define P-var (lambda (p) (cadr p)))
(define P-cmd (lambda (p) (caddr p)))
(define P-expr (lambda (p) (cadddr p)))

(define C-comp
  (lambda (c1 c2) (list 'C-comp c1 c2)))
(define C-comp? (lambda (c) (eq? (car c) 'C-comp)))
(define C-comp-1 (lambda (c) (cadr c)))
(define C-comp-2 (lambda (c) (caddr c)))

(define C-ass
  (lambda (var expr) (list 'C-ass var expr)))
(define C-ass? (lambda (c) (eq? (car c) 'C-ass)))
(define C-ass-var (lambda (c) (cadr c)))
(define C-ass-expr (lambda (c) (caddr c)))

(define C-loop
  (lambda (expr cmd) (list 'C-loop expr cmd)))
(define C-loop? (lambda (c) (eq? (car c) 'C-loop)))
(define C-loop-expr (lambda (c) (cadr c)))
(define C-loop-cmd (lambda (c) (caddr c)))

(define E-0 (lambda () (list 'E-0)))
(define E-0? (lambda (e) (eq? (car e) 'E-0)))

(define E-var (lambda (e) (list 'E-var e)))
(define E-var? (lambda (e) (eq? (car e) 'E-var)))
(define E-var-name (lambda (e) (cadr e)))

(define E-succ (lambda (e) (list 'E-succ e)))
(define E-succ? (lambda (e) (eq? (car e) 'E-succ)))
(define E-succ-expr (lambda (e) (cadr e)))

;;; Semantic Domains
;;;  v : N              (non-negative integers)
;;;  o : S = V -> N     (states)

;;; Semantic Functions
;;;  Mp : Prog -> N -> N
;;;  Mc : Cmd -> S -> S
;;;  Me : Exp -> S -> N

;;;+fs
;;; An auxillary function that enriches the environment `state' with
;;; the binding of `variable' to `value'
;;;-
(define update
  (lambda (state variable value)
    (lambda (id) (if (eq? id variable) value (state id)))))

;;;+fs
;;; Composes the function `f' with itself `n' times.
;;; `state' is the inital environment.
;;;-
(define iterate
  (lambda (f n)
    (lambda (state)
      (if (zero? n)
	  state
	  (let ((g (iterate f (- n 1))))
	    (f (g state)))))))

;;;+fs
;;; Semantic Function
;;;  Me : Exp -> S -> N
;;; Defines the meaning of expressions in LOOP
;;;-
(define Me
  (lambda (expr)
    (lambda (state)
      (cond ((E-0? expr) 0)
	    ((E-var? expr) (state (E-var-name expr)))
	    ((E-succ? expr) (+ ((Me (E-succ-expr expr)) state) 1))))))

;;;+fs
;;; Semantic function
;;;  Mc : Cmd -> S -> S
;;; Defines the meaning of commands in LOOP.
;;;-
(define Mc
  (lambda (cmd)
    (lambda (state)
      (cond ((C-comp? cmd)
	     ((Mc (C-comp-2 cmd)) ((Mc (C-comp-1 cmd)) state)))
	    ((C-ass? cmd)
	     (update state (C-ass-var cmd) ((Me (C-ass-expr cmd)) state)))
	    ((C-loop? cmd)
	     ((iterate (Mc (C-loop-cmd cmd))
		       ((Me (C-loop-expr cmd)) state))
	      state))))))

;;;+fs
;;; Semantic Functions
;;;  Mp : Prog -> N -> N
;;; Defines the meaning of a LOOP program.
;;;-
(define Mp
  (lambda (p)
    (lambda (n)
      (let* ((initial-state (lambda (v) 0))
	     (final-state ((Mc (P-cmd p)) (update initial-state (P-var p) n))))
	((Me (P-expr p)) final-state)))))

;;;+fs
;;; An simple LOOP program.
;;; This has the same effect as (lambda (x) (+ (* 2 x) 2))
;;;- +fe
;;; > ((Mp test) 3)
;;; 8
;;;-
(define test
  (P 'x
     (C-comp (C-ass 'y (E-var 'x))
	     (C-loop (E-var 'x) (C-ass 'y (E-succ (E-var 'y)))))
     (E-succ (E-succ (E-var 'y)))))
