;;; Customization file for the citi conversion program.
;;;
;;; March 15, 1991
;;;
;;; Karl O. Pinc
;;;
;;; August 7, 1991 Karl O. Pinc: Added sybase-message/error-handler.
;;;
;;; Low level access to selected C library functions, to datatype predicates, typecasts
;;; and type related utilities required for Sybase calls, to Sybase database calls,
;;; to get pointers to sybase error and message handlers written in C, and to
;;; VMS string allocation routines.
;;;
;;; See the section titled "Sybase procedures." below for information on how to use the
;;; db-lib calls.
;;;
;;; WARNING: Some of the constants of sybdbtoken.h are hardcoded into token->datatype.
;;; If Sybase ever changes these values (which would require everybody to recompile
;;; and relink their C programs) this code will need to change.
;;;
;;; WARNING: The Sybase message and error handling is dependent upon the
;;; undocumented-but-should-be-documented temporal behavior of sybase
;;; installed C message and error handlers.  In particular, the code relys on
;;; the C message and error handlers being called by sybase before the db-lib
;;; function raising the error/message returns.  Sybase is not likely to change this
;;; as it could break all the customer's error handling code, on the other hand
;;; not too many programmers seem to handle errors anyway -- the sybase example
;;; code surely dosen't.  See also the note under "Bugs:" below re: failure to
;;; deal with INT_EXIT, INT_CANCEL, and INT_CONTINUE.
;;;
;;; WARNING:  The integer values for Sybase #defines like SUCCEED and NO_MORE_ROWS are
;;; shamelessly hardcoded (in obvious places) because only the value for FAIL appears
;;; more than once in the code.  None the less, this should really be tidyed up and
;;; symbols declared.  See also the first warning above.

;; C library functions.

;; The C-library free function to free memory allocated with malloc.
;; Indiscriminate use in applications is scaaareeey.
(define free (foreign-procedure "free" (unsigned-32) void))

(define malloc (foreign-procedure "malloc" (unsigned-32) unsigned-32))

(define strcpy 
  (foreign-procedure "strcpy" (unsigned-32 unsigned-32) unsigned-32))

(define memcpy
  (foreign-procedure "memcpy" (unsigned-32 unsigned-32 unsigned-32) unsigned-32))

;; Functions to peek at data through pointers
(define peek.integer-32 
  (foreign-procedure "dref_ptr_int_32" (unsigned-32) integer-32))

(define peek.integer-16
  (foreign-procedure "dref_ptr_int_16" (unsigned-32) integer-32))

(define peek.unsigned-32
  (foreign-procedure "dref_ptr_int_32" (unsigned-32) unsigned-32))

(define peek.unsigned-16
  (foreign-procedure "dref_ptr_unsigned_16" (unsigned-32) unsigned-32))

(define peek.unsigned-8
  (foreign-procedure "dref_ptr_unsigned_8" (unsigned-32) unsigned-32))

(define peek.double-float
  (foreign-procedure "dref_ptr_double_float" (unsigned-32) double-float))

(define peek.boolean
  (foreign-procedure "dref_ptr_unsigned_8" (unsigned-32) boolean))

(define peek.string
  (foreign-procedure "id" (unsigned-32) string))

; Takes a string, a pointer to vms memory, and a length.  Mutates the
; first length characters of the scheme string to the first length bytes of
; what's in the vms memory location.  Woe betide you if length is longer than
; the scheme string.
(define vms-memory->string!
  (foreign-procedure "memcpy" (string unsigned-32 unsigned-32) void))

(define poke.unsigned-32 
  (foreign-procedure "poke_int_32" (unsigned-32 unsigned-32) void))

(define poke.unsigned-8
  (foreign-procedure "poke_int_8" (unsigned-32 unsigned-32) void))

(define dbproc? '())
(define loginrec? '())
(define sybase-error-handler? '())
(define sybase-message-handler? '())
(define computeid? '())
(define vms-string? '())
(define closed-object? '())
(define mark-closed! '())
; The Sybase documentation does not define what a computeid is very well,
; the implication is that it is a sequential integer starting with one.
; If you wish to program using this assumption, these procedures
; will be useful.
(define integer->computeid '()) ; Typecast an integer to a compute id.
(define computeid->integer '()) ; Typecast a computeid to an integer.

(define dbcmd '())
(define dbdata '())
(define dbadata '())
(define dbadlen '())
(define dbalttype '())
(define dbcancel '())
(define dbcanquery '())
(define dbclose '())
(define dbcoltype '())
(define dbdatlen '())
(define dbdead '())
(define dberrhandle '())
(define dbexit '())
(define dbfreebuf '())
(define dbinit '())
(define dblogin '())
(define dbmsghandle '())
(define dbnextrow '())
(define dbnumalts '())
(define dbopen '())
(define dbresults '())
(define dbrows '())
(define dbsetlapp '())
(define dbsetlhost '())
(define dbsetlpwd '())
(define dbsetluser '())
(define dbsqlexec '())
(define dbsqlok '())
(define dbsqlsend '())

(define get-error-handler '())
(define get-message-handler '())

(define allocate-vms-string '())
(define free-vms-string '())

(letrec ()
  ;; Typechecking procedures to keep from crashing the program by inadvertently
  ;; passing the wrong sort of argument.
  
  (define dbproc-key (string->uninterned-symbol "dbproc"))
  (define loginrec-key (string->uninterned-symbol "dblogin"))
  (define error-handler-key (string->uninterned-symbol "error-handler"))
  (define message-handler-key (string->uninterned-symbol "message-handler"))
  (define computeid-key (string->uninterned-symbol "computeid"))
  (define vms-string-key (string->uninterned-symbol "vms-string"))
  (define closed-object-key (string->uninterned-symbol "closed-object"))
  
  (define tag-object cons) ; (tag-object type-key object) ==> tagged-object
  (define untag-object cdr); (untag-object tagged-object) ==> object
  (define make-type-pred  ; (make-type-pred type-key) ==> type checker predicate
    (lambda (key)
      (lambda (obj)
        (and (pair? obj) (eq? (car obj) key)))))
  
  
  ;; Sybase procedures.
  ;;
  ;; Sybase procedures which return SUCCEED and FAIL return #t and #f.
  ;; Sybase procedures which return null pointers, return #f instead
  ;; of a null pointer.  Likewise, #f is accecpted by the routines in place
  ;; of a null pointer.  Sybase NULL data is returned as '(), the empty
  ;; list.  Other numbers returned by sybase procedures
  ;; which are #defined to C symbols return a scheme symbol comprised
  ;; of the same character sequence.  For example NO_MORE_ROWS (the number -2)
  ;; comes back as 'NO_MORE_ROWS.
  ;;
  ;;
  ;; The dbcoltype and dbalttype procedures return the datatype symbols:
  ;;
  ;;   'int
  ;;   'smallint
  ;;   'tinyint
  ;;   'float
  ;;   'char
  ;;   'binary
  ;;   'text
  ;;   'image
  ;;   'bit
  ;;   'money
  ;;   'datetime
  ;;
  ;; There is no way to distinguish between the variable
  ;; length columns and the fixed length columns.  These procedures always return the
  ;; datatype for the fixed length columns.  These symbols are the same as
  ;; those accecpted by the high level data getter procedures.
  ;;
  ;; There is no use for dbptrtype and it is not implimented.
  ;;
  ;; The dbmsghandle and dberrorhandle procedures are used to install
  ;; C error and message handlers, currently there is only 1 of each
  ;; so these are not very useful.  The C handlers are available
  ;; through get-message-handler and get-error-handler procedures (no
  ;; arguments.)  It is probably better to use the scheme globals
  ;; sybase-error-handler and sybase-message-handler.
  ;; The scheme error handler is bound to the global sybase-error-handler.
  ;; The scheme message handler is bound to the global
  ;; sybase-message-handler.  These functions take arguments which
  ;; corrsepond to the arguments which a C function would take.
  ;; The results of both these functions are ignored.  
  ;;
  ;; Fluid-let is reccommended as the way to rebind sybase-error-handler and
  ;; sybase-message-handler.
  ;;
  ;; The C error handler always returns INT_CANCEL to sybase so that
  ;; whenever a db-library error occurs the call which caused the error
  ;; will always return #f.
  ;;
  ;; The default scheme error handler has the same effect as the
  ;; default sybase error handler.  The behavior of the default Sybase
  ;; message handler is not documented.  The default scheme message handler
  ;; behaves similar to the default scheme error handler.  If the dbprocess
  ;; is dead or the program is otherwise unable to proceed, the connection
  ;; with sybase is terminated and procedure "error" is called.  Otherwise, the error is
  ;; displayed on the default output port and the program is allowed to proceed.
  ;;
  ;; Note that the standard Chez Scheme procedure "error" will bypass the current
  ;; continuation and return to the current cafe.  This will leave sybase initialized,
  ;; and discard all pending errors and messages from sybase.
  ;; While executing call-with-sybase-ready, you may want to use fluid-let to
  ;; redefine the "error" procedure to evaluate dbexit so that sybase is reset
  ;; along with Scheme.
  ;;
  ;; As an alternative to using sybase-message-handler and sybase-error-handler,
  ;; there is sybase-message/error-handler.  This procedure is called with two
  ;; arguments, a list of messages and a list of errors, whenever there are messages
  ;; or errors.  The order of the list elements correspond to the order in which the
  ;; messages or errors were receieved.  The structure of the list elements
  ;; correspond to the parameters
  ;; passed to sybase-message-handler and sybase-error-handler.  The default
  ;; sybase-message/error-handler applys the handlers to the elements of the lists.
  ;; sybase-message/error-handler can be rebound to a procedure of your choosing.
  ;; The advantage of using this procedure in place of the 'standard' sybase
  ;; handlers is that it allows action to be taken after all the messages and/or
  ;; errors are handled; using the standard mechanisim, there is no way to tell
  ;; which message/error is the last.
  ;;
  ;; Bugs:
  ;;
  ;; The error
  ;; handler could be modified to respond to the return of symbols
  ;; 'INT_EXIT and 'INT_CANCEL (but not 'INT_CONTINUE), but I didn't
  ;; bother.  (Perhaps, to better conform to standard sybase, I should
  ;; change dbmsghandle & dberrorhandle to install scheme error
  ;; handlers and do something else for C error handlers.)
  
  
  ; Helper functions to convert sybase results into more scheme-like values.
  ; Make a function which calls Sybase to return a pointer,
  ; and returns #f if the sybase call returns a null pointer, otherwise
  ; returns a typed object of type 'type'.
  (define sybase-ptr
    (lambda (type p)
      (lambda args
        (let ((result (apply p args)))
          (if (zero? result)
              #f
              (tag-object type result))))))
	
  ;
  ; Error traps, these could be implemented as macros for speed but ...    
  ;
  
  ; Make a function which makes a sybase call and traps any errors generated.
  (define trap-errors
    (lambda (p)
      (lambda args
        (let ((result (apply p args)))
          (save-messages/errors)
          (handle-messages/errors)
          result))))
  
  ; Make a function which makes a sybase call and requires the
  ; first argument to be a dbproc.
  (define require-dbproc
    (lambda (p)
      (lambda args
        (if (dbproc? (car args))
            (apply p (cons (untag-object (car args))
                           (cdr args)))
            (error 'require-dbproc
                   "the first argument of ~a was ~a but should be a dbproc"
                   p
                   (car args))))))
  
  ; Make a function which makes a sybase call and requires the first
  ; argument to be a loginrec.
  (define require-loginrec
    (lambda (p)
      (lambda args
        (if (loginrec? (car args))
            (apply p (cons (untag-object (car args))
                           (cdr args)))
            (error 'require-loginrec
                   "the first argument of ~a was ~a but should be a loginrec"
                   p
                   (car args))))))
  
  ; Make a function which makes a sybase call and requires the second
  ; argument to be a computeid.
  (define require-computeid-2
    (lambda (p)
      (lambda args
        (if (computeid? (cadr args))
            (apply p (cons (car args)
                           (cons (untag-object (cadr args))
                                 (cddr args))))
            (error 'require-computeid
                   "The second argument of ~a was ~a but should be a computeid:"
                   p
                   (cadr args))))))
	
  ;
  ; end of error trap helper functions.
  ;
  
  ; A function which translates Sybases integer token values for datatypes into
  ; Scheme symbols.
  (define token->datatype
    (lambda (token)
      (case token
        (#x2f 'char) ;SYBCHAR
        (#x23 'text) ;SYBTEXT
        (#x2d 'binary) ;SYBBINARY
        (#x22 'image) ;SYBIMAGE
        (#x30 'tinyint) ;SYBINT1
        (#x34 'smallint) ;SYBINT2
        (#x38 'int) ;SYBINT4
        (#x3e 'float) ;SYBFLT8
        (#x32 'bit) ;SYBBIT
        (#x3c 'money) ;SYBMONEY
        (#x3d 'datetime) ;SYBDATETIME
        (else
         (error 'token->datatype 
                "Unknown token ~a will not convert to a sybase datatype symbol"
                token)))))
  
  ;; Functions to get errors and messages from C.
  
  ;; The variables sybase-message and sybase-error are called
  ;; (in that order) after every sybase call to pick up any errors which may
  ;; have occurred.
  
  (define save-messages/errors '())
  (define handle-messages/errors '())
  (let ((saved-messages '())
        (saved-errors '()))
    ;; Helper function to make handlers which iterate through a C linked list.
    ;; This is designed to free all the memory allocated in the vax before calling
    ;; any of the handlers.  This is so that the handlers can be exited via
    ;; continuations and we don't have to worry about leaving vax memory
    ;; littered about.  (The only alternative to this requires that
    ;; procedures be associated with scheme
    ;; memory and that these procedures be called when the associated memory is
    ;; garbagecollected.)
    (define make-saver
      (lambda (getter ; get the head of the C list.
               freer) ; free the element, save the contents.
        (letrec ((free-elements
                  (lambda (head) ; The address of the head of the list.
                    (if (not (= 0 head))
                        (let ((next-element (peek.unsigned-32 head))
                              (contents (peek.unsigned-32 (+ head 4))))
                          (free head)
                          (freer contents)
                          (free-elements next-element))))))
          (lambda ()
            (let* ((&anchor.head (getter))
                   (anchor.head (peek.unsigned-32 &anchor.head)))
              (poke.unsigned-32 &anchor.head 0)
              (free-elements anchor.head))))))
    
    ;; Helper functions which call scheme-error-handler and
    ;; scheme-message-handler.  Watch the order in which things are cons-ed here.
    (define save-errors
          (make-saver (foreign-procedure "get_saved_errors" () unsigned-32)
                      (lambda (error)
                        (let ((dbproc (tag-object dbproc-key
                                                  (peek.unsigned-32 error)))
                              (severity (peek.integer-32 (+ error 4)))
                              (dberr (peek.integer-32 (+ error 8)))
                              (oserr (peek.integer-32 (+ error 12)))
                              (dberrstr (peek.string (peek.integer-32 (+ error 16))))
                              (oserrstr (peek.string (peek.integer-32 (+ error 20)))))
                          (free (peek.integer-32 (+ error 16)))
                          (free (peek.integer-32 (+ error 20)))
                          (free error)
                          (set! saved-errors
                                (cons (list dbproc
                                            severity
                                            dberr
                                            oserr
                                            dberrstr
                                            oserrstr)
                                      saved-errors))))))
    
    (define save-messages
          (make-saver (foreign-procedure "get_saved_messages" () unsigned-32)
                      (lambda (message)
                        (let ((dbproc (tag-object dbproc-key
                                                  (peek.unsigned-32 message)))
                              (msgno (peek.integer-32 (+ message 4)))
                              (msgstate (peek.integer-32 (+ message 8)))
                              (severity (peek.integer-32 (+ message 12)))
                              (msgtext (peek.string (peek.integer-32 (+ message 16))))
                              (srvname (peek.string (peek.integer-32 (+ message 20))))
                              (procname (peek.string (peek.integer-32 (+ message 24))))
                              (line (peek.unsigned-16 (+ message 28))))
                          (free (peek.integer-32 (+ message 16)))
                          (free (peek.integer-32 (+ message 20)))
                          (free (peek.integer-32 (+ message 24)))
                          (free message)
                          (set! saved-messages
                                (cons (list dbproc
                                            msgno
                                            msgstate
                                            severity
                                            msgtext
                                            srvname
                                            procname
                                            line)
                                      saved-messages))))))
    
    (set! save-messages/errors
          (lambda ()
            (save-messages) ; order of these calls is important! (to get messages first).
            (save-errors)))
    
    (set! handle-messages/errors
          (lambda ()
            (if (or (not (null? saved-errors)) 
                    ; I believe every error results in a message but just in case...
                    (not (null? saved-messages)))
                (let ((saved-error-stuff (reverse saved-errors))
                      (saved-message-stuff (reverse saved-messages)))
                  (set! saved-errors '())
                  (set! saved-messages '())
                  (sybase-message/error-handler saved-message-stuff saved-error-stuff))))))
  
  (set! dbproc? (make-type-pred dbproc-key))
  (set! loginrec? (make-type-pred loginrec-key))
  (set! sybase-error-handler? (make-type-pred error-handler-key))
  (set! sybase-message-handler? (make-type-pred message-handler-key))
  (set! computeid? (make-type-pred computeid-key))
  (set! vms-string? (make-type-pred vms-string-key))
  (set! closed-object? (make-type-pred closed-object-key))
  (set! integer->computeid
        (lambda (integer)
          (if (and (integer? integer)
                   (positive? integer))
              (tag-object computeid-key integer)
              (error 'integer->computeid
                     "Not a positive integer: ~a"
                     integer))))
  (set! computeid->integer
        (lambda (computeid)
          (if (computeid? computeid)
              (untag-object computeid)
              (error 'computeid->integer
                     "Not a computeid: ~a"
                     computeid))))
  (set! mark-closed!
        (lambda (obj)
          (if (and (pair? obj)
                   (symbol? (car obj)))
              (set-car! obj closed-object-key)
              (error 'mark-closed! "Not a typed object: ~a" obj))))
  
  (set! dbinit
        (trap-errors
         (foreign-procedure "dbinit" () boolean)))
  
  (set! dberrhandle 
        (sybase-ptr
         error-handler-key
         (trap-errors
          (let ((dberrhandle
                 (foreign-procedure "dberrhandle" (unsigned-32) unsigned-32)))
            (lambda (handler)
              (cond ((sybase-error-handler? handler)
                     (dberrhandle (untag-object handler)))
                    ((eq? handle #f)
                     (dberrhandle 0))
                    (else
                     (error 'dberrhandle 
                            "not a sybase error handler ~a"
                            handler))))))))
  
  (set! dbmsghandle 
        (sybase-ptr
         message-handler-key
         (trap-errors
          (let ((dbmsghandle
                 (foreign-procedure "dbmsghandle" (unsigned-32) unsigned-32)))
            (lambda (handler)
              (cond ((sybase-message-handler? handler)
                     (dbmsghandle (untag-object handler)))
                    ((eq? handle #f)
                     (dbmsghandle 0))
                    (else
                     (error 'dbmsghandle
                            "not a sybase message handler ~a"
                            handler))))))))
  
  
  (set! dblogin 
        (sybase-ptr loginrec-key
                    (trap-errors
                     (foreign-procedure "dblogin" () unsigned-32))))
  
  (set! dbopen 
        (sybase-ptr dbproc-key
                    (trap-errors
                     (require-loginrec
                       (foreign-procedure "dbopen" 
                                          (unsigned-32 string) 
                                          unsigned-32)))))
  
  (set! dbcmd
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbcmd" (unsigned-32 string) boolean))))
  
  (set! dbsqlexec 
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbsqlexec" (unsigned-32) boolean))))
  
  (set! dbresults 
        (trap-errors
         (require-dbproc
          (let ((dbresults
                 (foreign-procedure "dbresults" (unsigned-32) integer-32)))
            (lambda (dbproc)
              (let ((result (dbresults dbproc)))
                (case result
                  (0 #f) ; FAIL
                  (1 #t) ; SUCCEED
                  (2 'NO_MORE_RESULTS) ; NO_MORE_RESULTS
                  (else
                   (error 'dbresults
                          "The C call to dbresults returned an unexpected result: ~a"
                          result)))))))))
  
  (set! dbdata 
        (trap-errors
         (require-dbproc
                (foreign-procedure "dbdata"
                                   (unsigned-32 unsigned-32)
                                   unsigned-32))))
  
  (set! dbnextrow 
        (trap-errors
         (require-dbproc
          (let ((dbnextrow
                 (foreign-procedure "dbnextrow" (unsigned-32) integer-32)))
            (lambda (dbproc)
              (let ((result (dbnextrow dbproc)))
                (case result
                  (-1 'REG_ROW)
                  (-2 'NO_MORE_ROWS)
                  (0 #f) ; FAIL
                  (-3 'BUF_FULL)
                  (else (tag-object computeid-key result)))))))))
  
  (set! dbexit 
        (trap-errors
         (foreign-procedure "dbexit" () void)))
  
  (set! dbadlen
        (trap-errors
         (require-dbproc
          (require-computeid-2
           (foreign-procedure "dbadlen" 
                              (unsigned-32 unsigned-32 unsigned-32)
                              integer-32)))))
  
  (set! dbalttype
        (trap-errors
         (require-dbproc
          (require-computeid-2
           (let ((dbalttype
                  (foreign-procedure "dbalttype" 
                                     (unsigned-32 unsigned-32 unsigned-32) 
                                     unsigned-32)))
             (lambda (dbproc computeid column)
               (token->datatype (dbalttype dbproc computeid column))))))))
  
  (set! dbclose 
        (trap-errors
         (let ((dbclose
                (require-dbproc
                 (let ((dbclose
                        (foreign-procedure "dbclose" (unsigned-32) void)))
                   (lambda (dbproc)
                     (dbclose dbproc))))))
           (lambda (dbproc)
             (dbclose dbproc)
             (mark-closed! dbproc)))))
  
  (set! dbcoltype
        (trap-errors
         (require-dbproc
          (let ((dbcoltype
                 (foreign-procedure "dbcoltype" (unsigned-32 unsigned-32) unsigned-32)))
            (lambda (dbproc column)
              (token->datatype (dbcoltype dbproc column)))))))
  
  (set! dbdatlen
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbdatlen" (unsigned-32 unsigned-32) integer-32))))
  
  (set! dbnumalts
        (trap-errors
         (require-dbproc
          (require-computeid-2
           (foreign-procedure "dbnumalts" (unsigned-32 unsigned-32) unsigned-32)))))
  
  (set! dbsetluser 
        (trap-errors
         (require-loginrec
           (foreign-procedure "call_dbsetluser" 
                              (unsigned-32 string) 
                              boolean))))
  
  (set! dbsetlpwd 
        (trap-errors
         (require-loginrec
           (foreign-procedure "call_dbsetlpwd" 
                              (unsigned-32 string) 
                              boolean))))
  
  (set! dbsetlapp 
        (trap-errors
         (require-loginrec
           (foreign-procedure "call_dbsetlapp" 
                              (unsigned-32 string) 
                              boolean))))
  
  (set! dbsetlhost 
        (trap-errors
         (require-loginrec
           (foreign-procedure "call_dbsetlhost" 
                              (unsigned-32 string) 
                              boolean))))
  
  (set! dbadata 
        (trap-errors
         (require-dbproc
          (require-computeid-2
                (foreign-procedure "dbadata"
                                   (unsigned-32 unsigned-32 unsigned-32)
                                   unsigned-32)))))
  
  (set! dbcancel
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbcancel" (unsigned-32) boolean))))
  
  (set! dbcanquery
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbcanquery" (unsigned-32) boolean))))
  
  (set! dbdead
        (trap-errors
         (require-dbproc
          (foreign-procedure "call_dbdead" (unsigned-32) boolean))))
  
  (set! dbfreebuf
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbfreebuf" (unsigned-32) void))))
  
  (set! dbrows
        (trap-errors
         (require-dbproc
          (foreign-procedure "call_dbrows" (unsigned-32) boolean))))
  
  (set! dbnumcols
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbnumcols" (unsigned-32) unsigned-32))))
  
  (set! dbsqlok
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbsqlok" (unsigned-32) boolean))))
  
  (set! dbsqlsend
        (trap-errors
         (require-dbproc
          (foreign-procedure "dbsqlsend" (unsigned-32) boolean))))
  
  
  ;; Functions to provide access to the error and message handlers
  ;; written in C.
  
  (set! get-error-handler
        (sybase-ptr
         error-handler-key
         (foreign-procedure "get_error_handler" () unsigned-32)))
  
  (set! get-message-handler
        (sybase-ptr
         message-handler-key
         (foreign-procedure "get_message_handler" () unsigned-32)))
  
  ; Copy a Scheme string into VMS.
  (set! allocate-vms-string
        (let ((copy (foreign-procedure "strcpy" 
                                       (unsigned-32 string) 
                                       unsigned-32)))
          (lambda (string)
            (tag-object vms-string-key
                        (copy (malloc (+ 1 (string-length string)))
                              string)))))
  
  ; Dalocate the storate allocated by allocate-vms-string
  (set! free-vms-string
        (lambda (vms-string)
          (if (vms-string? vms-string)
              (begin
               (free (untag-object vms-string))
               (mark-closed! vms-string))
              (error 'free-vms-string
                     "Not a vms-string ~a"
                     vms-string))))
  )



;; Install the default Scheme error handlers.

(define sybase-message/error-handler 
  (lambda (messages errors)
    (for-each (lambda (message)
                (apply sybase-message-handler message))
              messages)
    (for-each (lambda (error)
                (apply sybase-error-handler error))
              errors)))

(define sybase-error-handler
  (lambda (dbproc severity dberr oserr dberrstr oserrstr)
    (let* ((newline (string #\newline))
           (message (format (string-append
                             "Error from DB-Lib:" 
                             newline
                             "dbproc: ~a"
                             newline
                             "severity: ~a"
                             newline
                             "dberr: ~a"
                             newline
                             "oserr: ~a"
                             newline
                             "dberrstr: ~a"
                             newline
                             "oserrstr: ~a")
                            dbproc severity dberr oserr dberrstr oserrstr)))
      (fluid-let ((sybase-error-handler ; Prevent re-invocation of this handler
                   (lambda args #t))    ; while we attempt to clean up.
                  (sybase-message-handler
                   (lambda args #t)))
                 
                 (if (or (<= severity 7)
                         (and (= severity 8)
                              (not (dbdead dbproc))))
                     (begin
                      (display message)
                      (display newline))
                     (begin
                      (dbclose dbproc)
                      (error 'sybase-error-handler
                             msg)))))))

(define sybase-message-handler
  (lambda (dbproc msgno msgname severity msgtext srvname procname line)
    (let* ((newline (string #\newline))
           (message (format (string-append
                             "Error from Sybase server:"
                             newline
                             "dbproc: ~a"
                             newline
                             "msgno: ~a"
                             newline
                             "msgname: ~a"
                             newline
                             "severity: ~a"
                             newline
                             "msgtext: ~a"
                             newline
                             "srvname: ~a"
                             newline
                             "procname: ~a"
                             newline
                             "line: ~a")
                            dbproc msgno msgname severity msgtext srvname procname line)))
      (if (<= severity 19)
          (begin
           (display message)
           (display newline))
          (fluid-let ((sybase-error-handler ; Prevent re-invocation of this handler
                       (lambda args #t))    ; while we attempt to clean up.
                      (sybase-message-handler
                       (lambda args #t)))
                     
                     (dbclose dbproc)
                     (error 'sybase-message-handler
                            message))))))