/* custom.c */

/* This is the customization file for the citi conversion program.*/

/* March 15, 1991
 *
 * Karl O. Pinc
*/

/* Elements of structures which are referenced by Scheme are assumed
 * to be aligned on byte boundries.  See the VAX C Guide sections
 * on Structures and Unions and the member_alignment pragma. */

#include <stdlib.h>
#include <string>
#include <sybfront.h>
#include <sybdb.h>

/* Functions to call standard C procedures.
 * For some reason we can't call them directly (something to do
 * with being in a shared library) and this works. */
static void *my_malloc(size_t size) {
   return malloc(size);}

static void my_free(void *ptr) {
   free(ptr);}

static char * my_strcpy(char * c1, char * c2) {
   return strcpy(c1, c2);}

static void * my_memcpy(void *s1, void *s2, size_t size) {
   return memcpy(s1, s2, size);}

/* Functions for Scheme to access C data. */
static int dref_ptr_int_32(int *ptr) { /* good for signed and unsigned ints */
   return *ptr;}

static int dref_ptr_int_16(short *ptr) { /* only for signed shorts */
   return *ptr;}

static unsigned int dref_ptr_unsigned_16(unsigned short *ptr) { /* for unsigned shorts */
   return *ptr;}

static unsigned int dref_ptr_unsigned_8(unsigned char *ptr) { /* only for unsigned chars */
   return *ptr;}

static double float dref_ptr_double_float(double float *ptr) {
   return *ptr;}

static int id(int x) {
   return x;}

static void poke_int_32 (int *ptr, int data) {
   *ptr = data;}

static void poke_int_8(char *ptr, int data) {
   *ptr = data;}


/* Functions used in this code to store messages which sybase generates. */
struct cell /* A linked list element. */
                             /*This structure is referenced by Scheme. */
   {  struct cell *link;   /* Link to next cell */
      void        *data;   /* The contents of the element */
   };
   
struct anchor /* What we need to keep track of a linked list. */
   {   struct cell *head;  /* pointer to the first list element */
       struct cell *tail;  /* pointer to the last list element */
   };

/* Save a pointer in a linked list */
static void save(struct anchor *anchor, void *data) {
   /* As this code is called asyncrounously with respect to the code
    * which removes elements from the list, we need to make sure that
    * we won't add elements to the list after the removal code has
    * already examined it.  We do this by having the removal code
    * 'take' the list, whereupon this code starts a new list.  The
    * removal code 'takes' the list by setting the head of the list
    * to the null pointer.  It must be careful to do this before it
    * examines the elements of the list. */
   struct cell *this_cell;

   /* Create new cell. */
   this_cell = malloc(sizeof(struct cell));
   this_cell->data = data;
   this_cell->link = 0;
   /* Link into list. */
   if (anchor->head == 0)
      anchor->head = this_cell; /* Start a new list. */
   else
      anchor->tail->link = this_cell;  /* Append to the old list. */
   /* Keep track of the end of the list. */
   anchor->tail = this_cell;
   };

/* Allocate more storage, save a pointer to it in *to,
 * and copy the from string into it. */
static void dup(char **to, char *from) {
   *to = malloc(strlen(from) + 1);
   strcpy(*to, from); }


/* Functions called by sybase. */

/* The saved_error and saved_msg structures are accessed by Scheme code. */
struct saved_error /* The stuff that comes with an error */
   {  DBPROCESS *dbproc;
      int        severity;
      int        dberr;
      int        oserr;
      char      *dberrstr;
      char      *oserrstr;
   };

struct saved_msg /* the stuff that comes with a messsage */
   {  DBPROCESS   *dbproc;
      DBINT        msgno;
      int          msgstate;
      int          severity;
      char        *msgtext;
      char        *srvname;
      char        *procname;
      DBUSMALLINT  line;
   };

/* The beginning of our linked list of errors and messages. */
static struct anchor saved_errors,
                     saved_messages;

/* The error handler.  Saves the error to be picked up later. */
static int error_handler(DBPROCESS *dbproc,
                         int        severity,
                         int        dberr,
                         int        oserr,
                         char      *dberrstr,
                         char      *oserrstr) {
   struct saved_error *this_error;

   this_error = malloc(sizeof(struct saved_error));
   this_error->dbproc = dbproc;
   this_error->severity = severity;
   this_error->dberr = dberr;
   this_error->oserr = oserr;
   dup(&this_error->dberrstr, dberrstr);
   if (oserr == DBNOERR)
     {
      this_error->oserrstr = malloc(1);
      *this_error->oserrstr = 0;
     }
   else
      dup(&this_error->oserrstr, oserrstr);

   save(&saved_errors, this_error);

   return INT_CANCEL;}

/* The message handler. Saves the message to be picked up later. */
static int message_handler(DBPROCESS   *dbproc,
                           DBINT        msgno,
                           int          msgstate,
                           int          severity,
                           char        *msgtext,
                           char        *srvname,
                           char        *procname,
                           DBUSMALLINT  line) {
   struct saved_msg *this_msg;

   this_msg = malloc(sizeof(struct saved_msg));
   this_msg->dbproc = dbproc;
   this_msg->msgno = msgno;
   this_msg->msgstate = msgstate;
   this_msg->severity = severity;
   dup(&this_msg->msgtext, msgtext);
   dup(&this_msg->srvname, srvname);
   dup(&this_msg->procname, procname);
   this_msg->line = line;

   save(&saved_messages, this_msg);

   return 0;}

/* Functions to return pointers to C stuff back to scheme. */
static int get_error_handler(void) {
   return error_handler;}

static int get_message_handler(void) {
   return message_handler;}

static struct cell **get_saved_errors(void) {
   return &saved_errors.head;}

static struct cell **get_saved_messages(void) {
   return &saved_messages.head;}


/* Turn Sybase macros into procedures so we have something to link with. */
static RETCODE call_dbsetluser(LOGINREC *logrec, char *username) {
   return DBSETLUSER(logrec, username);}

static RETCODE call_dbsetlpwd(LOGINREC *logrec, char *pwd) {
   return DBSETLPWD(logrec, pwd);}

static RETCODE call_dbsetlapp(LOGINREC *logrec, char *app) {
   return DBSETLAPP(logrec, app);}

static RETCODE call_dbsetlhost(LOGINREC *logrec, char *host) {
   return DBSETLHOST(logrec, host);}

static RETCODE call_dbdead(DBPROCESS *dbproc) {
   return DBDEAD(dbproc);}

static RETCODE call_dbrows(DBPROCESS *dbproc) {
   return DBROWS(dbproc);}


/* Supply link information to Scheme. */
custom_init() {
   foreign_symbol("free", my_free);
   foreign_symbol("malloc", my_malloc);
   foreign_symbol("strcpy", my_strcpy);
   foreign_symbol("memcpy", my_memcpy);
                 
   foreign_symbol("dref_ptr_int_32", dref_ptr_int_32);
   foreign_symbol("dref_ptr_int_16", dref_ptr_int_16);
   foreign_symbol("dref_ptr_unsigned_16", dref_ptr_unsigned_16);
   foreign_symbol("dref_ptr_unsigned_8", dref_ptr_unsigned_8);
   foreign_symbol("dref_ptr_double_float", dref_ptr_double_float);
   foreign_symbol("id", id);
   foreign_symbol("poke_int_32", poke_int_32);
   foreign_symbol("poke_int_8", poke_int_8);
                 
   foreign_symbol("get_error_handler", get_error_handler);
   foreign_symbol("get_message_handler", get_message_handler);
   foreign_symbol("get_saved_errors", get_saved_errors);
   foreign_symbol("get_saved_messages", get_saved_messages);

   foreign_symbol("dbadata", dbadata);
   foreign_symbol("dbadlen", dbadlen);
   foreign_symbol("dbalttype", dbalttype);
   foreign_symbol("dbcancel", dbcancel);
   foreign_symbol("dbcanquery", dbcanquery);
   foreign_symbol("dbclose", dbclose);
   foreign_symbol("dbcmd", dbcmd);
   foreign_symbol("dbcoltype", dbcoltype);
   foreign_symbol("dbdata", dbdata);
   foreign_symbol("dbdatlen", dbdatlen);
   foreign_symbol("call_dbdead", call_dbdead);
   foreign_symbol("dberrhandle", dberrhandle);
   foreign_symbol("dbexit", dbexit);
   foreign_symbol("dbfreebuf", dbfreebuf);
   foreign_symbol("dbinit", dbinit);
   foreign_symbol("dblogin", dblogin);
   foreign_symbol("dbmsghandle", dbmsghandle);
   foreign_symbol("dbnextrow", dbnextrow);
   foreign_symbol("dbnumalts", dbnumalts);
   foreign_symbol("dbnumcols", dbnumcols);
   foreign_symbol("dbopen", dbopen);
   foreign_symbol("dbresults", dbresults);
   foreign_symbol("call_dbrows", call_dbrows);
   foreign_symbol("call_dbsetluser", call_dbsetluser);
   foreign_symbol("call_dbsetlpwd", call_dbsetlpwd);
   foreign_symbol("call_dbsetlapp", call_dbsetlapp);
   foreign_symbol("call_dbsetlhost", call_dbsetlhost);
   foreign_symbol("dbsqlexec", dbsqlexec);
   foreign_symbol("dbsqlok", dbsqlok);
   foreign_symbol("dbsqlsend", dbsqlsend);
}
