;;; ==================================================================
;;;
;;; MODULE:	Adobe.scm -- Portable 2-D gray-scale graphics for Scheme.
;;;
;;; AUTHOR:	J. Bartholdi
;;;		ISyE, Georgia Tech
;;;		Atlanta, GA 30332-0205 USA
;;;		(john.bartholdi@isye.gatech.edu)
;;;
;;;		Based on a similar, earlier module by O. Schlageter.
;;;
;;;		Anyone can use this for anything; just do not hold me  
;;;             responsible!
;;;
;;;
;;; VERSION:	1.0	12 March 93
;;;		1.0.1	22 March 93
;;;
;;;
;;; DESCRIPTION:	See companion file Adobe.doc.
;;;
;;; ==================================================================

(define Make-Adobe-Illustrator-Device
  ;
  (lambda (file-name llx lly urx . optional-ury)
    ;
    (let* ((open? #T)		; State of device
           (output-port	'())	; Where graphics will be written
           (clip-region	'())	; default: AI clips at art board
           ;
           ; The following assume that AI is in Single Full Page mode 

           ; (its default) and that you are using 8.5" x 11.0" paper 

           ;in portrait orientation.
           ;
           (paper-width	612)	; PaperRect: change as necessary
           (paper-height792)	; PaperRect: change as necessary
           ;
           ; The following values depend on what printer AI assumes. 

           ; These values are for Apple LaserWriter II and the NeXT 

           ; printer.  They are probably close enough for others.
           ;
           (margin-left		30)	; Margin: change as necessary
           (margin-right	30)	; Margin: change as necessary
           (margin-top		31)	; Margin: change as necessary
           (margin-bottom	31)	; Margin: change as necessary
           ;
           ; Must be exact integers:
           ;
           (page-width	(- paper-width margin-left margin-right))
           (page-height	(- paper-height margin-top margin-bottom))
           ;
           (points/unit-x (/ page-width	; scale in x-direction
                             (- urx llx)))
           (ury (if (null? optional-ury)
                    (+ lly (/ page-height points/unit-x))
                    (car optional-ury)))
           (points/unit-y (/ page-height	; scale in y-direction
                             (- ury lly)))
           ;
           ; Graphics defaults (chosen to be identical to the 

           ; defaults of AI):
           ;
           (line-width		1)		; default = 1 point
           (dash-pattern	(list 1 0))	; default = solid
           (stroke-color	0)		; default = black
           (fill-color		1)		; default = white
           ;
           ; Determines the size at which "points" will be drawn:
           ;
           (point-size	0.0025)	; default radius = 0.0025 x page-width
           ;
           ; Adobe Illustrator will not accept numbers in scientific 

           ; format, therefore we replace any number less than 

           ; <tolerance> with 0.
           ;
           (tolerance 0.00001))
      ;
      (letrec (
               ;
               ; A utility to make output slightly less tedious.
	       ;
               (wrtln
                (lambda args
                  (for-each (lambda (x) (display x output-port)) args)
                  (newline output-port)))
               ;
               ;-- Functions to transform the user's coordinates
               ;-- to coordinates describing the page on which
               ;-- Adobe Illustrator places the drawing.
               ;
               ; Replaces numbers less than <tolerance> with 0.
               ; This is to avoid writing numbers in scientific 

               ; notation because AI cannot read them.
               ;
               (chop
                (lambda (n tolerance)
                  (if (< (abs n) tolerance)
                      0
                      n)))
               ;
               ; Coordinates are transformed to inexact numbers so 

               ; they can be read by AI. (AI cannot read rational 

               ; numbers produced by some Scheme systems.)
               ;
               ; There is no need to translate the coordinates because
               ; one corner of the "bounding box" is (0,0) so AI will
               ; position the drawing appropriately.
               ;
               (virtual->page-coordinates-x
                (lambda (x)
                  (chop (exact->inexact 

                          (* (- x llx) points/unit-x)) tolerance)))

               (virtual->page-coordinates-y
                (lambda (y)
                  (chop (exact->inexact 

                          (* (- y lly) points/unit-y)) tolerance)))

               ;--- Functions to set the graphics state
               ;
               (set-line-width!
                (lambda (w)
                  (cond ((negative? w)
                         (error "Line width out of range: " w))
                        (else
                         (set! line-width w)
                         (wrtln (number->string w) #\space #\w)
                         w))))
               ;
               (set-dash-pattern!
                (lambda (pattern)
                  (set! dash-pattern pattern)
                  (wrtln (string-append
                          "["
                          (apply string-append
                                 (map (lambda (x)
                                        (string-append 

                                          (number->string x) " "))
                                      pattern))
                          "] 0 d"))))
               ;
               (set-stroke-color!
                (lambda (gray-scale)
                  (cond ((or (negative? gray-scale)
                             (> gray-scale 1))
                         (error "Stroke color out of range: " 

                                gray-scale))
                        (else
                         (wrtln (number->string gray-scale) 

                                #\space #\G)
                         (set! stroke-color gray-scale)
                         gray-scale))))
               ;
               (set-fill-color!
                (lambda (gray-scale)
                  (cond ((or (negative? gray-scale)
                             (> gray-scale 1))
                         (error "Fill color out of range: " 

                                gray-scale))
                        (else
                         (wrtln (number->string gray-scale) #\space 

                                #\g)
                         (set! fill-color gray-scale)
                         gray-scale))))
               ;
               (set-point-size!
                (lambda (size)
                  (cond ((or (negative? size)
                             (> size 0.5))
                         (error "Point size out of range: " size))
                        (else
                         (set! point-size size)
                         size))))
               ;
               (set-clip-region!
                (lambda (coord-list)
                  (set! clip-region (cons coord-list clip-region))
                  (wrtln #\q)
                  (draw-polygon coord-list)
                  (wrtln #\h)
                  (wrtln #\W)
                  (wrtln #\n)
                  #T))
               ;
               (unset-clip-region!
                (lambda ()
                  (if (not (null? clip-region))
                      (begin
                       (set! clip-region (cdr clip-region))
                       (wrtln #\Q)
                       #T))))
               ;
               ;--- Drawing routines
               ;
               ; An internal utility function: returns a list of the
               ; "average" x and the average y of a list of
               ; alternating x,y coordinates.  This is the 

               ; "barycenter" of the polygon whose corner points are 

               ; given by the list.  It is helpful to draw this to aid
               ; in aligning other graphics under AI.
               ;
               (barycenter
                (lambda (coord-list)
                  (letrec ((take-every-other
                            (lambda (coords)
                              (cond ((null? coords) '())
                                    (else
                                     (cons (car coords)
                                           (take-every-other
                                              (cddr coords))))))))
                    (let ((average
                           (lambda (<list>)
                             (/ (apply + <list>)
                                (length <list>)))))
                      (list (average (take-every-other coord-list))
                            (average (take-every-other 

                                        (reverse coord-list))))))))
               ;
               ;
               ; An internal utility function: takes as input a list 
               ; of form (x1 y1 x2 y2 ...xn yn). Moves pen to position 
               ; (x1,y1) and connects to (x2,y2) with a straight line, 
               ; then then same from (x2,y2) to (x3,y3), and so on. 
               ; Neither strokes the path nor fills it so that calling
               ; routines can choose what they do.
               ;
               (connect
                (lambda (coord-list)
                  ; move to first point:
                  (wrtln (virtual->page-coordinates-x 

                           (car coord-list))
                         #\space
                         (virtual->page-coordinates-y
                           (cadr coord-list))
                         #\space
                         #\m)
                  ; Connect successive points with straight lines:
                  (let loop ((pts (cddr coord-list)))
                    (cond ((null? pts) #T)
                          (else
                           (wrtln (virtual->page-coordinates-x 

                                    (car pts))
                                  #\space
                                  (virtual->page-coordinates-y
                                  (cadr pts))
                                  #\space
                                  #\L)
                           (loop (cddr pts)))))))
               ;
               (stroke-path
                (lambda (coord-list)
                  (wrtln stroke-color #\space #\G); set stroke color
                  (connect coord-list)
                  (wrtln #\S)			; stroke the path
                  #T))
               ;
               (fill-path
                (lambda (coord-list)
                  (wrtln fill-color #\space #\g); set fill color
                  (connect coord-list)
                  (wrtln #\F)			; stroke the path
                  #T))
               ;
               (stroke-and-fill-path
                (lambda (coord-list)
                  (wrtln stroke-color #\space #\G); set stroke color
                  (wrtln fill-color #\space #\g); set fill color
                  (connect coord-list)
                  (wrtln #\B)			; stroke, fill path
                  #T))

               ; Just like connect, except closes the path.
               ;
               (draw-polygon
                (lambda (coord-list)
                  (let ((x1 (car coord-list))
                        (y1 (cadr coord-list)))
                    (connect coord-list)
                    (wrtln (virtual->page-coordinates-x x1)
                           #\space
                           (virtual->page-coordinates-y y1)
                           #\space
                           #\L))))
               ;
               (stroke-polygon
                (lambda (coord-list)
                  (wrtln #\u)			; begin group
                  (wrtln stroke-color #\space #\G)
                  (draw-polygon coord-list)
                  (wrtln #\s)			; stroke path
                  (let ((c (barycenter coord-list)))
                    (wrtln (virtual->page-coordinates-x (car c))
                           #\space
                           (virtual->page-coordinates-y (cadr c))
                           #\space
                           #\m))
                  (wrtln #\S)		; alignment point at center
                  (wrtln #\U)		; end group
                  #T))
               ;
               (fill-polygon
                (lambda (coord-list)
                  (wrtln #\u)			; begin group
                  (wrtln fill-color #\space #\g)
                  (draw-polygon coord-list)
                  (wrtln #\f)			; fill path
                  (let ((c (barycenter coord-list)))
                    (wrtln (virtual->page-coordinates-x (car c))
                           #\space
                           (virtual->page-coordinates-y (cadr c))
                           #\space
                           #\m))
                  (wrtln #\F)		; alignment point at center
                  (wrtln #\U)		; end group
                  #T))
               ;
               (stroke-and-fill-polygon
                (lambda (coord-list)
                  (wrtln #\u)		; begin group
                  (wrtln stroke-color #\space #\G)
                  (wrtln fill-color #\space #\g)
                  (draw-polygon coord-list)
                  (wrtln #\b)			; stroke-and-fill path
                  (let ((c (barycenter coord-list)))
                    (wrtln (virtual->page-coordinates-x (car c))
                           #\space
                           (virtual->page-coordinates-y (cadr c))
                           #\space
                           #\m))
                  (wrtln #\B)		; alignment point at center
                  (wrtln #\U)		; end group
                  #T))
               ;
               (stroke-rectangle
                (lambda (x1 y1 x2 y2)
                  (stroke-polygon (list x1 y1 x1 y2 x2 y2 x2 y1))))
               ;
               (fill-rectangle
                (lambda (x1 y1 x2 y2)
                  (fill-polygon (list x1 y1 x1 y2 x2 y2 x2 y1))))
               ;
               (stroke-and-fill-rectangle
                (lambda (x1 y1 x2 y2)
                  (stroke-and-fill-polygon 

                     (list x1 y1 x1 y2 x2 y2 x2 y1))))
               ;
               ; Connect four points at periphery of circle with the
               ; appropriate Bezier curves to form that circle.
               ;
               (draw-ellipse
                (let (
                      ;
                      ; This is determined by the mathematics of 

                      ; Bezier curves and will be used to produce a 

                      ; good approximation to an ellipse:
                      ;
                      (bezier-factor (/ (* 4 (- (sqrt 2) 1))
                                        3)))
                  ;
                  (lambda (raw-x raw-y raw-x-radius raw-y-radius)
                    (let* ((x (virtual->page-coordinates-x raw-x))
                           (y (virtual->page-coordinates-y raw-y))
                           (rx (chop (exact->inexact 

                                       (* raw-x-radius points/unit-x))
                                     tolerance))
                           (ry (chop (exact->inexact
                                       (* raw-y-radius points/unit-y))
                                     tolerance))
                           ;
                           ; Compute distances (dx,dy) of Bezier
                           ; points from (x,y). The Bezier points are 

                           ; determined by requiring the Bezier curve 

                           ; to pass through (rx,0), (0,ry), and 

                           ; (rx/sqrt(2),ry/sqrt(2)).
                           ;
                           (dx (* rx bezier-factor))
                           (dy (* ry bezier-factor)))
                      ;
                      ; Move to initial position:
                      (wrtln x #\space (- y ry) #\space #\m)
                      ;
                      ; Draw SE quarter arc:
                      (wrtln (+ x dx)	#\space
                             (- y ry)	#\space
                             (+ x rx)	#\space
                             (- y dy)	#\space
                             (+ x rx)	#\space
                             y	#\space
                             #\c)
                      ; Draw NE quarter arc:
                      (wrtln (+ x rx)	#\space
                             (+ y dy)	#\space
                             (+ x dx)	#\space
                             (+ y ry)	#\space
                             x	#\space
                             (+ y ry)	#\space
                             #\c)
                      ; Draw NW quarter arc:
                      (wrtln (- x dx) #\space
                             (+ y ry)	#\space
                             (- x rx)	#\space
                             (+ y dy)	#\space
                             (- x rx)	#\space
                             y	#\space
                             #\c)
                      ; Draw SW quarter arc:
                      (wrtln (- x rx)	#\space
                             (- y dy)	#\space
                             (- x dx)	#\space
                             (- y ry)	#\space
                             x	#\space
                             (- y ry)	#\space
                             #\c)))))
               ;
               (stroke-ellipse
                (lambda (x y rx ry)
                  (wrtln #\u)			; begin group
                  (wrtln stroke-color #\space #\G)
                  (draw-ellipse x y rx ry)
                  (wrtln #\s)			; stroke path
                  (wrtln (virtual->page-coordinates-x x)
                         #\space
                         (virtual->page-coordinates-y y)
                         #\space
                         #\m)
                  (wrtln #\S)		; alignment point at center
                  (wrtln #\U)		; end group
                  #T))
               ;
               (fill-ellipse
                (lambda (x y rx ry)
                  (wrtln #\u)		; begin group
                  (wrtln fill-color #\space #\g)
                  (draw-ellipse x y rx ry)
                  (wrtln #\f)		; fill path
                  (wrtln (virtual->page-coordinates-x x)
                         #\space
                         (virtual->page-coordinates-y y)
                         #\space
                         #\m)
                  (wrtln #\F)		; alignment point at center
                  (wrtln #\U)		; end group
                  #T))
               ;
               (stroke-and-fill-ellipse
                (lambda (x y rx ry)
                  (wrtln #\u)			; begin group
                  (wrtln stroke-color #\space #\G)
                  (wrtln fill-color #\space #\g)
                  (draw-ellipse x y rx ry)
                  (wrtln #\b)			; stroke-and-fill path
                  (wrtln (virtual->page-coordinates-x x)
                         #\space
                         (virtual->page-coordinates-y y)
                         #\space
                         #\m)
                  (wrtln #\B)		; alignment point at center
                  (wrtln #\U)		; end group
                  #T))
               ;
               (stroke-circle
                (lambda (x y r)
                  (stroke-ellipse x y r r)))
               ;
               (fill-circle
                (lambda (x y r)
                  (fill-ellipse x y r r)))
               ;
               (stroke-and-fill-circle
                (lambda (x y r)
                  (stroke-and-fill-ellipse x y r r)))
               ;
               ; A point is guaranteed to appear as a circle 

               ; independent of the aspect ratio of the drawing.
               ;
               (stroke-point
                (lambda (x y)
                  (let ((radius-on-page (* point-size page-width)))
                    (stroke-ellipse x 

                                    y 

                                    (+ (/ radius-on-page 

                                          points/unit-x) 

                                       llx)
                                    (+ (/ radius-on-page 

                                          points/unit-y) 

                                    lly))
                    #T)))
               ;
               (fill-point
                (lambda (x y)
                  (let ((radius-on-page (* point-size page-width)))
                    (fill-ellipse x
                                  y 

                                  (+ (/ radius-on-page 

                                        points/unit-x) 

                                     llx)
                                  (+ (/ radius-on-page 

                                        points/unit-y) 

                                     lly))
                    #T)))
               ;
               (stroke-and-fill-point
                (lambda (x y)
                  (let ((radius-on-page (* point-size page-width)))
                    (stroke-and-fill-ellipse 

                      x
                      y 

                      (+ (/ radius-on-page points/unit-x) llx)
                      (+ (/ radius-on-page points/unit-y) lly))
                    #T)))
               ;
               (end-graphics
                (lambda ()
                  ; close clip regions:
                  (let loop ((to-close clip-region))
                    (cond ((null? to-close) #T)
                          (else
                           (wrtln #\Q)
                           (loop (cdr to-close)))))
                  ;--- Begin required Adobe trailer:
                  (wrtln	"%%Trailer")
                  (wrtln	"%%EOF")
                  ;--- End required Adobe trailer:
                  (close-output-port output-port)
                  (set! open? #F) ; to prevent further writing to file
                  #T))
               ;
               ; The message dispatcher:
               ;
               (Adobe-Illustrator-Device
                (lambda (msg . args)
                  (cond ((eq? msg 'open?) open?)
                        ((not open?)
                         (error "Adobe Illustrator device has been closed!"))
                        ;
                        (else			; process message
                         (case msg
                           ;
                           ((stroke-path)
                            (apply stroke-path args))
                           ((fill-path)
                            (apply fill-path args))
                           ((stroke-and-fill-path)
                            (apply stroke-and-fill-path args))
                           ;
                           ((stroke-polygon)
                            (apply stroke-polygon args))
                           ((fill-polygon)
                            (apply fill-polygon args))
                           ((stroke-and-fill-polygon)
                            (apply stroke-and-fill-polygon args))
                           ;
                           ((stroke-rectangle)
                            (apply stroke-rectangle args))
                           ((fill-rectangle)
                            (apply fill-rectangle args))
                           ((stroke-and-fill-rectangle)
                            (apply stroke-and-fill-rectangle args))
                           ;
                           ((stroke-ellipse)
                            (apply stroke-ellipse args))
                           ((fill-ellipse)
                            (apply fill-ellipse args))
                           ((stroke-and-fill-ellipse)
                            (apply stroke-and-fill-ellipse args))
                           ;
                           ((stroke-circle)
                            (apply stroke-circle args))
                           ((fill-circle)
                            (apply fill-circle args))
                           ((stroke-and-fill-circle)
                            (apply stroke-and-fill-circle args))
                           ;
                           ((set-point-size!)
                            (set-point-size! (car args)))
                           ((get-point-size)
                            point-size)
                           ((stroke-point)
                            (apply stroke-point args))
                           ((fill-point)
                            (apply fill-point args))
                           ((stroke-and-fill-point)
                            (apply stroke-and-fill-point args))
                           ;
                           ((get-virtual-coordinates)
                            (list llx lly urx ury))
                           ((get-page-coordinates)
                            (list 0 0 page-width page-height))
                           ;
                           ((set-clip-region!)
                            (if (null? args)
                              (set-clip-region! 

                                (list
                                  llx lly llx ury urx ury urx lly))
                              (apply set-clip-region! args)))
                           ((unset-clip-region!)
                            (unset-clip-region!))
                           ((get-clip-region)
                            clip-region)
                           ((set-line-width!)
                            (set-line-width! (car args)))
                           ((get-line-width)
                            line-width)
                           ((set-dash-pattern!)
                            (set-dash-pattern! (car args)))
                           ((get-dash-pattern)
                            dash-pattern)
                           ((set-stroke-color!)
                            (set-stroke-color! (car args)))
                           ((get-stroke-color)
                            stroke-color)
                           ((set-fill-color!)
                            (set-fill-color! (car args)))
                           ((get-fill-color)
                            fill-color)
                           ;
                           ((end-graphics)
                            (end-graphics))
                           ;
                           (else
                            (error "Unknown message to Adobe Illustrator Device: " msg))))))))
        ;
        ; Open output file and write AI prolog:
        ;
        (set! output-port (open-output-file file-name))
        (if (not (output-port? output-port))
            (error "Unable to open output file!" output-port))
        ;
        ;--- Begin required Adobe prolog:
        (wrtln	"%!PS-Adobe-3.0")
        (wrtln	"%%Creator: Scheme (Adobe.scm 1.0.1)")
        (wrtln	"%%BoundingBox: 0 0"	#\space
		paper-width		#\space
		paper-height)
        (wrtln	"%%EndComments")
        (wrtln	"%%EndProlog")
        ;--- End required Adobe prolog
        ;;
        ;; Return New Object
        ;;
        Adobe-Illustrator-Device))))

