;;
;; rcasm.scm
;;
;; A Redcode assembler.
;;
;; This is a 2 pass assembler.  First, we scan through the text
;; accumulating the labels, then we process the text again substituting
;; label values for the labels.
;;

(module rcasm)
(include "cw.sch")

(define (rca text)
  (define (iter text code symtab pc)
    (cond ((null? text) code)
	  ((or (label? (car text)) (comment? (car text)))
	   (iter (cdr text) code symtab pc))
	  ((instruction? (car text))
	   (iter (cdr text)
		 (cons (assemble-instruction (car text) symtab pc) code)
		 symtab (inc pc)))))
  (let ((symbol-table (collect-symbols text)))
    (reverse (iter text '() symbol-table 0))))

;;
;; Collect all the symbols in text.  Return an association list, key
;; is symbol, value is program-counter.
;;
(define (collect-symbols text)
  (let ((pc 0)
	(tab (list)))
    (foreach
     text
     (lambda (stmt)
       (cond ((label? stmt) (set! tab (cons (cons (cadr stmt) pc) tab)))
	     ((instruction? stmt) (set! pc (inc pc))))))
    tab))

;;
;; Assemble an instruction to a length 5 vector.
;; #(op-code mode-a operand-a mode-b operand-b)
;; 
(define (assemble-instruction i-list symtab pc)
  (let ((i (lookup-instruction-by-name (car i-list)))
	(a (cons 0 0))
	(b (cons 0 0)))
    (cond ((one-op-instruction? i)
	   (if (= (length i-list) 2)
	       (set! a (assemble-mode&operand (cadr i-list) symtab pc))
	       (error
		"Wrong number of arguments to instruction (supposed to be 1):"
		i-list)))
	  ((zero-op-instruction? i)
	   (if (not (= (length i-list) 1))
	       (error
		"Wrong number of arguments to instruction (supposed to be 0):"
		i-list)))
	  (else
	   (if (= (length i-list) 3)
	       (begin
		 (set! a (assemble-mode&operand (cadr i-list) symtab pc))
		 (set! b (assemble-mode&operand (caddr i-list) symtab pc)))
	       (error
		"Wrong number of arguments to instruction (supposed to be 2):"
		i-list))))
    (vector (opcode i) (car a) (cdr a) (car b) (cdr b))))

;;
;; This is required because we don't scale immediate values to the pc.
;;
(define (assemble-mode&operand m&o symtab pc)
  (cons (assemble-mode (car m&o))
	(assemble-operand (cadr m&o) pc symtab)))

(define (assemble-mode m)
  (cond ((relative? m) 0)
	((indirect? m) 1)
	((immediate? m) 2)
	((auto-decrement? m) 3)
	(else
	 (error "Unknown addressing mode:" m))))

(define (assemble-operand o pc symtab)
  (cond ((number? o) (scale-to-core o))
	((symbol? o)
	 (let ((symbol-entry (assq o symtab)))
	   (if symbol-entry
	       (scale-to-core (- (cdr symbol-entry) pc))
	       (error "Uknown symbol:" o))))
	(else
	 (error "Can't figure out what kind of operand this is:" o))))

;;
;; Boring data abstraction stuff.
;;

; comments start with /*
(define (comment? thing) (if (pair? thing) (eq? (car thing) '/*) #f))

(define (label? thing) (if (pair? thing) (eq? (car thing) 'label) #f))

; an instruction is not a comment or a label.  This *should* be better
; but is good enough.
(define (instruction? thing) (and (not (comment? thing))
				  (not (label? thing))))

(define (relative? op) (eq? op '+))
(define (immediate? op) (eq? op '%))
(define (indirect? op) (eq? op '^))
(define (auto-decrement? op) (eq? op '<))
